// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "role_rokuro.h"

#include "bhv_dango.h"
#include "bhv_simple_pass.h"
#include "bhv_offense.h"
#include "strategy.h"
#include "opuci_neck.h"
#include "opuci_search_ball.h"
#include "bhv_defense.h"

#include "bhv_opuci_defensivehalf_move.h"
#include "bhv_opuci_defensive_kick.h"
#include "neck_opuci_turn_to_next_receiver.h"
#include "body_kick_to_corner.h"

#include <rcsc/formation/formation.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>


#include <rcsc/action/body_smart_kick.h>
#include <rcsc/action/body_pass.h>
#include <rcsc/action/body_dribble.h>
#include <rcsc/action/body_hold_ball.h>
#include <rcsc/action/body_advance_ball.h>
#include <rcsc/action/body_kick_one_step.h>
#include <rcsc/action/neck_turn_to_ball.h>
#include <rcsc/action/neck_scan_field.h>
//#include <rcsc/action/neck_turn_to_low_conf_teammate.h>
#include <rcsc/player/say_message_builder.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/player/say_message_builder.h>

const std::string RoleRokuro::NAME( "Rokuro" );

int RoleRokuro::M_passMate = -1;
int RoleRokuro::M_neck_cnt = -1;

/*-------------------------------------------------------------------*/
/*!

 */
namespace {
rcss::RegHolder role = SoccerRole::creators().autoReg( &RoleRokuro::create,
                                                       RoleRokuro::NAME );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
RoleRokuro::execute( rcsc::PlayerAgent * agent )
{
    bool kickable = agent->world().self().isKickable();
    if ( agent->world().existKickableTeammate()
         && agent->world().teammatesFromBall().front()->distFromBall()
         < agent->world().ball().distFromSelf() )
    {
        kickable = false;
    }

    if ( kickable )
    {
        doKick( agent );
    }
    else
    {
        doMove( agent );
        Bhv_defense().defenseMessage( agent );
    }
    Bhv_defense().defenseAttention( agent );

/*
//Bhv_Dango().execute( agent );

if( agent->world().ball().posCount() > 5 )
{
Opuci_SearchBall().execute( agent );
}
//    else if( agent->world().self().pos().dist( agent->world().ball().pos() ) < agent->world().self().kickableArea() )
//    {
//        Bhv_Offense( rcsc::Vector2D( 51.5, 0.0 ) ).execute( agent );
//    }
else 
{
Bhv_SimplePass().execute( agent );
Opuci_Neck().execute( agent );
}
*/
     return;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
RoleRokuro::doKick( rcsc::PlayerAgent * agent )
{
    switch ( Strategy::get_ball_area( agent->world().ball().pos() ) ) {
    case Strategy::BA_CrossBlock:
    case Strategy::BA_Stopper:
    case Strategy::BA_Danger:
    case Strategy::BA_DribbleBlock:
        doOffensiveKick( agent );
        break;
    case Strategy::BA_DefMidField:
    case Strategy::BA_DribbleAttack:
    case Strategy::BA_OffMidField:
    case Strategy::BA_Cross:
    case Strategy::BA_ShootChance:
        doOffensiveKick( agent );
        break;
    default:
        Bhv_opuciDefensiveKick().execute( agent );
        break;
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
void
RoleRokuro::doOffensiveKick( rcsc::PlayerAgent * agent )
{

    const rcsc::WorldModel & wm = agent->world();

    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    const rcsc::PlayerObject * nearest_opp
        = ( opps.empty()
            ? static_cast< rcsc::PlayerObject * >( 0 )
            : opps.front() );
    const double nearest_opp_dist = ( nearest_opp
                                      ? nearest_opp->distFromSelf()
                                      : 1000.0 );
    const rcsc::Vector2D nearest_opp_pos = ( nearest_opp
                                             ? nearest_opp->pos()
                                             : rcsc::Vector2D( -1000.0, 0.0 ) );

    const rcsc::Rect2D penalty_area( rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength(),
                                                     -rcsc::ServerParam::i().penaltyAreaHalfWidth() ),
                                     rcsc::Vector2D( -rcsc::ServerParam::i().pitchHalfLength() + rcsc::ServerParam::i().penaltyAreaLength(),
                                                     rcsc::ServerParam::i().penaltyAreaHalfWidth() ) );
    
    rcsc::Vector2D pass_point;
    const rcsc::PlayerPtrCont & mates = wm.teammatesFromSelf();
    const rcsc::PlayerPtrCont::const_iterator mates_end = mates.end();

    int unumTarget;
    for( int i = mates.size() - 1 ; i >= 0 ; --i )
    {
        if( mates[i]->pos().x < wm.self().pos().x )
            continue;

        unumTarget = mates[i]->unum();
	if( doDirectPass( agent, unumTarget ) )
        {
	    agent->debugClient().addMessage("target:%d", unumTarget );
	    return;
        }
	if( doThroughPass( agent, unumTarget ) )
        {
	    agent->debugClient().addMessage("target:%d", unumTarget );
	    return;
        }

    }

    int receiver;
    if ( rcsc::Body_Pass::get_best_pass( wm, &pass_point, NULL, &receiver ) )
    {
        if ( pass_point.x > wm.self().pos().x - 1.0 )
        {
            bool safety = true;
            const rcsc::Sector2D sector( wm.ball().pos(),
                                         1.0, wm.ball().pos().dist( pass_point ) + 3.0,
                                         ( pass_point - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                         ( pass_point - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );
            const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
            for ( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                  it != opps_end;
                  ++it )
            {
                if ( sector.contains( (*it)->pos() ) )
                {
                    safety = false;
                }
            }

            if ( safety )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: do best pass"
                                    ,__FILE__, __LINE__ );
                agent->debugClient().addMessage( "DefKickPass(1)" );
		agent->debugClient().setTarget( pass_point );
		agent->addSayMessage( new rcsc::PassMessage( receiver,
							     pass_point,
							     agent->effector().queuedNextBallPos(),
							     agent->effector().queuedNextBallVel() ) );
                rcsc::Body_Pass().execute( agent );
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
                return;
            }
        }
    }

    if ( nearest_opp_dist < 7.0 )
    {
        if ( rcsc::Body_Pass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: do best pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(2)" );
            agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
            return;
        }
    }

    //don't dribble in penalty area
    if( ! penalty_area.contains( wm.ball().pos() ) )
    {
        // dribble to my body dir
        if ( nearest_opp_dist < 5.0
             && nearest_opp_dist > ( rcsc::ServerParam::i().tackleDist()
                                     + rcsc::ServerParam::i().defaultPlayerSpeedMax() * 1.5 )
             && wm.self().body().abs() < 70.0 )
        {
            const rcsc::Vector2D body_dir_drib_target
                = wm.self().pos()
                + rcsc::Vector2D::polar2vector(5.0, wm.self().body());
            
            int max_dir_count = 0;
            wm.dirRangeCount( wm.self().body(), 20.0, &max_dir_count, NULL, NULL );
            
            if ( body_dir_drib_target.x < rcsc::ServerParam::i().pitchHalfLength() - 1.0
                 && body_dir_drib_target.absY() < rcsc::ServerParam::i().pitchHalfWidth() - 1.0
                 && max_dir_count < 3
                )
            {
                // check opponents
                // 10m, +-30 degree
                const rcsc::Sector2D sector( wm.self().pos(),
                                             0.5, 10.0,
                                             wm.self().body() - 30.0,
                                             wm.self().body() + 30.0 );
                // opponent check with goalie
                if ( ! wm.existOpponentIn( sector, 10, true ) )
                {
                    rcsc::dlog.addText( rcsc::Logger::TEAM,
                                        "%s:%d: dribble to my body dir"
                                        ,__FILE__, __LINE__ );
                    agent->debugClient().addMessage( "DefKickDrib(1)" );
                    rcsc::Body_Dribble( body_dir_drib_target,
                                        1.0,
                                        rcsc::ServerParam::i().maxDashPower(),
                                        2
                        ).execute( agent );
                    agent->setNeckAction( new Opuci_Neck() );
                    return;
                }
            }
        }
        
        rcsc::Vector2D drib_target( 50.0, wm.self().pos().absY() );
        if ( drib_target.y < 20.0 ) drib_target.y = 20.0;
        if ( drib_target.y > 29.0 ) drib_target.y = 27.0;
        if ( wm.self().pos().y < 0.0 ) drib_target.y *= -1.0;
        const rcsc::AngleDeg drib_angle = ( drib_target - wm.self().pos() ).th();
        
        // opponent is behind of me
        if ( nearest_opp_pos.x < wm.self().pos().x + 1.0 )
        {
            // check opponents
            // 15m, +-30 degree
            const rcsc::Sector2D sector( wm.self().pos(),
                                         0.5, 15.0,
                                         drib_angle - 30.0,
                                         drib_angle + 30.0 );
            // opponent check with goalie
            if ( ! wm.existOpponentIn( sector, 10, true ) )
            {
                const int max_dash_step
                    = wm.self().playerType()
                    .cyclesToReachDistance( wm.self().pos().dist( drib_target ) );
                if ( wm.self().pos().x > 35.0 )
                {
                    drib_target.y *= ( 10.0 / drib_target.absY() );
                }
                
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: fast dribble to (%.1f, %.1f) max_step=%d"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y,
                                    max_dash_step );
                agent->debugClient().addMessage( "DefKickDrib(2)" );
                rcsc::Body_Dribble( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower(),
                                    std::min( 5, max_dash_step )
                    ).execute( agent );
            }
            else
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "%s:%d: slow dribble to (%f, %f)"
                                    ,__FILE__, __LINE__,
                                    drib_target.x, drib_target.y );
                agent->debugClient().addMessage( "DefKickDrib(3)" );
                rcsc::Body_Dribble( drib_target,
                                    1.0,
                                    rcsc::ServerParam::i().maxDashPower(),
                                    2
                    ).execute( agent );
                
            }
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 5.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(4)" );
            rcsc::Body_Dribble( drib_target,
                                1.0,
                                rcsc::ServerParam::i().maxDashPower() * 0.4,
                                1
                ).execute( agent );
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }

        // opp is near

        // can pass
        if ( rcsc::Body_Pass().execute( agent ) )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: pass"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickPass(3)" );
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }
        
        // opp is far from me
        if ( nearest_opp_dist > 3.0 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: opp far. dribble(%f, %f)"
                                ,__FILE__, __LINE__,
                                drib_target.x, drib_target.y );
            agent->debugClient().addMessage( "DefKickDrib(5)" );
            rcsc::Body_Dribble( drib_target,
                                1.0,
                                rcsc::ServerParam::i().maxDashPower() * 0.2,
                                1
                ).execute( agent );
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }

        if ( nearest_opp_dist > 2.5 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: hold"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickHold" );
            rcsc::Body_HoldBall().execute( agent );
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }
    }

    if ( wm.self().pos().x > wm.offsideLineX() - 10.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: kick near side"
                            ,__FILE__, __LINE__ );
        agent->debugClient().addMessage( "DefKickToCorner" );
        Body_KickToCorner( (wm.self().pos().y < 0.0) ).execute( agent );
        agent->setNeckAction( new rcsc::Neck_ScanField() );
    }
    else
    {
        rcsc::Vector2D target_point;
        double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
        double best_angle = 90.0 * wm.ball().pos().y / wm.ball().pos().absY();
        for( double angle = -90.0; angle <= 90.0; angle += 10.0 )
        {
            bool angle_safety = true;
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   angle - 20.0,
                                   angle + 20.0 );
            for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
                 it != opps_end;
                 ++it )
            {
                if( sector.contains( (*it)->pos() ) )
                {
                    angle_safety = false;
                }
            }
            if( angle_safety
                && fabs( angle ) < fabs( best_angle ) )
            {
                best_angle = angle;
            }
        }
        target_point.x = wm.ball().pos().x + rcsc::AngleDeg( best_angle ).cos();
        target_point.y = wm.ball().pos().y + rcsc::AngleDeg( best_angle ).sin();
        for( rcsc::PlayerPtrCont::const_iterator it = mates.begin();
             it != mates_end;
             ++it )
        {
            rcsc::Sector2D sector( wm.ball().pos(),
                                   1.0, 30.0,
                                   best_angle - 20.0,
                                   best_angle + 20.0 );
            if( sector.contains( (*it)->pos() ) )
            {
                target_point = rcsc::Line2D( wm.ball().pos(), rcsc::AngleDeg( best_angle ) ).projection( (*it)->pos() );
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(1):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new Opuci_Neck() );
                return;
            }
            else if( (*it)->pos().dist( target_point ) < 20.0 * rcsc::AngleDeg( 20.0 ).sin() * 0.8 )
            {
                rcsc::Body_KickOneStep( target_point, ball_speed_max ).execute( agent );
                agent->debugClient().addMessage( "DefAdvancePass(2):%lf",best_angle );
                agent->addSayMessage( new rcsc::PassMessage( (*it)->unum(),
                                                             target_point,
                                                             agent->effector().queuedNextBallPos(),
                                                             agent->effector().queuedNextBallVel() ) );
                agent->setNeckAction( new Opuci_Neck() );
                return;
            }
        }
        if( rcsc::Body_KickOneStep( target_point,
                                    ball_speed_max ).execute( agent ) )
        {
            agent->debugClient().addMessage( "DefKickAdvance(1):%lf",best_angle );
            agent->setNeckAction( new Opuci_Neck() );
            return;
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: advance"
                                ,__FILE__, __LINE__ );
            agent->debugClient().addMessage( "DefKickAdvance(2)" );
            rcsc::Body_AdvanceBall().execute( agent );
            agent->setNeckAction( new rcsc::Neck_ScanField() );
        }
    }
    return;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
RoleRokuro::doDirectPass( rcsc::PlayerAgent * agent, const int target_unum )
{
    const rcsc::WorldModel & wm = agent->world();

    rcsc::Vector2D pass_point;
    rcsc::Vector2D modified_tgt;
    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    //const double ball_speed_max08 = ball_speed_max * 0.8;

    M_passMate = -1;

    if( wm.self().unum() == target_unum )
    {
        return false;
    }

    if( ! wm.teammate( target_unum ) )
    {
        return false;
    }

    int reach_cycle = (int)(log( 1.0 - ( 1.0 - ball_decay ) * wm.self().pos().dist( wm.teammate( target_unum )->pos() ) / ball_speed_max ) / log( ball_decay ));
    modified_tgt = wm.teammate( target_unum )->pos();
    if( wm.teammate( target_unum )->pos().x > wm.self().pos().x )
        modified_tgt.x += wm.teammate( target_unum )->playerTypePtr()->kickableArea() * 0.5 * reach_cycle; // 0.5 is magic number
    else
        modified_tgt.x += wm.teammate( target_unum )->playerTypePtr()->kickableArea() * 0.8 * 0.5 * reach_cycle; // 0.5 is magic number, 0.8 is internal point

    bool target_is_free = true;
    const rcsc::Sector2D sector( wm.ball().pos(),
                                 1.0, modified_tgt.dist( wm.ball().pos() ) + 3.0,
                                 ( modified_tgt - wm.ball().pos() ).th() - rcsc::AngleDeg( 20.0 ),
                                 ( modified_tgt - wm.ball().pos() ).th() + rcsc::AngleDeg( 20.0 ) );

    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps_end;
         ++it )
    {
        if( sector.contains( (*it)->pos() )
            && ! (*it)->isTackling() )
	{
            target_is_free = false;
	}
    }
  
    if( target_is_free
        && wm.teammate( target_unum )->posCount() < 3
        && modified_tgt.x > -25.0
        && modified_tgt.x > wm.self().pos().x - 5.0
        && modified_tgt.dist( wm.self().pos() ) > ( wm.self().playerType().kickableArea() 
                                                    + wm.teammate( target_unum )->playerTypePtr()->kickableArea() ) * 1.5 )
    {
        if( wm.teammate( target_unum )->seenPosCount() == 0 )
        {
            if( wm.teammate( target_unum )->pos().x > wm.self().pos().x )
            {
                pass_point = modified_tgt;
            }
            else
            {
                pass_point = modified_tgt * 0.8 + wm.ball().pos() * 0.2;
            }
            
            double ball_reach_speed = rcsc::ServerParam::i().kickPowerRate() * rcsc::ServerParam::i().maxPower() * 0.5;
            double ball_speed = ball_reach_speed / pow( ball_decay, reach_cycle - 1 );
            double first_speed = std::min( ball_speed_max, ball_speed );
            if( rcsc::Body_SmartKick( pass_point, first_speed, first_speed * 0.9, 3 ).execute( agent ) )
            {
                rcsc::KickTable::Sequence seq;
                rcsc::KickTable::instance().simulate( wm,
                                                      pass_point,
                                                      first_speed,
                                                      first_speed * 0.9,
                                                      3,
                                                      seq );
                if( seq.pos_list_.size() == 1 )
                {
                    agent->addSayMessage( new rcsc::PassMessage( target_unum, pass_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                }
            }
            agent->debugClient().addMessage( "DirectPass" );
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: reach_cycle %d, modified mate%d(%0.00f, %0.00f), pass_point(%0.00f, %0.00f)"
                                ,__FILE__, __LINE__, reach_cycle, target_unum, modified_tgt.x, modified_tgt.y, pass_point.x, pass_point.y );
            agent->debugClient().setTarget( pass_point );

            if( M_passMate != -1 )
            {
                Opuci_Neck().lookAtPassReceiver( agent, target_unum );
                M_neck_cnt--;
                if( M_neck_cnt <= 0 )
                    M_passMate = -1;
            }
            else
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
        }
        else if( M_passMate != -1 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Keep locking on %d at the next cycle."
                                ,__FILE__, __LINE__, M_passMate );
            rcsc::Body_HoldBall().execute( agent );
            Opuci_Neck().lookAtPassReceiver( agent, target_unum );
            M_neck_cnt--;
            if( M_neck_cnt <= 0 )
                M_passMate = -1;
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                ,__FILE__, __LINE__, wm.teammate(target_unum)->seenPosCount(), wm.teammate(target_unum)->unum() );
            rcsc::Body_HoldBall().execute( agent );
            Opuci_Neck().lookAtPassReceiver( agent, target_unum );
            M_passMate = target_unum;
            M_neck_cnt = 3;
        }
        return true;
    }

    return false;
}


/*-------------------------------------------------------------------*/
/*!

 */
bool
RoleRokuro::doThroughPass( rcsc::PlayerAgent * agent, const int target_unum )
{
    const rcsc::WorldModel & wm = agent->world();

    if( wm.self().unum() == target_unum )
    {
        return false;
    }
    if( ! wm.teammate( target_unum ) )
    {
        return false;
    }
    if( wm.teammate( target_unum )->pos().x < wm.self().pos().x )
    {
        return false;
    }
    if( wm.self().pos().x > 30.0 )
    {
        return false;
    }
    if( wm.offsideLineX() - wm.teammate( target_unum )->pos().x > 15.0 )
    {
        return false;
    }

    rcsc::Vector2D target_pos = wm.teammate( target_unum )->pos();

    rcsc::Vector2D first_opp_pos( 500.0, 0.0 );
    rcsc::Vector2D second_opp_pos( 1000.0, 0.0 );
    const rcsc::PlayerPtrCont & opps = wm.opponentsFromSelf();
    const rcsc::PlayerPtrCont::const_iterator opps_end = opps.end();
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps_end;
         ++it )
    {
        bool is_defend_line = true;
        rcsc::Vector2D offside_point( wm.offsideLineX(), (*it)->pos().y );
        for( rcsc::PlayerPtrCont::const_iterator itr = opps.begin();
             itr != opps_end;
             ++itr )
	{
            if( offside_point.dist( (*itr)->pos() ) < offside_point.dist( (*it)->pos() ) )
	    {
                is_defend_line = false;
                break;
	    }
	}
        if( is_defend_line )
	{
            if( target_pos.dist( (*it)->pos() ) < target_pos.dist( first_opp_pos )
                && target_pos.dist( (*it)->pos() ) < target_pos.dist( second_opp_pos ) )
	    {
                second_opp_pos = first_opp_pos;
                first_opp_pos = (*it)->pos();
	    }
            else if( target_pos.dist( (*it)->pos() ) < target_pos.dist( second_opp_pos ) )
	    {
                second_opp_pos = (*it)->pos();
	    }
	}
    }

    rcsc::Vector2D pass_point = first_opp_pos * 0.6 + second_opp_pos * 0.4;
    pass_point.assign( 35.0, 
                       wm.ball().pos().y + ( 35.0 - wm.ball().pos().x ) 
                       * ( pass_point.y - wm.ball().pos().y ) / ( pass_point.x - wm.ball().pos().x ) );
    double target_angle = fabs( ( first_opp_pos - wm.ball().pos() ).th().degree() 
                                - ( second_opp_pos - wm.ball().pos() ).th().degree() );
    if( target_angle < 30.0 )
    {
        return false;
    }
    rcsc::Triangle2D target_triangle( wm.ball().pos(), 
                                      first_opp_pos * 0.9 + pass_point * 0.1, 
                                      second_opp_pos * 0.9 + pass_point * 0.1 );
    for( rcsc::PlayerPtrCont::const_iterator it = opps.begin();
         it != opps_end;
         ++it )
    {
        if( target_triangle.contains( (*it)->pos() ) )
	{
            return false;
	}
        if( wm.ball().pos().dist( (*it)->pos() ) < 5.0 )
	{
            return false;
	}
    }

    double line_dist = ( pass_point - wm.ball().pos() ).r();
    if( line_dist < 10.0 )
    {
        return false;
    }

    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const double ball_speed_max = rcsc::ServerParam::i().ballSpeedMax();
    rcsc::Vector2D mate_pos = wm.teammate( target_unum )->pos();
    int mate_poscount = wm.teammate( target_unum )->posCount();
    int mate_reach_cycle = wm.teammate( target_unum )->playerTypePtr()->cyclesToReachDistance( wm.self().pos().dist( pass_point ) ) + 2;
    double first_speed = wm.ball().pos().dist( pass_point ) * ( 1.0 - ball_decay ) / ( 1.0 - pow( ball_decay, mate_reach_cycle ) );
    double ball_speed = std::min( ball_speed_max, first_speed );
    if( pass_point.absY() < 25.0
        && mate_poscount < 3 )
    {
        if( wm.teammate(target_unum)->seenPosCount() == 0 )
        {
            if( rcsc::Body_SmartKick( pass_point, ball_speed, ball_speed * 0.9, 3 ).execute( agent ) )
            {
                rcsc::KickTable::Sequence seq;
                rcsc::KickTable::instance().simulate( wm,
                                                      pass_point,
                                                      ball_speed,
                                                      ball_speed * 0.9,
                                                      3,
                                                      seq );
                if( seq.pos_list_.size() == 1 )
                {
                    agent->addSayMessage( new rcsc::PassMessage( target_unum, pass_point,
                                                                 agent->effector().queuedNextBallPos(),
                                                                 agent->effector().queuedNextBallVel() ) );
                }
            }
            
            agent->debugClient().addMessage( "ThroughPass:%lf", target_angle );
            agent->debugClient().setTarget( pass_point );
            agent->debugClient().addLine( wm.ball().pos(), first_opp_pos );
            agent->debugClient().addLine( wm.ball().pos(), second_opp_pos );
            if( M_passMate != -1 )
            {
                Opuci_Neck().lookAtPassReceiver( agent, target_unum );
                M_neck_cnt--;
                if( M_neck_cnt <= 0 )
                    M_passMate = -1;
            }
            else
                agent->setNeckAction( new rcsc::Neck_OpuciTurnToNextReceiver() );
        }
        else if( M_passMate != -1 )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: Keep locking on %d at the next cycle."
                                ,__FILE__, __LINE__, M_passMate );
            M_passMate = wm.teammate(target_unum)->unum();
            rcsc::Body_HoldBall().execute( agent );
            Opuci_Neck().lookAtPassReceiver( agent, target_unum );
            M_neck_cnt--;
            if( M_neck_cnt <= 0 )
                M_passMate = -1;
        }
        else
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s:%d: seenCount = %d. Lock on %d at the next cycle."
                                ,__FILE__, __LINE__, wm.teammate(target_unum)->seenPosCount(), wm.teammate(target_unum)->unum() );
            M_passMate = wm.teammate(target_unum)->unum();
            rcsc::Body_HoldBall().execute( agent );
            Opuci_Neck().lookAtPassReceiver( agent, target_unum );
            M_neck_cnt = 3;
        }

        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

void
RoleRokuro::doMove( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();

    int ball_reach_step = 0;
    if ( ! wm.existKickableTeammate()
         && ! wm.existKickableOpponent() )
    {
        ball_reach_step
            = std::min( wm.interceptTable()->teammateReachCycle(),
                        wm.interceptTable()->opponentReachCycle() );
    }
    const rcsc::Vector2D base_pos
        = wm.ball().inertiaPoint( ball_reach_step );
    boost::shared_ptr< const rcsc::Formation > formation 
        = Strategy::i().getFormation( wm );
    rcsc::Vector2D home_pos
        = (*formation).getPosition( agent->config().playerNumber(),
                                   base_pos );
    if ( rcsc::ServerParam::i().useOffside() )
    {
        home_pos.x = std::min( home_pos.x, wm.offsideLineX() - 1.0 );
    }

    rcsc::dlog.addText( rcsc::Logger::ROLE,
                        "%s: HOME POSITION=(%.2f, %.2f) base_point(%.1f %.1f)"
                        ,__FILE__,
                        home_pos.x, home_pos.y,
                        base_pos.x, base_pos.y );
    Bhv_opuciDefensiveHalfMove( home_pos ).execute( agent );
    return;
}

/*-------------------------------------------------------------------*/
/*!

 */
/*-------------------------------------------------------------------*/

