// -*-c++-*-

/*!
  \file stamina_model.h
  \brief player's stamina model Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 3 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_STAMINA_MODEL_H
#define RCSC_PLAYER_STAMINA_MODEL_H

#include "fullstate_sensor.h"

namespace rcsc {

class BodySensor;
class PlayerType;
class GameTime;

/*!
  \class StaminaModel
  \brief stamina management class
*/
class StaminaModel {
private:

    //! stamina value
    double M_stamina;
    //! effort rate
    double M_effort;
    //! recover rate
    double M_recovery;
    //! remained stamina capacity
    double M_stamina_capacity;

public:
    /*!
      \brief init members by built-in values
    */
    StaminaModel();

    /*!
      \brief initialize internal variables with server settings
      \param player_type heterogeneous player type
    */
    void init( const PlayerType & player_type );

    //
    // accessor method
    //

    /*!
      \brief get current stamina value
      \return const reference to the current stamina value
    */
    const
    double & stamina() const
      {
          return M_stamina;
      }

    /*!
      \brief get current recovery value
      \return const reference to the current recovery value
    */
    const
    double & recovery() const
      {
          return M_recovery;
      }

    /*!
      \brief get current effort value
      \return const reference to the current effort value
    */
    const
    double & effort() const
      {
          return M_effort;
      }

    /*!
      \brief get the current remained stamina capacity
      \return stamina capacity value
     */
    const
    double & staminaCapacity() const
      {
          return M_stamina_capacity;
      }

    //
    // interface for update
    //

    /*!
      \brief update stamina related variables
      \param player_type PlayerType parameter
      \param dash_power previous dash command parameter

      update without sense_body message.
      this means that this method try to update variables
      only with internal values and estimated previous action effect,
      but we can update the stamina information very safety.
    */
    void update( const PlayerType & player_type,
                 const double & dash_power );

    /*!
      \brief update with sense_body message
      \param body_sensor reference to the body sensor instance
      \param current game time that sense_body was received
    */
    void updateAfterSense( const BodySensor & body_sensor,
                           const GameTime & current );

    /*!
      \brief update with fullstate message
      \param state reference to the fullstate sensor instance
     */
    void updateAfterFullstate( const FullstateSensor::PlayerT & state );


    //
    // utility for prediction
    //

    /*!
      \brief simulate stamina variables after nr waits.
      \param player_type heterogeneous player type
      \param n_wait number of wait cycles (kick, turn, tackle...)
     */
    void simulateWait( const PlayerType & player_type,
                       const int n_wait );

    /*!
      \brief simulate stamina variables after nr dashes.
      \param player_type heterogeneous player type
      \param n_dash number of dash cycles
      \param dash_power dash power for simulation
     */
    void simulateDash( const PlayerType & player_type,
                       const int n_dash,
                       const double & dash_power );


    /*!
      \brief simulate stamina variables
      \param player_type heterogeneous player type
      \param n_wait number of wait cycles (kick, turn, tackle...)
      \param n_dash number of dash cycles
      \param dash_power dash power for simulation
     */
    void simulate( const PlayerType & player_type,
                   const int n_wait,
                   const int n_dash,
                   const double & dash_power );
};

}

#endif
