/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.map.CSVDataMapper;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.NullBoolean;
import net.morilib.db.relations.DefaultRelationTuple;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.SingleTableRelation;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.sqlcs.ddl.SqlColumnAttribute;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.db.sqlcs.ddl.SqlTypeVarchar;

public class MemorySqlSchema implements SqlSchema {

	private Map<String, NamedRelation> relations;
	private Map<String, SqlCreateTable> ddls;
	private Map<String, Relation> rels;
	private Map<String, Void> lock;

	MemorySqlSchema() {
		lock = new HashMap<String, Void>();
	}

	/**
	 * 
	 * @param name
	 * @param ins
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static MemorySqlSchema readCSV(SqlCreateTable d,
			InputStream ins) throws IOException, SQLException {
		MemorySqlSchema m = new MemorySqlSchema();
		CSVDataMapper p = new CSVDataMapper();
		BufferedReader b;
		NamedRelation r;

		m.ddls = new HashMap<String, SqlCreateTable>();
		m.relations = new HashMap<String, NamedRelation>();
		b = new BufferedReader(new InputStreamReader(ins));
		b.readLine();
		r = p.read(d.getName(), null, d, b);
		m.ddls.put(d.getName(), d);
		m.relations.put(d.getName(), r);
		return m;
	}

	/**
	 * 
	 * @param tn
	 * @param kn
	 * @param vn
	 * @param ins
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static MemorySqlSchema readProperties(String tn, String kn,
			String vn,
			InputStream ins) throws IOException, SQLException {
		MemorySqlSchema m = new MemorySqlSchema();
		Properties p = new Properties();
		List<SqlColumnDefinition> f;
		List<RelationTuple> l;
		Map<String, Object> n;
		SqlCreateTable d;

		m.ddls = new HashMap<String, SqlCreateTable>();
		m.relations = new HashMap<String, NamedRelation>();
		f = new ArrayList<SqlColumnDefinition>();
		f.add(new SqlColumnDefinition(kn,
				new SqlTypeVarchar(SqlTypeVarchar.UNLIMITED),
				EnumSet.of(SqlColumnAttribute.PRIMARY_KEY)));
		f.add(new SqlColumnDefinition(vn,
				new SqlTypeVarchar(SqlTypeVarchar.UNLIMITED),
				EnumSet.noneOf(SqlColumnAttribute.class)));
		d = new SqlCreateTable(tn, f);

		p.load(ins);
		l = new ArrayList<RelationTuple>();
		n = new LinkedHashMap<String, Object>();
		for(Map.Entry<Object, Object> o : p.entrySet()) {
			n.put(kn, o.getKey().toString());
			n.put(vn, o.getValue().toString());
			l.add(new DefaultRelationTuple(n));
		}
		m.ddls.put(tn, d);
		m.relations.put(tn, new SingleTableRelation(d, tn, l));
		return m;
	}

	/**
	 * 
	 * @param csvs
	 * @return
	 * @throws IOException
	 * @throws SQLException
	 */
	public static MemorySqlSchema readCSVs(int k,
			String... csvs) throws IOException, SQLException {
		MemorySqlSchema m = new MemorySqlSchema();
		CSVDataMapper p = new CSVDataMapper();
		BufferedReader b;
		SqlCreateTable d;
		NamedRelation r;

		m.ddls = new HashMap<String, SqlCreateTable>();
		m.relations = new HashMap<String, NamedRelation>();
		for(int i = k; i < csvs.length; i++) {
			d = SqlSchemata.guessTable(csvs[i]);
			b = new BufferedReader(new InputStreamReader(
					new FileInputStream(csvs[i])));
			b.readLine();
			r = p.read(d.getName(), null, d, b);
			m.ddls.put(d.getName(), d);
			m.relations.put(d.getName(), r);
		}
		return m;
	}

	@Override
	public NamedRelation readRelation(String name,
			String as) throws IOException, SQLException {
		String s = name.toUpperCase();

		if(rels != null && rels.containsKey(name.toUpperCase())) {
			return new TableRenameRelation(
					rels.get(name.toUpperCase()),
					(as != null ? as : name).toUpperCase());
		} else if(!relations.containsKey(s)) {
			throw ErrorBundle.getDefault(10015, name);
		} else if(as == null) {
			return relations.get(s);
		} else {
			return new TableRenameRelation(relations.get(s), as);
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> z) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		String s = name.toUpperCase();

		if(!ddls.containsKey(s)) {
			throw ErrorBundle.getDefault(10015, name);
		} else {
			return ddls.get(s);
		}
	}

	@Override
	public Collection<String> getTableNames() {
		return Collections.unmodifiableCollection(relations.keySet());
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return relations.containsKey(name);
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		throw ErrorBundle.getDefault(10017);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#fork()
	 */
	@Override
	public SqlSchema fork() {
		return this;
	}

	@Override
	public NullBoolean isReadonly() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFiles() {
		return NullBoolean.FALSE;
	}

	@Override
	public NullBoolean usesLocalFilePerTable() {
		return NullBoolean.FALSE;
	}

	@Override
	public FabriqueDeFichier fabrique() {
		return FabriqueDeFichier.getDefault();
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#bindSchema(java.lang.String, net.morilib.db.relations.Relation)
	 */
	@Override
	public void bindSchema(String name, Relation r) {
		rels.put(name, r);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isAutoCommit()
	 */
	@Override
	public boolean isAutoCommit() {
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isLocked(java.lang.String)
	 */
	@Override
	public boolean isLocked(String name) {
		return lock.containsKey(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isUsed(java.lang.String)
	 */
	@Override
	public boolean isUsed(String name) {
		return lock.containsKey(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#lock(java.lang.String)
	 */
	@Override
	public void lock(String name) throws IOException {
		lock.put(name, null);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#unlock(java.lang.String)
	 */
	@Override
	public void unlock(String name) throws IOException {
		lock.remove(name);
	}

}
