/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.schema;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import net.morilib.db.fichier.FabriqueDeFichier;
import net.morilib.db.fichier.Fichier;
import net.morilib.db.misc.ErrorBundle;
import net.morilib.db.misc.HTMLTermException;
import net.morilib.db.misc.NullBoolean;
import net.morilib.db.misc.Rational;
import net.morilib.db.misc.XlsxSharedString;
import net.morilib.db.misc.XmlReplaceTag;
import net.morilib.db.misc.XmlTags;
import net.morilib.db.relations.NamedRelation;
import net.morilib.db.relations.Relation;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.relations.TableRenameRelation;
import net.morilib.db.sqlcs.ddl.SqlColumnDefinition;
import net.morilib.db.sqlcs.ddl.SqlColumnType;
import net.morilib.db.sqlcs.ddl.SqlCreateTable;
import net.morilib.parser.html.HTMLHandler;
import net.morilib.parser.html.HTMLParseException;
import net.morilib.parser.html.HTMLParser;

public class XlsxSqlSchema implements SqlSchema {

	private Fichier xlsx;
	private Map<String, Relation> rels;

	/**
	 * 
	 * @param xlsxfile
	 */
	public XlsxSqlSchema(Fichier xlsxfile) {
		xlsx = xlsxfile;
	}

	private static void _parse(HTMLHandler h,
			BufferedReader b) throws IOException, SQLException {
		try {
			HTMLParser.parse(h, b);
		} catch(HTMLTermException e) {
			// do nothing
		} catch(HTMLParseException e) {
			throw (SQLException)e.getCause();
		}
	}

	private List<List<String>> gettbl(
			String name) throws SQLException, IOException {
		BufferedReader b = null;
		ZipInputStream z = null;
		XlsxSharedString s;
		XlsxReadTable h;
		XlsxTables y;
		ZipEntry t;
		String n;

		y = _getTableNames();
		if((n = y.getSheet(name.toUpperCase())) == null) {
			throw ErrorBundle.getDefault(10015, name);
		}
		s = XlsxSharedString.fromXlsx(xlsx);

		try {
			z = new ZipInputStream(xlsx.openInputStream());
			while((t = z.getNextEntry()) != null) {
				if(t.getName().equalsIgnoreCase("xl/worksheets/" + n)) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					h = new XlsxReadTable(s);
					_parse(h, b);
					return h.values;
				}
			}
			return null;
		} finally {
			if(z != null)  z.close();
		}
	}

	@Override
	public NamedRelation readRelation(String name,
			String as) throws IOException, SQLException {
		List<List<String>> l;

		if(rels != null && rels.containsKey(name.toUpperCase())) {
			// with clause
			return new TableRenameRelation(
					rels.get(name.toUpperCase()),
					(as != null ? as : name).toUpperCase());
		} if((l = gettbl(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		} else {
			return SqlSchemata.readRelation(name, as, l);
		}
	}

	private static void _copy(InputStream in,
			OutputStream os) throws IOException {
		byte[] b = new byte[4096];
		int l;

		while((l = in.read(b)) >= 0)  os.write(b, 0, l);
	}

	void _writeRelation(SqlCreateTable c,
			Collection<RelationTuple> l
			) throws IOException, SQLException {
		List<SqlColumnDefinition> q;
		ZipOutputStream o = null;
		BufferedReader b = null;
		ZipInputStream z = null;
		OutputStream os = null;
		InputStream is = null;
		XlsxSharedString s;
		XlsxWriteTable h;
		PrintWriter w;
		XlsxTables y;
		ZipEntry t;
		Fichier f;
		String n;

		y = _getTableNames();
		if((n = y.getSheet(c.getName())) == null) {
			throw ErrorBundle.getDefault(10015, c.getName());
		}

		s = XlsxSharedString.fromListOfTuple(xlsx, l);
		q = c.getColumnDefinitions();
		for(int i = 0; i < q.size(); i++) {
			s.addString(q.get(i).getName());
		}

		f = fabrique().createTempFile("table", ".xlsx");
		try {
			z = new ZipInputStream(xlsx.openInputStream());
			o = new ZipOutputStream(f.openOutputStream());
			while((t = z.getNextEntry()) != null) {
				o.putNextEntry(new ZipEntry(t.getName()));
				if(t.getName().equalsIgnoreCase("xl/worksheets/" + n)) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					w = new PrintWriter(new BufferedWriter(
							new OutputStreamWriter(o, "UTF-8")), true);
					h = new XlsxWriteTable(w, s, c, l, null);
					_parse(h, b);
					w.flush();
				} else if(t.getName().equalsIgnoreCase(
						"xl/sharedStrings.xml")) {
					s.write(o);
				} else {
					_copy(z, o);
				}
				z.closeEntry();
				o.closeEntry();
			}
		} finally {
			if(z != null)  z.close();
			if(o != null)  o.close();
		}

		try {
			is = new BufferedInputStream(f.openInputStream());
			os = new BufferedOutputStream(xlsx.openOutputStream());
			_copy(is, os);
		} finally {
			if(is != null)  is.close();
			if(os != null)  os.close();
		}
	}

	@Override
	public void writeRelation(String name,
			Collection<RelationTuple> l
			) throws IOException, SQLException {
		_writeRelation(getCreateTable(name), l);
	}

	@Override
	public SqlCreateTable getCreateTable(
			String name) throws IOException, SQLException {
		List<List<String>> l;

		if((l = gettbl(name)) == null) {
			throw ErrorBundle.getDefault(10015, name);
		}
		return SqlSchemata.guessTable(name, l);
	}

	@Override
	public boolean isTable(
			String name) throws IOException, SQLException {
		return getTableNames().contains(name);
	}

	@Override
	public void putCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		List<SqlColumnDefinition> q;
		ZipOutputStream o = null;
		BufferedReader b = null;
		ZipInputStream z = null;
		OutputStream os = null;
		InputStream is = null;
		Map<String, String> m;
		XlsxSharedString s;
		XlsxWriteTables g;
		XlsxWriteTable h;
		List<String> y;
		XlsxRelsAdd a;
		PrintWriter w;
		ZipEntry t;
		Fichier f;
		String d;

		// add to workbook
		y = new ArrayList<String>(getTableNames());
		y.add(name);

		// add to shared string
		s = XlsxSharedString.fromXlsx(xlsx);
		q = table.getColumnDefinitions();
		for(int i = 0; i < q.size(); i++) {
			s.addString(q.get(i).getName());
		}

		// get relationship id
		try {
			d = XlsxRelsMax.fromXlsx(xlsx);
			d = d.replaceFirst("^rId", "");
			d = "rId" + (Integer.parseInt(d) + 1);
			m = XlsxRelsSearch.fromXlsx(xlsx);
			m.put(y.size() + "", d);
		} catch(NumberFormatException e) {
			throw ErrorBundle.getDefault(10046);
		}

		f = fabrique().createTempFile("table", ".xlsx");
		try {
			z = new ZipInputStream(xlsx.openInputStream());
			o = new ZipOutputStream(f.openOutputStream());
			while((t = z.getNextEntry()) != null) {
				o.putNextEntry(new ZipEntry(t.getName()));
				if(t.getName().equalsIgnoreCase("xl/workbook.xml")) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					w = new PrintWriter(new BufferedWriter(
							new OutputStreamWriter(o, "UTF-8")));
					g = new XlsxWriteTables(w, m, y);
					_parse(g, b);
					w.flush();
				} else if(t.getName().equalsIgnoreCase(
						"xl/sharedStrings.xml")) {
					s.write(o);
				} else if(t.getName().equalsIgnoreCase(
						"xl/_rels/workbook.xml.rels")) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					w = new PrintWriter(new BufferedWriter(
							new OutputStreamWriter(o, "UTF-8")));
					a = new XlsxRelsAdd(w, y.size(), d);
					_parse(a, b);
					w.flush();
				} else {
					_copy(z, o);
				}
				z.closeEntry();
				o.closeEntry();
			}

			// add new sheet
			o.putNextEntry(new ZipEntry(
					"xl/worksheets/sheet" + y.size() + ".xml"));
			b = new BufferedReader(new InputStreamReader(
					is = XlsxSqlSchema.class.getResourceAsStream(
							"/net/morilib/db/schema/sheettemplate.xml"),
					"UTF-8"));
			w = new PrintWriter(new BufferedWriter(
					new OutputStreamWriter(o, "UTF-8")), true);
			h = new XlsxWriteTable(w, s, table,
					Collections.<RelationTuple>emptyList(), d);
			_parse(h, b);
			w.flush();
			o.closeEntry();
		} finally {
			if(is != null)  is.close();
			if(z != null)   z.close();
			if(o != null)   o.close();
			is = null;
		}

		try {
			is = new BufferedInputStream(f.openInputStream());
			os = new BufferedOutputStream(xlsx.openOutputStream());
			_copy(is, os);
		} finally {
			if(is != null)  is.close();
			if(os != null)  os.close();
		}
	}

	XlsxTables _getTableNames() throws IOException, SQLException {
		BufferedReader b = null;
		ZipInputStream z = null;
		XlsxTables h;
		ZipEntry t;

		try {
			z = new ZipInputStream(xlsx.openInputStream());
			while((t = z.getNextEntry()) != null) {
				if(t.getName().equalsIgnoreCase("xl/workbook.xml")) {
					b = new BufferedReader(new InputStreamReader(
							z, "UTF-8"));
					h = new XlsxTables();
					_parse(h, b);
					return h;
				}
			}
			return null;
		} finally {
			if(z != null)  z.close();
		}
	}

	@Override
	public Collection<String> getTableNames(
			) throws IOException, SQLException {
		return Collections.unmodifiableSet(
				_getTableNames().map.keySet());
	}

	@Override
	public void truncateTable(
			String name) throws IOException, SQLException {
		writeRelation(name, Collections.<RelationTuple>emptyList());
	}

	@Override
	public void removeCreateTable(
			String name) throws IOException, SQLException {
		ZipOutputStream o = null;
		ZipInputStream z = null;
		OutputStream os = null;
		InputStream is = null;
		Map<String, String> m;
		SqlCreateTable c;
		XlsxTables y;
		String d, n;
		ZipEntry t;
		Fichier f;

		c = getCreateTable(name);
		y = _getTableNames();
		if((n = y.getSheet(c.getName())) == null) {
			throw ErrorBundle.getDefault(10015, c.getName());
		}

		// get relationship id
		m = XlsxRelsSearch.fromXlsx(xlsx);
		d = n.replaceFirst("^sheet", "");
		d = d.replaceFirst("\\.xml$", "");
		d = m.get(d);

		f = fabrique().createTempFile("table", ".xlsx");
		try {
			z = new ZipInputStream(xlsx.openInputStream());
			o = new ZipOutputStream(f.openOutputStream());
			while((t = z.getNextEntry()) != null) {
				if(t.getName().equalsIgnoreCase("xl/workbook.xml")) {
					o.putNextEntry(new ZipEntry(t.getName()));
					XmlTags.removeTag(z, o, "sheet", "r:id", d);
					o.closeEntry();
				} else if(t.getName().equalsIgnoreCase(
						"xl/_rels/workbook.xml.rels")) {
					o.putNextEntry(new ZipEntry(t.getName()));
					XmlTags.removeTag(z, o, "Relationship", "Id", d);
					o.closeEntry();
				} else if(!t.getName().equalsIgnoreCase(
						"xl/worksheets/" + n)) {
					o.putNextEntry(new ZipEntry(t.getName()));
					_copy(z, o);
					o.closeEntry();
				}
				z.closeEntry();
			}
		} finally {
			if(z != null)   z.close();
			if(o != null)   o.close();
		}

		try {
			is = new BufferedInputStream(f.openInputStream());
			os = new BufferedOutputStream(xlsx.openOutputStream());
			_copy(is, os);
		} finally {
			if(is != null)  is.close();
			if(os != null)  os.close();
		}
	}

	@Override
	public void alterCreateTable(String name,
			SqlCreateTable table) throws IOException, SQLException {
		_writeRelation(table, Collections.<RelationTuple>emptyList());
	}

	@Override
	public SqlSchema fork() {
		return this;
	}

	@Override
	public NullBoolean isReadonly() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFiles() {
		return NullBoolean.TRUE;
	}

	@Override
	public NullBoolean usesLocalFilePerTable() {
		return NullBoolean.FALSE;
	}

	@Override
	public FabriqueDeFichier fabrique() {
		return FabriqueDeFichier.getDefault();
	}

	@Override
	public void bindSchema(String name, Relation r) {
		rels.put(name, r);
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isAutoCommit()
	 */
	@Override
	public boolean isAutoCommit() {
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isLocked(java.lang.String)
	 */
	@Override
	public boolean isLocked(String name) {
		Fichier f = xlsx.getParent();

		f = fabrique().newInstance(f, name + ".lock");
		return f.isFile();
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#isUsed(java.lang.String)
	 */
	@Override
	public boolean isUsed(String name) {
		Fichier f = xlsx.getParent();

		f = fabrique().newInstance(f, name + ".lock");
		return f.isFile();
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#lock(java.lang.String)
	 */
	@Override
	public void lock(String name) throws IOException {
		Fichier f = xlsx.getParent();

		f = fabrique().newInstance(f, name + ".lock");
		f.createNewFile();
	}

	/* (non-Javadoc)
	 * @see net.morilib.db.schema.SqlSchema#unlock(java.lang.String)
	 */
	@Override
	public void unlock(String name) throws IOException {
		Fichier f = xlsx.getParent();

		f = fabrique().newInstance(f, name + ".lock");
		f.delete();
	}

}

class XlsxReadTable implements HTMLHandler {

	private static enum S {
		INIT, WORKSHEET, SHEET_DATA, ROW, C, V
	}

	private static enum F {
		NONE, SHARED, DATE
	}

	private static final Pattern PT1 = Pattern.compile("[A-Z]+");
	private static final BigDecimal ML1 = BigDecimal.valueOf(86400000);
	private static final BigDecimal SU1 = BigDecimal.valueOf(25569);
	private static final SimpleDateFormat FM1 =
			new SimpleDateFormat("yyyy/MM/dd");

	List<List<String>> values;
	private XlsxSharedString share;
	private List<String> value;
	private F sflg;
	private int colno;
	private S stat;

	XlsxReadTable(XlsxSharedString s) {
		values = new ArrayList<List<String>>();
		stat = S.INIT;
		share = s;
	}

	private static int colno(String s) throws HTMLParseException {
		Matcher m;
		int x = 0;
		String v;

		if((m = PT1.matcher(s.toUpperCase())).lookingAt()) {
			v = m.group();
			for(int i = v.length() - 1; i >= 0; i--) {
				x = x * 26 + (v.charAt(i) - 'A');
			}
			return x;
		} else {
			throw _geterr(10046);
		}
	}

	private static String serialToYMD(
			String serial) throws HTMLParseException {
		java.util.Date d;
		BigDecimal b;

		try {
			b = new BigDecimal(serial);
			b = b.subtract(SU1).multiply(ML1);
			d = new java.util.Date(b.longValue());
			return FM1.format(d);
		} catch(NumberFormatException e) {
			throw _geterr(10046);
		}
	}

	private void _put(int x, String s) {
		if(x < value.size()) {
			value.set(x, s);
		} else {
			for(int i = value.size(); i < x; i++)  value.set(i, "");
			value.add(s);
		}
	}

	private static HTMLParseException _geterr(int c, Object... a) {
		return new HTMLParseException(ErrorBundle.getDefault(c, a));
	}

	@Override
	public void string(String s) throws HTMLParseException {
		switch(stat) {
		case INIT:
		case WORKSHEET:
		case SHEET_DATA:
		case ROW:
		case C:
			break;
		case V:
			if(colno < 0) {
				throw _geterr(10046);
			} else if(sflg == F.SHARED) {
				try {
					_put(colno, share.getString(Integer.parseInt(s)));
				} catch(NumberFormatException e) {
					throw _geterr(10046);
				}
			} else if(sflg == F.DATE) {
				_put(colno, serialToYMD(s));
			} else {
				_put(colno, s);
			}
			break;
		}
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		switch(stat) {
		case INIT:
			if(s.equals("worksheet"))  stat = S.WORKSHEET;
			break;
		case WORKSHEET:
			if(s.equals("sheetData"))  stat = S.SHEET_DATA;
			break;
		case SHEET_DATA:
			if(s.equals("row")) {
				stat = S.ROW;
				value = new ArrayList<String>();
			}
			break;
		case ROW:
			if(s.equals("c")) {
				stat = S.C;
				sflg = F.NONE;
				colno = -1;
			}
			break;
		case C:
			if(s.equals("v"))  stat = S.V;
			break;
		case V:
			break;
		}
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		switch(stat) {
		case INIT:
			break;
		case WORKSHEET:
			if(s.equals("worksheet"))  stat = S.INIT;
			break;
		case SHEET_DATA:
			if(s.equals("sheetData"))  stat = S.WORKSHEET;
			break;
		case ROW:
			if(s.equals("row")) {
				values.add(value);
				stat = S.SHEET_DATA;
			}
			break;
		case C:
			if(s.equals("c"))  stat = S.ROW;
			break;
		case V:
			if(s.equals("v"))  stat = S.C;
			break;
		}
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		switch(stat) {
		case INIT:
		case WORKSHEET:
		case SHEET_DATA:
		case ROW:
		case V:
			break;
		case C:
			if(k.equals("r")) {
				colno = colno(v);
			} else if(k.equals("t")) {
				if(v.equals("s")) {
					sflg = F.SHARED;
				}
			} else if(k.equals("s")) {
				if(v.equals("1")) {
					sflg = F.DATE;
				}
			}
			break;
		}
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		// do nothing
	}

}

class XlsxWriteTable implements HTMLHandler {

	private static enum S {
		INIT, WORKSHEET, COLS, SHEET_DATA
	}

	private static final String COLS =
			"<col customWidth=\"1\"" +
			" bestFit=\"1\" width=\"11.625\"" +
			" max=\"%d\" min=\"%d\" />";

	private static final BigDecimal ML1 = BigDecimal.valueOf(86400000);
	private static final BigDecimal SU1 = BigDecimal.valueOf(25569);

	Collection<RelationTuple> tuples;
	SqlCreateTable table;
	String rid;

	private XlsxSharedString share;
	private PrintWriter wr;
	private boolean tagw;
	private S stat = S.INIT;

	XlsxWriteTable(PrintWriter o,
			XlsxSharedString s,
			SqlCreateTable c,
			Collection<RelationTuple> t,
			String i) throws SQLException, IOException {
		wr     = o;
		table  = c;
		tuples = t;
		share  = s;
		rid    = i;
	}

	private void puttagw() {
		if(tagw) {
			wr.print('>');
			tagw = false;
		}
	}

	private static String colno(int a) throws SQLException {
		String v = "";

		if(a == 0) {
			return "A";
		} else {
			for(int x = a; x > 0; x /= 26) {
				v = Character.toString((char)(x % 26 + 'A')) + v;
			}
			return v;
		}
	}

	private static String toSerialFromYMD(
			java.util.Date d) throws SQLException {
		BigDecimal b;

		try {
			b = new BigDecimal(d.getTime());
			b = b.divide(ML1, 6, RoundingMode.HALF_UP).add(SU1);
			return b.toString();
		} catch(NumberFormatException e) {
			throw ErrorBundle.getDefault(10046);
		}
	}

	@Override
	public void string(String s) throws HTMLParseException {
		puttagw();
		switch(stat) {
		case INIT:
		case WORKSHEET:
			wr.print(HTMLParser.escape(s));
			break;
		case SHEET_DATA:
		case COLS:
			break;
		}
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		puttagw();
		switch(stat) {
		case INIT:
			wr.format("<%s", HTMLParser.escape(s));  tagw = true;
			if(s.equals("worksheet"))  stat = S.WORKSHEET;
			break;
		case WORKSHEET:
			wr.format("<%s", HTMLParser.escape(s));  tagw = true;
			if(s.equals("sheetData")) {
				stat = S.SHEET_DATA;
			} else if(s.equals("cols")) {
				stat = S.COLS;
			}
			break;
		case SHEET_DATA:
		case COLS:
			break;
		}
	}

	private static final String ROW =
			"<row r=\"%d\" x14ac:dyDescent=\"0.15\" spans=\"1:%d\">";

	private void _putSheets() throws HTMLParseException {
		List<SqlColumnDefinition> l;
		SqlColumnType p;
		int j = 1, k;

		try {
			// header
			l = table.getColumnDefinitions();
			wr.format(ROW, 1, l.size());
			for(int i = 0; i < l.size(); i++) {
				wr.format("<c r=\"%s1\" t=\"s\">", colno(i));
				wr.format("<v>%s</v></c>",
						share.getNumber(l.get(i).getName()));
			}
			wr.println("</row>");

			// body
			for(RelationTuple t : tuples) {
				k = 0;
				wr.format(ROW, ++j, l.size());
				for(Object o : t.toMap().values()) {
					if(o instanceof String) {
						wr.format("<c r=\"%s%d\" t=\"s\">", colno(k++),
								j);
						wr.format("<v>%s</v></c>",
								share.getNumber((String)o));
					} else if(o instanceof java.util.Date) {
						wr.format("<c r=\"%s%d\" s=\"1\">", colno(k++),
								j);
						wr.format("<v>%s</v></c>",
								toSerialFromYMD((java.util.Date)o));
					} else if(o instanceof Rational) {
						p = l.get(k - 1).getType();
						wr.format("<c r=\"%s%d\">", colno(k++), j);
						wr.format("<v>%s</v></c>", p.string(o));
					} else {
						throw ErrorBundle.getDefault(10046);
					}
				}
				wr.println("</row>");
			}
		} catch(SQLException e) {
			throw new HTMLParseException(e);
		}
	}

	private void _putCols() {
		wr.format(COLS,
				table.getColumnDefinitions().size(),
				table.getColumnDefinitions().size());
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		puttagw();
		switch(stat) {
		case INIT:
			wr.format("</%s>", HTMLParser.escape(s));
			break;
		case WORKSHEET:
			wr.format("</%s>", HTMLParser.escape(s));
			if(s.equals("worksheet"))  stat = S.INIT;
			break;
		case SHEET_DATA:
			if(s.equals("sheetData")) {
				_putSheets();
				wr.format("</%s>", HTMLParser.escape(s));
				stat = S.WORKSHEET;
			}
			break;
		case COLS:
			if(s.equals("cols")) {
				_putCols();
				wr.format("</%s>", HTMLParser.escape(s));
				stat = S.WORKSHEET;
			}
			break;
		}
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		switch(stat) {
		case INIT:
		case WORKSHEET:
			wr.format(" %s='%s'",
					HTMLParser.escape(k),
					HTMLParser.escape(v));
			break;
		case SHEET_DATA:
		case COLS:
			break;
		}
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		wr.println(s);
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		wr.format("<!--%s-->", s);
	}

}

class XlsxTables implements HTMLHandler {

	private static enum S {
		INIT, WORKBOOK, SHEETS, SHEET
	}

	Map<String, Integer> map = new HashMap<String, Integer>();
	Map<Integer, String> inv = new HashMap<Integer, String>();
	private S stat = S.INIT;
	private String name;
	private int id;

	String getSheet(String name) {
		return "sheet" + map.get(name) + ".xml";
	}

	private static HTMLParseException _geterr(int c, Object... a) {
		return new HTMLParseException(ErrorBundle.getDefault(c, a));
	}

	@Override
	public void string(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		switch(stat) {
		case INIT:
			if(s.equals("workbook"))  stat = S.WORKBOOK;
			break;
		case WORKBOOK:
			if(s.equals("sheets"))  stat = S.SHEETS;
			break;
		case SHEETS:
			if(s.equals("sheet")) {
				id = -1;
				stat = S.SHEET;
			}
			break;
		case SHEET:
			break;
		}
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		switch(stat) {
		case INIT:
			break;
		case WORKBOOK:
			if(s.equals("workbook"))  stat = S.INIT;
			break;
		case SHEETS:
			if(s.equals("sheets"))  stat = S.WORKBOOK;
			break;
		case SHEET:
			if(s.equals("sheet")) {
				if(id < 0) {
					throw _geterr(10046);
				}
				map.put(name, id);
				inv.put(id, name);
				stat = S.SHEETS;
			}
			break;
		}
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		if(stat != S.SHEET) {
			// do nothing
		} else if(k.equals("sheetId")) {
			try {
				id = Integer.parseInt(v);
			} catch(NumberFormatException e) {
				throw _geterr(10046);
			}
		} else if(k.equals("name")) {
			name = v;
		}
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		// do nothing
	}

}

class XlsxWriteTables extends XmlReplaceTag {

	private Map<String, String> map;
	private Collection<String> tnames;

	public XlsxWriteTables(PrintWriter w, Map<String, String> m,
			Collection<String> t) {
		super("sheets", w);
		tnames = t;
		map = m;
	}

	@Override
	public void insertFragment(PrintWriter w) {
		int j = 1;

		for(String s : tnames) {
			w.format("<sheet r:id=\"%s\" sheetId=\"%d\" name=\"%s\"/>",
					map.get(j + ""), j, s);
			j++;
		}
	}

}

class XlsxRelsMax implements HTMLHandler {

	private boolean tagr = false;
	private String rid = null;

	/**
	 * 
	 * @param ins
	 * @return
	 * @throws SQLException
	 * @throws IOException
	 */
	static String fromXlsx(
			Fichier f) throws SQLException, IOException {
		XlsxRelsMax h = new XlsxRelsMax();
		ZipInputStream zip = null;
		ZipEntry ent;

		try {
			zip = new ZipInputStream(f.openInputStream());
			while((ent = zip.getNextEntry()) != null) {
				if(ent.getName().equals(
						"xl/_rels/workbook.xml.rels")) {
					h = new XlsxRelsMax();
					HTMLParser.parse(h,
							new InputStreamReader(zip, "UTF-8"));
					return h.rid;
				}
			}
			throw ErrorBundle.getDefault(10046);
		} catch(HTMLParseException e) {
			throw (SQLException)e.getCause();
		} finally {
			if(zip != null)  zip.close();
		}
	}

	@Override
	public void string(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		tagr = s.equals("Relationship");
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		tagr = false;
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		if(tagr && k.equals("Id")) {
			rid = rid == null || rid.compareTo(v) < 0 ? v : rid;
		}
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		// do nothing
	}

}

class XlsxRelsSearch implements HTMLHandler {

	Map<String, String> vals = new HashMap<String, String>();
	private boolean tagr = false;
	private String tid, tval;

	/**
	 * 
	 * @param ins
	 * @return
	 * @throws SQLException
	 * @throws IOException
	 */
	static Map<String, String> fromXlsx(
			Fichier f) throws SQLException, IOException {
		XlsxRelsSearch h = new XlsxRelsSearch();
		ZipInputStream zip = null;
		ZipEntry ent;

		try {
			zip = new ZipInputStream(f.openInputStream());
			while((ent = zip.getNextEntry()) != null) {
				if(ent.getName().equals(
						"xl/_rels/workbook.xml.rels")) {
					h = new XlsxRelsSearch();
					HTMLParser.parse(h,
							new InputStreamReader(zip, "UTF-8"));
					return h.vals;
				}
			}
			throw ErrorBundle.getDefault(10046);
		} catch(HTMLParseException e) {
			throw (SQLException)e.getCause();
		} finally {
			if(zip != null)  zip.close();
		}
	}

	@Override
	public void string(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		tagr = s.equals("Relationship");
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		if(tagr)  vals.put(tval, tid);
		tagr = false;
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		if(tagr && k.equals("Id")) {
			tid = v;
		} else if(tagr && k.equals("Target")) {
			tval = v.replaceFirst("^worksheets/sheet", "");
			tval = tval.replaceFirst("\\.xml$", "");
		}
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		// do nothing
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		// do nothing
	}

}

class XlsxRelsAdd implements HTMLHandler {

	private PrintWriter wr;
	private boolean tagw, tagr = false;
	private String rid;
	private int num;

	XlsxRelsAdd(PrintWriter o, int n, String x) {
		wr = o;
		rid = x;
		num = n;
	}

	private void puttagw() {
		if(tagw) {
			wr.print('>');
			tagw = false;
		}
	}

	@Override
	public void string(String s) throws HTMLParseException {
		puttagw();
		wr.print(HTMLParser.escape(s));
	}

	@Override
	public void startTag(String s) throws HTMLParseException {
		puttagw();
		wr.format("<%s", HTMLParser.escape(s));
		tagr = tagr || s.equals("Relationships");
		tagw = true;
	}

	@Override
	public void endTag(String s) throws HTMLParseException {
		puttagw();
		if(tagr && s.equals("Relationships")) {
			wr.format("<Relationship Target=\"worksheets/sheet%d.xml\" " +
					"Type=\"http://schemas.openxmlformats.org/officeDocument" +
					"/2006/relationships/worksheet\" Id=\"%s\"/>",
					num, rid);
			tagr = false;
		}
		wr.format("</%s>", HTMLParser.escape(s));
	}

	@Override
	public void tagAttribute(String k,
			String v) throws HTMLParseException {
		wr.format(" %s=\"%s\"",
				HTMLParser.escape(k),
				HTMLParser.escape(v));
	}

	@Override
	public void meta(String s) throws HTMLParseException {
		wr.print(s);
	}

	@Override
	public void comment(String s) throws HTMLParseException {
		wr.format("<!--%s-->", s);
	}

}
