package com.sanpudo.formula;

import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.util.Hashtable;

/**
 * ユーザ定義関数を実装する際に拡張するためのアダプタクラス。
 * 
 * @author Sanpudo.
 */
public class FunctionAdaptor implements FunctionImplementation {

	private Hashtable<String, Integer> argNums;
	private Hashtable<String, Method> dMethods;
	private Hashtable<String, Method> bdMethods;

	/** コンストラクタ。 */
	protected FunctionAdaptor() {
		setCompatibleMethods();
	}

	/** 指定した名前の関数のdoubleの関数実行結果を返す。 
	 * @param name 関数名
	 * @param args 関数に渡す引数
	 * @return 関数実行結果
	 * @exception 関数実行時に評価できない状況が発生
	 * */
	public double value(String name, double[] args)
			throws FunctionEvalException {
		if (dMethods.containsKey(name)) {
			Object[] arg = new Double[args.length];
			for (int i = 0; i < arg.length; i++) {
				arg[i] = new Double(args[i]);
			}
			try {
				Double d = (Double) dMethods.get(name).invoke(this, arg);
				return d.doubleValue();
			} catch (IllegalAccessException e) {
				throw new FunctionException(e.getMessage(), name);
			} catch (IllegalArgumentException e) {
				throw new FunctionException(e.getMessage(), name);
			} catch (InvocationTargetException e) {
				if (e.getCause() instanceof FunctionEvalException) {
					throw (FunctionEvalException) (e.getCause());
				} else {
					throw new FunctionException(Messages.UNKOWN_EXCEPTION, e
							.getMessage());
				}
			}
		} else {
			throw new FunctionEvalException(Messages.UNSUPPORT_DOUBLE);
		}

	}

	/** 指定した名前の関数のBigDecimalの関数実行結果を返す。 
	 * @param name 関数名
	 * @param args 関数に渡す引数
	 * @return 関数実行結果
	 * @exception 関数実行時に評価できない状況が発生
	 * */
	public BigDecimal value(String name, BigDecimal[] args)
			throws FunctionEvalException {
		if (bdMethods.containsKey(name)) {
			try {
				Object[] arg = (Object[]) args;
				return (BigDecimal) bdMethods.get(name).invoke(this, arg);
			} catch (IllegalAccessException e) {
				throw new FunctionException(e.getMessage(), name);
			} catch (IllegalArgumentException e) {
				throw new FunctionException(e.getMessage(), name);
			} catch (InvocationTargetException e) {
				if (e.getCause() instanceof FunctionEvalException) {
					throw (FunctionEvalException) (e.getCause());
				} else {
					throw new FunctionException(Messages.UNKOWN_EXCEPTION, e
							.getMessage());
				}
			}
		} else {
			throw new FunctionEvalException(Messages.UNSUPPORT_DECIMAL);
		}

	}

	/** このクラスで定義されている関数名の配列を返す。 
	 * @return このクラスで定義されてる関数名の配列
	 * */
	public String[] names() {
		return argNums.keySet().toArray(new String[argNums.size()]);
	}

	/** 指定した名前の関数の引数の個数を返す。 
	 * @param name 関数名
	 * @return 関数の引数の個数
	 * */
	public int numberOfArgs(String name) {
		return argNums.get(name);
	}

	/** このクラスで定義されている関数名であるときtrueを返す。 
	 * @param name 関数名
	 * @return このクラスで定義されている関数である時true
	 * */
	public boolean supports(String name) {
		return argNums.containsKey(name);
	}

	/** 関数実装に適合するメソッドをリストアップする。 */
	private void setCompatibleMethods() {
		argNums = new Hashtable<String, Integer>();
		dMethods = new Hashtable<String, Method>();
		bdMethods = new Hashtable<String, Method>();

		Method[] methods = this.getClass().getMethods();

		// doubleに適合する関数をリストアップ
		for (int i = 0; i < methods.length; i++) {
			// 関数名が適合
			if (Lex.isValidFuncName(methods[i].getName())) {
				// 戻り値がdoubleであること
				if (methods[i].getReturnType().getName().equals("double")) {
					// 引数が全てdoubleであること
					Class<?>[] argClasses = methods[i].getParameterTypes();
					int j;
					for (j = 0; j < argClasses.length; j++) {
						if (!argClasses[j].getName().equals("double")) {
							break;
						}
					}
					if (j >= argClasses.length) {
						String name = methods[i].getName();
						// 同名の関数が登録済みでないこと
						if (dMethods.containsKey(name)) {
							throw new FunctionException(
									Messages.SAME_NAME_USED, name);
						}
						argNums.put(name, argClasses.length);
						dMethods.put(name, methods[i]);
					}
				}
			}
		}
		// BigDecimalに適合する関数をリストアップ
		for (int i = 0; i < methods.length; i++) {
			// 関数名が適合
			if (Lex.isValidFuncName(methods[i].getName())) {
				// 戻り値がBigDecimalであること
				if (methods[i].getReturnType().getName().equals(
						"java.math.BigDecimal")) {
					// 引数が全てBigDecimalであること
					Class<?>[] argClasses = methods[i].getParameterTypes();
					int j;
					for (j = 0; j < argClasses.length; j++) {
						if (!argClasses[j].getName().equals(
								"java.math.BigDecimal")) {
							break;
						}
					}
					if (j >= argClasses.length) {
						String name = methods[i].getName();
						// 同名の関数が登録済みでないこと
						if (bdMethods.containsKey(name)) {
							throw new FunctionException(
									Messages.SAME_NAME_USED, name);
						}
						// doubleも実装されている場合は、引数の数が等しい事
						if (argNums.containsKey(name)) {
							if (argNums.get(name) != argClasses.length) {
								throw new FunctionException(
										Messages.ARG_NO_CONFLICT, name);
							}
						}
						argNums.put(methods[i].getName(), argClasses.length);
						bdMethods.put(name, methods[i]);
					}
				}
			}
		}
	}

}
