/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: SymbolList.cpp,v 1.12 2005/10/28 07:23:34 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "SL_header.h"

#include "libsatellite.h"

using namespace std;

#define  __EXPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"        
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#undef   __EXPORTSYMBOL__

#define SYMLIST_CBFUNC_VAR    0
#define SYMLIST_CBFUNC_MODULE 1
#define SYMLIST_CBFUNC_SIZE   2


typedef struct _symbol_callback_node_t {
  symbol_callback_t  cb;
  void              *data;
} symbol_callback_node_t;

/* #define PRINT_BUILTIN_SYMBOLS 1 */ /* for debug */

static void _hrule(tty_console *con, char c);
static int  _symbol_set_name(symbol_t *sym, const char *name);
static void _symbol_delete_object(symbol_t *sym);
static int  _symbol_release_depend_sym(symbol_t *sym);
static int  _symbol_append_reference_sym(symbol_t *sym, symbol_t *parent);
static void _symbol_table_hash_free(void *void_sym, void *arg);
static void _symbol_table_hash_set_undef(void *void_sym, void *void_arg);
static void _symbol_table_symbol_print(void *void_sym, void *void_arg);

static int _symbol_callback_eqfunc(void *data1, void *data2);

static symlist_event_callback_t symlist_cb_funcs[SYMLIST_CBFUNC_SIZE] = {
  (symlist_event_callback_t)NULL, /* SYMLIST_CBFUNC_VAR    */
  (symlist_event_callback_t)NULL  /* SYMLIST_CBFUNC_MODULE */
};

/* do callback event handlers */
#define _symbol_do_callback_event_handlers(event, sym, type) { \
  if((type) == SYMBOL_TYPE_VAR && symlist_cb_funcs[SYMLIST_CBFUNC_VAR]){ \
    symlist_cb_funcs[SYMLIST_CBFUNC_VAR]((event), (sym)); \
  } \
  if((type) == SYMBOL_TYPE_MODULE && \
    symlist_cb_funcs[SYMLIST_CBFUNC_MODULE]){ \
    symlist_cb_funcs[SYMLIST_CBFUNC_MODULE]((event), (sym)); \
  } \
}

static int _symbol_set_name(symbol_t *sym, const char *name){
  if(sym->name) return -1;
  sym->name = strdup(name);
  if(sym->name == NULL) return -1;
  return 0;
}

DLLEXPORT int symbol_undef(symbol_t *sym){
  symbol_callback_node_t *node;
  int pre_type,i;
  if(sym == NULL) return -1;
  /* do delete callback event and remove callback events */
  if(sym->cblist){
    while(sl4_list_empty(sym->cblist) == 0){
      node = (symbol_callback_node_t*)sl4_list_top(sym->cblist);
      node->cb(SYMBOL_EVENT_DELETE, sym, node->data);
      free(node);
      sl4_list_pop_front(sym->cblist);
    }
    sl4_list_delete(sym->cblist);
    sym->cblist = NULL;
  }
  /* do delete callback event handler */
  pre_type = symbol_get_type(sym);
  _symbol_do_callback_event_handlers(SYMLIST_EVENT_DELETE, sym, pre_type);

  /* remove mysymbol pointer from referenced symbol */
  if(sym->ref_cnt != 0){
    for(i=0;i<sym->ref_cnt;i++)
      sym->ref[i]->dep = NULL;
    free(sym->ref);
    sym->ref = NULL;
    sym->ref_cnt = 0;
  }
  /* release depending symbol */
  _symbol_release_depend_sym(sym);
  /* delete object */
  _symbol_delete_object(sym);
  sym->type = SYMBOL_TYPE_UNDEF;
  return 0;
}

DLLEXPORT symbol_t *symbol_new(const char *name, SL_Object *obj, int type){
  symbol_t *sym;
  if(name == NULL || *name=='\0') return NULL;
  sym = (symbol_t*)malloc(sizeof(symbol_t));
  if(sym == NULL) return NULL;
  sym->type    = type;
  sym->name    = NULL;
  sym->ref     = NULL;
  sym->dep     = NULL;
  sym->ref_cnt = 0;
  sym->cblist  = NULL;
  if(_symbol_set_name(sym, name) == -1){ free(sym); return NULL; }
  sym->object = obj;
  if(sym->object != NULL){
    sym->object->obj_ref();
  }
  return sym;
}

DLLEXPORT int symbol_get_type(symbol_t *sym){
  if(sym->dep != NULL) return symbol_get_type(sym->dep);
  return sym->type;
}

DLLEXPORT SL_Object *symbol_get_object(symbol_t *sym){
  if(sym->dep != NULL) return symbol_get_object(sym->dep);
  return sym->object;
}

DLLEXPORT SL_Object *symbol_get_single_object(symbol_t *sym){
  SL_Object *obj;
  int type;
  obj = symbol_get_object(sym);
  type = symbol_get_type(sym);
  if(obj == NULL) return NULL;
  obj->obj_unref();
  if(!obj->empty()){
    obj->obj_ref();
    obj = obj->obj_dup();
    if(obj == NULL)
      return NULL;
    symbol_set_object(sym, obj, type);
  }else{
    obj->obj_ref();
  }
  return obj;
}

static void _symbol_delete_object(symbol_t *sym){
  if(sym->object){
    sym->object->obj_unref();
    if(sym->object->empty()) delete sym->object;
    sym->object = NULL;
  }
}

static int _symbol_release_depend_sym(symbol_t *sym){
  int i,j;
  symbol_t  *dep;
  symbol_t **tmp;
  if(sym->dep != NULL){
    dep = sym->dep;
    if(dep->ref_cnt-1 != 0){
      tmp = (symbol_t**)malloc(sizeof(symbol_t*)*(dep->ref_cnt-1));
      if(tmp == NULL){ return -1; }
      for(i=0,j=0;i<dep->ref_cnt;i++)
        if(dep->ref[i] != sym)
          tmp[j++] = dep->ref[i];
      free(dep->ref);
      dep->ref = tmp;
    }else{
      free(dep->ref);
      dep->ref = NULL;
    }
    dep->ref_cnt--;
    sym->dep = NULL;
  }
  return 0;
}


DLLEXPORT void symbol_delete(symbol_t *sym){
  /* undef symbol */
  if(symbol_undef(sym) != 0) return;
  /* free name */
  if(sym->name) free(sym->name);
  free(sym);
}

DLLEXPORT int symbol_set_object(symbol_t *sym, SL_Object *obj, int type){
  int pre_type;

  if(sym->dep != NULL)
    return symbol_set_object(sym->dep, obj, type);

  /* do callback delete event handler */
  pre_type = symbol_get_type(sym);
  if(pre_type != type)
    _symbol_do_callback_event_handlers(SYMLIST_EVENT_DELETE, sym, pre_type);

  sym->type = type;
  if(sym->object == obj) return 0;
  _symbol_delete_object(sym);
  sym->object = obj;
  if(sym->object != NULL)
    sym->object->obj_ref();

  /* do callback add event handler */
  if(type != pre_type)
    _symbol_do_callback_event_handlers(SYMLIST_EVENT_ADD, sym, type);

  /* do update event */
  symbol_do_event(sym, SYMBOL_EVENT_UPDATE);
  return 0;
}

static int _symbol_append_reference_sym(symbol_t *sym, symbol_t *parent){
  symbol_t **tmp;
  int i;
  tmp = (symbol_t**)malloc(sizeof(symbol_t*)*(parent->ref_cnt+1));
  if(tmp == NULL) return -1;
  for(i=0;i<parent->ref_cnt;i++)
    tmp[i] = parent->ref[i];
  tmp[i]=sym;
  if(parent->ref_cnt!=0) free(parent->ref);
  parent->ref = tmp;
  parent->ref_cnt++;
  sym->dep = parent;
  return 0;
}

DLLEXPORT int symbol_set_depend(symbol_t *sym, symbol_t *dep){
  if(sym == dep) return 0;
  /* release already depended symbol */
  if(_symbol_release_depend_sym(sym) != 0) return -1;
  /* delete already setted object */
  _symbol_delete_object(sym);
  /* append reference symbol */
  if(_symbol_append_reference_sym(sym, dep) != 0) return -1;
  return 0;
}

DLLEXPORT symbol_table_t *symbol_table_new(){
  symbol_table_t *symtab;
  symtab = (symbol_table_t*)malloc(sizeof(symbol_table_t));
  if(symtab == NULL) return NULL;
  symtab->table = hash_table_str_new();
  if(symtab->table == NULL){
    free(symtab);
    return NULL;
  }
  return symtab;
}


static void _symbol_table_hash_free(void *void_sym, void *arg){
  symbol_t *sym = (symbol_t*)void_sym;
  symbol_delete(sym);
}

DLLEXPORT void symbol_table_delete(symbol_table_t *symtab){
  hash_table_delete(symtab->table, _symbol_table_hash_free, NULL);
  free(symtab);
}

DLLEXPORT symbol_t *symbol_table_lookup(symbol_table_t *symtab,
                                        const char *name){
  return (symbol_t*)hash_table_lookup(symtab->table, name);
}

DLLEXPORT int symbol_table_install(symbol_table_t *symtab, symbol_t *sym){
  int status;
  status = hash_table_insert(symtab->table, sym->name, sym);
  /* do callback add event handler */
  if(status == 0){
    int type = symbol_get_type(sym);
    _symbol_do_callback_event_handlers(SYMLIST_EVENT_ADD, sym, type);
  }
  return status;
}

DLLEXPORT int symbol_table_move(symbol_table_t *dstst, symbol_table_t *srcst,
                                symbol_t *sym){
  symbol_t *msym;
  msym = (symbol_t*)hash_table_remove(srcst->table, sym->name);
  if(msym != sym) return -1;
  if(hash_table_insert(dstst->table, sym->name, sym) != 0)
    return -1;
  return 0;
}

static void _symbol_table_hash_set_undef(void *void_sym, void *void_arg){
  symbol_t *sym      = (symbol_t*)void_sym;
  int      *var_only = (int*)void_arg;
  if( *var_only == 0 || symbol_get_type(sym) == SYMBOL_TYPE_VAR )
    symbol_undef(sym);
}

DLLEXPORT void symbol_table_set_undef(symbol_table_t *symtab){
  int var_only = 0;
  hash_table_foreach(symtab->table, _symbol_table_hash_set_undef, &var_only);
}

DLLEXPORT int symbol_table_undef(symbol_table_t *symtab, const char *name){
  symbol_t *sym;
  sym = (symbol_t*)hash_table_lookup(symtab->table, name);
  return symbol_undef(sym);
}

DLLEXPORT int symbol_table_undefall(symbol_table_t *symtab){
  int var_only = 1;
  hash_table_foreach(symtab->table, _symbol_table_hash_set_undef, &var_only);
  return 0;
}

typedef struct _table_listup_args_t {
  tty_console *con;
  SL_OBJ::TYPE  obj_type;
  int           cls_type;
  const char   *name;
  int flag;
  int mode;
} table_listup_args_t;

static void _symbol_table_symbol_print(void *void_sym, void *void_arg){
  static const char *excepts[]={"LittleEndian","BigEndian","+1",NULL};
  symbol_t            *sym = (symbol_t*) void_sym;
  table_listup_args_t *arg = (table_listup_args_t*) void_arg;
  int          i;
  Base_Buffer *buf;
  SL_Object   *obj;
  switch(arg->mode){
  case 0: /* print variable names */
    if(symbol_get_type(sym) == SYMBOL_TYPE_VAR){
      obj = symbol_get_object(sym);
      if(obj != NULL && obj->TypeofOBJ() == arg->obj_type){
        if(arg->flag == 0){
          arg->con->tty_printf("Defined %s objects:\n", arg->name);
          _hrule(arg->con, '-');
          arg->flag = 1;
        }
        arg->con->tty_printf("%s", sym->name);
        buf = obj->GetBufferPointer();
        if(buf->GetDim() > 0){
          if(!(buf->GetDim() == 1 && buf->GetIndex(0) == 1))
              for(i = 0; i < buf->GetDim(); i++)
                arg->con->tty_printf("[%d]", buf->GetIndex(i));
        }
        arg->con->tty_printf("  ");
      }
    }
    break;
  case 1: /* print func/proc names */
    if(symbol_get_type(sym) == arg->cls_type){
      if(arg->flag == 0){
        arg->con->tty_printf("Defined %s:\n", arg->name);
        _hrule(arg->con, '-');
        arg->flag = 1;
      }
      arg->con->tty_printf("%s  ", sym->name);
    }
    break;
  case 2:  /* print constants */
    if(symbol_get_type(sym) == SYMBOL_TYPE_CONST){
      for(i = 0; excepts[i] != NULL; i++){
        if(!strcmp(sym->name, excepts[i])) break;
      }
      if(excepts[i] == 0){
        if(arg->flag == 0){
          arg->con->tty_printf("Defined constants:\n");
          _hrule(arg->con, '-');
          arg->flag = 1;
        }
        arg->con->tty_printf("%s  ", sym->name);
      }
    }
    break;
  case 3:  /* print modules */
    if(symbol_get_type(sym) == SYMBOL_TYPE_MODULE){
      if(arg->flag == 0){
        arg->con->tty_printf("Defined command modules:\n");
        _hrule(arg->con, '-');
        arg->flag = 1;
      }
      arg->con->tty_printf("%s  ", sym->name);
    }
    break;
  case 4:  /* print buildin objects */
    if(symbol_get_type(sym) == SYMBOL_TYPE_BLTIN || 
       symbol_get_type(sym) == SYMBOL_TYPE_OPCODE){
      if(arg->flag == 0){
        arg->con->tty_printf("Builtin symbols:\n");
        _hrule(arg->con, '-');
        arg->flag = 1;
      }
      arg->con->tty_printf("%s  ", sym->name);
    }
    break;
  }
}

DLLEXPORT void symbol_table_listup(symbol_table_t *symtab, tty_console *con,
                                   int is_global){
  static struct {SL_OBJ::TYPE type; const char *name; } types[] = {
    {SL_OBJ::SCALAR_O,   "scalar"},   {SL_OBJ::SERIES_O, "series"},
    {SL_OBJ::SNAPSHOT_O, "snapshot"}, {SL_OBJ::STRING_O, "string"},
    {SL_OBJ::UNDEF_O, NULL}
  };
  /*
  static struct {int type; const char *name; } classes[] = {
    {SYMBOL_TYPE_FUNC, "functions"}, {SYMBOL_TYPE_PROC, "procedures"}, 
    {0, NULL}
  };
  */
  table_listup_args_t arg;
  int i;
  arg.con     = con;
  arg.flag    = 0;
  if(is_global == 0){
    /* variable */
    arg.mode = 0;
    for(i=0; types[i].name != NULL; i++){
      arg.obj_type = types[i].type;   arg.name = types[i].name;
      hash_table_foreach(symtab->table, _symbol_table_symbol_print, &arg);
      if(arg.flag == 1){ arg.flag = 0; con->tty_printf("\n\n"); }
    }
  }else{
    /* func/proc */
    /*
    arg.mode = 1;
    for(i=0; classes[i].name != NULL; i++){
      arg.cls_type = classes[i].type; arg.name = classes[i].name;
      hash_table_foreach(symtab->table, _symbol_table_symbol_print, &arg);
      if(arg.flag == 1){ arg.flag = 0; con->tty_printf("\n\n"); }
    }
    */
    /* constants */
    arg.mode = 2;
    hash_table_foreach(symtab->table, _symbol_table_symbol_print, &arg);
    if(arg.flag == 1){ arg.flag = 0; con->tty_printf("\n\n"); }
    /* modules */
    arg.mode = 3;
    hash_table_foreach(symtab->table, _symbol_table_symbol_print, &arg);
    if(arg.flag == 1){ arg.flag = 0; con->tty_printf("\n\n"); }
    /* bultin symbols */
#ifdef PRINT_BUILTIN_SYMBOLS
    arg.mode = 4;
    hash_table_foreach(symtab->table, _symbol_table_symbol_print, &arg);
    if(arg.flag == 1){ arg.flag = 0; con->tty_printf("\n\n"); }
#endif
  }
}

static void _hrule(tty_console *con, char c){
  int y,x,i;
  con->term_getmaxyx(&y,&x);
  for(i=0;i<x;i++)
    con->tty_putc((int)c);
}


DLLEXPORT void symbol_do_event(symbol_t *sym, int event){
  sl4_list_iterator_t it;
  symbol_callback_node_t *node;
  if(sym->cblist == NULL) return;
  sl4_list_begin(sym->cblist, &it);
  while(sl4_list_it_is_end(&it) == 0){
    node = (symbol_callback_node_t*)sl4_list_it_data(&it);
    node->cb(event, sym, node->data);
    sl4_list_it_next(&it);
  }
}

DLLEXPORT int  symbol_add_callback(symbol_t *sym, symbol_callback_t cb,
                                   void *data){
  symbol_callback_node_t *node;
  if(sym->cblist == NULL){
    sym->cblist = sl4_list_new();
    if(sym->cblist == NULL) return -1;
  }
  node = (symbol_callback_node_t*)malloc(sizeof(symbol_callback_node_t));
  if(node == NULL){
    if(sl4_list_empty(sym->cblist)){ free(sym->cblist); sym->cblist = NULL; }
    return -1;
  }
  node->cb   = cb;
  node->data = data;
  if(sl4_list_push_back(sym->cblist, node) != 0){
    free(node);
    if(sl4_list_empty(sym->cblist)){ 
      sl4_list_delete(sym->cblist);
      sym->cblist = NULL;
    }
    return -1;
  }
  return 0;
}

static int _symbol_callback_eqfunc(void *data1, void *data2){
  symbol_callback_node_t *node1,*node2;
  node1 = (symbol_callback_node_t*)data1;
  node2 = (symbol_callback_node_t*)data2;
  if(node1->cb == node2->cb && node1->data == node2->data)
    return 1;
  return 0;
}

DLLEXPORT int  symbol_remove_callback(symbol_t *sym, symbol_callback_t cb,
                                      void *data){
  sl4_list_iterator_t it;
  symbol_callback_node_t node;
  if(sym->cblist == NULL)
    return -1;
  node.cb   = cb;
  node.data = data;
  if(sl4_list_find(sym->cblist, &node, &it, _symbol_callback_eqfunc) != 0)
    return -1;
  if(sl4_list_erase(sym->cblist, &it) != 0)
    return -1;
  if(sl4_list_empty(sym->cblist)){
    sl4_list_delete(sym->cblist);
    sym->cblist = NULL;
  }
  return 0;
}

/* set callback event handlers */
DLLEXPORT void symlist_set_var_callback(symlist_event_callback_t cb){
  symlist_cb_funcs[SYMLIST_CBFUNC_VAR] = cb;
}

DLLEXPORT void symlist_set_module_callback(symlist_event_callback_t cb){
  symlist_cb_funcs[SYMLIST_CBFUNC_MODULE] = cb;
}

