/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: SystemCommon.cpp,v 1.7 2005/10/29 09:21:04 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif

#include "SL_header.h"
#define __SYSTEMCOMMON_MAIN__
#define __SYSTEMFLAG_MAIN__
#include <cstdarg>

#define  __EXPORTSYMBOL__
#include "libsatellite.h"
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __EXPORTSYMBOL__
#include "mathfunc.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#define VALIDDATAOBJ(x) ( (x) != 0 && (x)->GetBufferPointer() != 0 && \
                          (x)->GetBufferPointer()->GetDataPointer() != 0 )

using namespace std;

SystemCommon::SystemCommon(tty_console *con):  console(con) {
  narg=0;
  argv=0;
  ret=0;
  syscom=this;
  compat2x=false;
  gl_symtab = symbol_table_new();
  lo_symtab = symbol_table_new();
  cur_symtab = lo_symtab;
}

SystemCommon::~SystemCommon(){
  if(ret!=NULL) delete ret;
  symbol_table_delete(gl_symtab);
  symbol_table_delete(lo_symtab);
  free_syscom();
}

void SystemCommon::ComCallInit(int oargc,SL_Object **oargv, 
                               symbol_t **oargv_sym){
  narg=oargc;
  argv=oargv;
  argv_sym=oargv_sym;
  ret=0;
}

SL_Object *SystemCommon::ComCallQuit(){
  SL_Object *tmp=ret;
  while(mem_struct.size()!=0){
    mem_it=mem_struct.begin();
    delete [] (char*)(*mem_it).ptr;
    mem_struct.erase(mem_it);
  }
  ret=0;
  return tmp;
}

symbol_t *SystemCommon::get_symbol(int n){
  if(n<narg){
    return argv_sym[n];
  }else
    return 0;
}

char *SystemCommon::SCsl4_getstr(char *buf, int siz){
  return console->tty_gets(buf,siz);
}

char *SystemCommon::SCGetArgType(int n){
  if(n<narg){
    return argv[n]->TypeName();
  }else
    return 0;
}

double SystemCommon::SCGetScalar(int n){
  double val;
  val=0.0;
  if(n<narg){
    if(VALIDDATAOBJ(argv[n]) &&
       (argv[n]->TypeofOBJ()==SL_OBJ::SNAPSHOT_O || 
	argv[n]->TypeofOBJ()==SL_OBJ::SERIES_O  ||
	argv[n]->TypeofOBJ()==SL_OBJ::SCALAR_O )){
      val=((double*)argv[n]->GetBufferPointer()->GetDataPointer())[0];
    }
  }
  return val;
}

string *SystemCommon::SCGetStringMem(int n, int *dim, int *index){
  int i, d;
  string *ret = NULL;
  if(n<narg){
    if(VALIDDATAOBJ(argv[n]) && argv[n]->TypeofOBJ()==SL_OBJ::STRING_O){
      d = argv[n]->GetBufferPointer()->GetDim();
      if(dim) *dim = d;
      if(index){
        for(i=0; i<d; i++){
          index[i]=argv[n]->GetBufferPointer()->GetIndex(i);
        }
      }
      ret = (string*)argv[n]->GetBufferPointer()->GetDataPointer();
    }
  }
  return ret;
}

char *SystemCommon::SCGetString(int n){
  string *str;
  char *ret=NULL;
  str = SCGetStringMem(n, NULL, NULL);
  if(str != NULL){
    string::size_type len;
    len = str[0].length();
    ret=(char*)SCmalloc(len+1);
    if(ret!=0){
      if(len==0)
        ret[0]='\0';
      else
        strcpy(ret,str[0].c_str());
    }
  }
  return ret;
}

double *SystemCommon::SCGetSeriesMem(int n, int *dim, int *index){
  int i, d;
  double *ret = NULL;
  if(n<narg){
    if(VALIDDATAOBJ(argv[n]) &&
       (argv[n]->TypeofOBJ()==SL_OBJ::SERIES_O || 
	argv[n]->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
      d = argv[n]->GetBufferPointer()->GetDim();
      if(dim) *dim = d;
      if(index){
        for(i=0; i<d; i++){
          index[i] = argv[n]->GetBufferPointer()->GetIndex(i);
        }
      }
      ret=(double*)argv[n]->GetBufferPointer()->GetDataPointer();
    }
  }
  return ret;
}

double *SystemCommon::SCGetSeries(int n, int *dim, int *index){
  int size;
  double *val,*ret;
  val = SCGetSeriesMem(n, dim, index);
  ret = NULL;
  if(val != NULL){
    size=IndexSize(*dim,index)*sizeof(double);
    ret=(double*)SCmalloc(size);
    if(ret!=0)memcpy(ret,val,size);
  }
  return ret;
}

void SystemCommon::SCReturnSeries(double *buf,int dim,int *index){
  static char *where="ReturnSeries():";
  int i, stat=0;
  SL_OBJ::TYPE type=SL_OBJ::SERIES_O;
  Base_Buffer *ret_buf=0;
  Index idx;
  if(dim<=0 || dim> MAX_INDEX){
    console->tty_printf("%s %s",where,"Illegal demension\n");
    stat=1;
  }
  if(index==0){
    console->tty_printf("%s %s",where,"Illegal index array\n");
    stat=1;
  } else{
    for(i=0;i<dim;i++){
      if(index[i]<0){
	console->tty_printf("%s %s",where,"Illegal index\n");
	stat=1;
	break;
      }
    }
  }
  if(stat==0){
    idx.SetIndexInfo(dim,index);
    if(dim==0) idx.SetIndex(0,1);
    try{
      if(dim==0){ type=SL_OBJ::SCALAR_O; ret_buf=new Scalar_Buffer; }
      else{ ret_buf=new Series_Buffer; }
      ret_buf->CopyIndex(idx);
      ret_buf->InitBuffer();
    }catch(bad_alloc){stat=1;}
    catch(buffer_exception){stat=1;}
    catch(execerr_exception){stat=1;}
    if(stat==0){
      if(ret!=0)ret->obj_unref();
      memcpy(ret_buf->GetDataPointer(),buf,idx.IndexSize()*sizeof(double));
      try{ ret=new SL_Object(type,ret_buf); }
      catch(bad_alloc){}
      catch(buffer_exception){}
      catch(execerr_exception){}
    }
  }
}

void SystemCommon::SCReturnSnapshot(double *buf,int dim,int *index){
  static char *where="ReturnSnapshot():";
  int i, stat=0;
  SL_OBJ::TYPE type=SL_OBJ::SNAPSHOT_O;
  Base_Buffer *ret_buf=0;
  Index idx;
  if(dim<=0 || dim> MAX_INDEX){
    console->tty_printf("%s %s",where,"Illegal demension\n");
    stat=1;
  }
  if(index==0){
    console->tty_printf("%s %s",where,"Illegal index array\n");
    stat=1;
  } else{
    for(i=0;i<dim;i++){
      if(index[i]<0){
	console->tty_printf("%s %s",where,"Illegal index\n");
	stat=1;
	break;
      }
    }
  }
  if(stat==0){
    idx.SetIndexInfo(dim,index);
    if(dim==0) idx.SetIndex(0,1);
    try{
      if(dim==0){ type=SL_OBJ::SCALAR_O; ret_buf=new Scalar_Buffer; }
      else	{ ret_buf=new Series_Buffer; }
      ret_buf->CopyIndex(idx);
      ret_buf->InitBuffer();
    }catch(bad_alloc){stat=1;}
    catch(buffer_exception){stat=1;}
    catch(execerr_exception){stat=1;}
    if(stat==0){
      if(ret!=0)ret->obj_unref();
      memcpy(ret_buf->GetDataPointer(),buf,idx.IndexSize()*sizeof(double));
      try{
	ret=new SL_Object(type,ret_buf);
      }catch(bad_alloc){}
      catch(buffer_exception){}
      catch(execerr_exception){}
    }
  }
}

void SystemCommon::SCReturnScalar(double sc){
  Scalar_Buffer *buf=0;
  if(ret!=0){ delete ret; }
  buf=new Scalar_Buffer;
  try{
    buf->InitBuffer();
    buf->SetScalar(sc);
    ret=new SL_Object(SL_OBJ::SCALAR_O,buf);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
}

void SystemCommon::SCReturnString(const char *str){
  String_Buffer *buf=0;
  Index idx;
  if(ret!=0){
    delete ret;
  }
  buf=new String_Buffer;
  buf->SetDim(1);
  buf->SetIndex(0,1);
  try{
    buf->InitBuffer();
    ret=new SL_Object(SL_OBJ::STRING_O,buf);
    idx.SetIndex(0,0);
    ((String_Buffer *)ret->GetBufferPointer())->SetString(idx,str);
  }catch(execerr_exception){
    delete buf;
    ret=0;
    throw;
  }catch(bad_alloc){
    delete buf;
    ret=0;
    throw;
  }
}

void SystemCommon::sl4_syscom_return_string(int dim, int *idx, 
                                            sl4_list_t *lst){
  int i;
  sl4_list_iterator_t it;
  sl4_string_t *str;
  const char *cstr;
  String_Buffer *buf=0;
  Index ptr;
  if(ret!=0){
    delete ret;
  }
  buf = new String_Buffer;
  buf->SetDim(dim);
  try{
    for(i = 0; i < dim; i++)
      buf->SetIndex(i,idx[i]);
    buf->InitBuffer();
    ret = new SL_Object(SL_OBJ::STRING_O,buf);
    if(sl4_list_size(lst) == (unsigned int)buf->IndexSize()) {
      sl4_list_begin(lst, &it);
      for(i = 0; i < buf->IndexSize(); i++){
        ptr = buf->rIndex(i);
        str = (sl4_string_t*)sl4_list_it_data(&it);
        cstr = sl4_string_get_cstr(str);
        ((String_Buffer *)ret->GetBufferPointer())->SetString(ptr, cstr);
        sl4_list_it_next(&it);
      }
    }
  }catch(execerr_exception){
    delete buf;
    ret=0;
    throw;
  }catch(bad_alloc){
    delete buf;
    ret=0;
    throw;
  }
}

sl4_list_t *SystemCommon::sl4_syscom_get_string(int n, int *dim, 
                                                  int *index) {
  int i, d, size;
  sl4_list_t *lst;
  string *str = NULL;
  sl4_string_t *tmp = NULL;
  lst = sl4_list_new();
  if(lst == NULL)
    return NULL;
  if(n<narg){
    if(VALIDDATAOBJ(argv[n]) && argv[n]->TypeofOBJ()==SL_OBJ::STRING_O){
      d = argv[n]->GetBufferPointer()->GetDim();
      *dim = d;
      for(i=0; i<d; i++){
        index[i]=argv[n]->GetBufferPointer()->GetIndex(i);
      }
      size = IndexSize(*dim, index);
      str = (string*)argv[n]->GetBufferPointer()->GetDataPointer();
      for(i = 0; i<size; i++){
        tmp = sl4_string_new_cstr(str[i].c_str());
        sl4_list_push_back(lst, tmp);
      }
    }
  }
  return lst;
}

int   SystemCommon::SCGetBufferID(int n){
  if(n-1<=narg && argv[n] != 0) 
    return n;
  return -1;
}

SL_Object *SystemCommon::GetBufferFromID(int n){
  if(narg>n && n>=0){
    if(argv[n])
      return argv[n];
    return 0;
  }else{
    return 0;
  }
}

// int SystemCommon::SCDestroyBuffer(int n){
//   SL_Object *obj;
//   obj=GetBufferFromID(n);
//   if(obj==0)return -1;
//   obj->obj_unref();
//   return (int)obj->empty();
// }

Buffer *SystemCommon::SCAllocBuffer(unsigned  buf_size)
{
  double *ptr;
  ErrorNO = SUCCESS;
  ptr=(double*)SCmalloc(sizeof(double)*buf_size);
  if(ptr==0)ErrorNO = OUT_MEM;
  return ptr;
}

Buffer *SystemCommon::SCCAllocBuffer(unsigned buf_size){
  ErrorNO = SUCCESS;
  Buffer *buf = SCAllocBuffer(buf_size);
  if(buf!=0)
    memset(buf,0,buf_size*sizeof(double));
  return buf;
}

int SystemCommon::SCFreeBuffer(Buffer *area){
  ErrorNO = SUCCESS;
  if (area != 0) SCfree(area);
  return 0;
}

int SystemCommon::SCInitBuffer(int n,int dim,int *index){
  SL_Object *obj;
  Index idx;
  ErrorNO = SUCCESS;
  if (dim < 1 ) {
    ErrorNO = ILL_DIM;
    return(-1);
  }
  if ( index == 0) {
    ErrorNO = ILL_INDEX;
    return (-1);
  }
  obj=GetBufferFromID(n);
  if(obj==0){
    ErrorNO = OPEN_ER;
    return (-1);
  }
  return (int)obj->Init(SL_OBJ::SERIES_O,idx);
}

Buffer *SystemCommon::SCReadBuffer(int n,int *dim,int *index){
  SL_Object  *obj;
  Base_Buffer *buf;
  Buffer *ret;
  ErrorNO = SUCCESS;
  if (index == 0) {
    ErrorNO = ILL_INDEX;
    return 0;
  }
  obj=GetBufferFromID(n);
  if(obj==0 &&
     (obj->TypeofOBJ()==SL_OBJ::SERIES_O ||
      obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O) &&
     obj->GetBufferPointer()==0 ){
    ErrorNO = OPEN_ER;
    return 0;
  }
  buf=obj->GetBufferPointer();
  buf->GetIndexInfo(dim,index);
  ret=SCAllocBuffer(buf->IndexSize());
  memcpy(ret,buf->GetDataPointer(),buf->IndexSize()*sizeof(double));
  return ret;
}

int SystemCommon::SCWriteBuffer(int n,int dim,int *index,Buffer *area){
  SL_Object *obj;
  Base_Buffer *buf;
  Index idx;
  ErrorNO = SUCCESS;
  if (index == 0) {
    ErrorNO = ILL_INDEX;
    return (-1);
  }
  obj=GetBufferFromID(n);
  if(obj==0){
    ErrorNO = OPEN_ER;
    return (-1);
  }
  idx.SetIndexInfo(dim,index);
  obj->Init(SL_OBJ::SERIES_O, idx);
  buf=obj->GetBufferPointer();
  memcpy(buf->GetDataPointer(),area,buf->IndexSize()*sizeof(Buffer));
  return buf->IndexSize();
}

int SystemCommon::SCAppendBuffer(int n,int sub_dim,int *sub_index,
			       Buffer *area){
  static char *where="(AppendBuffer)";
  SL_Object *obj=0;
  ErrorNO = SUCCESS;
  Series_Buffer *sub=0;
  Base_Buffer *buf=0;
  int stat;
  stat=0; //for compiler
  obj=GetBufferFromID(n);
  if(obj==0 && obj->GetBufferPointer()==0 &&
     !(obj->TypeofOBJ()==SL_OBJ::SERIES_O ||
       obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
    ErrorNO = OPEN_ER;
    return (-1);
  }
  buf=obj->GetBufferPointer();
  sub=new Series_Buffer;
  sub->SetIndexInfo(sub_dim,sub_index);
  try{
    sub->InitBuffer();
    memcpy(buf->GetDataPointer(),area,buf->IndexSize()*sizeof(double));
    stat=((Series_Buffer *)buf)->AppendBuffer(sub);
  }catch(buffer_exception err){
    delete sub;
    console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete sub;
    throw;
  }
  delete sub;
  return stat;
}

Buffer *SystemCommon::SCReadSubBuffer(int n,int t,int *sub_dim,int *sub_index){
  static char *where="(ReadSubBuffer)";
  SL_Object *obj;
  Series_Buffer *buf,*ret;
  Buffer  *area;
  ErrorNO = SUCCESS;
  ret=0; // for compiler
  obj=GetBufferFromID(n);
  if(obj==0 && obj->GetBufferPointer()==0 &&
     !(obj->TypeofOBJ()==SL_OBJ::SERIES_O || 
       obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
    ErrorNO = OPEN_ER;
    return 0;
  }
  buf=(Series_Buffer *)obj->GetBufferPointer();
  try{
    ret=buf->GetSubBuffer(t);
  }catch(buffer_exception err){
    console->execerror(err.what(),where);
  }
  ret->GetIndexInfo(sub_dim,sub_index);
  area=SCAllocBuffer(ret->IndexSize());
  memcpy(area,ret->GetDataPointer(),ret->IndexSize()*sizeof(double));
  delete ret;
  return area;
}

int SystemCommon::SCWriteSubBuffer(int n,int t,int sub_dim,int *sub_index,
				 Buffer *area){
  static char *where="(WriteSubBuffer)";
  SL_Object *obj=0;
  Series_Buffer *sub=0;
  int stat=0;
  ErrorNO = SUCCESS;
  obj=GetBufferFromID(n);
  if(obj==0 && obj->GetBufferPointer()==0 &&
     !(obj->TypeofOBJ()==SL_OBJ::SERIES_O || 
       obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
    ErrorNO = OPEN_ER;
    return 0;
  }
  sub=new Series_Buffer;
  sub->SetIndexInfo(sub_dim,sub_index);
  try{
    sub->InitBuffer();
    memcpy(sub->GetDataPointer(),area,sub->IndexSize()*sizeof(double));
    stat=((Series_Buffer *)obj->GetBufferPointer())->PutSubBuffer(t,sub);
  }catch(buffer_exception err){
    delete sub;
    console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete sub;
    throw;
  }
  delete sub;
  return stat;
}

Buffer *SystemCommon::SCReadTimeSeries(int n,int sub_dim,int *sub_index,
				     int *length){
  static char *where="(ReadTimeSeries)";
  Series_Buffer *buf=0;
  Buffer        *area=0;
  Index idx;
  SL_Object *obj=0;
  ErrorNO = SUCCESS;
  obj=GetBufferFromID(n);
  if(obj==0 && obj->GetBufferPointer()==0 &&
     !(obj->TypeofOBJ()==SL_OBJ::SERIES_O ||
       obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
    ErrorNO = OPEN_ER;
    return 0;
  }
  idx.SetIndexInfo(sub_dim,sub_index);
  try{
    buf=((Series_Buffer *)obj->GetBufferPointer())->GetTimeSeries(idx);
  }catch(buffer_exception err){
    console->execerror(err.what(),where);
  }
  try{
    area=SCAllocBuffer(buf->IndexSize());
    memcpy(area,buf->GetDataPointer(),buf->IndexSize()*sizeof(double));
    *length=buf->GetIndex(0);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  delete buf;
  return area;
}

int SystemCommon::SCWriteTimeSeries(int n,int sub_dim,int *sub_index,
				  Buffer *area,int length){
  static char *where="(WriteTimeSeries)";
  Series_Buffer *buf=0;
  SL_Object *obj=0;
  Index idx;
  int stat=0;
  ErrorNO = SUCCESS;
  obj=GetBufferFromID(n);
  if(obj==0 && obj->GetBufferPointer()==0 &&
     !(obj->TypeofOBJ()==SL_OBJ::SERIES_O ||
       obj->TypeofOBJ()==SL_OBJ::SNAPSHOT_O)){
    ErrorNO = OPEN_ER;
    return 0;
  }
  idx.SetIndexInfo(sub_dim,sub_index);
  buf=new Series_Buffer;
  buf->SetDim(1);
  buf->SetIndex(0,length);
  try{
    buf->InitBuffer();
    memcpy(buf->GetDataPointer(),area,buf->IndexSize()*sizeof(double));
    stat=((Series_Buffer*)obj->GetBufferPointer())->PutTimeSeries(idx,buf);
  }catch(buffer_exception err){
    delete buf;
    console->execerror(err.what(),where);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  delete buf;
  return stat;
}

void *SystemCommon::SCmalloc(size_t size){
  void *ptr;
  __SystemCommonMemoryStruct mem;
  if(size<=0)return 0;
  try{
    ptr=(void*)new char[size];
  }catch(bad_alloc){
    return 0;
  }
  mem.ptr=ptr;
  mem.size=size;
  mem_struct.push_front(mem);
  return ptr;
}

void *SystemCommon::SCcalloc(size_t num, size_t size){
  void *ptr;
  __SystemCommonMemoryStruct mem;
  if(size*num<=0)return 0;
  try{
    ptr=(void*)new char[size*num];
  }catch(bad_alloc){
    return 0;
  }
  memset(ptr,0,num*size);
  mem.ptr=ptr;
  mem.size=size*num;
  mem_struct.push_front(mem);
  return ptr;
}

void SystemCommon::SCfree(void *ptr){
  for(mem_it=mem_struct.begin();mem_it!=mem_struct.end();mem_it++)
    if(ptr==(*mem_it).ptr)break;
  if(mem_it==mem_struct.end()){
    //console->tty_printf("%s %s","SATELLITE in free():",
	//	       "warning: junk pointer, too high to make sense.\n");
  }else{
    mem_struct.erase(mem_it);
    delete [] (char*)ptr;
  }
}

void *SystemCommon::SCrealloc(void *ptr, size_t size){
  void *dup_ptr;
  size_t cp_size,i;
  __SystemCommonMemoryStruct mem;
  if(size<=0)return 0;
  try{
    dup_ptr=(void*)new char[size];
  }catch(bad_alloc){return 0;}
  for(mem_it=mem_struct.begin();mem_it!=mem_struct.end();mem_it++)
    if(ptr==(*mem_it).ptr)break;
  if(mem_it!=mem_struct.end()){
    cp_size=Min((*mem_it).size,size);
    for(i=0;i<cp_size;i++)
      ((char*)dup_ptr)[i]=((char*)ptr)[i];
    mem_struct.erase(mem_it);
  }
  mem.ptr=dup_ptr;
  mem.size=size;
  mem_struct.push_front(mem);
  return dup_ptr;
}

int SystemCommon::SCprintf(const char *fmt, ...){
  int ret;
  char buf[1024];
  va_list marker;
  va_start( marker, fmt ); // initialize
  ret=vsprintf(buf,fmt,marker);
  va_end(marker);
  console->tty_printf(buf);
  return ret;
}

int SystemCommon::SCfprintf(int fp,const char *fmt, ...){
  int ret;
  char buf[1024];
  if(fp!=1 && fp!=2)return 0;
  va_list marker;
  va_start( marker, fmt ); // initialize
  ret=vsprintf(buf,fmt,marker);
  va_end(marker);
  console->tty_printf(buf);
  return ret;
}

int SystemCommon::SCfprintf(FILE *fp,const char *fmt, ...){
  int ret;
  va_list marker;
  va_start( marker, fmt ); // initialize
  ret=vfprintf(fp,fmt,marker);
  va_end(marker);
  return ret;
}
