/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: hash.c,v 1.2 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libsatellite.h"

/* dynamic resizeing range of hash bucket size */
#define HASH_BUCKETSIZE_MIN        13
#define HASH_BUCKETSIZE_MAX   5099893

#define emalloc(x) malloc(x)
#define efree(x)   free(x)

#ifdef __cplusplus
extern "C" {
#endif

static const unsigned int _hash_primes[]={
  13,      23,      43,      83,      163,     317,     631,
  1259,    2503,    5003,    9973,    19937,   39869,   79699,
  159389,  318751,  637499,  1274989, 2549951, 5099893
};

static unsigned int _hash_create_prime(unsigned int nkeys);
static int          _hash_table_resize(hash_table_t *table);

static void        *_hash_key_copy_int(const void *key);
static void         _hash_key_free_int(void *key);
static void        *_hash_key_copy_str(const void *key);
static void         _hash_key_free_str(void *key);

static unsigned int _hash_func_ptr(const void *key);
static int          _hash_comp_ptr(const void *key1, const void *key2);
static unsigned int _hash_func_int(const void *key);
static int          _hash_comp_int(const void *key1, const void *key2);
static unsigned int _hash_func_str(const void *key);
static int          _hash_comp_str(const void *key1, const void *key2);

DLLEXPORT hash_table_t *hash_table_new(hash_func_t hash_func,
                                       hash_comp_t hash_comp,
                                       hash_key_copy_t hash_key_copy,
                                       hash_key_free_t hash_key_free){
  hash_table_t *table;
  unsigned int i, size;
  table = (hash_table_t*)emalloc(sizeof(hash_table_t));
  if(table == NULL) return NULL;
  size = HASH_BUCKETSIZE_MIN;
  table->buckets = (hash_bucket_t**)emalloc(sizeof(hash_bucket_t*) * size);
  if(table->buckets == NULL){ efree(table); return NULL; }
  for(i=0; i<size; i++) table->buckets[i] = NULL;
  table->nbucks = size;
  if(hash_func) table->hash_func = hash_func;
  else          table->hash_func = _hash_func_ptr;
  if(hash_comp) table->hash_comp = hash_comp;
  else          table->hash_comp = _hash_comp_ptr;
  if(hash_key_copy) table->hash_key_copy = hash_key_copy;
  else              table->hash_key_copy = NULL;
  if(hash_key_free) table->hash_key_free = hash_key_free;
  else              table->hash_key_free = NULL;
  table->nkeys = 0;
  table->dyn_resize = 1;
  return table;
}

DLLEXPORT hash_table_t *hash_table_str_new(){
  return hash_table_new(_hash_func_str, _hash_comp_str,
                        _hash_key_copy_str, _hash_key_free_str);
}

DLLEXPORT hash_table_t *hash_table_int_new(){
  return hash_table_new(_hash_func_int, _hash_comp_int,
                        _hash_key_copy_int, _hash_key_free_int);
}

DLLEXPORT hash_table_t *hash_table_ptr_new(){
  return hash_table_new(NULL, NULL, NULL, NULL);
}

DLLEXPORT void hash_table_delete(hash_table_t *table, 
                                 hash_foreach_t hash_del, void *arg){
  unsigned int i;
  hash_bucket_t *buck,*next;
  next = NULL;
  for(i=0; i < table->nbucks; i++){
    for(buck=table->buckets[i]; buck; buck=next){
      next = buck->next;
      if(hash_del) hash_del(buck->data, arg);
      if(table->hash_key_free) table->hash_key_free(buck->key);
      efree(buck);
    }
  }
  if(table->buckets)
    efree(table->buckets);
  efree(table);
}

static unsigned int _hash_create_prime(unsigned int nkeys){
  unsigned int i;
  if(nkeys < HASH_BUCKETSIZE_MIN) return HASH_BUCKETSIZE_MIN;
  for(i=1; nkeys > _hash_primes[i]; i++)
    if(_hash_primes[i] == HASH_BUCKETSIZE_MAX)
      return HASH_BUCKETSIZE_MAX;
  return _hash_primes[i-1];
}

static int _hash_table_resize(hash_table_t *table){
  float ratio;
  unsigned int i, new_size;
  hash_bucket_t *buck, *next, **new_bucks;
  ratio =  (float)table->nkeys / (float)table->nbucks;
  if(ratio > 0.3 && ratio < 3.0) return 0;
  new_size = _hash_create_prime(table->nkeys);
  if(new_size == table->nbucks) return 0;
  new_bucks = (hash_bucket_t**)emalloc(sizeof(hash_bucket_t*)*new_size);
  if(new_bucks == NULL) return -1;
  for(i=0; i < new_size; i++) new_bucks[i] = NULL;
  for(i=0; i < table->nbucks; i++)
    for(buck=table->buckets[i]; buck; buck=next){
      next = buck->next;
      buck->next = new_bucks[buck->val%new_size];
      new_bucks[buck->val%new_size] = buck;
    }
  efree(table->buckets);
  table->buckets = new_bucks;
  table->nbucks  = new_size;
  return 0;
}

DLLEXPORT void *hash_table_lookup(hash_table_t *table, const void *key){
  unsigned int hval;
  hash_bucket_t *buck;
  hval = (table->hash_func(key) % table->nbucks);
  for(buck = table->buckets[hval]; buck; buck = buck->next)
    if(table->hash_comp(buck->key, key) == 0) break;
  if(buck == NULL) return NULL;
  return buck->data;
}

DLLEXPORT void *hash_table_remove(hash_table_t *table, const void *key){
  void *data;
  unsigned int hval;
  hash_bucket_t *buck, *last;
  hval = (table->hash_func(key) % table->nbucks);
  last = NULL;
  for(buck = table->buckets[hval]; buck; buck = buck->next){
    if(table->hash_comp(buck->key, key) == 0) break;
    last = buck;
  }
  if(buck == NULL) return NULL;
  if(last) last->next = buck->next;
  else      table->buckets[hval] = buck->next;
  table->nkeys--;
  data = buck->data;
  if(table->hash_key_free) table->hash_key_free(buck->key);
  efree(buck);
  if(table->dyn_resize) _hash_table_resize(table);
  return data;
}

DLLEXPORT int hash_table_insert(hash_table_t *table, void *key, void *data){
  unsigned int  hval;
  hash_bucket_t *buck;
  buck = hash_table_lookup(table, key);
  if(buck) return -1;
  buck = (hash_bucket_t*)emalloc(sizeof(hash_bucket_t));
  if(buck == NULL) return -1;
  if(table->hash_key_copy){
    buck->key = table->hash_key_copy(key);
    if(buck->key == NULL){ efree(buck); return -1; }
  }else{
    buck->key = key;
  }
  buck->data = data;
  buck->val  = table->hash_func(key);
  hval = (buck->val % table->nbucks);
  buck->next = table->buckets[hval];
  table->buckets[hval] = buck;
  table->nkeys++;
  if(table->dyn_resize) _hash_table_resize(table);
  return 0;
}

DLLEXPORT void hash_table_foreach(hash_table_t *table,
                                  hash_foreach_t hash_foreach, void *arg){
  unsigned int i;
  hash_bucket_t *buck;
  if(table->nkeys != 0){
    for(i=0; i < table->nbucks; i++)
      for(buck=table->buckets[i]; buck; buck=buck->next)
        hash_foreach(buck->data, arg);
  }
}

DLLEXPORT void hash_table_erase(hash_table_t *table,
                                hash_foreach_t hash_del, void *arg){
  unsigned int i;
  hash_bucket_t *buck,*next;
  next = NULL;
  for(i=0; i < table->nbucks; i++){
    for(buck=table->buckets[i]; buck; buck=next){
      next = buck->next;
      if(hash_del) hash_del(buck->data, arg);
      if(table->hash_key_free) table->hash_key_free(buck->key);
      efree(buck);
    }
    table->buckets[i]=NULL;
  }
  table->nkeys = 0;
  if(table->dyn_resize) _hash_table_resize(table);
}


/* hash functions - pointer version */
static unsigned int _hash_func_ptr(const void *key){
#ifdef _MSC_VER
/* Truncate 'const void *' to 'unsigned int' */
# pragma  warning(disable: 4311) 
#endif
  return (unsigned int)key;
#ifdef _MSC_VER
# pragma  warning(default: 4311)
#endif
}

static int _hash_comp_ptr(const void *key1, const void *key2){
#ifdef _MSC_VER
/* Truncate 'const void *' to 'unsigned int' */
# pragma  warning(disable: 4311) 
#endif
  return (int)((unsigned int)key1 - (unsigned int)key2);
#ifdef _MSC_VER
# pragma  warning(default: 4311)
#endif
}

/* hash functions - int version */
static void *_hash_key_copy_int(const void *key){
  int *ret;
  ret = (int*)emalloc(sizeof(int));
  if(ret) *ret = *(int*)key;
  return ret;
}

static void _hash_key_free_int(void *key){
  efree(key);
}

static unsigned int _hash_func_int(const void *key){
  return (unsigned int)(*(int*)key);
}

static int _hash_comp_int(const void *key1, const void *key2){
  return (*(int*)key1) - (*(int*)key2);
}


/* hash functions - string version */

static void *_hash_key_copy_str(const void *key){
  char *ret,*k;
  k = (char*) key;
  ret = (char*)emalloc(strlen(k)+1);
  if(ret) strcpy(ret,k);
  return ret;
}

static void _hash_key_free_str(void *key){
  efree(key);
}

/*
 * Fowler/Noll/Vo hash
 *
 * The basis of this hash algorithm was token from an idea sent
 *  by email to the the IEEE POSIX P1003.2 mailing list from
 *
 *  Phong Vo (kpv@research.att.com)
 *  Glenn Flowler (gsf@research.att.com),
 *  Landon Curt Noll (chongo@toad.com) 
 *
 * later improved on their algorithm.
 *
 */

static unsigned int _hash_func_str(const void *key){
  unsigned int h;
  const unsigned char *k;
  k = (const unsigned char *)key;
  for(h=0; *k; k++){
    h += 16777619; /* 2^24 + 403 */
    h ^= *k;
  }
  return h;
}

static int _hash_comp_str(const void *key1, const void *key2){
  return strcmp((const char*)key1,(const char*)key2);
}

#ifdef __cplusplus
}
#endif
    
