/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: history.c,v 1.2 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#ifdef HAVE_SYS_FILE_H
# include <sys/file.h>
#endif

#define __EXPORTSYMBOL__
#include "history.h"
#undef  __EXPORTSYMBOL__

#define HISTORY_SIZE 256
#define HISTORY_BUFSIZE 1024

/* new method - written by c */

static history_cell_t *_history_cell_new(time_t htime, 
                                                   const char *cmd);
static void _history_cell_delete(history_cell_t *cell);
static void _history_push(history_t *hist,
			       history_cell_t *cell);
static void _history_read_file(history_t *hist);
static void _history_write_file(history_t *hist);

#define emalloc(x) malloc((x))
#define efree(x)   free((x))

static history_cell_t *_history_cell_new(time_t htime, const char *cmd){
  size_t len;
  history_cell_t *cell;
  if(cmd == NULL || *cmd == '\0') return NULL;
  cell=(history_cell_t *)emalloc(sizeof(history_cell_t));
  if(cell == NULL) return NULL;
  len=strlen(cmd);
#ifdef HAVE_STRDUP
  cell->cmd=strdup(cmd);
  if(cell->cmd == NULL){ efree(cell); return NULL; }
#else
  cell->cmd=(char*)malloc(len+1);
  if(cell->cmd == NULL){ efree(cell); return NULL; }
  strcpy(cell->cmd, cmd);
#endif
  if(cell->cmd[len-1] == '\n') cell->cmd[len-1]='\0';
  cell->htime=htime;
  cell->next=NULL;
  cell->prev=NULL;
  return cell;
}

static void _history_cell_delete(history_cell_t *cell){
  if(cell->cmd!=NULL) efree(cell->cmd);
  efree(cell);
}

static void _history_push(history_t *hist, history_cell_t *cell){
  history_cell_t *tmp;
  hist->num++;
  tmp=hist->tail;
  hist->tail=cell;
  cell->next=NULL;
  if(tmp!=NULL){
    tmp->next=cell;
    cell->prev=tmp;
  }else{
    hist->top=cell;
    cell->prev=NULL;
  }
  if(hist->size == hist->max){
    tmp=hist->top->next;
    _history_cell_delete(hist->top);
    hist->top=tmp;
    tmp->prev=NULL;
  }else{
    hist->size++;
  }
  hist->pos=NULL;
}

static void _history_read_file(history_t *hist){
  int i;
  FILE *fp;
  long htime;
  char buf[HISTORY_BUFSIZE];
  history_cell_t *cell;
  if(hist->fname == 0 || *hist->fname == '\0') return;
  fp=fopen(hist->fname,"r");
  if(fp == NULL) return;
#ifdef HAVE_FLOCK
  flock(fileno(fp),LOCK_SH); /* shared lock */
#else
# ifdef HAVE_LOCKF
  lockf(fileno(fp),F_LOCK,0);
# endif
#endif
  while(fscanf(fp,"%ld ", &htime) == 1){
    for(i=0;i<HISTORY_BUFSIZE-1;i++){
      if(fread(buf+i,1,1,fp)<1) break;
      if(buf[i] == '\n') break;
    }
    buf[i]='\0';
    if(i==HISTORY_BUFSIZE-1){ /* skip file to next line */
      char c;
      while(1){
        if(fread(&c,1,1,fp)<1) break;
        if(c=='\n') break;
      }
    }
    cell=_history_cell_new((time_t)htime,buf);
    if(cell!=NULL) _history_push(hist, cell);
  }
#ifdef HAVE_FLOCK
  flock(fileno(fp),LOCK_UN); /* un lock */
#else
# ifdef HAVE_LOCKF
  lockf(fileno(fp),F_ULOCK,0);
# endif
#endif
  fclose(fp);
}

static void _history_write_file(history_t *hist){
  FILE *fp;
  history_cell_t *cell;
  if(hist->fname == 0 || *hist->fname == '\0') return;
  fp=fopen(hist->fname,"w");
  if(fp == NULL) return;
#ifdef HAVE_FLOCK
  flock(fileno(fp),LOCK_EX); /* exclusive lock */
#else
# ifdef HAVE_LOCKF
  lockf(fileno(fp),F_LOCK,0);
# endif
# endif
  for(cell=hist->top;cell!=NULL;cell=cell->next){
    fprintf(fp,"%ld %s\n", (long)cell->htime, cell->cmd);
  }
#ifdef HAVE_FLOCK
  flock(fileno(fp),LOCK_UN); /* un lock */
#else
# ifdef HAVE_LOCKF
  lockf(fileno(fp),F_ULOCK,0);
# endif
#endif
  fclose(fp);
}

DECLSPEC void history_init(history_t *hist){
  hist->size=0;
  hist->key=NULL;
  hist->max=HISTORY_SIZE;
  hist->top=NULL;
  hist->tail=NULL;
  hist->pos=NULL;
  hist->num=1;
  hist->fname=NULL;
}

DECLSPEC void history_clean(history_t *hist){
  history_cell_t *tmp;
  _history_write_file(hist);
  while(hist->top!=NULL){
    tmp=hist->top->next;
    _history_cell_delete(hist->top);
    hist->top=tmp;
  }
  if(hist->key!=NULL) efree(hist->key);
  if(hist->fname!=NULL) efree(hist->fname);
}

DECLSPEC void history_add(history_t *hist, const char *cmd){
  time_t tstamp;
  history_cell_t *tmp;
  if(strlen(cmd) == 1 && cmd[0] == '\n') return;
  tstamp=time(NULL);
  tmp=_history_cell_new(tstamp,cmd);
  if(tmp!=NULL) _history_push(hist,tmp);
}

DLLEXPORT int history_resize(history_t *hist, int size){
  history_cell_t *tmp;
  history_standby(hist);
  if(size <= 0) return -1;
  /* if current history size larger than 'size' then remove few cells */
  while(hist->size > size){
    tmp=hist->top->next;
    _history_cell_delete(hist->top);
    hist->top=tmp;
    tmp->prev=NULL;
    hist->size--;
  }
  hist->max = size;
  return 0;
}

DECLSPEC void history_standby(history_t *hist){
  if(hist->key!=NULL){
    efree(hist->key);
    hist->key=NULL;
  }
  hist->pos=NULL;
  hist->lock=0;
}

DECLSPEC void history_regist_key(history_t *hist, const char *k){
  if(hist->key!=NULL) efree(hist->key);
  if(k == NULL || *k == '\0') k=""; 
#ifdef HAVE_STRDUP
  hist->key=strdup(k);
#else
  hist->key=(char*)malloc(strlen(k)+1);
  if(hist->key)
    strcpy(hist->key,k);
#endif
  hist->pos=NULL;
  hist->lock=1;
}

DECLSPEC void history_set_fname(history_t *hist, const char *f){
  if(hist->fname!=NULL) efree(hist->fname);
  if(f == NULL || *f == '\0') f=""; 
#ifdef HAVE_STRDUP
  hist->fname=strdup(f);
  if(hist->fname == NULL) return;
#else
  hist->fname=(char*)malloc(strlen(f)+1);
  if(hist->fname == NULL) return;
  strcpy(hist->fname,f);
#endif
  _history_read_file(hist);
}

DECLSPEC const char *history_search_prev(history_t *hist){
  size_t len;
  history_cell_t *tmp;
  if(hist->key == NULL || *hist->key == '\0')
    return history_prev(hist);
  len=strlen(hist->key);
  tmp=hist->pos;
  if(tmp == NULL){
    tmp=hist->tail; 
  }else{
    tmp=hist->pos->prev;
  }
  for(;tmp!=NULL;tmp=tmp->prev){
    if(tmp->cmd == NULL || *tmp->cmd == '\0') continue;
    if(!strncmp(tmp->cmd,hist->key,len)){ /* found */
      hist->pos=tmp;
      break;
    }
  }
  if(tmp == NULL) return NULL;
  return tmp->cmd;
}

DECLSPEC const char *history_search_next(history_t *hist){
  size_t len;
  history_cell_t *tmp;
  if(hist->key == NULL || *hist->key == '\0')
    return history_next(hist);
  len=strlen(hist->key);
  tmp=hist->pos;
  if(tmp == NULL) return NULL;
  tmp=hist->pos->next;
  for(;tmp!=NULL;tmp=tmp->next){  /* TODO CHECK!!! maybe.. bug in here */
    if(tmp->cmd == NULL || *tmp->cmd == '\0') continue;
    if(!strncmp(tmp->cmd,hist->key,len)){ /* found */
      hist->pos=tmp;
      break; 
    }
  }
  if(tmp == NULL) return NULL;
  return hist->pos->cmd;
}

DECLSPEC const char *history_prev(history_t *hist){
  char *tmp=NULL;
  if(hist->pos == NULL){
    hist->pos=hist->tail;
    if(hist->tail!=NULL)
      tmp=hist->tail->cmd;
  }else{
    if(hist->pos->prev!=NULL){
      hist->pos=hist->pos->prev;
      tmp=hist->pos->cmd;
    }
  }
  return tmp;
}

DECLSPEC const char *history_next(history_t *hist){
  char *tmp=NULL;
  if(hist->pos!=NULL){
    hist->pos=hist->pos->next;
    if(hist->pos!=NULL){
      tmp=hist->pos->cmd;
    }
  }
  return tmp;
}
