/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: mathfunc.c,v 1.2 2004/07/29 18:07:53 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>

#ifdef HAVE_FLOAT_H
# include <float.h>
#endif

#define CALC_MACHINE_EPSILON 0

#ifndef M_PI
# define M_PI            3.14159265358979323846  /* pi */
#endif

#ifndef DBL_EPSILON
# define DBL_EPSILON     2.220446049250313080847263336181640625E-16
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if CALC_MACHINE_EPSILON
static double epsilon=1.0;
#else
static double epsilon=DBL_EPSILON;
#endif /* CALC_MACHINE_EPSILON */

#if CALC_MACHINE_EPSILON
void calc_machine_epsilon(){
  static int is_first=1;
  double x=1.0;
  if(is_first){
    while(x+epsilon>1.0)
      epsilon/=2.0;
    epsilon*=2.0;
    is_first=0;
  }
}
#endif /* CALC_MACHINE_EPSILON */

static int is_equal_double(double a, double b){
  return (fabs(a-b) <= epsilon);
}

#define EQUAL(a,b) is_equal_double((a),(b))

/* basic operators */
double math_add(double a, double b){ return a+b; }
double math_sub(double a, double b){ return a-b; }
double math_mul(double a, double b){ return a*b; }
double math_div(double a, double b){ return a/b; }
double math_mod(double a, double b){ return (double)((long)a%(long)b); }
double math_gt(double a, double b){ return (double)(a>b); }
double math_lt(double a, double b){ return (double)(a<b); }
double math_ge(double a, double b){ return (double)(a>=b); }
double math_le(double a, double b){ return (double)(a<=b); }
double math_eq(double a, double b){ return (double)(EQUAL(a,b)); }
double math_ne(double a, double b){ return (double)(!EQUAL(a,b)); }
double math_and(double a, double b){
  return (double)(!EQUAL(a,0.0) && !EQUAL(b,0.0)); 
}
double math_or(double a, double b){
  return (double)(!EQUAL(a,0.0) || !EQUAL(b,0.0));
}
double math_not(double a, double dmy){ return (double)(EQUAL(a,0.0)); }
double math_negate(double a,double dmy){ return -a; }

/* mathematical functions */
double math_int(double a, double dmy){
  double sgn, b;
  sgn = ( a < 0.0 ) ? -1.0 : 1.0;
  b = floor(fabs(a));
  if ( !EQUAL(b,0.0) )
    b *= sgn;
  return b;
}

double math_floor(double a, double dmy){
  return floor(a);
}

double math_ceil(double a, double dmy){
  return ceil(a);
}

double math_round(double a, double dmy){
  static const double magic=0.5;
  return (a > 0.0) ? math_int(a + magic, 0.0) : math_int(a - magic, 0.0);
}

double math_pow(double a,double b){ return (!EQUAL(a,0.0)) ? pow(a,b) : 1.0; }

double math_log2(double a, double dmy){
  return log(a)/log(2.0);
}


double math_sign(double a, double dmy){ 
  if( !EQUAL(a,0.0) ) return (a>0.0)?1.0:-1.0; return 0.0; 
}

// mathematical function wrappers
double math_abs(double a, double dmy){ return fabs(a); }
double math_exp(double a, double dmy){ return exp(a); }
double math_log(double a, double dmy){ return log(a); }
double math_log10(double a, double dmy){ return log10(a); }
double math_sqrt(double a, double dmy){ return sqrt(a); }
double math_sin(double a, double dmy){ return sin(a); }
double math_cos(double a, double dmy){ return cos(a); }
double math_tan(double a, double dmy){ return tan(a); }
double math_asin(double a, double dmy){ return asin(a); }
double math_acos(double a, double dmy){ return acos(a); }
double math_atan(double a, double dmy){ return atan(a); }
double math_atan2(double a, double b){
  double ret;
  if( !EQUAL(b,0.0) ) {
    if( EQUAL(a,0.0) ){
      ret = ( b > 0.0 ) ? 0.0 : M_PI;
    }
    else{ 
      ret = atan(a/b);
    }
  }else{
    if( !EQUAL(a,0.0) ){
      ret = ( a > 0.0 ) ? M_PI/2.0 : -M_PI/2.0; 
    }else{
      ret = 0.0;
      errno = 0;
      printf("WARNING.. atan2(0,0) illegal instruction.");
    }
  }
  return ret;
}


#ifdef __cplusplus
}
#endif
