/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: process.c,v 1.3 2005/10/29 20:05:44 orrisroot Exp $ */
#define LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libsatellite.h"
#include "path.h"

#ifdef RETSIGTYPE
typedef RETSIGTYPE (*sigfunc_t)(int);
#else
typedef void (*sigfunc_t)(int);
#endif

#ifndef WIN32
static char **build_argv(const char *bin, int *status);
#endif

/* process id manegement */
static sl4_list_t *pid_manager = NULL;
static int pid_manager_regist(int pid);
static int pid_manager_eq_func(void *p1, void *p2);
static int pid_manager_remove(int pid);

#ifdef WIN32
#define APP_NONE  0
#define APP_DOS   1
#define APP_WIN3X 2
#define APP_WIN32 3

static BOOL HasConsole();
static int  GetApplicationType(const char *bin);

DLLEXPORT int create_child_process(const char *bin, sl4_fd_t ifd, sl4_fd_t ofd,
                                   sl4_fd_t efd, const char *libpath){
  int ret,AppType,createFlags;
  size_t len;
  char *argv0, *next, *path, *tmp;
  const char *comspec;

  STARTUPINFO         startInfo;
  PROCESS_INFORMATION procInfo;
  SECURITY_ATTRIBUTES secAtts;
  HANDLE hProcess, h, iHandle, oHandle, eHandle;
  OSVERSIONINFO os;

  path = NULL;
  ret = -1;

  /* get argv0 */
  argv0 = pathname_get_token(bin, &next);
  if(argv0 == NULL) return -1;
  tmp = pathname_convert_normal(argv0);
  free(argv0);
  if(tmp == NULL) return -1;
  argv0 = pathname_get_executable(tmp);
  free(tmp); tmp = NULL;
  if(argv0 == NULL) return -1;

  AppType = GetApplicationType(argv0);

  os.dwOSVersionInfoSize = sizeof(os);
  GetVersionEx(&os);

  hProcess = GetCurrentProcess();

  ZeroMemory(&startInfo, sizeof(startInfo));
  startInfo.cb = sizeof(startInfo);
  startInfo.dwFlags = STARTF_USESTDHANDLES;
  startInfo.hStdInput  = INVALID_HANDLE_VALUE;
  startInfo.hStdOutput = INVALID_HANDLE_VALUE;
  startInfo.hStdError  = INVALID_HANDLE_VALUE;

  secAtts.nLength = sizeof(SECURITY_ATTRIBUTES);
  secAtts.lpSecurityDescriptor = NULL;
  secAtts.bInheritHandle = TRUE;

  iHandle = ifd.handle;
  oHandle = ofd.handle;
  eHandle = efd.handle;

  /* duplicate each handles */
  if(iHandle == INVALID_HANDLE_VALUE){
    if(CreatePipe(&startInfo.hStdInput, &h, &secAtts, 0) != FALSE)
      CloseHandle(h);
  }else
    DuplicateHandle(hProcess, iHandle, hProcess, &startInfo.hStdInput,
                    0, TRUE, DUPLICATE_SAME_ACCESS);
  if(oHandle == INVALID_HANDLE_VALUE){
    /* Win32 & DOS application */
    if(os.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS && AppType == APP_DOS){
      if(CreatePipe(&h, &startInfo.hStdOutput, &secAtts, 0) != FALSE)
        CloseHandle(h);
    }else{
      /* else */
      startInfo.hStdOutput = CreateFileA("NUL:", GENERIC_WRITE, 0,
                                         &secAtts, OPEN_ALWAYS, 
                                         FILE_ATTRIBUTE_NORMAL, NULL);
    }
  }else
    DuplicateHandle(hProcess, oHandle, hProcess, &startInfo.hStdOutput,
                    0, TRUE, DUPLICATE_SAME_ACCESS);
  if(eHandle == INVALID_HANDLE_VALUE){
    /* PLATFORM is Win32 & 16 bit DOS application */
    if(os.dwPlatformId == VER_PLATFORM_WIN32_WINDOWS && AppType == APP_DOS){
      if(CreatePipe(&h, &startInfo.hStdError, &secAtts, 0) != FALSE)
        CloseHandle(h);
    }else{
      /* else */
      startInfo.hStdOutput = CreateFileA("NUL:", GENERIC_WRITE, 0,
                                         &secAtts, OPEN_ALWAYS, 
                                         FILE_ATTRIBUTE_NORMAL, NULL);
    }
  }else
    DuplicateHandle(hProcess, eHandle, hProcess, &startInfo.hStdError,
                    0, TRUE, DUPLICATE_SAME_ACCESS);

  /* PLATFORM is WinNT */
  if(os.dwPlatformId == VER_PLATFORM_WIN32_NT){
    if(HasConsole()){
      createFlags = 0;
    }else if(AppType == APP_DOS){
      startInfo.wShowWindow = SW_HIDE;
      startInfo.dwFlags |= STARTF_USESHOWWINDOW;
      createFlags = CREATE_NEW_CONSOLE;
    }else
      createFlags = DETACHED_PROCESS;
  }else{
    /* Win95 */
    if(HasConsole()){
      createFlags = 0;
    }else{
      createFlags = DETACHED_PROCESS;
      if(AppType == APP_DOS){
        /* TODO: now not working */
        /* startInfo.wShowWindow = SW_HIDE; */
        /* startInfo.dwFlags |= STARTF_USESHOWWINDOW; */
        /* createFlags = CREATE_NEW_CONSOLE; */
        goto finalize;
      }
    }
  }
  len = 0;
  if(AppType == APP_DOS){
    comspec = getenv("ComSpec");
    if(comspec == NULL) goto finalize;
    len += strlen(comspec);
    len += strlen(" /c ");
  }
  len += strlen(argv0) + 2; /* 2 means '""' escape  */
  if(next) len += strlen(next) + 1;    /* 1 means ' ' */
  tmp = (char*)malloc(len + 1); /* 1 means '\0' */
  if(tmp == NULL) goto finalize;
  if(AppType == APP_DOS){
    strcpy(tmp, comspec);
    strcat(tmp, " /c ");
  }else{
    tmp[0]='\0';
  }
  strcat(tmp, "\""); strcat(tmp, argv0); strcat(tmp, "\"");
  if(next){ strcat(tmp, " "); strcat(tmp, next); }
  /* set libpath */
  if(libpath){
    path = strdup(getenv("PATH"));
    if(path != NULL){
      char *buf, *p;
      len = strlen(path) + strlen(libpath) + 7;
      /* 7 means 'PATH=' + ';' + '\0' */
      buf = (char*)malloc(sizeof(char)*len);
      if(buf){
        strcpy(buf,"PATH=");
        strcat(buf,libpath);
        strcat(buf,";");
        strcat(buf,path);
        /* replace directory separetor */
        for(p=buf;*p!='\0';p++) if(*p=='/')*p='\\';
        putenv(buf);
        free(buf);
      }
    }
  }
  if(CreateProcess(NULL, tmp, NULL, NULL, TRUE, createFlags,
                   NULL, NULL, &startInfo, &procInfo) == FALSE){
    goto finalize;
  }
  ret = procInfo.dwProcessId;

  if(AppType == APP_DOS){
    WaitForSingleObject(procInfo.hProcess, 50);
  }
  WaitForInputIdle(procInfo.hProcess, 5000);
  CloseHandle(procInfo.hThread);
finalize:
  /* free memory */
  if(tmp) free(tmp);
  if(argv0) free(argv0);
  /* restore 'PATH' environment */
  if(path != NULL){
    char *buf;
    size_t len;
    len = strlen(path) + 6;
    /* 6 means 'PATH=' + '\0' */
    buf = (char*)malloc(sizeof(char)*len);
    if(buf){
      strcpy(buf,"PATH=");
      strcat(buf,path);
      putenv(path);
      free(buf);
    }
    free(path);
  }
  /* close standard I/O handles for new process creation */
  if(startInfo.hStdInput != INVALID_HANDLE_VALUE)
    CloseHandle(startInfo.hStdInput);
  if(startInfo.hStdOutput != INVALID_HANDLE_VALUE)
    CloseHandle(startInfo.hStdOutput);
  if(startInfo.hStdError != INVALID_HANDLE_VALUE)
    CloseHandle(startInfo.hStdError);
  if(ret != -1)
    pid_manager_regist(ret);
  return ret;
}

DECLSPEC int wait_child_process(int pid, int *status){
  int ret;
  DWORD st;
  HANDLE h;
  h = OpenProcess(PROCESS_ALL_ACCESS, FALSE, pid);
  if(h == NULL){ return -1; }
  WaitForSingleObject(h, INFINITE);
  GetExitCodeProcess(h, &st);
  CloseHandle(h);
  ret = 0;
  *status = (int)st;
  /* if child process is console application and it was force 
     terminated by the signal (^C) then exit code will be set
     to number 3. */
  if(st == 3){  ret = 1; }
  pid_manager_remove(pid);
  return ret;
}

static BOOL HasConsole(){
  HANDLE h;
  h = CreateFileA("CONOUT$", GENERIC_WRITE, FILE_SHARE_WRITE, NULL,
                  OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if(h != INVALID_HANDLE_VALUE){
    CloseHandle(h);
    return TRUE;
  }
  return FALSE;
}

static int GetApplicationType(const char *bin){
  int   ret;
  char *ext, buf[2];
  IMAGE_DOS_HEADER head;
  DWORD attr, read;
  HANDLE h;
  ret = APP_NONE;
  /* exist check */
  attr = GetFileAttributes(bin);
  if(attr == (DWORD) -1 || attr & FILE_ATTRIBUTE_DIRECTORY) goto done;
  /* file found */
  /* file extension check */
  ext = strrchr(bin, '.');
  if(ext != NULL && stricmp(ext,".bat") == 0){
    ret = APP_DOS; goto done;
  }
  /* DOS header check */
  h = CreateFile(bin,GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING,
                 FILE_ATTRIBUTE_NORMAL, NULL);
  if(h == INVALID_HANDLE_VALUE) goto done;
  head.e_magic = 0;
  ReadFile(h, &head, sizeof(head), &read, NULL);
  if(head.e_magic != IMAGE_DOS_SIGNATURE){
    CloseHandle(h);
    if(ext != NULL && stricmp(ext,".com") == 0)
      ret = APP_DOS;
    goto done;
  }
  if(head.e_lfarlc != sizeof(head)){
    CloseHandle(h); ret = APP_DOS; goto done;
  }
  buf[0] = '\0';
  SetFilePointer(h, head.e_lfanew, NULL, FILE_BEGIN);
  ReadFile(h, buf, 2, &read, NULL);
  CloseHandle(h);
  if(buf[0] == 'N' && buf[1] == 'E'){
    ret = APP_WIN3X;
  }else if(buf[0] == 'P' && buf[1] == 'E'){
    ret = APP_WIN32;
  }else{
    ret = APP_DOS;
  }
 done:
  return ret;
}

#else
/* UNIX version */
DLLEXPORT int create_child_process(const char *bin, sl4_fd_t ifd, sl4_fd_t ofd,
                                   sl4_fd_t efd, const char *libpath){
  int pid, status;
  sigfunc_t isigfunc, qsigfunc, tsigfunc;
  char **argv;
  int    i;

  /* build argv */
  argv = build_argv(bin, &status);
  if(argv == NULL){
    return -1; 
  }

  /* save and ignore signal */
  isigfunc = (sigfunc_t) signal(SIGINT,  SIG_IGN);
  qsigfunc = (sigfunc_t) signal(SIGQUIT, SIG_IGN);
  tsigfunc = (sigfunc_t) signal(SIGTERM, SIG_IGN);

  pid = fork();
  switch(pid){
  case -1: /* fork fail */
    break;
  case 0: /* child */
    if(ifd.fd != 0){ close(0); dup(ifd.fd); }
    if(ofd.fd != 1){ close(1); dup(ofd.fd); }
    if(efd.fd != 2){ close(2); dup(efd.fd); }
    /* append libpath to "PATH" environment variable */
    if(libpath != NULL){
#ifdef HAVE_SETENV
      setenv(SHLIBPATH_VAR,libpath,1);
#else
      char *path, *tmp;
      size_t len;
      len = strlen(SHLIBPATH_VAR) + strlen(libpath) + 2;
      /* 2 means '=' + '\0' */
      tmp = (char*)malloc(sizeof(char)*len);
      if(tmp != NULL){
        strcpy(tmp, SHLIBPATH_VAR); strcat(tmp, "="); strcat(tmp, libpath);
        putenv(tmp);
        free(tmp);
      }
#endif
    }
    /* initialize signal */
    signal(SIGINT,  SIG_DFL);
    signal(SIGQUIT, SIG_DFL);
    signal(SIGTERM, SIG_DFL);
    execvp(argv[0],argv);
    /* perror("failed to execute child"); */
    _exit(0);
  default: /* parent*/
    pid_manager_regist(pid);
    break;
  }
  /* restore signal */
  signal(SIGINT,  isigfunc);
  signal(SIGQUIT, qsigfunc);
  signal(SIGTERM, tsigfunc);
  /* free 'argv' memory */
  for(i=0; argv[i] != NULL; i++) free(argv[i]);
  free(argv);
  return pid;
}

DLLEXPORT int wait_child_process(int pid, int *status){
  int wpid,st,ret,num;
  wpid = waitpid(pid, &st, 0);
  if(wpid != pid) return -1;
  ret = 4; num = 0;
  if(WIFEXITED(st)){
    num = WEXITSTATUS(st); /* normal exit */
    ret = 0;
#ifdef WCOREDUMP
  }else if(WCOREDUMP(st)){
    num = 0;  /* core dumped */
    ret = 3;
#endif
  }else if(WIFSIGNALED(st)){
    num = WTERMSIG(st);    /* caught signal */
    ret = 1;
  }else if(WIFSTOPPED(st)){
    num = WSTOPSIG(st);    /* stopped */
    ret = 2;
  }
  *status = num;
  pid_manager_remove(pid);
  return ret;
}
#endif

DLLEXPORT int wait_all_process(){
  int pid,status;
  sl4_list_iterator_t it;
  if(pid_manager == NULL) return 0;
  while(pid_manager->size != 0){
    sl4_list_begin(pid_manager, &it);
    pid = *(int*)sl4_list_it_data(&it);
    wait_child_process(pid, &status);
  }
  free(pid_manager);
  pid_manager = NULL;
  return 0;
}

#ifndef WIN32
/* status                                             */
/*         -1 : out of memory                         */
/*          0 : success                               */
/*          1 : command not found                     */
/*          2 : illegal string                        */
static char **build_argv(const char *bin, int *status){
  sl4_list_t *alist;
  const char *p,*next;
/* unsigned int len; */
  char *tmp1,*tmp2, **argv;
  unsigned int i;

  /* inlitialize */
  argv = NULL;
  *status = 0; 
  alist = sl4_list_new();
  if(alist == NULL) goto fatal; /* out of memory */

  /* append arguments to list */
  tmp1 = pathname_get_token(bin, &next);
  while(tmp1 != NULL){
    tmp2 = pathname_convert_normal(tmp1);
    free(tmp1);
    if(tmp2 == NULL){ *status = 2; goto fatal; } /* illegal string */
    if(sl4_list_push_back(alist, tmp2) != 0){
      free(tmp2); goto fatal; /* out of memory */
    }
    if(next == NULL) break;
    p = next;
    tmp1 = pathname_get_token(p, &next);
  }
  if(alist->size == 0){ *status = 2;  goto fatal; } /* illegal string */

  /* create argv */
  argv = (char**)malloc(sizeof(char*)*(alist->size+1));
  if(argv == NULL){ goto fatal; } /* out of memory */
  for(i=0; i<alist->size+1; i++){ argv[i] = NULL; } /* for fatal */
  for(i=0; alist->size > 0; i++){
    tmp1 = (char*)sl4_list_top(alist);
    argv[i] = tmp1;
    sl4_list_pop_front(alist);
  }
  sl4_list_delete(alist);
  alist = NULL;

  /* fixed path of argv[0] */
  tmp1 = pathname_get_executable(argv[0]);
  if(tmp1 == NULL){ *status = 1; goto fatal; }
  free(argv[0]);
  argv[0] = tmp1;
  return argv;

 fatal:
  /* error recovery */
  /* free alist */
  if(alist){
    for(i=0; alist->size > 0; i++){
      tmp1 = (char*)sl4_list_top(alist);
      sl4_list_pop_front(alist);
      free(tmp1);
    }
    sl4_list_delete(alist);
    alist = NULL;
  }
  /* free argv */
  if(argv){
    for(i=0;argv[i]!=NULL;i++) free(argv[i]);
    free(argv);
  }
  /* set status for out of memory error */
  if(*status == 0) *status = -1;
  return NULL;
}
#endif

static int pid_manager_init(){
  pid_manager = sl4_list_new();
  if(pid_manager == NULL) return -1;
  return 0;
}

static int pid_manager_regist(int pid){
  int *dat;
  /* check already initilized of pid_manager */
  if(pid_manager == NULL)
    if(pid_manager_init() == -1) return -1;
  dat = (int*)malloc(sizeof(int));
  if(dat == NULL) return -1;
  *dat = pid;
  return sl4_list_push_front(pid_manager, dat);
}

static int pid_manager_eq_func(void *p1, void *p2){
  int *pid1=(int*)p1;
  int *pid2=(int*)p2;
  return(*pid1 == *pid2);
}

static int pid_manager_remove(int pid){
  sl4_list_iterator_t it;
  int *target;
  static const char *w1 = 
    "SL4_WARNING - pid(%d) is not registed in pid_manager.\n";
  static const char *w2 = 
    "SL4_WARNING - pid(%d) could not remove from pid_manager.\n";
  if(pid_manager == NULL){
    fprintf(stderr, w1, pid);
    return 0;
  }
  if(sl4_list_find(pid_manager, &pid, &it, pid_manager_eq_func) != 0){
    fprintf(stderr, w1, pid);
    return -1;
  }
  target = (int*)sl4_list_it_data(&it);
  if(sl4_list_erase(pid_manager, &it) != 0){
    fprintf(stderr, w2, pid);
    return -1;
  }
  free(target);
  return 0;
}

