/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: rwfile1.c,v 1.6 2005/10/29 21:23:09 orrisroot Exp $ */
/******************************************************************
**                                                               **
**      File Name : rwfile1.c                                    **
**                                                               **
**              Data File Read / Write Routines                  **
**                for SATELITE Basic Library                     **
**                                                               **
**                                      Coded by T.Kobayashi     **
**                                      Modified by Y.OKUMURA    **
**                                                               **
******************************************************************/
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_SYS_FILE_H
#include <sys/file.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifndef S_IRUSR
# define S_IRUSR S_IREAD
#endif
#ifndef S_IWUSR
# define S_IWUSR S_IWRITE
#endif
#ifndef S_IRGRP
# define S_IRGRP 0
#endif
#ifndef S_IROTH
# define S_IROTH 0
#endif
#ifndef O_BINARY
# define O_BINARY 0
#endif

#include "libsatellite.h"

#define REC_SIZE 256

/* int syscom_buff_leng = 4096; */

/***** Local Functions *****/
static void Iswap(unsigned char *data);
static void Fswap(unsigned char *data);
static void PC_head(h_type *hdr);
static void PC_data(float *data, int d_point);
static int  cpu_type();

/*************************************
      Make New File
*************************************/
DLLEXPORT void m_file(char *f_name){
  char  buffer[1024];
  int   fp, count;
  for (count = 0; count < 1024; count++)
    buffer[count] = (char) 0;
  /* File Creat  mode -> 644     */
  /* ( Owner : Group : User  )   */
  /* ( r w x : r w x : r w x )   */
  /* ( 1 1 0 : 1 0 0 : 1 0 0 )   */
  fp = open(f_name, O_RDWR | O_CREAT | O_TRUNC | O_BINARY, 
            S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH);
  write(fp, buffer, 1024);
  close(fp);
}

/*********************************************
      Read from File( Low Level : Type 1  ) 
      input      s_blk   : Start  Record No 
      num_blk : Number of Records 
      f_name  : File Name
      output     data    : Read Data 
      return     : Number of Read Record 
**********************************************/
DLLEXPORT int r_file(int s_blk, int num_blk, char *f_name, char *data){
  int   fd;
  long  ll = 0;
  int   read_blk, count, point;
  char  buff[REC_SIZE];
  if ((fd = open(f_name, O_RDONLY | O_BINARY)) < 0)  return (-1);
  read_blk = 0;
  ll = s_blk * REC_SIZE;
  lseek(fd, ll, 0);
  while (read_blk < num_blk) {
    for (count = 0; count <= REC_SIZE - 1; count++)
      buff[count] = 0x00;
    if (read(fd, buff, REC_SIZE) != REC_SIZE)
      break;
    point = read_blk * REC_SIZE;
    for (count = 0; count <= REC_SIZE - 1; count++)
      data[point + count] = buff[count];
    ++read_blk;
  }
  close(fd);
  return (read_blk);
}

/*************************************************
      Write to File( Low Level : Type 1 )
      input	s_blk    : Start Record No
      num_blk  : Number of Write Record 
      f_name   : File Name
      data     : Write Data 
      return : Number of Write Record 
*************************************************/
DLLEXPORT int w_file(int s_blk, int num_blk, char *f_name, char *data){
  int   fd;
  long  ll = 0;
  int   write_blk, count, point;
  char  buff[REC_SIZE + 1];
  if ((fd = open(f_name, O_RDWR | O_BINARY)) < 0) return (-1);
  write_blk = 0;
  ll = (s_blk) * REC_SIZE;
  lseek(fd, ll, 0);
  while (write_blk < num_blk) {
    point = write_blk * REC_SIZE;
    for (count = 0; count < REC_SIZE; count++)
      buff[count] = data[point + count];
    if (write(fd, buff, REC_SIZE) != REC_SIZE)
      break;
    ++write_blk;
  }
  close(fd);
  return (write_blk);
}


/*************************************************************
      SATELITE File Initialize 
      input	f_name	: File Name 
      type    : CPU type  0 - 680x0 or SPARC (EWS)
      1 - 80x86 (PC)
**************************************************************/
DLLEXPORT void stor_init(char *f_name, int type){
  h_type  head;
  int     y, m, d;
  /* Read Header from file .. */
  load_header(f_name, &head);
  head.type_flag = (short) type;	/* Default = 0 : UNIX Format */
  head.data_flag = (short) 0;
  getusername(head.opr_name, sizeof(head.opr_name));
  get_date(&y, &m, &d);
  head.date[0] = (short) y;
  head.date[1] = (short) m;
  head.date[2] = (short) d;
  head.rec_leng = 0;
  head.rec_num = 0;
  head.data_num = 0;
  /* Write Header */
  stor_header(f_name, &head);
}

/****************************************************
      Write to SATELITE File
      input	data	: Write Data 
      d_point : Data Point
      f_name  : File Name 
      numb    : Write Block No 
*****************************************************/
DLLEXPORT int stor_data(float *data, int d_point, char *f_name, int numb){
  h_type  head;
  int     rec_leng, rec_num, data_num, s_blk, write_blk;
  /* Read Header Block */
  load_header(f_name, &head);
  if (head.type_flag != cpu_type()) PC_data(data, d_point);
  rec_leng = head.rec_leng;
  rec_num = head.rec_num;
  data_num = head.data_num;
  if (data_num == 0) {
    data_num = d_point;
    rec_leng = d_point / 64;
    if (d_point % 64 != 0) ++rec_leng;
  }
  if (data_num != d_point) return (-2);
  s_blk = (numb-1)* rec_leng + 4;
  write_blk = w_file(s_blk, rec_leng, f_name, (char *) data);
  if (write_blk != rec_leng) return (-1);
  rec_num = Max(rec_num, numb);
  head.rec_leng = rec_leng;
  head.rec_num = rec_num;
  head.data_num = data_num;
  stor_header(f_name, &head);
  if (head.type_flag != cpu_type()) PC_data(data, d_point);
  return( d_point );
}

/****************************************************
      Read Data from SATELITE File
      input	f_name  : File Name
      numb    : Read Block No
      output	data    : Read Data
      return  : Data Point
*****************************************************/
DLLEXPORT int load_data(float *data, char *f_name, int numb){
  h_type  head;
  int     rec_leng, rec_num, data_num, s_blk, read_blk;
  /* Read Header Block */
  load_header(f_name, &head);
  rec_leng = head.rec_leng;
  rec_num = head.rec_num;
  data_num = head.data_num;
  /* if (syscom_buff_leng == 0) syscom_buff_leng = 4096;
	 if (data_num > syscom_buff_leng) return (-2); */
  s_blk = (numb-1) * rec_leng + 4;
  read_blk = r_file(s_blk, rec_leng, f_name, (char *)data);
  if (read_blk != rec_leng) return (-1);
  if (head.type_flag != cpu_type()) PC_data(data, data_num);
  return (data_num);
}

/****************************************************
      Write Header to SATELITE Data File
      input	f_name  : File Name 
      head    : Header Information 
***************************************************/
DLLEXPORT void stor_header(char *f_name, h_type *head){
  if (head->type_flag != cpu_type()) PC_head(head);
  w_file(0, 1, f_name, (char *) head);
  if (head->type_flag != cpu_type()) {
    PC_head(head);
  }
}

/****************************************************
      Read Header to SATELITE Data File
      input	f_name  : File Name 
      head    : Header Information 
**************************************************/
DLLEXPORT int load_header(char *f_name, h_type *head){
  if (r_file(0, 1, f_name, (char *) head) == -1)  return (-1);
  if (head->type_flag != cpu_type()) PC_head(head);
  return (0);
}

/**********************************
 ***********************************
 **                               **
 **        Local Functions        **
 **                               **
 ***********************************
 **********************************/

/**********************************
 *   A|B                           *
 *               Type of short     *
 *   B|A                           *
 **********************************/
static void Iswap(unsigned char *data){
  unsigned char   temp;
  temp = data[0];     /* temp = A */
  data[0] = data[1];  /* A = B    */
  data[1] = temp;     /* B = temp */
}

/**********************************
 *   A|B|C|D                       *
 *                Type of float    *
 *   D|C|B|A                       *
 **********************************/
static void Fswap(unsigned char *data){
  unsigned char   temp;
  temp = data[0];    /* temp = A */
  data[0] = data[3]; /* A = D    */
  data[3] = temp;    /* D = temp */
  temp = data[1];    /* temp = B */
  data[1] = data[2]; /* B = C    */
  data[2] = temp;    /* C = temp */
}

static void PC_data(float *data, int d_point){
  int             i;
  float           temp;
  for (i = 0; i < d_point; i++) {
    temp = data[i];
    Fswap((unsigned char *) &temp);
    data[i] = temp;
  }
}

static void PC_head(h_type *hdr){
  Iswap((unsigned char *) &hdr->type_flag);
  Iswap((unsigned char *) &hdr->data_flag);
  Iswap((unsigned char *) &hdr->date[0]);
  Iswap((unsigned char *) &hdr->date[1]);
  Iswap((unsigned char *) &hdr->date[2]);
  Fswap((unsigned char *) &hdr->samf);
  Iswap((unsigned char *) &hdr->rec_leng);
  Iswap((unsigned char *) &hdr->rec_num);
  Iswap((unsigned char *) &hdr->data_num);
}

/* LITTLE_ENDIAN : TRUE */
/* BIG_ENDAIN    : FALSE */
static int cpu_type(){
  short i = 0x0001;
  char *c = (char *)&i;
  return (int)c[0];
}


char *read_old_file(const char *fname, int numb, int *length){
/*      fname    data file name  */
/*      numb     number of block */
/*      length   data point      */
  int             dpt, size;
  h_type          head; /* old header type */
  float          *fdata;

  load_header((char*)fname, &head);
  *length = head.data_num;

  size = head.rec_leng * REC_SIZE;
  fdata = (float*)malloc(size);

/*   before = syscom.buff_leng;  /\* take *\/ */
/*   syscom.buff_leng = size;  /\* take *\/ */

  dpt = load_data(fdata, (char*)fname, numb+1);
  if(dpt == -1)
    return NULL;
    /* execerror("data file read error :", fname); */
  if(dpt == -2)
    /* execerror("data point too large :", fname); */
    return NULL;

/*   syscom.buff_leng = before;  /\* take *\/ */
  return (char *) fdata;
}


char *read_old_file2(const char *fname, int *dim, int *length){
/*      fname    data file name */
/*      dim      data dimension */ 
/*      length   data point     */
  int    numb, max;
  int    dpt, size;
  h_type head;  /* old header type */
  float  *fdata, *tmp;

  load_header((char*)fname, &head);

  max = *dim = head.rec_num;
  size = head.data_num;

  if ( max == 1 ) {
    *length = head.data_num;
    *dim = 1;
  } else {
    *length = head.rec_num;
    *(length+1) = head.data_num;
    *dim = 2;
  }

  fdata = (float *)malloc( head.rec_leng * max * REC_SIZE );

/*   before = syscom.buff_leng;  /\* take *\/ */
/*   syscom.buff_leng = head.data_num;    /\* take *\/ */

  for ( numb = 0, tmp = fdata; numb < max; numb++, tmp += size ) {
    dpt = load_data( tmp, (char*)fname, numb+1);

    if (dpt == -1)
      return NULL;
/*       execerror("data file read error :", fname); */
    if (dpt == -2)
      return NULL;
/*       execerror("data point too large :", fname); */
  }
  /*
    printf(" \"%s\":[%d] ---> series(%d points)\n", fname, numb,
    *length);
    printf(" %x \n", tmp );
    */
/*   syscom.buff_leng = before;  /\* take *\/ */

  return (char *) fdata;
}
