/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: rwfile2.c,v 1.5 2005/10/29 21:23:09 orrisroot Exp $ */
/******************************************************************
**                                                               **
**      File Name : rwfile2.c                                    **
**                                                               **
**              Data File Read / Write Routines                  **
**                for SATELLITE Basic Library                    **
**                                                               **
**                                      Coded by S.Hitomi        **
**                                      Modified by Y.OKUMURA    **
**                                                               **
******************************************************************/
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_STRINGS_H
# include <strings.h>
#endif
#ifdef HAVE_SYS_FILE_H
# include <sys/file.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#else
# ifdef HAVE_MALLLOC_H
#  include <malloc.h>
# endif
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifndef S_IRUSR
# define S_IRUSR S_IREAD
#endif
#ifndef S_IWUSR
# define S_IWUSR S_IWRITE
#endif
#ifndef S_IRGRP
# define S_IRGRP 0
#endif
#ifndef S_IROTH
# define S_IROTH 0
#endif
#ifndef O_BINARY
# define O_BINARY 0
#endif

#define __EXPORTSYMBOL__
#include "libsatellite.h"

/* TODO fprintf handler ? */

/* old function of SATELITE */
static int    write_data(int fd, char *data, int elsize, int siz, int flag);
static char  *read_data(int fd, int elsize, int siz, int flag);

/* private functions */
static int   _WriteFile_with_mode(const char *fname, int dim, int *index, 
                                  char *data, int mode);
static int    openfile(const char *fname, Header *head);
static int    fillgap(int fd, int bsize);
static int    LoadHeader_sub(int fd, Header *head);
static int    StoreHeader_sub(int fd, Header *head);
static int    TotalDataSize(Header *head);
static int    cpu_type();
static void   ReverseData(char *data, int siz, int length);
static void   ReverseHead(Header *head);

/* interface of rwfile1.c */
extern char  *read_old_file(const char *fname, int numb, int *idx);
extern char  *read_old_file2(const char *fname, int *dim, int *idx);

/**************************************************************
  --------- Public Functions for Data File Operation  -----------
  ReadFile()  : read data file
  WriteFile() : write data fIle
  LoadData()  : read series-block from data file
  StoreData() : write series-block to data file
  FreeData()  : free memory
  ---------------------------------------------------------------
  ***************************************************************/
DLLEXPORT char *_ReadFile(const char *fname, Header *head){
/* fname  : data file name */
/* head   : header of data */
  char *area;
  int   siz, fd;
  fd = open(fname, O_RDONLY | O_BINARY);
  if(fd < 0) return NULL;
  if(LoadHeader_sub(fd, head) == -1) return NULL;
  if(head->type_flag == 0 || head->type_flag == 1){
    int  dim;
    close(fd);
    area = read_old_file2(fname, &dim, head->index);
    if(area == NULL) return NULL;
    head->dim = dim;
    head->data_size = 4;
    return area;
  }
  siz = TotalDataSize(head);
  area = read_data(fd, head->data_size, siz, head->type_flag);
  if(area == NULL) return NULL;
  close(fd);
  return area;
}

DLLEXPORT int _WriteFile(const char *fname, int dim, int *index, char *data){
  int ret;
  if(is_compat2x_mode())
    ret = _WriteFile_with_mode(fname, dim, index, data, 1);
  else
    ret = _WriteFile_with_mode(fname, dim, index, data, 0);
  return ret;
}

DLLEXPORT int _WriteFile2(const char *fname, int dim, int *index, char *data){
  return _WriteFile_with_mode(fname, dim, index, data, 1);
}

DLLEXPORT int _WriteFile4(const char *fname, int dim, int *index, char *data){
  return _WriteFile_with_mode(fname, dim, index, data, 0);
}

static int _WriteFile_with_mode(const char *fname, int dim, int *index, 
                                char *data, int mode){
/* fname : data file name  */
/* dim    : dimension      */
/* index  : index          */
/* data   : data           */
/* mode   : index fixing   */
  int  fd, siz;
  Header head;
  InitHeader(&head);
  if(dim == 1 && mode == 1) {
    head.dim = 2;
    head.index[0] = index[0];
    head.index[1] = 1;
  } else {
    head.dim = dim;
    CopyIndex(head.index, index, head.dim);
  }
  /* file create */
  fd = open(fname, O_RDWR | O_CREAT | O_TRUNC | O_BINARY, 
            S_IWUSR | S_IRUSR | S_IRGRP | S_IROTH);
  if (fd == -1)
    return -1;
  siz = TotalDataSize(&head);
  if (StoreHeader_sub(fd, &head) == -1)
    return -1;
  if (write_data(fd, data, head.data_size, siz, head.type_flag) == -1)
    return -1;
  close(fd);
  return 0;
}

DLLEXPORT char *LoadData(const char *fname, int numb, Header *head){
/* fname : data file name      */
/* numb   : number of block     */
/* head   : header of read data */
  int     i;
  char   *area;
  long    offset;
  int     aBlockSize = 0;
  int     fd = open(fname, O_RDONLY | O_BINARY);
  if (fd == -1)
    return NULL;
  if(LoadHeader_sub(fd, head) == -1)
    return NULL;
  if(head->type_flag == 0 || head->type_flag == 1){
    close(fd);
    area = read_old_file(fname, numb, head->index);
    if(area == NULL)
      return NULL;
    head->dim = 1;
    head->data_size = 4;
    return area;
  }
#if 0
  PrintHeader(head);
#endif
  if(head->dim == 0)
    return NULL;
  if(head->index[0] < numb)
    return NULL;
  aBlockSize = TotalDataSize(head) / head->index[0];  /* [byte] */
  offset = sizeof(Header) + (long) aBlockSize *(long) numb;
  lseek(fd, offset, 0);  /* set the seek pointer */
  area = read_data(fd, head->data_size, aBlockSize, head->type_flag);
  if(area == NULL)
    return NULL;
  head->dim--;
  for(i=0;i<head->dim;i++)
    head->index[i]=head->index[i+1];
  close(fd);
  return area;
}

DLLEXPORT int StoreData(const char *fname, int numb, int dim, 
                        int *index, char *data){
/* fname  : data file name  */
/* numb    : number of block */
/* dim     : dimension       */
/* index   : index           */
/* data    : write data      */
  int     aBlockSize;
  Header  head;
  int     bsiz, fsiz, old_numb;
  int     fd = openfile(fname, &head);
  if(fd == -1) return -1;
  if(head.data_size != sl2_get_data_size()) {
    fprintf(stderr, "%s : a data size mismatch", fname);
    return -1;
  }
  if(head.dim == 0) {
    head.dim = dim + 1;
    CopyIndex(&head.index[1], index, dim);
  }
  old_numb = head.index[0];
  head.index[0] = Max(old_numb, numb + 1);
  bsiz = IndexSize(dim, index);
  fsiz = IndexSize(head.dim-1, SubIndex(head.index));
  if(bsiz != fsiz) {
    fprintf(stderr, "%s : block size mismatch", fname);
    return -1;
  }
  aBlockSize = bsiz * sl2_get_data_size(); /* [byte] */
  if (StoreHeader_sub(fd, &head) == -1)
    return -1;
  if (numb > head.index[0]) {
    fillgap(fd, aBlockSize * (numb - old_numb));
  } else {
    long            offset;
    offset = sizeof(Header) + (long) aBlockSize *(long) numb;
    lseek(fd, offset, 0);  /* set the seek pointer */
  }
  if (write_data(fd, data, head.data_size, aBlockSize, head.type_flag) == -1)
    return -1;
  close(fd);
  return 0;
}

DLLEXPORT int StoreTimeData(const char *fname, int sub_dim, int *sub_index, 
                            char *data, int length){
/* fname     : data file name  */
/* sub_dim   : dimension       */
/* sub_index : index           */
/* data      : write data      */
/* length    : writing point   */

  register int  i;
  int           fd;
  int           sub_siz, aBlockSize, zero_point;
  long          offset;
  Header        head;

  fd = openfile(fname, &head);
  if(fd == -1) return -1;
  if(head.data_size != sl2_get_data_size()) {
    fprintf(stderr, "%s : a data size mismatch", fname);
    return -1;
  }
  if(head.dim - 1 != sub_dim){
    fprintf(stderr, "%s : dimension mismatch", fname);
    return -1;
  }
  if(sub_index == NULL){
    fprintf(stderr, "%s : illegal index", fname);
    return -1;
  }

  sub_siz = (sub_dim == 0) ? 1 : IndexSize(head.dim - 1, SubIndex(head.index));
  aBlockSize = sub_siz * head.data_size;

  /**** Append Data ******/
  if(head.index[0] < length){
    fillgap(fd, aBlockSize * (length - head.index[0]));
    head.index[0] = length;
    /* store resizing header */
    if(StoreHeader_sub(fd, &head) == -1){
      return -1;
    }
  }

  /* set the seek pointer at a point of zero [time] */
  zero_point = _Index(sub_index, head.dim - 1, SubIndex(head.index));
  offset = sizeof(Header) + zero_point * head.data_size;
  lseek(fd, offset, 0);
  for(i=0; i < length; i++){
    if(write_data(fd, data + (head.data_size * i), head.data_size, 
                  head.data_size, head.type_flag) == -1){
      return -1;
    }
    offset += aBlockSize;
    lseek(fd, offset, 0);       /* increment the seek pointer */
  }

  close(fd);
  return 0;
}

/****************************************************************
  --------- Private Functions for Data File Operation  ----------
  openfile()  : open exist file
  fillgap()   : fill a gap ( zero set )
  write_data(): write data block
  read_data() : read data block
  ---------------------------------------------------------------
  ***************************************************************/

static int openfile(const char *fname, Header *head){
/* fname : data file name */
/* head   : header of data */
  int fd;
  if (Access(fname, SL_FATTR_FOK) != 0) {
    if((fd = open(fname, O_RDWR | O_CREAT | O_TRUNC | O_BINARY, 
                  S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)) == -1)
      return(-1);
    InitHeader(head);
  } else {
    if((fd = open(fname, O_RDWR | O_BINARY)) == -1)
      return (-1);
    if (LoadHeader_sub(fd, head) == -1)
      return (-1);
  }
  return fd;
}

static int fillgap(int fd, int bsize){
/* fd    : file descriptor   */
/* bsize : block size of gap */
  char *area = (char *) malloc(bsize);
  if (area == NULL)
    return -1;
  memset(area, 0, bsize);
  lseek(fd, 0L, 2);  /* extend the file size */
  if (write(fd, area, bsize) != bsize)
    return -1;
  free(area);
  return 0;
}


static int write_data(int fd, char *data, int elsize, int siz, int flag){
/* fd     : file descriptor     */
/* data   : write  data         */
/* elsize : element size [byte] */
/* siz    : total reading size  */
/* flag   : cpu type flag       */
  if (flag != cpu_type())
    ReverseData((char *) data, elsize, siz / elsize);
  if (write(fd, data, siz) != siz){
    close(fd);
    return -1;
  }
  if (flag != cpu_type())
    ReverseData((char *) data, elsize, siz / elsize);
  return 0;
}

static char *read_data(int fd, int elsize, int siz, int flag){
/* fd     : file descriptor     */
/* elsize : element size [byte] */
/* siz    : total reading size  */
/* flag   : cpu type flag       */
  char *area = (char *)malloc(siz);
  if(area == NULL)
    return NULL;
  if(read(fd, area, siz) != siz)
    return NULL;
  if(flag != cpu_type())
    ReverseData((char *) area, elsize, siz / elsize);
  return area;
}


/**************************************************************
  ---------- Public Functions for Header Operation  -------------
  StoreHeader(): Store Header //include ReverseHeader()
  LoadHeader() : Load Header  //include ReverseHeader()
  ---------------------------------------------------------------
  ***************************************************************/
#if 0
DLLEXPORT void PrintHeader(Header *head){
/* head : header of data */
  int  i;
  printf("type_flag = %d\n", (int)head->type_flag);
  printf("data_size = %d\n", (int)head->data_size);
  printf("opr_name  = [%s]\n", head->opr_name);
  printf("comment   = [%s]\n", head->comment);
  printf("date      = [%d:%d:%d]\n",
    head->date[0], head->date[1], head->date[2]);
  printf("dim       = %d\n", (int)head->dim);
  printf("index     = ");
  for(i = 0; i < 10; i++)
    printf("[%d]", head->index[i]);
  puts("");
  printf("samf      = %f\n", head->samf);
}
#endif

DLLEXPORT void InitHeader(Header *head){
/* head : header of data */
  int             y, m, d;
  memset((char *)head, 0, sizeof(Header));
  head->type_flag = (char) sl2_get_data_type();  /* Default = 0 */
  head->data_size = (char) sl2_get_data_size();  /* Default = 4 byte */
  getusername(head->opr_name, sizeof(head->opr_name));
  get_date(&y, &m, &d);
  head->date[0] = (char) y;
  head->date[1] = (char) m;
  head->date[2] = (char) d;
}


DLLEXPORT int LoadHeader(const char *fname, Header *head){
/* fname : data file name */
/* head  : header of data */
  int fd = open(fname, O_RDONLY | O_BINARY);
  if (fd == -1)
    return -1;
  LoadHeader_sub(fd, head);
  close(fd);
  return 0;
}

DLLEXPORT int StoreHeader(const char *fname, Header *head){
/* fname : data file name */
/* head  : header of data */
  int fd = open(fname, O_RDWR | O_BINARY);
  if (fd == -1)
    return -1;
  StoreHeader_sub(fd, head);
  close(fd);
  return 0;
}

DLLEXPORT void FreeData(void *mem){
/* mem : memory */
  free(mem);
}

static int LoadHeader_sub(int fd, Header *head){
/* fd   : file discripter */
/* head : header of data  */
  lseek(fd, 0L, 0);  /* begin of file */
  if(read(fd, head, sizeof(Header)) != sizeof(Header))
    return (-1);
  
  if(head->type_flag != cpu_type())
    ReverseHead(head);
  
  return 0;
}

static int StoreHeader_sub(int fd, Header *head){
/* fd   : file discripter */
/* head : header of data  */
  int             status = 0;

  if (head->type_flag != cpu_type())
    ReverseHead(head);
  
  lseek(fd, 0L, 0);  /* begin of file */
  if (write(fd, head, sizeof(Header)) != sizeof(Header))
    status = -1;
  
  if (head->type_flag != cpu_type())
    ReverseHead(head);
  return status;
}

/**************************************************************
  -----------------------  Private Functions --------------------
  TotalDataSize()  : Total Data Size
  ReverseData()   : Reverse the byte order (Data)
  ReverseHead()   : Reverse the byte order (Header)
  ---------------------------------------------------------------
  local function
  reverse() : reverse the byte order
  (2, 4, and 8 byte data)
  ***************************************************************/

static int TotalDataSize(Header *head){
/* head : header of data  */
  int  isiz = IndexSize(head->dim, head->index);
  return isiz * head->data_size; /* [byte] */
}

static int cpu_type(){
  short i = 0x0001;
  char *c = (char *)&i;
  return ((int)c[0] == 1) ? 
    SL2_DATAFILE_TYPE_LITTLEENDIAN : SL2_DATAFILE_TYPE_BIGENDIAN;
}

static void reverse(unsigned char *s, int siz){
  register int      i;
  register unsigned char c;
  int  half = siz / 2, j = siz - 1;
  for (i = 0; i < half; i++) {
    c = s[i];
    s[i] = s[j - i];
    s[j - i] = c;
  }
}

static void ReverseData(char *data, int siz, int length){
/* data   : pointer of data area */
/* siz    : size of data         */
/* length : length of data       */
  register int    i;
  for (i = 0; i < length; i++)
    reverse((unsigned char*)&(data[i * siz]), siz);
}

static void ReverseHead(Header *head){
/* head  : header of data */
  register int i;
  for (i = 0; i < 10; i++)
    reverse((unsigned char*)&(head->index[i]), sizeof(int));
  reverse((unsigned char*)&(head->samf), sizeof(float));
}

