/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: string.c,v 1.6 2005/10/27 04:19:27 orrisroot Exp $ */
#define LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libsatellite.h"

#define STRING_BLOCKSIZE 32

#define TRIM_MODE_RIGHT 0x01
#define TRIM_MODE_LEFT  0x02
#define TRIM_MODE_ALL   0x03

static int _sl4_string_resize(sl4_string_t *str, size_t len);
static int _sl4_string_trim_sub(sl4_string_t *str, int mode);

DLLEXPORT sl4_string_t *sl4_string_new(){
  sl4_string_t *str;
  str = (sl4_string_t*)malloc(sizeof(sl4_string_t));
  if(str == NULL) return NULL;
  str->bsize = 0;
  str->len = 0;
  str->mem = NULL;
  return str;
}

DLLEXPORT sl4_string_t *sl4_string_new_char(char c){
  sl4_string_t *ret;
  ret = sl4_string_new();
  if(ret == NULL) return NULL;
  if(sl4_string_set_char(ret, c) != 0){
    sl4_string_delete(ret);
    return NULL;
  }
  return ret;
}

DLLEXPORT sl4_string_t *sl4_string_new_cstr(const char *cstr){
  sl4_string_t *ret;
  ret = sl4_string_new();
  if(ret == NULL) return NULL;
  if(sl4_string_set_cstr(ret, cstr) != 0){
    sl4_string_delete(ret);
    return NULL;
  }
  return ret;
}

DLLEXPORT sl4_string_t *sl4_string_new_str(sl4_string_t *str){
  sl4_string_t *ret;
  if(str == NULL) return NULL;
  ret = sl4_string_new();
  if(ret == NULL) return NULL;
  if(sl4_string_set_str(ret, str) != 0){
    sl4_string_delete(ret);
    return NULL;
  }
  return ret;
}

DLLEXPORT int sl4_string_delete(sl4_string_t *str){
  if(str->mem != NULL) free(str->mem);
  free(str);
  return 0;
}

DLLEXPORT int sl4_string_set_cstr(sl4_string_t *dst, const char *src){
  size_t len;
  if(src == NULL) src = "";
  len = strlen(src);
  if(_sl4_string_resize(dst, len) != 0) return -1;
  if(len != 0){
    strcpy(dst->mem, src);
    dst->len = len;
  }
  return 0;
}

DLLEXPORT int sl4_string_set_char(sl4_string_t *dst, char c){
  char cstr[2];
  cstr[0] = c; cstr[1] = '\0';
  return sl4_string_set_cstr(dst, cstr);
}

DLLEXPORT int sl4_string_set_str(sl4_string_t *dst, sl4_string_t *src){
  return sl4_string_set_cstr(dst, src->mem);
}

DLLEXPORT const char *sl4_string_get_cstr(sl4_string_t *str){
  if(str->mem)
    return str->mem;
  return "";
}

DLLEXPORT char sl4_string_get_at(sl4_string_t *str, size_t offset){
  if(str->len > offset)
    return str->mem[offset];
  return '\0';
}

DLLEXPORT size_t sl4_string_length(sl4_string_t *str){
  return str->len;
}

DLLEXPORT int sl4_string_empty(sl4_string_t *str){
  return (str->len == 0);
}

DLLEXPORT int sl4_string_insert_cstr(sl4_string_t *dst, size_t offset, 
                                     const char *src){
  size_t alen, nlen, shiftlen;
  if(dst->len > offset) return -1;
  if(src == NULL) return 0;
  alen = strlen(src);
  /* if src string length is zero then return as success */
  if(alen == 0) return 0;
  nlen = dst->len + alen;
  if(_sl4_string_resize(dst, nlen) != 0) return -1;
  /* shift memory */
  shiftlen = dst->len - offset;
  if(shiftlen != 0)
    memmove(dst->mem+offset+alen, dst->mem+offset, shiftlen);
  /* insert memory */
  memcpy(dst->mem+offset, src, alen);
  /* null terminate */
  dst->mem[nlen] = '\0';
  /* set new string length */
  dst->len = nlen;
  return 0;
}

DLLEXPORT int sl4_string_insert_char(sl4_string_t *dst, size_t offset, 
                                     char c){
  char cstr[2];
  cstr[0] = c;  cstr[1] = '\0';
  return sl4_string_insert_cstr(dst, offset, cstr);
}

DLLEXPORT int sl4_string_insert_str(sl4_string_t *dst, size_t offset, 
                                    sl4_string_t *src){
  return sl4_string_insert_cstr(dst, offset, src->mem);
}

DLLEXPORT int sl4_string_append_cstr(sl4_string_t *dst, const char *src){
  return sl4_string_insert_cstr(dst, dst->len, src);
}

DLLEXPORT int sl4_string_append_char(sl4_string_t *dst, char c){
  return sl4_string_insert_char(dst, dst->len, c);
}

DLLEXPORT int sl4_string_append_str(sl4_string_t *dst, sl4_string_t *src){
  return sl4_string_insert_str(dst, dst->len, src);
}

DLLEXPORT int sl4_string_erase(sl4_string_t *str, size_t offset, size_t len){
  size_t nlen;
  if(offset >= str->len) return -1;
  if(len == 0) len = str->len - offset;
  if(offset + len > str->len) return -1;
  if(str->len == 0 || len == 0) return 0; /* success */
  nlen = str->len - len;
  if(nlen == 0) return sl4_string_set_cstr(str, NULL);
  /* shift */
  memmove(str->mem+offset, str->mem+offset+len, len);
  /* null terminate */
  str->mem[nlen] = '\0';
  /* set new string length */
  str->len = nlen;
  /* resize memory block */
  _sl4_string_resize(str, nlen); /* if an error occured then no problem */
  return 0;
}

DLLEXPORT int sl4_string_clear(sl4_string_t *str){
  return sl4_string_set_cstr(str, NULL);
}

DLLEXPORT sl4_string_t *sl4_string_substr(sl4_string_t *str, size_t offset,
                                          size_t len){
  sl4_string_t *ret;
  if(offset >= str->len) return NULL;
  if(len == 0) len = str->len - offset;
  if(offset + len > str->len) return NULL;
  /* allocate new string */
  ret = sl4_string_new();
  if(ret == NULL) return NULL;
  /* resize memory of new string */
  if(_sl4_string_resize(ret, len) != 0){
    sl4_string_delete(ret);
    return NULL;
  }
  memcpy(ret->mem, str->mem+offset, len);
  ret->mem[len] = '\0';
  ret->len = len;
  return ret;
}

DLLEXPORT int sl4_string_trim(sl4_string_t *str){
  return _sl4_string_trim_sub(str, TRIM_MODE_ALL);
}

DLLEXPORT int sl4_string_ltrim(sl4_string_t *str){
  return _sl4_string_trim_sub(str, TRIM_MODE_LEFT);
}

DLLEXPORT int sl4_string_rtrim(sl4_string_t *str){
  return _sl4_string_trim_sub(str, TRIM_MODE_RIGHT);
}

DLLEXPORT int sl4_string_compare(sl4_string_t *s1, sl4_string_t *s2){
  const char *c1 = sl4_string_get_cstr( s1 );
  const char *c2 = sl4_string_get_cstr( s2 );
  return strcmp(c1, c2);
}

DLLEXPORT size_t sl4_string_find(sl4_string_t *str, const char *target){
  return sl4_string_findn(str, target, 0);
}

DLLEXPORT size_t sl4_string_findn(sl4_string_t *str, const char *target,
                                  size_t spos){
  size_t i;
  const char *t, *p;
  const char *base = sl4_string_get_cstr( str );
  size_t len = sl4_string_length( str );
  if( len == 0 || spos > len || target == NULL || *target == '\0')
    return -1;
  for(i=spos; i<len; i++){
    p = base + i;
    t = target;
    while( *p == *t ){
      p++; t++;
      if( *t == '\0' ){ /* found ! */
        return i;
      }
    }
  }
  return -1;
}

DLLEXPORT size_t sl4_string_rfind(sl4_string_t *str, const char *target){
  return sl4_string_rfindn(str, target, 0);
}

DLLEXPORT size_t sl4_string_rfindn(sl4_string_t *str, const char *target,
                                   size_t spos){
  size_t i;
  const char *t, *p;
  const char *base = sl4_string_get_cstr( str );
  size_t len = sl4_string_length( str );
  if( len == 0 || spos > len || target == NULL || *target == '\0')
    return -1;
  for(i=spos; ;i--){
    p = base + i;
    t = target;
    while( *p == *t ){
      p++; t++;
      if( *t == '\0' ){ /* found ! */
        return i;
      }
    }
    if(i == 0)
      break;
  }
  return -1;
}



static int _sl4_string_resize(sl4_string_t *str, size_t len){
  size_t nlen, nbsi, olen, obsi;
  char  *nmem, *omem;
  omem = str->mem;
  olen = str->len;
  obsi = str->bsize;
  nbsi = (len == 0) ? 0 : len/STRING_BLOCKSIZE + 1;
  /* if new block size is same size then return as success */
  if( nbsi == obsi ) return 0;
  /* allocate new memory */
  if( nbsi == 0 ){
    nmem = NULL;
  } else {
    nmem = (char*)malloc(sizeof(char) * STRING_BLOCKSIZE * nbsi);
    if(nmem == NULL) return -1;
  }
  nlen = (len < olen) ? len : olen;
  /* copy memory */
  if(nlen != 0){
    memcpy(nmem, omem, nlen);
    /* null terminate new memory */
    nmem[nlen] = '\0';
  }
  /* assgin new variables to sl4_string_t */
  str->mem   = nmem;
  str->len   = nlen;
  str->bsize = nbsi;
  /* free old memory */
  if(omem != NULL) free(omem);
  /* success */
  return 0;
}

static int _sl4_string_trim_sub(sl4_string_t *str, int mode){
  static const char *trimchar     =" \t\n\r\0\x0b";
  static int         trimchar_num = 6;
  static int first = 1;
  static char mask[256];
  size_t i, len;
  int    j;
  size_t alltrim, ltrim=0, rtrim=0;
  if(str->mem == NULL) return 0;
  len = str->len;

  /* set trimming character map */
  if(first){
    memset(mask,0,256);
    for(j=0;j<trimchar_num;j++){
      mask[((unsigned char*)trimchar)[j]]=1;
    }
    first=0;
  }
  if(mode & TRIM_MODE_LEFT){ /* trim left of string */
    for(i=0; i<len; i++){
      if(mask[((unsigned char)str->mem[i])]) ltrim++;
      else break;
    }
  }
  if(mode & TRIM_MODE_RIGHT){ /* trim right of string */
    for(i=len-1;i>=0;i--){
      if(mask[((unsigned char)str->mem[i])]) rtrim++;
      else break;
    }
  }
  alltrim = ltrim + rtrim;
  if(alltrim > len){
    len = 0;
  } else {
    len -= alltrim;
  }
  if(len == 0)
    return _sl4_string_resize(str, len);
  for(i=0;i<len;i++){
    str->mem[i]=str->mem[ltrim+i];
  }
  str->mem[len] = '\0';
  str->len = len;
  /* success */
  return 0;
}
