/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: tty_console.cpp,v 1.10 2005/10/27 04:23:56 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32
#include <windows.h>
#endif
#include "SL_header.h"
#include <errno.h>
#include <stdarg.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_DIRECT_H
# include <direct.h>
#endif
#ifdef HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif
#ifdef HAVE_PROCESS_H
# include <process.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif

using namespace std;

#include "libsatellite.h"
#define  __EXPORTSYMBOL__
#include "SL_Tool.h"
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "path.h"

#define HISTORY_FILE "history.dat"

#ifndef STR_BUF_SIZE
#define STR_BUF_SIZE 256
#endif

#ifndef ONELINE
#define ONELINE     512
#endif

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#ifdef WIN32
#define lstat(x,y) stat((x),(y))
#define isatty(x)  _isatty((x))
#define fileno(x)  _fileno((x))
#endif

#define IS_KANJI(x) ( ((x)&0x80) == 0x80 )
#define IS_CTRL(x)  ( (x)>=0x00 && (x)<=0x1b )
#define POSX(x)     (((x)+prompt_length) % col)
#define POSY(y)     (((y)+prompt_length) / col)

static string prompt_fmt_p();
static void   split_path(string &buf, string &path, string &file,int mode);
static int    split_last_key(string &buf);
static int    is_quoted(const char *buf);
static void   trim_head_whitespace(string &buf);
static void   collect_file_list(list<string> &flist, const char *path, 
                                const char *fname, const char *arg);
static void   collect_satcom_list(list<string> *flist, hash_table_t *coms,
                                  const char *word, int mode);

static string      _current_func(const char *str, int *no);
static const char *_last_str(const char *str);

void tty_console::editbuf_standby(){
  ebuf.erase();
  pos = 0;
}

void tty_console::editbuf_trim(){
  size_t i;
  for(i=0;i<ebuf.length();i++){
    if(ebuf[i] == ' ' || ebuf[i] == '\t') ebuf.erase(i,1);
    else break;
  }
  for(i=ebuf.length()-1;i>0;i--){
    if(ebuf[i] == ' ' || ebuf[i] == '\t') ebuf.erase(i,1);
    else break;
  }
}

void tty_console::editbuf_addnewline(){
  ebuf.insert(ebuf.length(),"\n");
}

void tty_console::editbuf_insert(int c, size_t p){
  ebuf.insert(p,1,(char)c);
}

void tty_console::editbuf_delete(size_t p){
  ebuf.erase(p,1);
}

size_t tty_console::editbuf_length(size_t p){
 size_t i,len(0);
  int kanji(0);
  for(i=0;i<p;i++,len++){
    /* KANJI 2byte */
    if(kanji==1){ kanji=0; len++; continue; }
    /* KANJI 1byte */
    if(IS_KANJI(ebuf[i])){ kanji=1; len--; continue; }
    /* 0x01 "^A" - 0x1a "^Z" , 0x1b "^[" */
    if(IS_CTRL(ebuf[i])) len++;
  }
  return len;
}

void tty_console::editbuf_print(){
  size_t i;
  int tcol;
  int kanji=0;
  char c;
  cursor_goto_pos(0);
  tcol=POSX(0);
  for(i=0;i<ebuf.length();i++){
    c=ebuf[i];
    if(kanji){             /* KANJI 2 byte */
      kanji=0;
      if(tcol==0) c='?';
    }else if(IS_KANJI(c)){ /* KANJI 1 byte */
      kanji=1;
      if(tcol==col-1) c='?';
    }else if(IS_CTRL(c)){  /* 0x01 "^A" - 0x1a "^Z", 0x1b "^[" */
      term_putc('^');
      tcol++;
      if(tcol == col){ tcol=0; term_move_newline(); }
      c+='A'-1;
    }
    term_putc(c);
    tcol++;
    if(tcol == col){ tcol=0; term_move_newline(); }
  }
  pos = ebuf.length();
  cursor_goto_pos(0);
}

int tty_console::editbuf_is_kanji2(size_t p){
  size_t i;
  int kanji(0);
  for(i=0;i<p;i++){
    if(kanji){ kanji=0; continue; }
    if(IS_KANJI(ebuf[i])){ kanji=1; }
  }
  return kanji;
}

void tty_console::cursor_goto_pos(size_t p){
  int dummy,posy,posx;
  size_t cpos,tpos;
  term_getmaxyx(&dummy,&col);
  cpos=editbuf_length(pos);
  tpos=editbuf_length(p);
  posy=(int)(POSY(cpos)-POSY(tpos)); /* TODO: remove cast */
  posx=(int)(POSX(cpos)-POSX(tpos)); /* TODO: remove cast */
  if(posy<0){
    term_move_down(-posy);
  }else if(posy>0){
    term_move_up(posy);
  }
  if(posx<0){
    term_move_right(-posx);
  }else if(posx>0){
    term_move_left(posx);
  }
  pos=p;
}

void tty_console::_keyevent_ndelete(int n){
  size_t cpos,epos,tmp_pos;
  int i,posx,posy;
  if(ebuf.length()==0 || ebuf.length() < pos+n){ return; }
  cpos=editbuf_length(pos);
  epos=editbuf_length(ebuf.length());
  posy=(int)(POSY(epos)-POSY(cpos)); /* TODO: remove cast */
  posx=(int)POSX(cpos);              /* TODO: remove cast */
  term_clear_eol(posx);
  for(i=0;i<posy;i++){
    term_move_down(1);
    term_move_bol();
    term_clear_eol(0);
  }
  if(posy!=0){
    term_move_up(posy);
    term_move_right(posx);
  }
  for(i=0;i<n;i++){
    editbuf_delete(pos);
  }
  tmp_pos=pos;
  editbuf_print();
  cursor_goto_pos(tmp_pos);
}


int tty_console::keyevent_ignore(int c){
  if(input_lock){
    input_lock_len=0;
    input_lock_less=0;
    input_lock=false;
    term_hide_help();
  }
  /* ignore key */
  return 1;
}


int tty_console::keyevent_newline_or_help(int c){
  size_t i;
  int lin_argc,com_argc;
  const char *func,*tmp;
  module_command_t *tmp_cmd;
  tmp_cmd=0;
  string a;
  if(input_lock){
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    if(input_lock_less > 0)
      editbuf_insert(',',ebuf.length());
    if(input_lock_less == 0)
      editbuf_insert(')',ebuf.length());
    input_lock_less=0;
    editbuf_print();
    cursor_goto_pos(ebuf.length());
  }
  keyevent_move_eol(c);
 retry:
/* phase 1 : check missing quote */
  {
    int status;
    strutil_check_quote(ebuf.c_str(),'\0', &status);
    switch(status){
    case 0x01: /* in single quoted string */
      editbuf_insert('\'',ebuf.length());
      editbuf_print();
      cursor_goto_pos(ebuf.length());
      break;
    case 0x02: /* in double quoted string */
      editbuf_insert('"',ebuf.length());
      editbuf_print();
      cursor_goto_pos(ebuf.length());
      break;
    case 0x04: /* in escape */
      editbuf_insert('\\',ebuf.length());
      editbuf_print();
      cursor_goto_pos(ebuf.length());
      break;
    case 0x06: /* in double quoted string and in escape */
      editbuf_insert('\\',ebuf.length());
      editbuf_insert(':',ebuf.length());
      editbuf_print();
      cursor_goto_pos(ebuf.length());
      break;
    }
  }
  func = _last_str(ebuf.c_str());
  if(*func!='\0'){
    /* phase 2 : check '(' */
    tmp_cmd = (module_command_t*)hash_table_lookup(commands, func);
    if(tmp_cmd){
      editbuf_insert('(',ebuf.length());
      editbuf_print();
      cursor_goto_pos(ebuf.length());
    }
  }
  /* phase 3 : check ')' */
  a = _current_func(ebuf.c_str(), &lin_argc);
  if(!a.empty()){
    tmp_cmd = (module_command_t*)hash_table_lookup(commands, a.c_str());
    if(tmp_cmd){
      com_argc = tmp_cmd->argc;
      tmp = _last_str(ebuf.c_str());
      if(com_argc == 0 || (com_argc == lin_argc +1 && *tmp!='\0')){
        editbuf_insert(')',ebuf.length());
        editbuf_print();
        cursor_goto_pos(ebuf.length());
        tmp_cmd = 0;
        goto retry;
      }
  /* phase 4 : check ',' */
      if(*tmp!='\0'){
        editbuf_insert(',',ebuf.length());
        editbuf_print();
        cursor_goto_pos(ebuf.length());
        goto retry;
      }
    }
  }
  if(tmp_cmd && tmp_cmd->argc > lin_argc){
    const char *com_mesg,*com_argv;
    com_mesg = tmp_cmd->messages[lin_argc];
    com_argc = tmp_cmd->argc;
    com_argv = tmp_cmd->argv[lin_argc];
    input_lock_len  = strlen(com_argv);
    input_lock_less = com_argc-lin_argc-1;
    for(i=0;i<input_lock_len;i++)
      editbuf_insert(com_argv[i],pos+i);
    i=pos;
    editbuf_print();
    cursor_goto_pos(i);
    term_show_help(com_mesg);
    input_lock=true;
    return 1;
  }
  term_move_newline();
  editbuf_trim();
  editbuf_addnewline();
  return 0;
}

int tty_console::keyevent_newline(int c){
  term_move_newline();
  editbuf_trim();
  editbuf_addnewline();
  return 0;
}

int tty_console::keyevent_insert(int c){
  size_t to;
  int kanji(0),c2;
  if(input_lock){
    _keyevent_ndelete((int)input_lock_len);
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
//    return 1;
  }
  if(ebuf.length() == ONELINE-1){
    if(use_term_vbell) term_vbell();
    else               term_bell();
  }else{
    history_standby(hist);
    kanji=IS_KANJI(c);
    if(kanji){
      c2=term_getc();
      if(ebuf.length()==ONELINE-2){
        if(use_term_vbell) term_vbell();
        else               term_bell();
        return 1;
      }
      editbuf_insert(c,pos);
      editbuf_insert(c2,pos+1);
    }else{
      editbuf_insert(c,pos);
    }
    to=pos+1+kanji;
    editbuf_print();
    cursor_goto_pos(to);
  }
  return 1;
}

int tty_console::keyevent_delete(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }

  if(ebuf.length()==0 || ebuf.length()==pos){
    if(use_term_vbell) term_vbell();
    else               term_bell();
  }else{
    int kanji;
    history_standby(hist);
    kanji=IS_KANJI(ebuf[pos]);
    _keyevent_ndelete(1+kanji);
  }
  return 1;
}

int tty_console::keyevent_delete_or_eof(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  if(ebuf.length()==0){ return 0; }
  if(ebuf.length()==pos){
    if(use_term_vbell) term_vbell();
    else               term_bell();
  }else{
    int kanji;
    history_standby(hist);
    kanji=IS_KANJI(ebuf[pos]);
    _keyevent_ndelete(1+kanji);
  }
  return 1;
}

int tty_console::keyevent_delete_or_list_or_eof(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  if(ebuf.length()==0){ return 0; }
  if(ebuf.length()==pos){
    string buf,path,file;
    int mode;
    buf=ebuf;
    mode = split_last_key(buf);
    switch(mode){
    case 0: /* nothing */ 
      if(use_term_vbell) term_vbell();
      else               term_bell();
      break;
    case 1: /* satellite command list */
      {
        list<string> lst;
        buf = _last_str(ebuf.c_str());
        collect_satcom_list(&lst,commands,buf.c_str(),1);
        list_print(lst);  
      }
      break;
    case 2: /* file list */
      split_path(buf,path,file,0);
      ls_command(path.c_str(), file.c_str(), "Fa");
      break;
    case 3: /* external command list */
      split_path(buf,path,file,1);
      ls_command(path.c_str(), file.c_str(), "DEa");
      break;
    case 4: /* satellite & external command list */
      {
        list<string> lst;
        split_path(buf,path,file,1);
        collect_file_list(lst,path.c_str(),file.c_str(),"DEa");
        buf = _last_str(ebuf.c_str());
        collect_satcom_list(&lst,commands,buf.c_str(),1);
        list_print(lst);  
      }
        break;
    }
  }else{
    int kanji;
    history_standby(hist);
    kanji=IS_KANJI(ebuf[pos]);
    _keyevent_ndelete(1+kanji);
  }
  return 1;
}

int tty_console::keyevent_move_bol(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(pos!=0){ cursor_goto_pos(0); }
  return 1;
}

int tty_console::keyevent_move_eol(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(ebuf.length()!=pos){ cursor_goto_pos(ebuf.length()); }
  return 1;
}

int tty_console::keyevent_move_forward(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(ebuf.length()==pos){
    term_bell();
  }else{
    int kanji;
    kanji=IS_KANJI(ebuf[pos]);
    cursor_goto_pos(pos+1+kanji);
  }
  return 1;
}

int tty_console::keyevent_move_backward(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(pos==0){
    term_bell();
  }else{
    int kanji;
    kanji=editbuf_is_kanji2(pos-1);
    cursor_goto_pos(pos-1-kanji);
  }
  return 1;
}

int tty_console::keyevent_cancel(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  cursor_goto_pos(ebuf.length());
  term_move_newline();
  editbuf_standby();
  history_standby(hist);
  prompt_print();
  pos=0;
  return 1;
}

int tty_console::keyevent_backspace(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(pos==0){
    term_bell();
  }else{
    keyevent_move_backward(c);
    keyevent_delete(c);
  }
  return 1;
}

int tty_console::keyevent_mark(int c){
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  mark = pos;
  return 1;
}

int tty_console::keyevent_cut_region(int c){
  size_t min,max;
  int len;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(ebuf.length()==0) return 1;
  min = (pos<mark) ? pos : mark;
  max = (pos>mark) ? pos : mark;
  if(editbuf_is_kanji2(min)){ min--; }
  if(max>ebuf.length()) max=ebuf.length();
  len=(int)(max-min); /* TODO: remove cast */
  if(len==0) return 1;
  history_standby(hist);
  sl4_string_set_cstr(str_cutbuf, ebuf.substr(min,len).c_str());
  cursor_goto_pos(min);
  _keyevent_ndelete(len);
  return 1;
}

int tty_console::keyevent_yank(int c){
  size_t i,len,tmp_pos;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(sl4_string_empty(str_cutbuf)) return 1;
  len = sl4_string_length(str_cutbuf);
  if(ebuf.length()+len>=ONELINE) return 1;
  history_standby(hist);
  for(i=0;i<len;i++){
    editbuf_insert(sl4_string_get_at(str_cutbuf,i),pos+i);
  }
  tmp_pos=pos+len;
  editbuf_print();
  cursor_goto_pos(tmp_pos);
  return 1;
}

int tty_console::keyevent_kill_eol(int c){
  size_t min,max;
  int len;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(ebuf.length()==0) return 1; 
  min = pos;
  max = ebuf.length();
  len=(int)(max-min); /* TODO: remove cast */
  if(len==0) return 1;
  history_standby(hist);
  sl4_string_set_cstr(str_cutbuf, ebuf.substr(min,len).c_str());
  cursor_goto_pos(min);
  _keyevent_ndelete(len);
  return 1;
}

int tty_console::keyevent_trans_char(int c){
  size_t i;
  int kanji,kanji2;
  char tmp;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
  }
  if(ebuf.length()<=1 || pos==0) return 1;
  i=pos;
  if(pos == ebuf.length()){
    i--;
    kanji=editbuf_is_kanji2(i);
    if(kanji)i--;
  }else{
    kanji=IS_KANJI(ebuf[i]);
  }
  if(kanji && ebuf.length()==2) return 1;
  kanji2=editbuf_is_kanji2(i-1);
  if(kanji && kanji2){
    tmp=ebuf[i];
    ebuf[i]=ebuf[i-2];
    ebuf[i-2]=tmp;
    tmp=ebuf[i+1];
    ebuf[i+1]=ebuf[i-1];
    ebuf[i-1]=tmp;
  }else if(kanji){
    tmp=ebuf[i];
    ebuf[i]=ebuf[i+1];
    ebuf[i+1]=ebuf[i-1];
    ebuf[i-1]=tmp;
  }else if(kanji2){
    tmp=ebuf[i];
    ebuf[i]=ebuf[i-1];
    ebuf[i-1]=ebuf[i-2];
    ebuf[i-2]=tmp;
  }else{
    tmp=ebuf[i-1];
    ebuf[i-1]=ebuf[i];
    ebuf[i]=tmp;
  }
  editbuf_print();
  cursor_goto_pos(i+1+kanji);
  return 1;
}

int tty_console::keyevent_quote_input(int c){
  int n;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  n=term_getc();
  if(IS_KANJI(n)){
    n=term_getc(); /* ignore */
  }else{
    keyevent_insert(n);
  }
  return 1;
}

int tty_console::keyevent_history_prev(int c){
  int i=0;
  const char *tmp,*prev;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  prev=history_prev(hist);
  if(prev==NULL || *prev=='\0'){
    term_bell();
    return 1;
  }
  cursor_goto_pos(0);
  _keyevent_ndelete((int)ebuf.length()); /* TODO: remove cast */
  editbuf_standby();
  for(tmp=prev;*tmp!='\0';tmp++){
    editbuf_insert(*tmp,i);
    i++;
  }
  editbuf_print();
  cursor_goto_pos(ebuf.length());
  return 1;
}

int tty_console::keyevent_history_next(int c){
  int i=0;
  const char *tmp,*next;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  next=history_next(hist);
  if(next==NULL || *next=='\0'){
    next=hist->key;
    if(ebuf == next){
      term_bell();
      return 1;
    }
  }
  cursor_goto_pos(0);
  _keyevent_ndelete((int)ebuf.length()); /* TODO: remove cast */
  editbuf_standby();
  for(tmp=next;*tmp!='\0';tmp++){
    editbuf_insert(*tmp,i);
    i++;
  }
  editbuf_print();
  cursor_goto_pos(ebuf.length());
  return 1;
}

int tty_console::keyevent_history_search_prev(int c){
  int i=0;
  const char *tmp,*prev;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  if(hist->key == NULL || *hist->key == '\0'){
    if(hist->lock == 0)
      history_regist_key(hist,ebuf.c_str());
  }
  prev=history_search_prev(hist);
  if(prev==NULL || *prev=='\0'){
    term_bell();
    return 1;
  }
  cursor_goto_pos(0);
  _keyevent_ndelete((int)ebuf.length()); /* TODO: remove cast */
  editbuf_standby();
  for(tmp=prev;*tmp!='\0';tmp++){
    editbuf_insert(*tmp,i);
    i++;
  }
  editbuf_print();
  cursor_goto_pos(ebuf.length());
  return 1;
}

int tty_console::keyevent_history_search_next(int c){
  int i=0;
  const char *tmp,*next;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  if(hist->key == NULL || *hist->key == '\0'){
    if(hist->lock == 0)
      history_regist_key(hist,ebuf.c_str());
  }
  next=history_search_next(hist);
  if(next==NULL || *next=='\0'){
    next=hist->key;
    if(ebuf==next){
      term_bell();
      return 1;
    }
  }
  cursor_goto_pos(0);
  _keyevent_ndelete((int)ebuf.length()); /* TODO: remove cast */
  editbuf_standby();
  for(tmp=next;*tmp!='\0';tmp++){
    editbuf_insert(*tmp,i);
    i++;
  }
  editbuf_print();
  cursor_goto_pos(ebuf.length());
  return 1;
}

int tty_console::keyevent_complete_word(int c){
  char ch;
  int mode;
  size_t i,len;
  string buf(ebuf),path,keyword;
  list<string> lst;
  list<string>::iterator it;
  if(input_lock){
    input_lock_less=0;
    input_lock_len=0;
    input_lock=false;
    term_hide_help();
    return 1;
  }
  mode = split_last_key(buf);
  if(mode == 0) return 1;
  switch(mode){
  case 1: /* satellite command list */
    keyword = _last_str(ebuf.c_str());
    if(!keyword.empty())
      collect_satcom_list(&lst, commands, keyword.c_str(), 0);
    break;
  case 2: /* file list */
    split_path(buf,path,keyword,0);
    if(!keyword.empty())
      collect_file_list(lst,path.c_str(),keyword.c_str(),"a");
    break;
  case 3: /* external command list */
    split_path(buf,path,keyword,1);
    if(!keyword.empty())
      collect_file_list(lst,path.c_str(),keyword.c_str(),"DEa");
    break;
  case 4: /* satellite command & external command list */
    split_path(buf,path,keyword,1);
    if(!keyword.empty()){
      collect_satcom_list(&lst, commands, keyword.c_str(), 0);
      collect_file_list(lst,path.c_str(),keyword.c_str(),"DEa");
    }
  }
  if(lst.empty()){ return 1; }
  it=lst.begin();
  for(len=(*it).length();it!=lst.end();it++)
    if(len>(*it).length()) len=(*it).length();
  for(i=keyword.length();i<len;i++){
    it=lst.begin();
    for(ch=(*it)[i];it!=lst.end();it++)
      if(ch!=(*it)[i]) return 1;
    if(it==lst.end()){
      keyevent_insert(ch);
    }
  }
  return 1;
}

int tty_console::keyevent_list_satcom(int c){
  list<string> lst;
  collect_satcom_list(&lst, commands, _last_str(ebuf.c_str()), 0);
  if(!lst.empty()){
    list_print(lst);
  }
  return 1;
}

void tty_console::prompt_print(){
  char buf[_MAX_PATH];
  string pmes,tmp_str;
  size_t it, len;
  prompt_length = 0;
  len = sl4_string_length(str_prompt[cur_prompt_no]);
  for(it=0; it < len; it++){
    if(sl4_string_get_at(str_prompt[cur_prompt_no], it) == '%'){
      it++;
      if(it < len){
        switch ( sl4_string_get_at(str_prompt[cur_prompt_no], it) ){
        case '%': pmes+= "%"; prompt_length++; break;
        case '!': 
          snprintf(buf, _MAX_PATH, "%d", lineno);
          pmes += buf;
          prompt_length += strlen(buf);
          break;
        case 'P':
          if(getcwd(buf, _MAX_PATH)){
            pmes += buf;
            prompt_length += strlen(buf);
          }
          break;
        case 'p': 
          tmp_str += prompt_fmt_p();
          pmes += tmp_str;
          prompt_length += tmp_str.length();
          break;
        case 'B': 
          if(!pmes.empty()) term_print(pmes.c_str());
          pmes.erase();     tty_set_attr(SL_TTY::ATTR_BOLD);
          break;
        case 'R': 
          if(!pmes.empty()) term_print(pmes.c_str());
          pmes.erase();     tty_set_attr(SL_TTY::ATTR_REVERSE);
          break;
        case 'U': 
          if(!pmes.empty()) term_print(pmes.c_str());
          pmes.erase();     tty_set_attr(SL_TTY::ATTR_UNDERLINE);
          break;
        case 'N': 
          if(!pmes.empty()) term_print(pmes.c_str());
          pmes.erase();     tty_set_attr(SL_TTY::ATTR_NORMAL);
          break;
        default: break; // ignore other char.
        }
      }
    }else{
      prompt_length++;
      pmes += sl4_string_get_at(str_prompt[cur_prompt_no], it);
    }
  }
  if(!pmes.empty()) term_print(pmes.c_str());
}

void tty_console::term_show_help(const char *str){
  size_t len, y, tmp_pos;
  int mx, my;
  static char *mes="   ... ";
  if(hint_buf != 0){ term_hide_help(); }
  len = strlen(mes) + strlen(str);
  hint_buf = (char*)malloc(len+1);
  if(hint_buf == 0) return;
  strcpy(hint_buf,mes); /* safe */
  strcat(hint_buf,str); /* safe */
  hint_buf[len] = '\0';
  tmp_pos = pos;
  cursor_goto_pos(ebuf.length());
  term_getmaxyx(&my,&mx);
  term_move_newline();
  y=len/mx+1 + (prompt_length+ebuf.length())/mx;
  term_normal_mode();
  term_print(hint_buf);
  term_edit_mode();
  term_move_bol();
  term_move_up((int)y); /* TODO: remove cast */
  pos = 0;
  prompt_print();
  editbuf_print();
  cursor_goto_pos(tmp_pos);
  return;
}

void tty_console::term_hide_help(){
  size_t i, len, y, tmp_pos;
  int mx,my;
  if(hint_buf == 0) return;
  len = strlen(hint_buf);
  tmp_pos = pos;
  cursor_goto_pos(ebuf.length());
  term_getmaxyx(&my,&mx);
  y=len/mx+1;
  for(i=0;i<y;i++){ term_move_newline(); term_clear_eol(0); }
  y+=(prompt_length+ebuf.length())/mx;
  term_move_up((int)y); /* TODO: remove cast */
  pos = 0;
  prompt_print();
  editbuf_print();
  cursor_goto_pos(tmp_pos);
  free(hint_buf);
  hint_buf = 0;
  return;
}

void tty_console::set_keybind_default(){
  int i;
  for(i=0;i<0x20;i++)
    keymap[i]=SL_TTY::KEYEVENT_IGNORE;
  for(i=0x20;i<0xff;i++)
    keymap[i]=SL_TTY::KEYEVENT_INSERT;
  keymap[0x00]=SL_TTY::KEYEVENT_MARK;          /* ^@ */
  keymap[0x01]=SL_TTY::KEYEVENT_MOVE_BOL;      /* ^A */
  keymap[0x02]=SL_TTY::KEYEVENT_MOVE_BACKWARD; /* ^B */
  keymap[0x03]=SL_TTY::KEYEVENT_CANCEL;        /* ^C */
  keymap[0x04]=SL_TTY::KEYEVENT_DELETE_OR_LIST_OR_EOF; /* ^D */
  keymap[0x05]=SL_TTY::KEYEVENT_MOVE_EOL;      /* ^E */
  keymap[0x06]=SL_TTY::KEYEVENT_MOVE_FORWARD;  /* ^F */
  keymap[0x07]=SL_TTY::KEYEVENT_IGNORE;        /* ^G */
  keymap[0x08]=SL_TTY::KEYEVENT_BACKSPACE;     /* ^H */
  keymap[0x09]=SL_TTY::KEYEVENT_COMPLETE_WORD; /* ^I */
  keymap[0x0a]=SL_TTY::KEYEVENT_NEWLINE_OR_HELP; /* ^J */
  keymap[0x0b]=SL_TTY::KEYEVENT_KILL_EOL;      /* ^K */
  keymap[0x0c]=SL_TTY::KEYEVENT_IGNORE;        /* ^L */
  keymap[0x0d]=SL_TTY::KEYEVENT_NEWLINE_OR_HELP; /* ^M */
  keymap[0x0e]=SL_TTY::KEYEVENT_HISTORY_SEARCH_NEXT;  /* ^N */
  keymap[0x0f]=SL_TTY::KEYEVENT_IGNORE;        /* ^O */
  keymap[0x10]=SL_TTY::KEYEVENT_HISTORY_SEARCH_PREV;  /* ^P */
  keymap[0x11]=SL_TTY::KEYEVENT_IGNORE;        /* ^Q */
  //  keymap[0x12]=SL_TTY::KEYEVENT_IGNORE;        /* ^R */
  keymap[0x12]=SL_TTY::KEYEVENT_LIST_SATCOM;   /* ^R */
  keymap[0x13]=SL_TTY::KEYEVENT_IGNORE;        /* ^S */
  keymap[0x14]=SL_TTY::KEYEVENT_TRANS_CHAR;    /* ^T */
  keymap[0x15]=SL_TTY::KEYEVENT_IGNORE;        /* ^U */
  //  keymap[0x16]=SL_TTY::KEYEVENT_QUOTE_INPUT;   /* ^V */
  keymap[0x16]=SL_TTY::KEYEVENT_IGNORE;        /* ^V */
  keymap[0x17]=SL_TTY::KEYEVENT_CUT_REGION;    /* ^W */
  keymap[0x19]=SL_TTY::KEYEVENT_YANK;          /* ^Y */
  keymap[0x1a]=SL_TTY::KEYEVENT_IGNORE;        /* ^Z */
  keymap[0x1b]=SL_TTY::KEYEVENT_IGNORE;        /* ^[ */
  keymap[0x7f]=SL_TTY::KEYEVENT_DELETE;        /* DEL */

  for(i=0;i<SL_TTY::_FUNCTIONKEYS_SIZE;i++)
    fkeymap[i]=SL_TTY::KEYEVENT_IGNORE;
  fkeymap[SL_TTY::HOMEKEY]  = SL_TTY::KEYEVENT_MOVE_BOL;
  fkeymap[SL_TTY::ENDKEY]   = SL_TTY::KEYEVENT_MOVE_EOL;
  fkeymap[SL_TTY::UPKEY]    = SL_TTY::KEYEVENT_HISTORY_SEARCH_PREV;
  fkeymap[SL_TTY::DOWNKEY]  = SL_TTY::KEYEVENT_HISTORY_SEARCH_NEXT;
  fkeymap[SL_TTY::RIGHTKEY] = SL_TTY::KEYEVENT_MOVE_FORWARD;
  fkeymap[SL_TTY::LEFTKEY]  = SL_TTY::KEYEVENT_MOVE_BACKWARD;
}

tty_console::tty_console() :
  in_pipe(false), use_term_vbell(true), 
  input_lock(false), input_lock_len(0), input_lock_less(0), prompt_length(0), 
  cur_prompt_no(0){
  string hfile;
  char tmp[_MAX_PATH];
  static char *format1="[%R]SATELLITE[%N]%p:[%!]%% ";
  static char *format2="+ ";
  hint_buf = NULL;
  file_name = NULL;
  hist = (history_t*)malloc(sizeof(history_t));
  commands = hash_table_str_new();
  history_init(hist);
  if(GetUserResourceDirectory(tmp,_MAX_PATH)!=NULL){
    hfile = tmp;
    hfile += ((IsWindows())?'\\':'/');
    hfile += HISTORY_FILE;
    history_set_fname(hist,hfile.c_str());
  }
  lineno = hist->size+1;
  /* key event assign */
  kfuncs[SL_TTY::KEYEVENT_IGNORE]       = &tty_console::keyevent_ignore;
  kfuncs[SL_TTY::KEYEVENT_NEWLINE]      = &tty_console::keyevent_newline;
  kfuncs[SL_TTY::KEYEVENT_NEWLINE_OR_HELP] = 
    &tty_console::keyevent_newline_or_help;
  kfuncs[SL_TTY::KEYEVENT_INSERT]       = &tty_console::keyevent_insert;
  kfuncs[SL_TTY::KEYEVENT_DELETE]       = &tty_console::keyevent_delete;
  kfuncs[SL_TTY::KEYEVENT_DELETE_OR_EOF]= &tty_console::keyevent_delete_or_eof;
  kfuncs[SL_TTY::KEYEVENT_DELETE_OR_LIST_OR_EOF] = 
    &tty_console::keyevent_delete_or_list_or_eof;
  kfuncs[SL_TTY::KEYEVENT_MOVE_BOL]     = &tty_console::keyevent_move_bol;
  kfuncs[SL_TTY::KEYEVENT_MOVE_EOL]     = &tty_console::keyevent_move_eol;
  kfuncs[SL_TTY::KEYEVENT_MOVE_FORWARD] = &tty_console::keyevent_move_forward;
  kfuncs[SL_TTY::KEYEVENT_MOVE_BACKWARD]= &tty_console::keyevent_move_backward;
  kfuncs[SL_TTY::KEYEVENT_CANCEL]       = &tty_console::keyevent_cancel;
  kfuncs[SL_TTY::KEYEVENT_BACKSPACE]    = &tty_console::keyevent_backspace;
  kfuncs[SL_TTY::KEYEVENT_MARK]         = &tty_console::keyevent_mark;
  kfuncs[SL_TTY::KEYEVENT_CUT_REGION]   = &tty_console::keyevent_cut_region;
  kfuncs[SL_TTY::KEYEVENT_YANK]         = &tty_console::keyevent_yank;
  kfuncs[SL_TTY::KEYEVENT_KILL_EOL]     = &tty_console::keyevent_kill_eol;
  kfuncs[SL_TTY::KEYEVENT_TRANS_CHAR]   = &tty_console::keyevent_trans_char;
  kfuncs[SL_TTY::KEYEVENT_QUOTE_INPUT]  = &tty_console::keyevent_quote_input;  
  kfuncs[SL_TTY::KEYEVENT_HISTORY_PREV] = &tty_console::keyevent_history_prev;
  kfuncs[SL_TTY::KEYEVENT_HISTORY_NEXT] = &tty_console::keyevent_history_next;
  kfuncs[SL_TTY::KEYEVENT_HISTORY_SEARCH_PREV] = 
    &tty_console::keyevent_history_search_prev;
  kfuncs[SL_TTY::KEYEVENT_HISTORY_SEARCH_NEXT] = 
    &tty_console::keyevent_history_search_next;
  kfuncs[SL_TTY::KEYEVENT_COMPLETE_WORD] = 
    &tty_console::keyevent_complete_word;
  kfuncs[SL_TTY::KEYEVENT_LIST_SATCOM]  = &tty_console::keyevent_list_satcom;
  set_keybind_default();
  path_hash_build();

  input_filept = sl4_stack_new();
  input_lineno = sl4_stack_new();
  input_fname  = sl4_stack_new();

  str_cutbuf = sl4_string_new();   // cut buffer
  str_prompt[0] = sl4_string_new_cstr(format1); // prompt string 1
  str_prompt[1] = sl4_string_new_cstr(format2); // prompt string 2
}

void tty_console::term_clear_screen(){
  term_putc('\r');
  term_putc('\n');
}

void tty_console::term_print(const char *str){
  size_t len,i;
  if(str==0 || *str=='\0') return;
  len = strlen(str);
  for(i=0;i<len;i++)
    term_putc((int)str[i]);
}

char *tty_console::term_gets(char *buf, int size){
  int c,si;
  char *s;
  if(size == 0) return NULL;
  for(s=buf, si=1; (c=getchar()) != '\n'; si++){
    if(c==EOF)
      if(s==buf)
        return (NULL);
      else
        break;
    else
      *s++ = c;
    if(si==size) break;
  }
  if(si!=size) *s = 0;
  return (buf);
}

void tty_console::tty_putc(int c){
  term_putc(c);
  //  if(in_tty) term_putc(c);
  //  else       fputc(c,stdout);
}

void tty_console::tty_print(const char *mes){
  term_print(mes);
  //  if(in_tty) term_print(mes);
  //  else{
  //    int len;
  //    len = strlen(mes);
  //    fwrite(mes,1,len,stdout);
  //  }
}

char *tty_console::tty_gets(char *str, int size){
  str[size-1]='\0';
  return term_gets(str, size-1);
}

int tty_console::tty_getc(){
  if(file_name==0) return term_getc();
  else             return getc(stdin);
}

void tty_console::tty_printf(const char *fmt, ...){
  char msg[1024];
  va_list marker;
  va_start( marker, fmt ); // initialize
#ifdef HAVE_VSNPRINTF
  vsnprintf(msg,1023,fmt,marker);
#else
  vsprintf(msg,fmt,marker);
#endif
  va_end( marker );        // reset var list
  msg[1023]='\0';
  tty_print(msg);
}

void tty_console::tty_set_attr(SL_TTY::ATTRIBUTE a){
  //  if(in_tty){
  term_flush(ofd);
  if( a == SL_TTY::ATTR_NORMAL )
    term_set_attr_normal();
  if((a & SL_TTY::ATTR_BOLD) == SL_TTY::ATTR_BOLD)
    term_set_attr_bold();
  if((a & SL_TTY::ATTR_REVERSE) == SL_TTY::ATTR_REVERSE)
    term_set_attr_reverse();
  if((a & SL_TTY::ATTR_UNDERLINE) == SL_TTY::ATTR_UNDERLINE)
    term_set_attr_underline();
  //  }
}

tty_console::~tty_console(){
  history_clean(hist);
  free(hist);
  hash_table_delete(commands,NULL,NULL);
  path_hash_delete();
  while(input_file_pop());
  free(input_filept);
  free(input_lineno);
  free(input_fname);
  // free string buffer
  sl4_string_delete(str_cutbuf);
  sl4_string_delete(str_prompt[0]);
  sl4_string_delete(str_prompt[1]);
}

void tty_console::execerror(const char *str, const char *str2){
  tty_print("sl4: ");
  if(str!=0 && *str!='\0'){
    tty_set_attr(SL_TTY::ATTR_REVERSE);
    tty_print(str);
    tty_set_attr(SL_TTY::ATTR_NORMAL);
  }
  if(str2!=0 && *str2!='\0')
    tty_printf(" %s",str2);
  if(file_name != 0){
    tty_printf(" in %s",file_name);
  }
  tty_printf(" near line %d\n",lineno-1);
  throw execerr_exception();
}

void tty_console::warning(const char *str, const char *str2){
  tty_printf("sl4: warning, %s %s \n",str,str2);
}


int  tty_console::input_file_push(const char *fname){
  FILE *tmp_fp;
  char *tmp_fn;
  int  *tmp_lineno;
  if(*fname=='\0') fname=0;
  tmp_lineno  = (int*)malloc(sizeof(int));
  if(tmp_lineno == NULL) return 0;
  *tmp_lineno = lineno;
  if(fname==0){
    tmp_fn = 0;
    tmp_fp = stdin;
  }else{
    tmp_fn = (char*)malloc(strlen(fname)+1);
    if(tmp_fn == 0){ free(tmp_lineno); return 0; }
    strcpy(tmp_fn, fname);
    tmp_fp = fopen(fname,"r");
    if(tmp_fp == 0){ free(tmp_lineno); free(tmp_fn); return 0; }
  }
  sl4_stack_push(input_fname,  file_name);
  sl4_stack_push(input_filept, current_input);
  sl4_stack_push(input_lineno, tmp_lineno);
  file_name     = tmp_fn;
  lineno        = 1;
  current_input = tmp_fp;
  return 1;
}

int  tty_console::input_file_pop(){
  int *tmp_lineno;
  if(sl4_stack_empty(input_lineno)) return 0;
  if(file_name != 0){
    free(file_name);
    fclose(current_input);
  }
  file_name     = (char*)sl4_stack_top(input_fname);
  current_input = (FILE*)sl4_stack_top(input_filept);
  tmp_lineno    = (int*)sl4_stack_top(input_lineno);
  lineno = *tmp_lineno;
  free(tmp_lineno);
  sl4_stack_pop(input_fname);
  sl4_stack_pop(input_filept);
  sl4_stack_pop(input_lineno);
  return 1;
}

char *tty_console::get_filepath(size_t n, char *dir){
  char *c;
  size_t len;
  if(file_name == NULL) return NULL;
  c = strrchr(file_name, '/');
  if(c != NULL){
    len = c - file_name;
    strncpy(dir, file_name, ((n < len)?n:len));
    if(n > len) dir[len] = '\0';
  }else{
    char cwd[1024];
    getcwd(cwd,1024);
#ifdef WIN32
    for(char *p = cwd; *p!='\0'; p++) if(*p=='\\') *p='/';
#endif
    snprintf(dir, n, "%s/%s", cwd, file_name);
  }
  return dir;
}

char *tty_console::edit_readline(char *buf, int size){
  int ch,cnt(1),key;
  history_standby(hist);
  editbuf_standby();
  term_move_bol();
  prompt_print();
  term_edit_mode();
  while(cnt){
    ch = term_keypad_getc();
    if(ch == EOF) return 0;
    if(ch>0xff){
      key = fkeymap[ch >> 8];
    }else{
      key = keymap[ch];
    }
    cnt = (this->*kfuncs[key])(ch);
  }
  if(ebuf.empty()) buf[0]='\0';
  else {
    strncpy(buf,ebuf.c_str(),size);
    history_add(hist,ebuf.c_str());
  }
  term_normal_mode();
  return buf;
}

char *tty_console::readline(char *buf, int size, int n){
  char *tmp;
  cur_prompt_no = n;
  bool in_tty = (file_name == 0);
  if(in_tty){
    tmp = edit_readline(buf,size);
  }else{
    tmp = fgets(buf, size, current_input);
  }
  if(tmp != 0){
    size_t len = strlen(tmp);
    if(tmp[len-1] == '\n'){
      lineno++;
      if(in_tty && len == 1) lineno--;
    }
  }
  return tmp;
}

void tty_console::ls_command(const char *path, const char *fname, 
                             const char *arg){
  list<string> lst;
  collect_file_list(lst,path,fname,arg);
  list_print(lst);
}

void tty_console::list_print(list<string> &lst){
  if(!lst.empty()){
    string tmp;
    list<string>::iterator it;
    size_t i, maxlen(0), back;
    int x,y,no,mod,count(0);
    lst.sort();
    term_getmaxyx(&y,&x);
    for(it = lst.begin(); it != lst.end(); it++)
      if(maxlen<(*it).length())maxlen=(*it).length();
    no  = (int)(x / (maxlen + 1)); /* TODO: remove cast */
    mod = (int)(x % (maxlen + 1)); /* TODO: remove cast */
    if(no == 0) no = 1;
    for(it = lst.begin(); it != lst.end(); it++){
      tmp+=(*it);
      for(i=0 ; i < (maxlen - (*it).length() + 1); i++) tmp+=" "; 
      count++ ;
      if(count == no && mod != 0 ){
        tmp+="\n";
        count = 0;
      }
    }
    if(count != 0)tmp+="\n";
    term_normal_mode();
    term_print("\n");
    term_print(tmp.c_str());
    term_edit_mode();
    back = pos;
    pos=0;
    prompt_print();
    editbuf_print();
    cursor_goto_pos(back);
  }
}

void tty_console::history_listup(){
  int no;
  char           *tmp_key;
  history_cell_t *tmp_pos;
  struct tm *sttm;
  tmp_key = hist->key;
  hist->key = NULL;
  tmp_pos = hist->pos;
  hist->pos = hist->top;
  no  = lineno - hist->size;
  while(hist->pos != 0){
    sttm=localtime(&hist->pos->htime);
    if(hist->pos == hist->tail)
      tty_putc('>');
    else
      tty_putc(' ');
    tty_printf("%3d:[%02d:%02d:%02d]  %s\n", no, sttm->tm_hour, 
               sttm->tm_min, sttm->tm_sec, hist->pos->cmd);
    if(hist->pos == hist->tail) break;
    history_next(hist);
    no++;
  }
  hist->key = tmp_key;
  hist->pos = tmp_pos;
}

void tty_console::history_resize(int size){
  if(size <= 0)
    tty_printf("%d\n", hist->max);
  else
    ::history_resize(hist, size);
}

void tty_console::satcom_db_add(module_command_t *cmd){
  hash_table_insert(commands, cmd->command_name, cmd);
}

void tty_console::satcom_db_del(module_command_t *cmd){
  hash_table_remove(commands, cmd->command_name);
}

void tty_console::prompt_set(const char *fmt, int n){
  sl4_string_set_cstr(str_prompt[n], fmt);
}

void tty_console::prompt_get(char *fmt, int size, int n){
  strncpy(fmt, sl4_string_get_cstr(str_prompt[n]), size);
}

// prompt format - %p
static string prompt_fmt_p(){
  char curdir[256];
  string str;
  if(getcwd(curdir,256)){
    string::size_type pos1(string::npos),pos2(string::npos);
    size_t pos1_stop;
    string sep;
    sep = (IsWindows()) ? '\\' : '/';
    // pos1_stop - unix                 (/host/share)  : 0
    //           - win32 & network path (\\host\share) : 1
    //           - win32 & local drive  (c:\dir\dir2)  : 2
    if(IsWindows()) pos1_stop = (curdir[0] == '\\') ? 1 : 2 ;
    else       pos1_stop = 0;
    str=curdir;
    pos2 = str.rfind(sep.c_str());
    if(pos2 > 0)
      pos1 = str.rfind(sep.c_str(),pos2-1);
    if(pos1 > pos1_stop ){
      str.erase(0,pos1);
      str.replace(0,1,"~");
    }
  }
  return str;
}

/* check done.. */
static void trim_head_whitespace(string &buf){
  size_t i;
  for(i=0;i<buf.length();i++)
    if(buf[i]!=' ' && buf[i]!='\t')  break;
  if(i!=0) buf.erase(0,i);
}

static int is_quoted(const char *buf){
  int quoted(0),escaped(0);
  char tmp,prev('\0');
  size_t i,len;
  len = strlen(buf);
  for(i=0;i<len;i++){
    tmp = buf[i];
    if(quoted && tmp == '\\')
      if(escaped) escaped=0;
      else        escaped=1;
    if(tmp == '"'){
      if(quoted && !escaped) quoted=0;
      else{ quoted=1; if(escaped) escaped=0; }
    }
    prev=tmp;
  }
  if(quoted && escaped) return -1;
  return quoted;
}

static int split_last_key(string &buf){
  // mode 0: not needed
  //      1: satellite command
  //      2: file
  //      3: external command
  //      4: satellite & external command
  int quoted;
  char tmp;
  size_t len,i;
  int  mode(4);
  string debug_str; /* TODO: debug */
  debug_str = buf;
  trim_head_whitespace(buf);
  len = buf.length();
  if(len == 0) return 0;
  quoted = is_quoted(buf.c_str());
  if(quoted==-1){ /* printf("\r\n"); */ return 0; }
  for(i=len-1;i>0;i--){
    tmp = buf[i];
    if(tmp == ')'){ if(quoted == 0){ mode=0; break; } }
    if(tmp == ','){ if(quoted == 0){ mode=1; break; } }
    if(tmp == '='){ if(quoted == 0){ mode=1; break; } }
    if(tmp == '('){ if(quoted == 0){ mode=1; break; } }
    if(tmp == ' '){ if(quoted == 0){ mode=2; break; } }
    if(tmp == '"'){
      if(quoted){
        /* backslash escape */
        if(i<=0 || buf[i-1] != '\\'){ mode=2; break; }
      }else{ mode=0; break; }
    }
    if(tmp == ';'){ mode=4; break; }
  }
  if(i!=0)i++;
  if(len-i==0){ buf.erase(); }
  else {
    try {
      buf=buf.substr(i,len-i); /* TODO: this line has bug.. */
    }catch(std::exception){
      fprintf(stderr,"_DEBUG: buf(orig)[%s] buf(cur)[%s] quoted[%d] %d\n",
              debug_str.c_str(),buf.c_str(), quoted, i);
      throw;
    }
  }
  if(quoted){
    i=0; while(i=buf.find("\\\"",i),i!=string::npos) buf.erase(i++,1);
    i=0; while(i=buf.find("\\\\",i),i!=string::npos) buf.erase(i++,1);
  }
  //  printf("\r\nbuf[%s], mode(%d), quoted(%d)\r\n",
  //         buf.c_str(),mode,quoted);

  /* check directory separater */
  if(mode == 4){
    for(i=0;i<buf.length();i++){
      if(buf[i] == '/'){ mode = 3; break; }
      if(IsWindows() && buf[i] == '\\'){ mode = 3; break; }
    }
  }
  return mode;
}

static void split_path(string &buf, string &path, string &file,int mode){
  size_t i(0);
  string tmp;
  if(IsWindows()){
    while(i=buf.find("\\",i),i!=string::npos){ buf[i]='/'; }
  }
  i=buf.rfind('/');
  if(i==string::npos){ path.erase(); file=buf; }
  else {
    if(i==0) path ="/.";
    else path=buf.substr(0,i);
    if(i!=buf.length()-1)
      file=buf.substr(i+1,buf.length()-i-1);
  }
  if(mode==0 && path.empty()){
    path=".";
  }
  //  printf("\r\nbuf[%s], path[%s], file[%s], mode(%d)\r\n",
  //         buf.c_str(), path.c_str(), file.c_str(), mode);
}

// local functions
static void collect_file_list(list<string> &lst, const char *path, 
                              const char *fname, const char *arg){
  DIR *dirp;
  struct dirent *dp;
  stack<string> spath;
  string args;
  string tmp,fi;
  const char *sepa;
  bool mode, exec, dir, all;
  size_t len;
  if(fname == 0 || *fname=='\0') len = 0;
  else len = strlen(fname);
  sepa = (IsWindows())?"\\":"/";
  args = (arg!=0) ? arg : "";
  mode = (args.find("F")!=string::npos); /* mode flag */
  exec = (args.find("E")!=string::npos); /* executable only */
  dir  = (args.find("D")!=string::npos); /* including directory */
  all  = (args.find("a")!=string::npos); /* all files */
  if(path==0 || *path=='\0'){
    char *env=getenv("PATH");
    if(env!=0){
      string token(env);
      size_t p;
      const char *token_char;
      token_char = (IsWindows()) ? ";" : ":" ;
      for(p=token.find(token_char);p!=string::npos;p=token.find(token_char)){
        spath.push(token.substr(0,p));
        token.erase(0,p+1);
      }
    }
  }else{
    tmp = path;
    if(IsWindows()){ 
      size_t i; while(i=tmp.find('/'),i!=string::npos) tmp[i]='\\';
    }
    spath.push(tmp);
  }
  while(!spath.empty()){
    const char *s=spath.top().c_str();
    dirp = opendir(s);
    if(dirp != 0){
      while((dp = readdir(dirp)) != 0){
        // if(!strcmp(dp->d_name, ".") || !strcmp(dp->d_name, "..")) continue;
        if(strncmp(dp->d_name, fname, len)){ continue; }
        if(!all && !strncmp(dp->d_name, ".", 1)){ continue; }
        tmp=s; tmp+=sepa; tmp+=dp->d_name; fi=dp->d_name;
        if(exec){
          if(Access(tmp.c_str(), SL_FATTR_XOK)!=0) continue;
          if(!dir && IsDirectory(tmp.c_str())) continue;
          if(IsWindows()){
            size_t fnlen=strlen(dp->d_name);
            if(fnlen > 4){
              if(!strncmp(dp->d_name+(fnlen-4),".dll",4)) continue;
              if(!strncmp(dp->d_name+(fnlen-4),".sys",4)) continue;
            }
          }
        }
        // file mode check
        if(!mode){
          struct stat sb;
          stat(tmp.c_str(),&sb);
#ifdef S_IFDIR
          // directory
          if((sb.st_mode & S_IFMT) == S_IFDIR) fi+='/';
#endif // S_IFDIR
        }else{
          struct stat sb;
          bool suf_flag=false;
          lstat(tmp.c_str(),&sb);
#ifdef S_IFLNK
          // symbolic link
          if(!suf_flag && (sb.st_mode & S_IFMT) == S_IFLNK){ 
            fi+='@'; suf_flag=true; 
          }
#endif // S_IFLNK
#ifdef S_IFDIR
          // directory
          if(!suf_flag && (sb.st_mode & S_IFMT) == S_IFDIR){ 
            fi+='/'; suf_flag=true;
          }
#endif // S_IFDIR
#ifdef S_IFCR
          // character special
          if(!suf_flag && (sb.st_mode & S_IFMT) == S_IFCHR){
            fi+='%'; suf_flag=true;
          }
#endif // S_IFCHR
          // execute file
          if(!suf_flag &&// !flag && 
             (Access(tmp.c_str(),SL_FATTR_XOK) == 0)) fi+='*';
        }
        lst.push_back(fi);
      }
      closedir(dirp);
    }
    spath.pop();
  }
}

typedef struct _listup_satcom_args_t {
  int           mode; /* 1: append '()', else: nothing */
  size_t        len;
  const char   *word;
  list<string> *coms;
} listup_satcom_arg_t;

static void listup_satcom_for(void *void_cmd, void *void_arg){
  module_command_t    *cmd;
  listup_satcom_arg_t *arg;
  string word;
  cmd = (module_command_t*)void_cmd;
  arg = (listup_satcom_arg_t*)void_arg;
  if(strncmp(cmd->command_name, arg->word, arg->len) == 0){
    word=cmd->command_name;
    if(arg->mode!=0) word+="()";
    arg->coms->push_back(word);
  }
}

static void collect_satcom_list(list<string> *lst, hash_table_t *coms,
                                const char *word, int mode){
  listup_satcom_arg_t arg;
  arg.word   = word;
  arg.len    = strlen(arg.word);
  arg.coms   = lst;
  arg.mode   = mode;
  if(arg.len == 0) return;
  hash_table_foreach(coms, listup_satcom_for, &arg);
}

static const char *_last_str(const char *str){
  size_t i,len;
  int is_arg,in_brace;
  char *p;
  static char delimiter[]="!@#$%^&*(-+=\\|~`{}[;:,.<>/? ";
  if(str==0 || *str=='\0') return "";
  len=strlen(str);
  is_arg=0; in_brace=0;
  for(i=len-1;i>0;i--){ if(str[i]=='=') break; }
  if(i!=0) i++;
  for(;str[i]==' '||str[i] =='\t';i++);
  for(;i<len;i++){
    if(str[i] == '('){ in_brace++; continue; }
    if(str[i] == ')'){ in_brace--; continue; }
    if(in_brace == 0){
      if(str[i]==' '||str[i]=='\t'){ is_arg=1; break; }
    }
  }
  if(is_arg==1) return "";
  for(i=len-1;i>0;i--){
    for(p=delimiter;*p!='\0';p++) if(*p==str[i]) break;
    if(*p!='\0')break;
  }
  if(i!=0) i++;
  return &str[i];
}

static string _current_func(const char *str, int *no){
  static char delimiter[]="!@#$%^&*()-+=\\|~`{}[];:\'\",.<>/? ";
  stack<size_t> start_stack,len_stack;
  stack<int> cnt_stack;
  string b(str);
  int in_dquote(0),in_squote(0);
  int cnt(0);
  size_t i,j,pos;
  pos = strlen(str);
  for(i=0;i<pos;i++){
    /* check double quote */
    if(str[i] == '"'){
      if(in_squote) continue;
      if(in_dquote) in_dquote=(str[i-1]=='\\'); else in_dquote=1;
    }
    if(in_dquote) continue;
    /* check single quote */
    if(str[i] == '\''){
      if(in_squote) in_squote=(str[i-1]=='\\'); else in_squote=1;
    }
    if(in_squote) continue;
    
    if(str[i] == ')'){
      if(len_stack.empty()) return string(""); /* invalid strings */
      cnt = cnt_stack.top(); cnt_stack.pop();
      start_stack.pop(); len_stack.pop();
    }
    if(str[i] == ',' && !len_stack.empty()) cnt++;
    if(str[i] == '('){
      const char *p;
      for(j=i;j>0;j--){
        for(p=delimiter;*p!='\0';p++) if(str[j-1]==*p) break;
        if(*p!='\0') break;
      }
      start_stack.push(j);
      len_stack.push(i-j);
      cnt_stack.push(cnt);
      cnt=0;
    }
  }
  if(in_dquote || in_squote) return string("");
  if(!len_stack.empty()){
    if(len_stack.top()!=0){
      string b(str);
      size_t s_top = start_stack.top();
      size_t l_top = len_stack.top();
      *no  = cnt;
      return b.substr(s_top,l_top);
    }
  }
  return string("");
}

/* return code */
/*  0: success */
/* -1: file not found */
/* -2: can't create child process */
/* -3: process not found          */
/* -4: signal caught */
/* -5: process is stoping */
/* -6: core dumped */
/* -7: unknown error */
int tty_console::run_pager(const char *file){
  /* output ASCII file to console display */
  FILE *fp;
  int   code, pid, status;
  char  buf[_MAX_PATH];
  char *pager;

  pager = getenv("SL_PAGER");
  if(pager == NULL)
    pager = getenv("PAGER");

  code = 0;

  if(pager == NULL){
    fp = fopen(file, "r");
    if(fp == NULL){
      code = -1; /* file not found */
    }else{
      while(feof(fp) == 0){
        if(fgets(buf, _MAX_PATH-1 ,fp) == NULL) break;
        buf[_MAX_PATH-1] = '\0';
        tty_print(buf);
      }
      fclose(fp);
    }
  }else{
    snprintf(buf, _MAX_PATH, "%s %s", pager,file);
    pid = create_child_process(buf, ifd, ofd, efd, NULL);
    if(pid == -1){
      code = -2; /* could not create child process */
    }else{
      status = wait_child_process(pid, &code);
      if(status == -1){
        code = -3; /* process not found */
      }else if(status == 0){
        code = 0;  /* normal exit */
      }else if(status == 1){
        code = -4; /* signal caught */
      }else if(status == 2){
        code = -5; /* process is stoping */
      }else if(status == 3){
        code = -6; /* core dumped */
      }else{ 
        code = -7; /* unknown error */
      }
    }
  }
  return code;
}

/* return code */
/*  0: success */
/* -1: file not found */
/* -2: can't create child process */
/* -3: process not found          */
/* -4: signal caught */
/* -5: process is stoping */
/* -6: core dumped */
/* -7: unknown error */
int tty_console::run_editor(const char *file){
  int   code, pid, status;
  char  buf[_MAX_PATH];
  char *editor;

  editor = getenv("SL_EDITOR");
  if(editor == NULL)
    editor = getenv("EDITOR");
  if(editor == NULL){
#ifdef WIN32
    editor = "notepad.exe";
#else
    editor = "vi";
#endif
  }
  
  code = 0;

  snprintf(buf, _MAX_PATH, "%s %s", editor,file);
  pid = create_child_process(buf, ifd, ofd, efd, NULL);
  if(pid == -1){
    code = -2; /* could not create child process */
  }else{
    status = wait_child_process(pid, &code);
    if(status == -1){
      code = -3; /* process not found */
    }else if(status == 0){
      code = 0;  /* normal exit */
    }else if(status == 1){
      code = -4; /* signal caught */
    }else if(status == 2){
      code = -5; /* process is stoping */
    }else if(status == 3){
      code = -6; /* core dumped */
    }else{ 
      code = -7; /* unknown error */
    }
  }
  return code;
}
