/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: userinfo.c,v 1.3 2005/03/31 04:13:09 orrisroot Exp $ */
#define  LIBSATELLITE_EXPORTS

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32
# include <windows.h>
#endif

#define MAXUSERNAME 128

#include <stdio.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <errno.h>
#ifdef HAVE_PROCESS_H
# include <process.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_DIRENT_H
# include <dirent.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_PWD_H
# include <pwd.h>
#endif
#ifdef HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif

#include "libsatellite.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#define MAX_REGLEN 64
#define MAX_REGNUM 256

/* select WIN32 API wrapper functions. */
#define USE_SHELL_API 1

#ifdef WIN32
# define MYDOCUMENTS_SATELLITE_STRING      "My SATELLITE"
# define SATELLITE_TEMP_PREFIX             "SATELLITE-Lang"
# define SATELLITE_RESOURCE_PREFIX         "SATELLITE4"
# define SATELLITE_SYSTEM_RESOURCE_PREFIX  "/etc"
# define SATELLITE_DEFAULT_MODULE_PREFIX   "/modules"
# define SATELLITE_DEFAULT_SHARE_PREFIX    "/share"
#else
# define SATELLITE_TEMP_PREFIX             "SATELLITE-Lang"
# define SATELLITE_RESOURCE_PREFIX         ".sl4rc"
# define SATELLITE_DEFAULT_TEMP            "/tmp"
#endif
#define  SATELLITE_REGISTRY_FILE           "/reg.dat"

#ifdef WIN32
#if USE_SHELL_API
#include <shlobj.h>
static int GetShellSpecialFolder(char *str, DWORD mask);
#else /* USE_SHELL_API */
static int GetRegistoryString(char *pData, LPDWORD size, HKEY pkey, 
                              LPCTSTR skey, LPTSTR val);
#endif /* USE_SHELL_API */
static int GetMyDocumentsPath(char *buf, int len);
#endif /* WIN32 */
static void replace_escape(char *buf);
static void ErrorDialog(const char *mes1, const char *mes2);
static int  CheckKeyParam(const char *linebuf, const char *key, 
                          const char *param, char *val, int vallen);

DLLEXPORT char *GetCurrentUserName(char *buf, int len){
  static int  is_first=1;
  static char var[MAXUSERNAME];
  static int  size=0;
  if(is_first){
#ifdef WIN32
    DWORD usiz = MAXUSERNAME;
    if(GetUserName(var,&usiz)==0) return NULL;
#else
    uid_t uid;
    struct passwd *pass;
    uid = getuid();
    pass = getpwuid(uid);
    if(strlen(pass->pw_name) + 1 > MAXUSERNAME){
#ifdef HAVE_ENDPWENT
      endpwent();
#endif
      return NULL;
    }
    strcpy(var,pass->pw_name);
#ifdef HAVE_ENDPWENT
    endpwent();
#endif
#endif
    size=(int)strlen(var);
    is_first = 0;
  }
  if(size >= len) return NULL;
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetHomeDirectory(char *buf, int len){
  static int  is_first = 1;
  static char var[_MAX_PATH];
  static int  size = 0;
  if(is_first){
    const char *home;
    home = getenv("SL_HOME");
    if(home == NULL) home = getenv("HOME");
    if(home != NULL){
      size = (int)strlen(home);
      if(size >= _MAX_PATH) return 0;
      strcpy(var, home);
    }
#ifdef WIN32
    if(home == NULL){
      size = GetMyDocumentsPath(var, _MAX_PATH);
      if(size == 0) return 0;
      replace_escape(var);
      /* MyDoc + '/' + "SATELLITE-Lang" */
      size = size+1+(int)strlen(MYDOCUMENTS_SATELLITE_STRING);
      if(size + 1 > _MAX_PATH) return NULL;
      strcat(var,"/");
      strcat(var, MYDOCUMENTS_SATELLITE_STRING);
      if(IsDirectory(var) == 0)
        if(Mkdir(var)!=0) return NULL;
    }
#endif
    is_first = 0;
  }
  if(size == 0 || len <= size) return NULL;
  /* directory check */
  if(IsDirectory(var) == 0){
    ErrorDialog(var, ": no such directory.");
    return NULL;
  }
  if(Access(var, SL_FATTR_WOK) != 0){
    ErrorDialog(var, ": permission denied.");
    return NULL;
  }
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetTempDirectory(char *buf, int len){
  static int  is_first = 1;
  static char var[_MAX_PATH];
  static int  size = 0;
  if(is_first){
#ifdef WIN32
    char  tmp[_MAX_PATH];
    DWORD pid;
#else
    pid_t pid;
#endif
    char *temp;
    temp = getenv("SL_TEMP");
    if(temp == NULL) temp = getenv("TEMP");
    if(temp == NULL) temp = getenv("TMP");
#ifndef WIN32
    if(temp == NULL) temp = SATELLITE_DEFAULT_TEMP;
#endif
    if(temp == NULL) return NULL;
#ifdef WIN32
    /* expand path name */
    if(GetLongPathName(temp, tmp, _MAX_PATH) == 0) return NULL;
    if(GetFullPathName(tmp, _MAX_PATH, var, &temp) == 0) return NULL;
    strcpy(tmp, var);
    temp = tmp;
    pid = GetCurrentProcessId();
#else
    pid = getpid();
#endif
    /* formatting temp directory name */
    memset(var,0,_MAX_PATH);
#ifdef HAVE_SNPRINTF
    snprintf(var, _MAX_PATH-1, "%s/%s%d", temp, SATELLITE_TEMP_PREFIX, pid);
# else
    sprintf(var, "%s/%s%d", temp, SATELLITE_TEMP_PREFIX, pid);
#endif
    /* directory check */
    if(Access(var, SL_FATTR_FOK) == 0){
      if(Access(var, SL_FATTR_WOK) != 0){
        ErrorDialog(var, ": already exists.");
        return NULL;
      }
    }else{
      if(Mkdir(var) != 0){
        ErrorDialog(var, ": permission denied.");
        return NULL;
      }
    }
    size = (int)strlen(var);
    replace_escape(var);
    is_first = 0;
  }
  if(size == 0 || len <= size) return NULL;
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetUserResourceDirectory(char *buf, int len){
  static int is_first=1;
  static int size=0;
  static char var[_MAX_PATH];
  if(is_first){
    char tmp[_MAX_PATH];
#ifdef WIN32
# if USE_SHELL_API
    if(GetShellSpecialFolder(tmp,CSIDL_APPDATA) == 0)
      return 0; /* error */
# else /* USE_SHELL_API */
    /* tested on winxp */
    DWORD s = _MAX_PATH;
    if(GetRegistoryString(tmp, &s, HKEY_CURRENT_USER,
                          "Software\\Microsoft\\Windows\\"
                          "CurrentVersion\\Explorer\\Shell Folders", 
                          "Local AppData") == 0)
      return 0; /* error */
# endif
#else
    if(GetHomeDirectory(tmp, _MAX_PATH) == 0) return 0;
#endif
#ifdef HAVE_SNPRINTF
    snprintf(var, _MAX_PATH-1, "%s/%s", tmp, SATELLITE_RESOURCE_PREFIX);
#else
    sprintf(var, "%s/%s", tmp, SATELLITE_RESOURCE_PREFIX);
#endif
    size = (int)strlen(var);
    replace_escape(var);
    is_first = 0;
  }
  if(IsDirectory(var)==0){
    if(Mkdir(var)!=0){
      ErrorDialog(var, ": permission denied.");
      return NULL;
    }
  }else{
    if(Access(var,SL_FATTR_WOK) !=0 ){
      ErrorDialog(var, ": permission denied.");
      return NULL;
    }
  }
  if(size == 0 || len <= size) return NULL;
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetSystemResourceDirectory(char *buf, int len){
  static int is_first=1;
  static int size=0;
  static char var[_MAX_PATH];
  if(is_first){
#ifdef WIN32
    size_t i,len;
    if(GetModuleFileName(NULL,var,_MAX_PATH) == 0) return NULL;
    len = strlen(var);
    for(i=len-1;i>0;i--){ if(var[i] == '\\'){ var[i]='\0'; break; } }
    if(var[i]=='\\') var[i]='\0';
    replace_escape(var);
    len = strlen(var) + strlen(SATELLITE_SYSTEM_RESOURCE_PREFIX) + 1;
    if(len > _MAX_PATH) return NULL;
    strcat(var, SATELLITE_SYSTEM_RESOURCE_PREFIX);
#else
    char *tmp;
    tmp = DEFAULT_SYSTEM_FILE_PATH;
    if(strlen(tmp) + 1 > _MAX_PATH) return NULL;
    strcpy(var,tmp);
#endif
    size = (int)strlen(var);
    is_first = 0;
  }
  if(IsDirectory(var)==0){
    ErrorDialog(var, ": no such directory.");
    return NULL;
  }
  if(Access(var,SL_FATTR_ROK) !=0 ){
    ErrorDialog(var, ": permission denied.");
    return NULL;
  }
  if(size == 0 || len <= size) return NULL;
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetModuleDirectory(char *buf, int len){
  static int is_first=1;
  static int size=0;
  static char var[_MAX_PATH];
  if(is_first){
#ifdef WIN32
    size_t i,len;
    if(GetModuleFileName(NULL,var,_MAX_PATH) == 0) return NULL;
    len = strlen(var);
    for(i=len-1;i>0;i--){ if(var[i] == '\\'){ var[i]='\0'; break; } }
    if(var[i]=='\\') var[i]='\0';
    replace_escape(var);
    len = strlen(var) + strlen(SATELLITE_DEFAULT_MODULE_PREFIX) + 1;
    if(len > _MAX_PATH) return NULL;
    strcat(var, SATELLITE_DEFAULT_MODULE_PREFIX);
#else
    char *tmp;
    tmp = DEFAULT_MODULE_PATH;
    if(strlen(tmp) + 1 > _MAX_PATH) return NULL;
    strcpy(var,tmp);
#endif
    size = (int)strlen(var);
    is_first = 0;
  }
  if(IsDirectory(var)==0){
    ErrorDialog(var, ": no such directory.");
    return NULL;
  }
  if(Access(var,SL_FATTR_ROK) != 0){
    ErrorDialog(var, ": permission denied.");
    return NULL;
  }
  if(size == 0 || len <= size) return NULL;
  strcpy(buf,var);
  return buf;
}

DLLEXPORT char *GetShareDirectory(char *buf, int len){
  static int is_first=1;
  static int size=0;
  static char var[_MAX_PATH];
  if(is_first){
#ifdef WIN32
    size_t i,len;
    if(GetModuleFileName(NULL,var,_MAX_PATH) == 0) return NULL;
    len = strlen(var);
    for(i=len-1;i>0;i--){ if(var[i] == '\\'){ var[i]='\0'; break; } }
    if(var[i]=='\\') var[i]='\0';
    replace_escape(var);
    len = strlen(var) + strlen(SATELLITE_DEFAULT_SHARE_PREFIX) + 1;
    if(len > _MAX_PATH) return NULL;
    strcat(var, SATELLITE_DEFAULT_SHARE_PREFIX);
#else
    char *tmp;
    tmp = DEFAULT_SHARE_PATH;
    if(strlen(tmp) + 1 > _MAX_PATH) return NULL;
    strcpy(var,tmp);
#endif
    size = (int)strlen(var);
    is_first = 0;
  }
  if(IsDirectory(var)==0){
    ErrorDialog(var, ": no such directory.");
    return NULL;
  }
  if(Access(var,SL_FATTR_ROK) != 0){
    ErrorDialog(var, ": permission denied.");
    return NULL;
  }
  if(size == 0 || len <= size) return NULL;
  strcpy(buf,var);
  return buf;
}

/* get local date and time*/
DLLEXPORT int get_date(int *year, int *month, int *day){
  struct tm  *times;
  time_t      gtime;
  if(year == NULL || month == NULL || day == NULL) return -1;
  time(&gtime);
  times = localtime(&gtime);
  *year = times->tm_year;
  *month = times->tm_mon + 1;
  *day = times->tm_mday;
  return 0;
}

DLLEXPORT int get_time(int *hour, int *minute, int *second){
  struct tm  *times;
  time_t      gtime;
  if(hour == NULL || minute == NULL || second == NULL) return -1;
  time(&gtime);
  times = localtime(&gtime);
  *hour = times->tm_hour;
  *minute = times->tm_min;
  *second = times->tm_sec;
  return 0;
}

/* function of read / write user local registry file */
DLLEXPORT int ReadRegistInt(const char *key, const char* param, long *value){
  FILE *fp;
  char  pathname[_MAX_PATH], linebuf[_MAX_PATH], valbuf[MAX_REGLEN];
  int   is_found = 0;
  
  if(GetUserResourceDirectory(pathname, _MAX_PATH) == 0) return -1;
  if(strlen(pathname) + strlen(SATELLITE_REGISTRY_FILE) + 1 > _MAX_PATH){
    /* + 1 : means length of EOS char ('\0') */
    return -1; /* error: too short of temporary buffer length */
  }
  strcat(pathname, SATELLITE_REGISTRY_FILE);
  fp = fopen(pathname, "r");
  if(fp == NULL) return -1; /* error: could not open file. */
  while(!feof(fp)) {
    if(fgets(linebuf,_MAX_PATH,fp) == NULL) break;
    linebuf[_MAX_PATH-1] = '\0';
    /* checking.. is exists keyword and parameter in line buffer */
    if(CheckKeyParam(linebuf, key, param, valbuf, MAX_REGLEN) == 0) {
      long data;
      data = atol(valbuf);
      if(errno == ERANGE) break;
      *value = data;
      is_found = 1;
      break;
    }
  }
  fclose(fp);
  if(is_found) return 0;
  return -1;
}

DLLEXPORT int WriteRegistInt(const char *key, const char* param, long value){
  FILE *fp;
  char  pathname[_MAX_PATH], linebuf[_MAX_PATH], valbuf[MAX_REGLEN];
  char *pbuf[MAX_REGNUM];
  int   i, lcnt, giveup;
  
  if(GetUserResourceDirectory(pathname, _MAX_PATH) == 0) return -1;
  if(strlen(pathname) + strlen(SATELLITE_REGISTRY_FILE) + 1 > _MAX_PATH){
    /* + 1 : means length of EOS char ('\0') */
    return -1; /* error: too short of temporary buffer length */
  }
  strcat(pathname, SATELLITE_REGISTRY_FILE);
  if(Access(pathname, SL_FATTR_FOK) != 0){
    /* registry file not exists.. */
    /* try create a new file */
    fp = fopen(pathname, "w");
    if(fp == NULL){
      ErrorDialog(pathname, ": can not create.");
      return -1;
    }
    fprintf(fp,"%s,%s,%ld\n", key, param, value);
    fclose(fp);
    return 0;
  }
  /* registry file exists... */
  /* load all keyword and parameter and value */
  fp = fopen(pathname, "r");
  if(fp == NULL){
    ErrorDialog(pathname, ": can not read.");
    return -1;
  }
  lcnt = 0;
  giveup = 0;
  while(!feof(fp)){
    if(fgets(linebuf,_MAX_PATH,fp) == NULL) break;
    linebuf[_MAX_PATH-1] = '\0';
    if(lcnt >= MAX_REGNUM){
      giveup = 1; lcnt--;
      break;
    }
    pbuf[lcnt] = (char*) malloc(strlen(linebuf) + 1);
    if(pbuf[lcnt] == NULL){ /* can not allocate memory */
      giveup = 1;
      break;
    }
    strcpy(pbuf[lcnt], linebuf);
    lcnt++;
  }
  fclose(fp);
  if(giveup) {
    for(i=0;i<lcnt;i++) free(pbuf[i]);
    return -1;
  }

  /* restore all registry keys */
  fp = fopen(pathname, "w");
  if( fp == NULL){
    ErrorDialog(pathname, ": can not override.");
    return -1;
  }
  for(i=0; i<lcnt; i++){
    if(CheckKeyParam(pbuf[i], key, param, valbuf, MAX_REGLEN) == 0){
      /* this line is same keyword and parameter. override it */
      fprintf(fp,"%s,%s,%ld\n", key, param, value);
    }else{
      fprintf(fp,"%s\n",pbuf[i]);
    }
    free(pbuf[i]);
  }
  fclose(fp);
  return 0;
}

/* private functions */

#ifdef WIN32
#if USE_SHELL_API
static int GetShellSpecialFolder(char *str, DWORD mask){
  ITEMIDLIST *pIdList;
  IMalloc    *pMalloc;
  int         ret=0;
  if(NOERROR == SHGetMalloc(&pMalloc)){
    if(NOERROR == SHGetSpecialFolderLocation(HWND_DESKTOP, mask, &pIdList)){
      SHGetPathFromIDList(pIdList, str);
      pMalloc->lpVtbl->Free(pMalloc,pIdList);
      ret = 1;
    }
  }
  return ret;
}

#else /* USE_SHELL_API */

static int GetRegistoryString(char *pData, LPDWORD size, HKEY pkey, 
                               LPCTSTR skey, LPTSTR val){
  HKEY rkey;
  memset(pData,0,*size);
  if(ERROR_SUCCESS != RegOpenKeyEx(pkey, skey, 0L, KEY_READ, &rkey))
    return 0;
  if(ERROR_SUCCESS != RegQueryValueEx(rkey, val, NULL, NULL, 
                                      (BYTE*)pData, size))
    return 0;
  RegCloseKey(pkey);
  return 1;
}

#endif /* USE_SHELL_API */

/* My Documents */
static int GetMyDocumentsPath(char *buf, int len){
  static int is_first=1;
  static char var[_MAX_PATH];
  static DWORD size = _MAX_PATH;
  if(is_first){
#if USE_SHELL_API
    if(GetShellSpecialFolder(var,CSIDL_PERSONAL) == 0)
      return 0; /* error */
#else /* USE_SHELL_API */
    /* tested on w2k */
    if(GetRegistoryString(var, &size, HKEY_CURRENT_USER,
                          "Software\\Microsoft\\Windows\\"
                          "CurrentVersion\\Explorer\\Shell Folders", 
                          "Personal") == 0)
      return 0; /* error */
#endif /* USE_SHELL_API */
    size = (DWORD)strlen(var);
    is_first = 0;
  }
  if((int)size >= len) return 0;
  strcpy(buf,var);
  return (int)size;
}

static void ErrorDialog(const char *mes1,const char *mes2){
  size_t len;
  char *str;
  len = strlen(mes1)+strlen(mes2) + 2;
  str = (char*)malloc(len);
  if(str==NULL) return;
  strcpy(str,mes1);
  strcat(str," ");
  strcat(str,mes2);
  MessageBox(NULL, str, "ERROR", MB_OK | MB_ICONERROR | 
             MB_TASKMODAL | MB_SETFOREGROUND | MB_TOPMOST);
  free(str);
}

#else
static void ErrorDialog(const char *mes1, const char *mes2){
  fprintf(stderr, "Error: %s %s\n", mes1,mes2);
}
#endif /* WIN32 */

static void replace_escape(char *buf){
  char *p;
  for(p=buf;*p!='\0';p++)
    if(*p=='\\')*p='/';
}

static int CheckKeyParam( const char *linebuf, const char *key, 
                          const char *param, char *val, int vallen ){
  char buf[MAX_REGLEN];
  size_t i, len;
  int  stage,j;
  i=0;
  len = strlen(linebuf);
  for(stage = 0; stage < 3; stage++){
    j=0;
    /* pick up each value of linebuf */
    while(i<len){
      buf[j] = linebuf[i++];
      if(buf[j] == ',' || buf[j] == '\0'){ buf[j] = '\0'; break; }
      j++;
      if(j==MAX_REGLEN){
        ErrorDialog("line buffer is too long.", ": give up...");
        return -1;
      }
    }
    switch(stage){
    case 0: /* first stage. checking key.. */
      if(strcmp(buf, key) != 0) return -1;
      break;
    case 1: /* second stage. checking param.. */
      if(strcmp(buf, param) != 0) return -1;
      break;
    case 2: /* final stage. copying value.. */
      if(vallen <= (int)strlen(buf)){
        ErrorDialog("length of value is too long.", ": give up...");
        return -1;
      }
      strcpy(val,buf);
      break;
    }
  }
  return 0;
}

