/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: Builtin.cpp,v 1.3 2005/01/06 08:44:59 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "SL_header.h"

using namespace std;

#include <libsatellite.h>

#define  __IMPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __IMPORTSYMBOL__
#include "Program.h"
#include "Datum.h"
#include "Inline.h"
#include "CommandAlias.h"
#include "LexicalAnalizer.h"
#include "pipe.h"
#include "Builtin.h"
#include "vmstat.h"
#define  __EXPORTSYMBOL__
#include "StackMachine.h"
#undef   __EXPORTSYMBOL__
#include <cctype>
#ifdef WIN32
#include <direct.h>
#endif

int BuiltinFunctions::_symbols(const char *sym, size_t columns){
  size_t i,j,len,n;
  string symbol;
  if(sym==0) return -1;
  symbol=sym;
  len = symbol.length();
  if(len == 0) return 0;
  n = columns /= len;
  for(i=0; i<n; i++){
    for(j=0;j<len;j++)
      console->tty_putc((int)symbol[j]);
  }
  return 0;
}

SL_Object *BuiltinFunctions::alias(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:alias>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::alias()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  const char *nick=0, *real=0;
  if(argc>0){
    if(obj[0]==0 || obj[0]->GetBufferPointer()==0 ||
        obj[0]->TypeofOBJ()!=SL_OBJ::STRING_O ||
        obj[0]->GetBufferPointer()->GetDataPointer()==0 )
      console->execerror(e_where,"null object of arguments");
    nick=((String_Buffer *)obj[0]->GetBufferPointer())->GetString();
  }
  if(argc>1){
    if(obj[1]==0 || obj[1]->GetBufferPointer()==0 ||
       obj[1]->TypeofOBJ()!=SL_OBJ::STRING_O ||
       obj[1]->GetBufferPointer()->GetDataPointer()==0 )
      console->execerror(e_where,"null object of arguments");
    real=((String_Buffer *)obj[1]->GetBufferPointer())->GetString();
  }
  switch (argc) {
  case 0: stackmachine->lex.alias.Print(console); break;
  case 1: stackmachine->lex.alias.Delete(nick); break;
  case 2: stackmachine->lex.alias.Install(nick,real); break;
  default : console->execerror(e_where,"mismatch, number of arguments");
  }
  return (SL_Object *)0;
}

SL_Object *BuiltinFunctions::symbols(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:symbols>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::symbols()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  if (argc!=0 || obj!=0 )
    console->execerror(e_where,"mismatch, number of arguments");
  symbol_table_listup(syscom->cur_symtab,console,0);
  symbol_table_listup(syscom->gl_symtab,console,1);
  return (SL_Object *)0;
}

SL_Object *BuiltinFunctions::history(int argc,SL_Object *obj[]){
  static char *where="history()";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *e_where="Error [<BUILTIN:history>]";
  console->tty_printf("%s\n",e_where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  if(argc != 0 && argc != 1)
    console->execerror(where,"mismatch, number of arguments");
  if(argc == 0){
    console->history_listup();
  }else{
    double d;
    int    size;
    if(obj[0]==0 || obj[0]->GetBufferPointer()==0 ||
       obj[0]->TypeofOBJ()!=SL_OBJ::SCALAR_O ||
       obj[0]->GetBufferPointer()->GetDataPointer()==0 )
      console->execerror(where,"illegal data type");
    d = ((Scalar_Buffer*)obj[0]->GetBufferPointer())->GetScalar();
    size = (int)d;
    console->history_resize(size);
  }
  return 0;
}

SL_Object *BuiltinFunctions::lengthof(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:strlen>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::lengthof()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  Base_Buffer *buf,*retbuf=0;
  SL_Object *ret;
  ret=0; //for compiler
  if(argc!=1)
    console->execerror(e_where,"mismatch: number of arguments");
  if(obj[0]==0)
    console->execerror(e_where,"illegal object");
  buf=obj[0]->GetBufferPointer();
  if(buf==0)
    console->execerror(e_where,"null buffer");
  retbuf=new Scalar_Buffer;
  try{
    retbuf->InitBuffer();
    if(obj[0]->TypeofOBJ()==SL_OBJ::SCALAR_O){
      ((Scalar_Buffer*)retbuf)->SetScalar(1.0);
    }else{
      ((Scalar_Buffer*)retbuf)->SetScalar(buf->GetIndex(0));
    }
    
    ret=new_SL_Object(SL_OBJ::SCALAR_O,retbuf);
  }catch(buffer_exception){
    delete retbuf;
    console->execerror(e_where,"buffer exception");
  }catch(bad_alloc){
    delete retbuf;
    throw;
  }catch(execerr_exception){
    delete retbuf;
    throw;
  }
  return ret;
}

SL_Object *BuiltinFunctions::indexof(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:index>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::indexof()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  int    n;
  double *tmp;
  Base_Buffer *buf=0,*ret_buf=0;
  SL_Object *ret=0;
  if(argc!=1)
    console->execerror(e_where,"mismatch: number of arguments");
  if(obj[0]==0)
    console->execerror(e_where,"illegal object");
  buf=obj[0]->GetBufferPointer();
  if(buf == 0)
    console->execerror(e_where,"null buffer");
  if(buf->GetDim()==0 || buf->GetDim()==1){
    ret_buf=new Scalar_Buffer;
    try{
      ret_buf->InitBuffer();
    }catch(bad_alloc){
      delete ret_buf;
      throw;
    }
    ((Scalar_Buffer *)ret_buf)->SetScalar(buf->GetIndex(0));
  }else{
    ret_buf=new Series_Buffer;
    try{
      ret_buf->SetIndex(0,buf->GetDim());
      ret_buf->SetDim(1);
      ret_buf->InitBuffer();
    }catch(buffer_exception){
      delete ret_buf;
      console->execerror(e_where,"buffer exception");
    }catch(bad_alloc){
      delete ret_buf;
      throw;
    }
    tmp=(double *)ret_buf->GetDataPointer();
    for(n=0;n<buf->GetDim();n++)
      tmp[n]=buf->GetIndex(n);
  }
  try{
    if (buf->GetDim()<=1)
      ret=new_SL_Object(SL_OBJ::SCALAR_O,ret_buf);
    else
      ret=new_SL_Object(SL_OBJ::SERIES_O,ret_buf);
  }catch(bad_alloc){
    delete ret_buf;
    throw;
  }catch(execerr_exception){
    delete ret_buf;
    throw;
  }catch(buffer_exception){
    delete ret_buf;
    console->execerror(e_where,"buffer exception");
  }
  return ret;
}

SL_Object *BuiltinFunctions::Typeof(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:typeof>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::Typeof()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  int i;
  static char   *undefine = "undefine";
  char         *strp = undefine;
  Base_Buffer *retbuf=0;
  SL_Object   *ret=0;
  Index idx;
  static struct {
    SL_OBJ::TYPE no;
    char *name;
  } types[] = {
    {SL_OBJ::SCALAR_O, "scalar"},
    {SL_OBJ::STRING_O, "string"},
    {SL_OBJ::SERIES_O, "series"},
    {SL_OBJ::SNAPSHOT_O, "snapshot"},
    {SL_OBJ::CONSTANT_O, "constant"},
    {SL_OBJ::MODULE_NAME_O, "module"},
    {SL_OBJ::BUILTIN_O, "builtin"},
    {SL_OBJ::UNDEF_O, 0}
  };
  ret=0; // for compiler
  if(argc!=1)
    console->execerror(e_where,"mismatch: number of arguments");
  if(obj[0]==0)
    console->execerror(e_where,"illegal object");
  for(i=0;types[i].name!=0;i++){
    if (obj[0]->TypeofOBJ()==types[i].no ) {
      strp=types[i].name;
      break;
    }
  }
  retbuf=new String_Buffer;
  try{
    retbuf->SetDim(1);
    retbuf->SetIndex(0,1);
    retbuf->InitBuffer();
    idx.SetDim(1);
    idx.SetIndex(0,0);
    ((String_Buffer*)retbuf)->SetString(idx,strp);
    ret=new_SL_Object(SL_OBJ::STRING_O,retbuf);
  }catch(buffer_exception){
    delete retbuf;
    console->execerror(e_where,"buffer exception");
  }catch(bad_alloc){
    delete retbuf;
    throw;
  }catch(execerr_exception){
    delete retbuf;
    throw;
  }
  return ret;
}

SL_Object *BuiltinFunctions::Printf(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:printf>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::Printf()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  int wl,wc;
  console->term_getmaxyx(&wl,&wc);
  register int  i = 0, n = 0;
  const char *format;
  char  *p, fmt[10];
  SL_Object        *tmp;
  if(obj[n] == 0 || obj[n]->TypeofOBJ() != SL_OBJ::STRING_O || 
     obj[n]->GetBufferPointer()==0 || 
     obj[n]->GetBufferPointer()->GetDataPointer()==0) {
    console->execerror(e_where,"illegal formatter");
  }
  format=((string *)obj[n]->GetBufferPointer()->GetDataPointer())[0].c_str();
  if(format==0)
    console->execerror(e_where,"illegal formatter");
  for(p = (char*)format; *p; i = 0, p++) {
    if(*p != '%') {
      console->tty_printf("%c",*p);
      continue;
    }
    fmt[i++] = *p++;    // %
    if(*p == '-')
      fmt[i++] = *p++;
    while(isdigit(*p) || *p == '.')
      fmt[i++] = *p++;
    switch(*p){
    case 'l':
      fmt[i++] = *p++; // break through
    case 'e':
    case 'f': 
    case 'g':
    case 'd':
    case 's':
    case 'c':
      fmt[i++] = *p;
      fmt[i] = '\0';
      break;
    default:
      putchar(*p);
      continue;
    }
    if(argc==++n)
      console->execerror(e_where,"illegal number of argument");
    tmp=obj[n];
    if(tmp == 0 || tmp->GetBufferPointer()==0) {
      char  mes[20];
      sprintf(mes, "illegal %dth object", n+1);
      console->execerror(e_where,mes);
    }
    tmp->Print(fmt,wl);
  }
  return 0;
}

SL_Object *BuiltinFunctions::Abort(int argc,SL_Object *obj[]){
  //  static char *e_where="Error [<BUILTIN:abort>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::Abort()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  /* symlist->FreeSymbols(); */
  throw execerr_exception();
}

SL_Object *BuiltinFunctions::eval_program(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:eval>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::eval_program()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  String_Buffer *buf;
  if(argc!=1 || obj==0 || obj[0]==0 || 
     obj[0]->TypeofOBJ()!=SL_OBJ::STRING_O
     || obj[0]->GetBufferPointer()==0 ){
    console->execerror(e_where,"illegal program");
  }
  buf=(String_Buffer*)obj[0]->GetBufferPointer();
  if(buf->GetDataPointer()==0 ||
     ((string *)buf->GetDataPointer())[0].size()==0){
    console->execerror(e_where,"illegal string buffer");
  }
  return stackmachine->eval_program(((string*)buf->GetDataPointer())[0].c_str());
}

SL_Object *BuiltinFunctions::welcome(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:welcome>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::welcome()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  int   lines, columns;
  size_t col_tmp;
  static  char asta[2] = "*";
  char  sbuf[256];
  const char *sym = asta;

  if(argc != 0) {
    SL_Object *tmp = obj[0];
    if(tmp == 0 || tmp->TypeofOBJ() != SL_OBJ::STRING_O || 
       tmp->GetBufferPointer()==0 || 
       tmp->GetBufferPointer()->GetDataPointer()==0 ||
       ((string *)tmp->GetBufferPointer()->GetDataPointer())[0].size()==0)
      console->execerror(e_where,"illegal object");
    sym = ((string*)tmp->GetBufferPointer()->GetDataPointer())[0].c_str();
  }

  console->term_getmaxyx(&lines, &columns);
  col_tmp = columns;

  _symbols(sym, col_tmp);

  if(col_tmp > strlen(WELCOME))
    _symbols(" ", (col_tmp - strlen(WELCOME))/2);
  console->tty_printf("%s\n",WELCOME);

  sprintf(sbuf, "Copyright(C) %s, %s\n", YEAR, COPYRIGHT);
  if(col_tmp > strlen(sbuf))
    _symbols(" ", (col_tmp - strlen(sbuf))/2);     
  console->tty_print(sbuf);

  sprintf(sbuf, "Version %s\n", PACKAGE_VERSION);
  if(col_tmp > strlen(sbuf))
    _symbols(" ", (col_tmp - strlen(sbuf))/2);
  console->tty_print(sbuf);

  _symbols(sym, col_tmp);
  console->tty_putc('\n');
  return (SL_Object *)0;
}

SL_Object *BuiltinFunctions::goodbye(int argc,SL_Object *obj[]){
  static char *e_where="[<BUILTIN:goodbye>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::goodbye()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  if(argc!=0 || obj!=0)
    console->warning(e_where,"illegal arguments");
  console->tty_printf("\n\t\tExit The SATELLITE World      ... Thank you.\n");
  return (SL_Object *)0;
}

SL_Object *BuiltinFunctions::Strlen(int argc,SL_Object *obj[]){
  static char *e_where="[<BUILTIN:strlen>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::Strlen()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  int          i,n;
  string      *argv;
  double       d,*buf;
  Base_Buffer *retbuf=0,*tmp=0;
  SL_Object   *ret=0;
  if(argc!=1)
    console->execerror(e_where,"mismatch, number of arguments");
  if(obj[0]==0)
    console->execerror(e_where,"null object of arguments");
  if (obj[0]->TypeofOBJ() != SL_OBJ::STRING_O)
    console->execerror(e_where,"illegal object type (not string)");
  tmp=obj[0]->GetBufferPointer();
  if(tmp==0)
    console->execerror(e_where,"null buffer");
  n=tmp->IndexSize();
  argv=(string*)tmp->GetDataPointer();
  if(argv==0)
    console->execerror(e_where,"null data of buffer");
  if(n==1){
    retbuf=new Scalar_Buffer;
    try{
      retbuf->InitBuffer();
      d=(double)argv[0].size();
      ((Scalar_Buffer*)retbuf)->SetScalar(d);
      ret=new_SL_Object(SL_OBJ::SCALAR_O,retbuf);
    }catch(bad_alloc){
      delete retbuf;
      throw;
    }catch(buffer_exception){
      delete retbuf;
      console->execerror(e_where,"buffer exception");
    }catch(execerr_exception){
      delete retbuf;
      throw;
    }
  } else {
    retbuf=new Series_Buffer;
    try{
      retbuf->CopyIndex(tmp->GetBufferIndex());
      retbuf->InitBuffer();
      buf=(double*)retbuf->GetDataPointer();
      for(i=0; i<n; i++)
	buf[i]=(double)argv[i].size();
      ret=new_SL_Object(SL_OBJ::SERIES_O,retbuf);
    }catch(bad_alloc){
      delete retbuf;
      throw;
    }catch(buffer_exception){
      delete retbuf;
      console->execerror(e_where, "buffer exception");
    }catch(execerr_exception){
      delete retbuf;
      throw;
    }
  }
  return ret;
}

SL_Object *BuiltinFunctions::strict_grammar(int argc,SL_Object *obj[]){
  static char *e_where="[<BUILTIN:strict_grammar>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::strict_grammar()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  double d;
  if(argc>1)
    console->execerror(e_where,"mismatch, number of arguments");
  if(argc==0){
    console->tty_printf("%s\n",(is_compat2x_mode()==0)?"false":"true");
  }else{
    if(obj[0]==0)
      console->execerror(e_where,"null object of arguments");
    if(obj[0]->TypeofOBJ() != SL_OBJ::SCALAR_O)
      console->execerror(e_where,"illegal object type (not scalar)");
    d = ((Scalar_Buffer*)obj[0]->GetBufferPointer())->GetScalar();
    if(d == 0.0) set_compat2x_mode(1);
    else set_compat2x_mode(0);
  }
  return (SL_Object *)0;
}

SL_Object *BuiltinFunctions::sl4vars(int argc,SL_Object *obj[]){
  string *argv,ret_str;
  SL_Object   *ret;
  Base_Buffer *buf;
  static char *e_where="[<BUILTIN:sl4vars>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::sl4vars()";
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  if(argc!=1)
    console->execerror(e_where,"mismatch, number of arguments");
  if(obj[0]->TypeofOBJ() != SL_OBJ::STRING_O)
    console->execerror(e_where,"illegal object type (not string)");
  argv=(string*)obj[0]->GetBufferPointer()->GetDataPointer();
  if(argv == 0)
    console->execerror(e_where,"null string object");
  /* moduledir */
  if(argv[0] == "moduledir"){
    char tmp[1024];
    GetModuleDirectory(tmp, 1024);
    ret_str = tmp;
  }
  /* homedir */
  if(argv[0] == "homedir"){
    char tmp[1024];
    GetHomeDirectory(tmp, 1024);
    ret_str = tmp;
  }
  /* userrcdir */
  if(argv[0] == "userrcdir"){
    char tmp[1024];
    GetUserResourceDirectory(tmp, 1024);
    ret_str = tmp;
  }
  /* systemrcdir */
  if(argv[0] == "systemrcdir"){
    char tmp[1024];
    GetSystemResourceDirectory(tmp, 1024);
    ret_str = tmp;
  }
  /* sharedir */
  if(argv[0] == "sharedir"){
    char tmp[1024];
    GetShareDirectory(tmp, 1024);
    ret_str = tmp;
  }
//   /* examplesdir */
//   if(argv[0] == "examplesdir"){
//     char tmp[1024];
//     GetExamplesDirectory(tmp, 1024);
//     ret_str = tmp;
//   }
  /* scriptdir */
  if(argv[0] == "scriptdir"){
    char tmp[1024];
    if(console->get_filepath(1024,tmp) != NULL)
      ret_str = tmp;
  }
  /* script */
  if(argv[0] == "script"){
    ret_str = console->get_filename();
  }
  buf = new String_Buffer(ret_str);
  ret = new_SL_Object(SL_OBJ::STRING_O,buf);
  return ret;
}

#ifdef WIN32
SL_Object *BuiltinFunctions::Pwd(int argc,SL_Object *obj[]){
  static char *e_where="Error [<BUILTIN:pwd>]";
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  static char *where="BuiltinFunctions::Pwd()";
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  static char pwd[256];
  Index idx;
  String_Buffer *retbuf=0;
  SL_Object *ret=0;
   // get current directory
   if( _getcwd( pwd, 256 ) == 0 )
     console->execerror(e_where,"cann't get current directory");
  retbuf=new String_Buffer;
  try{
    retbuf->SetDim(1);
    retbuf->SetIndex(0,1);
    retbuf->InitBuffer();
    idx.SetDim(1);
    idx.SetIndex(0,0);
    ((String_Buffer*)retbuf)->SetString(idx,pwd);
    ret=new_SL_Object(SL_OBJ::STRING_O,retbuf);
  }catch(buffer_exception){
    delete retbuf;
    console->execerror(e_where,"buffer exception");
  }catch(bad_alloc){
    delete retbuf;
    throw;
  }catch(execerr_exception){
    delete retbuf;
    throw;
  }
  return ret;
}

#endif
