/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: StackMachine.cpp,v 1.22 2005/11/10 13:54:14 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "SL_header.h"

#include <libsatellite.h>

#define  __IMPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __IMPORTSYMBOL__
#include "Program.h"
#include "Datum.h"
#include "Inline.h"
#include "CommandAlias.h"
#include "LexicalAnalizer.h"
#include "pipe.h"
#include "Builtin.h"
#include "Builtin_Buffer.h"
#include "parse.h"
#include "vmstat.h"
#include "pcond.h"

#include "acc.h"

#define  __EXPORTSYMBOL__
#include "StackMachine.h"
#undef   __EXPORTSYMBOL__

using namespace std;

static module_t *symbol_to_module(symbol_t *sym){
  SL_Object *obj;
  Builtin_Buffer *buf;
  module_t *mod;
  if(sym == NULL || symbol_get_type(sym) != SYMBOL_TYPE_MODULE) return NULL;
  obj = symbol_get_object(sym);
  if(obj == NULL || obj->TypeofOBJ() != SL_OBJ::MODULE_NAME_O) return NULL;
  buf = (Builtin_Buffer*)obj->GetBufferPointer();
  if(buf == NULL || buf->bltin_type() !=  BLTINBUF_TYPE_MODULE) return NULL;
  mod = buf->GetModule();
  return mod;
}

static module_command_t *symbol_to_satcom(symbol_t *sym){
  SL_Object *obj;
  Builtin_Buffer *buf;
  module_command_t *com;
  if(sym == NULL || symbol_get_type(sym) != SYMBOL_TYPE_SATCOM) return NULL;
  obj = symbol_get_object(sym);
  if(obj == NULL || obj->TypeofOBJ() != SL_OBJ::SATCOM_O) return NULL;
  buf = (Builtin_Buffer*)obj->GetBufferPointer();
  if(buf == NULL || buf->bltin_type() !=  BLTINBUF_TYPE_SATCOM) return NULL;
  com = buf->GetSatCom();
  return com;
}

static struct {
  int st;
  SL_OBJ::TYPE ot;
} type_bind[] = {
  { CLASS_TYPE_SERIES,   SL_OBJ::SERIES_O },
  { CLASS_TYPE_SNAPSHOT, SL_OBJ::SNAPSHOT_O },
  { CLASS_TYPE_SCALAR,   SL_OBJ::SCALAR_O },
  { CLASS_TYPE_STRING,   SL_OBJ::STRING_O },
  //  { CLASS_TYPE_BUILTIN_T, SL_OBJ::BUILTIN_O },
  { -1, SL_OBJ::UNDEF_O }
};

extern int yyparse(void *);

StackMachine::StackMachine(tty_console *con) : 
  console(con), builtin(this,con), pipe_stream(con),
  in_eval(false), do_print(true),
  call_stack_depth(0),lex(this,con){
  // set stop valiable
  STOP=(STACKFUNC)0;
  vmstat.prog=&vmstat.mainprog;
  returning = breaking = continuing = false;
  parser_condition_init(&vmstat);
  acc_table_init();
};

StackMachine::~StackMachine(){
#if defined(_DEBUG) && defined(_DEBUG_STACK)
  const char *where="StackMachine::~StackMachine()";
  if(!eval_stack.empty())
    console->tty_printf("eval_stack does not empty in %s\n",where);
  if(!data_stack.empty())
    console->tty_printf("data_stack does not empty in %s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_STACK)
  while(!eval_stack.empty()){
    delete eval_stack.top();
    eval_stack.pop();
  }
  while(!data_stack.empty()){
    delete data_stack.top();
    data_stack.pop();
  }
  parser_condition_free();
  vmstat.mainprog.Program_Init();
  acc_table_clean();
}

// pop object from stack
SL_Object *StackMachine::pop_obj(){
  Datum *datum;
  SL_Object *obj;
  datum=data_stack.top();
  data_stack.pop();
  obj=datum->GetObject();
  delete datum;
  if(obj){
    obj->obj_unref();
  }
  return obj;
}

// pop symbol from stack
symbol_t *StackMachine::pop_sym(){
  Datum *datum;
  symbol_t *sym;
  datum=data_stack.top();
  if(datum->type() != DATUM_TYPE_SYMBOL) return 0;
  data_stack.pop();
  sym=datum->GetSymbol();
  delete datum;
  return sym;
}

// pop value from stack
double StackMachine::pop_val(){
  double d;
  SL_Object *obj;
  obj = pop_obj(); acc_add(obj);
  if(obj==0)
    syscom->console->execerror(0,"illegal object. (null object)");
  if(obj->TypeofOBJ() != SL_OBJ::SCALAR_O)
    syscom->console->execerror(0,"illegal object. (no scalar)");
  d = ((Scalar_Buffer*)obj->GetBufferPointer())->GetScalar();
  return d;
}

// push object to stack
void StackMachine::push_obj(SL_Object *obj){
  Datum *datum;
  datum=new Datum;
  datum->SetObject(obj);
  if(obj!=0) obj->obj_ref();
  data_stack.push(datum);
}

// push symbol to stack
void StackMachine::push_sym(symbol_t *sym){
  Datum *datum;
  datum=new Datum;
  datum->SetSymbol(sym);
  data_stack.push(datum);
}

// push quoted string symbol to stack
SL_Object *StackMachine::symbol_to_quoted_object(symbol_t *sym){
  SL_Object *obj;
  String_Buffer *buf=0;
  Index pt;
  buf=new String_Buffer;
  buf->SetIndex(0,1);
  buf->SetDim(1);
  try{
    buf->InitBuffer();
    pt.SetIndex(0,0);
    pt.SetDim(0);
    buf->SetString(pt,sym->name);
    obj=new_SL_Object(SL_OBJ::STRING_O,buf);
  }catch(buffer_exception err){
    delete buf;
    console->execerror(err.what(),"datum");
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  return obj;
}

void StackMachine::MainLoop(){
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::MainLoop()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  Program *sp;
  int yyparse_flag;  // yyparse_flag - ok: 1, fail: 0
  yyparse_flag=1;
  initcode();
  while(lex.moreinput(yyparse_flag)){
    try {
      if(lex.getstream(0)!=0){
        initcode();
        while((yyparse_flag=yyparse(this)),yyparse_flag){
          // sucess
          sp=vmstat.mainprog.Get_Top();
          if(parser_condition_sub_indef() == 0 && sp!=0){
            Program_Run(sp);
          if(lex.getstream(0)==0) break;
          initcode();
          }
        }
      }
    }catch(buffer_exception){
      console->tty_printf("buffer exception\n");
      recovery();
      initcode();
    }catch(bad_alloc){
      console->tty_printf("out of memory\n");
      recovery();
      initcode();
    }catch(execerr_exception){
      recovery();
      initcode();
    }
  }
  return;
}

void StackMachine::Program_Run(Program *p){
  STACKFUNC run;
  Program *tmp;
  vmstat.prog->Set_PC(p);
  setsigint();
  for(tmp=vmstat.prog->Get_PC();
      !returning && !breaking && !continuing && tmp!=0 && 
        tmp->GetProgram()!=STOP && tmp->GetProgram()!=(STACKFUNC)0;
      tmp=vmstat.prog->Get_PC()){
    run=tmp->GetProgram();
    if(run==0) return;
    vmstat.prog->Inc_PC();
    (this->*run)();
    if(isintcatch()){
      unsetsigint();
      throw execerr_exception();
    }
  }
  unsetsigint();
  // clean accumulator
  acc_gc();
}

// eval expression ex. eval("hoge(x)")
SL_Object *StackMachine::eval_program(const char *str){
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::eval_program()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
  console->tty_printf(" -> LexicalAnalizer::InitStream()\n");
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  if(lex.InitStream(str)){
    Program *pc=0;
    Program_List *plist=0;
    bool previneval=in_eval;
    in_eval = true;
    // push program counter
    pc=vmstat.prog->Get_PC();
    // push program list
    plist=vmstat.prog;
    // change program list
    vmstat.prog=new Program_List;
    // push eval program list 
    eval_stack.push(plist);
    // parse
#if defined(_DEBUG) && defined(_DEBUG_CODE)
    console->tty_printf(" -> LexicalAnalizer::yyparse()");
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
    if(yyparse(this)){
#if defined(_DEBUG) && defined(_DEBUG_CODE)
      console->tty_printf(" -> (program run) in %s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
      try{
        Program_Run(vmstat.prog->Get_Top()); // program run
      }catch(execerr_exception){
        delete vmstat.prog;
        vmstat.prog = eval_stack.top();
        eval_stack.pop();
        vmstat.prog->Set_PC(pc);
        in_eval = previneval;
        throw;
      }catch(bad_alloc){
        delete vmstat.prog;
        vmstat.prog = eval_stack.top();
        eval_stack.pop();
        vmstat.prog->Set_PC(pc);
        in_eval = previneval;
        throw;
      }catch(buffer_exception){
        delete vmstat.prog;
        vmstat.prog = eval_stack.top();
        eval_stack.pop();
        vmstat.prog->Set_PC(pc);
        in_eval = previneval;
        throw;
      }
    }
    if( eval_stack.empty() ){
      // parser error : recovery() already finised
      throw execerr_exception();
    }
    // pop eval program list
    delete vmstat.prog;
    // recover program list
    vmstat.prog = eval_stack.top();
    eval_stack.pop();
    // run next program
#if defined(_DEBUG) && defined(_DEBUG_CODE)
    console->tty_printf(" -> (program run next) in %s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
    vmstat.prog->Set_PC(pc);
    // recover sub program
    in_eval = previneval;
  }
  return 0;
}

// part of code.core.c
void StackMachine::initcode(){    // initialize for code generation
  if(console->pipe_status()){
    string a;
    pipe_stream.closepipe(a);
  }
  // reset flags
  do_print = true;
  returning = breaking = continuing = false;
  vmstat.sub = NULL;
  vmstat.prog = &vmstat.mainprog;      // reset program start address
  vmstat.prog->Program_Init();
  vmstat.module = NULL;
  //  syscom->symbol_list.FreeSymbols();
  if(call_stack_depth!=0){
    console->warning("call stack depth is not zero.","maybe bug");
    call_stack_depth=0;
  }
}

void StackMachine::recovery(){ // initialize for sub code generation
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::recovery()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  console->tty_printf("\n");
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  
  // if now running in FUNCTION or PROCEDURE then quit subprogram
  parser_condition_recovery();

  // if now running in eval then quit eval program
  if(in_eval){
    // error in eval
    while(!eval_stack.empty()){
      delete vmstat.prog;
      vmstat.prog = eval_stack.top();
      eval_stack.pop();
    }
    in_eval=false;
  }
  
  // if function stack isn't empty then pop all stack frames
  if(!data_stack.empty()){
    Datum     *datum;
    SL_Object *obj;
    while(!data_stack.empty()){
      datum = data_stack.top();
      if(datum->type() == DATUM_TYPE_OBJECT){
        obj = datum->GetObject(); 
        if(obj != NULL){
          obj->obj_unref();
          if(obj->empty()){
            acc_del(obj);
            delete_SL_Object(obj);
          }
        }
      }
      delete datum;
      data_stack.pop();
    }
  }
  vmstat.sub = NULL;
  /* if error at module definition then clear it */
  if(vmstat.module != NULL){
    module_delete(vmstat.module);
    vmstat.module = NULL;
  }
  
  // if now running in inline file quit inline
#if defined(_DEBUG) && defined(_DEBUG_CODE) 
  if(!lex.inline_stack.empty())
    console->tty_printf(" -> (error inline) in %s\n","inline");
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  while(!lex.inline_stack.empty()){
    Inline *inl;
    inl=lex.inline_stack.top();
    delete inl;
    lex.inline_stack.pop();
  }
  while(console->input_file_pop());
}

// regist code area 1
Program *StackMachine::code(STACKFUNC f){ 
  Program *ret;
  ret=vmstat.prog->Program_Add(f);
  return ret;
}


// regist code area 2
Program *StackMachine::code(symbol_t *f){
  Program *ret;
  ret=vmstat.prog->Symbol_Add(f);
  return ret;
}

// regist code area 3
Program *StackMachine::code(int f){ 
  Program *ret;
  ret=vmstat.prog->Data_Add(f);
  return ret;
}

Program *StackMachine::code(SL_Object *obj){
  Program *ret;
  ret=vmstat.prog->Obj_Add(obj);
  return ret;
}

// POP() st[-1]: obj(-1)
void StackMachine::POP(){
  SL_Object *obj = pop_obj();
  if(obj){ 
    if(obj->empty()){
      acc_del(obj);
      delete_SL_Object(obj);
    }
  }
}

// constpush(obj) st[+1]: obj(+1)
void StackMachine::constpush(){
  SL_Object *obj = vmstat.prog->Get_PC()->GetObj();
  vmstat.prog->Inc_PC();
  push_obj(obj);
}

// varpush(sym) st[+1]: sym(+1)
void StackMachine::varpush(){
  symbol_t *sym=vmstat.prog->Get_PC()->GetSymbol();
  vmstat.prog->Inc_PC();
  push_sym(sym);
}

// eval() st[0]: sym(-1), obj(+1)
void StackMachine::eval(){
  SL_Object *obj;
  symbol_t  *sym = pop_sym();
  if(symbol_get_type(sym) == SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable "; err_msg+=sym->name;
    console->execerror(0, err_msg.c_str());
  }
  obj = symbol_get_object(sym);
  if(obj == 0){
    err_msg="null object "; err_msg+=sym->name;
    console->execerror(0 ,err_msg.c_str());
  }
  push_obj(obj);
}

// assign() st[-1]: sym(-1), obj(-1), obj(+1)
void StackMachine::assign(){
  symbol_t  *sym;
  SL_Object *src, *tmp;
  sym = pop_sym();
  src = pop_obj(); acc_add(src);
  if(src == 0)
    console->execerror(0,"null object assignment");
  tmp = symbol_get_object(sym);
  if(tmp){
    if(src->TypeofOBJ() != tmp->TypeofOBJ())
      console->warning("illegal combination of %s and right, (op = )",
                       sym->name);
  }
  /* check object refcount */
  SL_Object *tobj = symbol_get_object(sym);
  if(tobj){
    tobj->obj_unref();
    if(tobj->empty())
      acc_del(tobj);
    tobj->obj_ref();
  }
  symbol_set_object(sym, src, SYMBOL_TYPE_VAR);
  acc_del(src);
  push_obj(src);
}

// selfasgn(func) st[-1]: sym(-1), obj(-1), obj(+1)
void StackMachine::selfasgn(){
  symbol_t  *sym;
  SL_Object *src,*tmp,*ret;
  int op_n = vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();
  sym = pop_sym();
  src = pop_obj(); acc_add(src);
  if(symbol_get_type(sym) != SYMBOL_TYPE_VAR){
    err_msg="assingment to non-variable "; err_msg+=sym->name;
    console->execerror(0, err_msg.c_str());
  }
  if(src == 0)
    console->execerror(0,"null object assignment");
  tmp = symbol_get_object(sym);
  if(tmp == 0)
    console->execerror(0,"assignment to null object");
  
  ret = tmp->Opecode(op_n, src);
  if(ret->TypeofOBJ() != tmp->TypeofOBJ()){
    delete_SL_Object(ret);
    console->execerror(0,"illegal combination of left and right, (op =)");
  }
  /* check object refcount */
  SL_Object *tobj = symbol_get_object(sym);
  if(tobj){
    tobj->obj_unref();
    if(tobj->empty())
      acc_del(tobj);
    tobj->obj_ref();
  }
  symbol_set_object(sym, ret, SYMBOL_TYPE_VAR);
  acc_del(src);
  push_obj(src);
}

// constant() st[-2]: sym(-1), obj(-1)
void StackMachine::constant(){ 
  symbol_t *sym;
  SL_Object *src;
  sym = pop_sym();
  src = pop_obj(); acc_add(src);
  if(symbol_get_type(sym) != CONSTANT &&
     symbol_get_type(sym) != SYMBOL_TYPE_UNDEF){
    err_msg="assingment to non-variable ";
    err_msg+=sym->name;
    console->execerror(0, err_msg.c_str());
  }
  if(symbol_get_type(sym) == SYMBOL_TYPE_UNDEF){
    symbol_table_move(syscom->gl_symtab, syscom->cur_symtab, sym);
    /* check object refcount */
    SL_Object *tobj = symbol_get_object(sym);
    if(tobj){
      tobj->obj_unref();
      if(tobj->empty())
        acc_del(tobj);
      tobj->obj_ref();
    }
    symbol_set_object(sym, src, SYMBOL_TYPE_CONST);
  }
}

// varread(sym) st[+1]: obj(+1)
void StackMachine::varread(){
  int kt;
  SL_Object   *obj;
  Base_Buffer *buf;
  symbol_t *sym = vmstat.prog->Get_PC()->GetSymbol();
  SL_OBJ::TYPE type;
  vmstat.prog->Inc_PC();
  kt=((Builtin_Buffer*)symbol_get_object(sym)->GetBufferPointer())->GetClassType();
  switch(kt){
  case CLASS_TYPE_SERIES:
    buf = new Series_Buffer;
    type = SL_OBJ::SERIES_O;
    break;
  case CLASS_TYPE_SNAPSHOT:
    buf = new Snapshot_Buffer;
    type = SL_OBJ::SNAPSHOT_O;
    break;
  case CLASS_TYPE_STRING:
    buf = new String_Buffer;
    type = SL_OBJ::STRING_O;
    break;
  case CLASS_TYPE_SCALAR:
    buf = new Scalar_Buffer;
    type = SL_OBJ::SCALAR_O;
    break;
  default:
    buf = 0; type = SL_OBJ::UNDEF_O;
    err_msg="not supported method ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str());
  }
  obj = new_SL_Object(type, buf);
  acc_add(obj);
  obj->Read();
  acc_del(obj);
  push_obj(obj);
}

// auto_print() st[0]: obj(-1), obj(+1)
void StackMachine::auto_print(){
  int l,c;
  SL_Object *obj;
  obj = pop_obj(); acc_add(obj);
  console->term_getmaxyx(&l,&c);
  if(obj){
    obj->Print(0,c);
    console->tty_printf("\n");
  }
  acc_del(obj);
  push_obj(obj);
}

// print() st[-1]: obj(-1)
void StackMachine::print(){
  int l,c;
  SL_Object *obj;
  obj = pop_obj(); acc_add(obj);
  console->term_getmaxyx(&l,&c);
  if(obj != (SL_Object *) 0){
    obj->Print(0,c);
    console->tty_printf("\n");
  }
}

// postfix(int) st[0]: sym(-1), obj(+1)
void StackMachine::postfix(){
  symbol_t  *sym;
  SL_Object *inc,*obj;
  sym = pop_sym();
  if(symbol_get_type(sym) == SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0, err_msg.c_str());
  }
  obj = symbol_get_object(sym);
  push_obj(obj);
  inc = increment(sym);
  /* check object refcount */
  SL_Object *tobj = symbol_get_object(sym);
  if(tobj){
    tobj->obj_unref();
    if(tobj->empty())
      acc_del(tobj);
    tobj->obj_ref();
  }
  symbol_set_object(sym, inc, SYMBOL_TYPE_VAR);
}

// prefix(int) st[0]: sym(-1), obj(+1)
void StackMachine::prefix(){
  symbol_t  *sym;
  SL_Object *inc;
  sym = pop_sym();
  if(symbol_get_type(sym) == SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0, err_msg.c_str());
  }
  inc = increment(sym);
  /* check object refcount */
  SL_Object *tobj = symbol_get_object(sym);
  if(tobj){
    tobj->obj_unref();
    if(tobj->empty())
      acc_del(tobj);
    tobj->obj_ref();
  }
  symbol_set_object(sym, inc, SYMBOL_TYPE_VAR);
  push_obj(inc);
}

// basic1() st[0]: obj(-1), obj(+1)
void StackMachine::basic1(){
  SL_Object *obj,*ret;
  int op_n = vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();
  obj = pop_obj(); acc_add(obj);
  if(obj == 0)
    console->execerror(0, "null object operation");
  ret = obj->Opecode(op_n, 0);
  push_obj(ret);
}

// basic2() st[-1]: obj(-1), obj(-1), obj(+1)
void StackMachine::basic2(){
  SL_Object *obj1,*obj2,*ret;
  int op_n = vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();
  obj1 = pop_obj(); acc_add(obj1);
  obj2 = pop_obj(); acc_add(obj2);
  if(obj1 == 0 || obj2 == 0)
    console->execerror(0, "null object operation");
  ret = obj2->Opecode(op_n, obj1);
  push_obj(ret);
}

// run builtin functions
void StackMachine::bltin(){
  int i,narg;
  BFUNC func;
  Builtin_Buffer *buf;
  SL_Object **argm,*obj;
  symbol_t *sym;
  sym = vmstat.prog->Get_PC()->GetSymbol();     vmstat.prog->Inc_PC();
  narg = (int)vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();
  if(narg!=0) argm = new SL_Object*[narg];
  else argm = 0;
  for(i=narg; i > 0; i--){
    argm[i-1] = pop_obj(); acc_add(argm[i-1]);
  }
  obj = symbol_get_object(sym);
  if(obj==0){
    if(argm)  delete [] argm;
    console->execerror(0,"null object operation");
  }
  buf=(Builtin_Buffer*)obj->GetBufferPointer();
  if(buf==0){
    if(argm)  delete [] argm;
    console->execerror(0,"null buffer operation");
  }
  func=buf->GetBFunction();
  try{
    obj = builtin.execute(func,narg, argm);
  }catch(bad_alloc){
    if(argm)  delete [] argm;
    throw;
  }catch(execerr_exception){
    if(argm)  delete [] argm;
    throw;
  }
  if(argm)  delete [] argm;
  push_obj(obj);
}

// dispatch(sym,int n) st[-n+1]: obj(-n),obj(+1)
void StackMachine::dispatch(){
  int i,narg;
  Builtin_Buffer *buf;
  SL_Object **arg,*obj=0,*ret;
  symbol_t *sym;
  sym   = vmstat.prog->Get_PC()->GetSymbol();  vmstat.prog->Inc_PC();
  narg = (int)vmstat.prog->Get_PC()->GetData();  vmstat.prog->Inc_PC();

  if(narg == 0)
    console->execerror(0,"null object operation");
  narg--;

  /* check operator function symbol */
  obj = symbol_get_object(sym);
  if(obj == 0){
    err_msg="null object operation ";  err_msg+=sym->name;
    console->execerror(0,err_msg.c_str());
  }
  buf = (Builtin_Buffer*)obj->GetBufferPointer();
  if(buf == 0){
    err_msg="null buffer operation ";  err_msg+=sym->name;
    console->execerror(0,err_msg.c_str());
  }

  if(narg != 0)
    arg = new SL_Object*[narg];
  else
    arg = 0;
  for(i=narg-1; i >= 0; i--){
    arg[i] = pop_obj(); acc_add(arg[i]);
  }
  obj = pop_obj(); acc_add(obj);
  if(obj == 0)
    console->execerror(0, "null object operation");
  ret = 0;
  try{
    int func;
    func=buf->GetOFunction();
    if(narg==0) ret = obj->Opecode(func,0);
    else        ret = obj->Opecode(func,arg[0]);
  }catch(execerr_exception){
    if(narg!=0) delete [] arg;
    throw;
  }catch(bad_alloc){
    if(narg!=0) delete [] arg;
    throw;
  }
  if(narg!=0) delete [] arg;
  push_obj(ret);
}

// array(int n) st[-1-n]: sym(-1),obj(-n),obj(+1)
//  -- pop stack --
//    1   - symbol : target symbol
//    2   - object : src object
//    3~n - object : array index object
//  -- push stack --
//    1   - object : src object
void StackMachine::array(){
  int i, dim, tmp;
  symbol_t *sym;
  SL_Object *obj,*ret;
  Index idx;
  // get dimension from next program counter. (ex. A[i] -- 'i')
  dim=vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();

  // pop
  sym = pop_sym();

  // error check
  if(symbol_get_type(sym)==SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  for(i=dim;i>0;i--){
    obj = pop_obj(); acc_add(obj);
    if(obj==0 || obj->TypeofOBJ()!=SL_OBJ::SCALAR_O){
      err_msg="illegal index ";
      err_msg+=sym->name;
      console->execerror(0,err_msg.c_str()); 
    }
    tmp = (int)((Scalar_Buffer *)obj->GetBufferPointer())->GetScalar();
    idx.SetIndex(i-1,tmp);
  }

  obj = symbol_get_object(sym);
  if(obj==0){
    err_msg="null object ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(obj->TypeofOBJ()!=SL_OBJ::SERIES_O && 
     obj->TypeofOBJ()!=SL_OBJ::SNAPSHOT_O && 
     obj->TypeofOBJ()!=SL_OBJ::STRING_O){
    err_msg="not supported method with ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(obj->TypeofOBJ()==SL_OBJ::SERIES_O){
    for(i=dim;i>0;i--)
      idx.SetIndex(i,idx.GetIndex(i-1));
    dim++;
    idx.SetIndex(0,0);
  }
  idx.SetDim(dim);

  ret = obj->Array(idx);
  push_obj(ret);
}

// array_asgn(int n) st[-1-n]: sym(-1),obj(-1),obj(-1)*n,obj(+1)
//  -- pop stack --
//    1   - symbol : target symbol
//    2   - object : src object
//    3~n - object : array index object
//  -- push stack --
//    1   - object : src object
void StackMachine::array_asgn(){
  int i, dim, tmp;
  symbol_t *sym;
  SL_Object *obj,*src;
  Index idx;
  dim = vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();

  // pop
  sym = pop_sym();
  src = pop_obj(); acc_add(src);

  // error check
  if(symbol_get_type(sym)!=SYMBOL_TYPE_VAR){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(src==0){
    err_msg="null object assignment ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }

  // get assignment index
  for(i = dim-1; i>=0; i--){
    obj = pop_obj(); acc_add(obj);
    if(obj==0 || obj->TypeofOBJ()!=SL_OBJ::SCALAR_O){
      err_msg="illegal index "; err_msg+=sym->name;
      console->execerror(0,err_msg.c_str()); 
    }
    tmp = (int)((Scalar_Buffer *)obj->GetBufferPointer())->GetScalar();
    idx.SetIndex(i, tmp);
  }

  obj = symbol_get_single_object(sym);
  if(obj==0){
    err_msg="null object ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(obj->TypeofOBJ()!=SL_OBJ::SERIES_O && 
     obj->TypeofOBJ()!=SL_OBJ::SNAPSHOT_O && 
     obj->TypeofOBJ()!=SL_OBJ::STRING_O){
    err_msg="not supported method with ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(obj->TypeofOBJ()==SL_OBJ::SERIES_O){
    for(i = dim; i>0; i--)
      idx.SetIndex(i,idx.GetIndex(i-1));
    dim++;
    idx.SetIndex(0,0);
  }

  idx.SetDim(dim);

  obj->ArrayAsgn(idx, src);
  acc_del(src);
  push_obj(src);
}

// snapshot() st[-1]: sym(-1),obj(-1),obj(+1)
//  -- pop stack --
//    1 - symbol : src symbol
//    2 - object : time index
//  -- push stack --
//    1 - object : snapshot object series x[idx]
void StackMachine::snapshot(){
  int time;
  symbol_t *sym;
  SL_Object *obj,*ret;

  // pop
  sym = pop_sym();
  obj = pop_obj(); acc_add(obj);

  // error check
  if(symbol_get_type(sym) == SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(obj==0 || obj->TypeofOBJ() != SL_OBJ::SCALAR_O){
    err_msg="illegal index ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  time=(int)((Scalar_Buffer *)obj->GetBufferPointer())->GetScalar();

  // get object from symbol
  obj = symbol_get_object(sym);
  if(obj == 0){
    err_msg="null object ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  // get snapshot of time from object
  if(obj->TypeofOBJ() != SL_OBJ::SERIES_O){
    err_msg="not supported method with ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  ret = obj->Snap(time);
  push_obj(ret);
}

// snapshot_asgn() st[-2]: sym(-1),obj(-1),obj(-1),obj(+1)
//  -- pop stack --
//    1 - symbol : to symbol
//    2 - object : src object
//    3 - object : target index
//  -- push stack --
//    1 - object : src object
void StackMachine::snapshot_asgn(){
  int time;
  symbol_t *sym;
  SL_Object *obj, *src, *time_obj;
  // pop
  sym      = pop_sym();
  src      = pop_obj(); acc_add(src);
  time_obj = pop_obj(); acc_add(time_obj);

  // check type
  if(symbol_get_type(sym)==SYMBOL_TYPE_UNDEF){
    err_msg="undefined variable ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(src == 0){
    err_msg="null object assignment ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  if(time_obj==0 || time_obj->TypeofOBJ() != SL_OBJ::SCALAR_O){
    err_msg="illegal index ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  time=(int)((Scalar_Buffer *)time_obj->GetBufferPointer())->GetScalar();

  // get object from symbol
  obj = symbol_get_single_object(sym);
  if(obj == 0){
    err_msg="null object ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }
  // get snapshot of time from object
  if(obj->TypeofOBJ()!=SL_OBJ::SERIES_O && 
     obj->TypeofOBJ()!=SL_OBJ::SNAPSHOT_O && 
     obj->TypeofOBJ()!=SL_OBJ::SCALAR_O){
    err_msg="not supported method with ";
    err_msg+=sym->name;
    console->execerror(0,err_msg.c_str()); 
  }

  obj->SnapAsgn(time,src);

  acc_del(src); push_obj(src);
}

// newobj(class<sym>,var<sym>,dim<int>) st[-dim]: obj(-1)*dim
void StackMachine::newobj(){
  int kt,i,dim,n,k,w;
  symbol_t *sym,*var;
  SL_Object *tmp,*obj;
  Index idx;
  // get new object class and it's dimension
  sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  var=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  dim=vmstat.prog->Get_PC()->GetData();   vmstat.prog->Inc_PC();
  n=dim;
  if(symbol_get_type(sym)!=SYMBOL_TYPE_CLASS){
    err_msg="illegal symbol type "; err_msg+=var->name;
    console->execerror(0,err_msg.c_str()); 
  }
  kt=((Builtin_Buffer*)symbol_get_object(sym)->GetBufferPointer())->GetClassType();
  if(kt==CLASS_TYPE_SCALAR  && dim != 0){
    err_msg="illegal dimension (scalar) ";
    err_msg+=var->name;
    console->execerror(0,err_msg.c_str()); 
  }
  idx.SetIndex(0,1);
  if(kt==CLASS_TYPE_SERIES){
    dim++;
    if(dim>MAX_INDEX){
      err_msg="illegal dimension (series) ";
      err_msg+=var->name;
      console->execerror(0,err_msg.c_str()); 
    }
  }
  if(kt==CLASS_TYPE_SNAPSHOT && (dim==0 || dim>MAX_INDEX)){
    err_msg="illegal dimension (snapshot) ";
    err_msg+=var->name;
    console->execerror(0,err_msg.c_str()); 
  }
  obj = symbol_get_object(var);
  k=dim-1;
  for(i=0;i<n;i++){
    tmp = pop_obj(); acc_add(tmp);
    if(tmp->TypeofOBJ()!=SL_OBJ::SCALAR_O){
      err_msg="illegal index "; err_msg+=var->name;
      console->execerror(0,err_msg.c_str()); 
    }
    w=(int)((Scalar_Buffer*)tmp->GetBufferPointer())->GetScalar();
    if(w<1){
      err_msg="illegal index "; err_msg+=var->name;
      console->execerror(0,err_msg.c_str()); 
    }
    idx.SetIndex(k--,w);
  }

  for(i=0;type_bind[i].st!=-1;i++){
    if(kt==type_bind[i].st) break;
  }
  if(type_bind[i].st==-1)
    console->execerror("fatal error", "StackMachine::newobj()");

  if(kt!=CLASS_TYPE_SCALAR && dim==0) dim=1;
  idx.SetDim(dim);

  if(obj){
    if(obj->TypeofOBJ()!=type_bind[i].ot)
      console->warning("different type object", var->name);
  }
  Base_Buffer *bb;
  switch(type_bind[i].ot){
  case SL_OBJ::SERIES_O:   bb=new Series_Buffer;   break;
  case SL_OBJ::SNAPSHOT_O: bb=new Snapshot_Buffer; break;
  case SL_OBJ::STRING_O:   bb=new String_Buffer;   break;
  case SL_OBJ::SCALAR_O:   bb=new Scalar_Buffer;   break;
  default: bb=0; break;
  }
  bb->CopyIndex(idx);
  bb->InitBuffer();
  obj = new_SL_Object(type_bind[i].ot,bb);
  /* check object refcount */
  SL_Object *tobj = symbol_get_object(var);
  if(tobj){
    tobj->obj_unref();
    if(tobj->empty())
      acc_del(tobj);
    tobj->obj_ref();
  }
  symbol_set_object(var, obj, SYMBOL_TYPE_VAR);
}

// create ramp '+ - 1' array
void StackMachine::ramp(){
  int i,to,from,size;
  Series_Buffer *buf=0;
  SL_Object *ret=0;
  double *pt=0;
  to =   (int)pop_val();
  from = (int)pop_val();
  buf=new Series_Buffer;
  size=Max(to,from)-Min(to,from)+1;
  buf->SetDim(1);
  buf->SetIndex(0,size);
  buf->InitBuffer();
  pt=(double *)buf->GetDataPointer();
  if(from<=to){
    for(i = 0; i < size; i++)
      pt[i] = from + (double)i;
  }else{
    for(i = 0; i < size; i++)
      pt[i] = from - (double)i;
  }
  try{
    ret=new_SL_Object(SL_OBJ::SERIES_O,buf);
  }catch(bad_alloc){
    delete buf;
    throw;
  }catch(execerr_exception){
    delete buf;
    throw;
  }
  push_obj(ret);
}

// combin(int n) st[-n+1]: obj(-n), obj(+1)
void StackMachine::combin(){
  SL_Object  **nobj, *ret;
  Base_Buffer *tmp, *buf;
  int i, j, size, narg, pos;
  const char *msg;
  double *data;
  narg = vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();
  if(narg==0)
    console->execerror(0,"null object operation");
  nobj = new SL_Object*[narg];
  size = 0;
  for(i=0; i<narg; i++){
    msg=0;
    nobj[i] = pop_obj(); acc_add(nobj[i]);
    if(nobj[i]==0) msg = "null object operation";
    tmp = nobj[i]->GetBufferPointer();
    if(tmp == 0)   msg = "null buffer in object";
    switch(nobj[i]->TypeofOBJ()){
    case SL_OBJ::SERIES_O:
    case SL_OBJ::SNAPSHOT_O:
    case SL_OBJ::STRING_O:
      if(tmp->GetDim()!=1)
        msg = "not supported dimension";
      break;
    case SL_OBJ::SCALAR_O:
      break;
    default:
      msg = "illegal object type";
      break;
    }
    size += tmp->IndexSize();
    if(msg){
      delete [] nobj;
      console->execerror(0,msg);
    }
  }
  try{
    buf = new Series_Buffer;
    buf->SetDim(1);
    buf->SetIndex(0,size);
    buf->InitBuffer();
  }catch(execerr_exception){
    delete [] nobj;
    throw;
  }catch(bad_alloc){
    delete [] nobj;
    throw;
  }
  pos = 0;
  data = (double*)buf->GetDataPointer();
  for(i=narg-1; i>=0; i--){
    tmp  = nobj[i]->GetBufferPointer();
    size = tmp->IndexSize();
    if(nobj[i]->TypeofOBJ() == SL_OBJ::STRING_O){
      string *str = (string*)tmp->GetDataPointer();
      for(j=0;j<size;j++) data[pos++] = atof(str[j].c_str());
    }else{
      double *d   = (double*)tmp->GetDataPointer();
      for(j=0;j<size;j++) data[pos++] = d[j];
    }
  }
  ret = new_SL_Object(SL_OBJ::SERIES_O,buf);
  delete [] nobj;
  push_obj(ret);
}

// undef_var(sym) st[-1]: sym(-1)
void StackMachine::undef_var(){
  symbol_t *sym;
  sym  = vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  if(symbol_table_undef(syscom->cur_symtab, sym->name) != 0)
    symbol_table_undef(syscom->gl_symtab, sym->name);
}

// undef_all() st[0]:
void StackMachine::undef_all(){
  vmstat.prog->Inc_PC();
  symbol_table_undefall(syscom->cur_symtab);
  symbol_table_undefall(syscom->gl_symtab);
}

// isdef_var(sym) st[0]: sym(-1),obj(+1)
void StackMachine::isdef_var(){
  SL_Object *ret;
  Scalar_Buffer *buf;
  double  flag = -1.0;
  symbol_t *sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  switch(symbol_get_type(sym)){
  case SYMBOL_TYPE_VAR:
  case CONSTANT:
    flag = 1.0;
    break;
  case SYMBOL_TYPE_UNDEF:
    flag = 0;
    break;
  }
  buf=new Scalar_Buffer;
  try{
    buf->InitBuffer();
    buf->SetScalar(flag);
    ret = new_SL_Object(SL_OBJ::SCALAR_O, buf);
  }catch(bad_alloc){
    delete buf;
    throw;
  }
  push_obj(ret);
}


static void *inv_cast(double *buf, int elsize, int length){
// Buffer *buf;       /* charactor type bufffer */
// int     elsize;    /* size of a data element [byte] */
// int     length;    /* total number of data   */
  register int    i;
  void     *data = 0;
  switch(elsize){
    case 2 /* [byte] */ :
      short *sdata;
      sdata = new short[length];
      for(i = 0; i < length; i++)
        sdata[i] = (short) buf[i];
      data = (void*)sdata;
      break;
    case 4 /* [byte] */ :
      float *fdata;
      fdata = new float[length];
      for(i = 0; i < length; i++)
        fdata[i] = (float)buf[i];
      data = fdata;
      break;
    case 8 /* [byte] */ :
      double *ddata;
      ddata = new double[length];
      for(i = 0; i < length; i++)
        ddata[i] = (double)buf[i];
      data = ddata;
      break;
    default:
      break;
  }
  return data;
}

static int del_inv_cast(void *data, int elsize){
  switch(elsize){
  case 2 /* [byte] */ :
    short *sdata;
    sdata = (short*)data;
    delete [] sdata;
    break;
  case 4 /* [byte] */ :
    float *fdata;
    fdata = (float*)data;
    delete [] fdata;
    break;
  case 8 /* [byte] */ :
    double *ddata;
    ddata = (double*)data;
    delete [] ddata;
    break;
  default:
    return -1;
    break;
  }
  return 0;
}

// load and store sub function
int StackMachine::get_index(){
  int    numb;
  SL_Object *tmp;
  tmp = pop_obj(); acc_add(tmp);
  if(tmp == 0 || tmp->TypeofOBJ()!=SL_OBJ::SCALAR_O){
    console->execerror(0,"illegal index type");
  }
  numb = (int)((Scalar_Buffer *)tmp->GetBufferPointer())->GetScalar();
  if(numb < 0){
    console->execerror(0,"illegal index");
  }
  return numb;
}

// load(int i) i==0 - st[0]:  obj(-1[fname]),obj(+1[ret])
//             i==1 - st[-1]: obj(-1[index]),obj(-1[fname]),obj(+1[ret])
void StackMachine::load(){
  int i, numb, idxsiz;
  SL_Object *obj;
  Series_Buffer *buf=0;
  Index pt;
  Header head;
  const char *tmp;
  char fname[256], *dat;
  int has_index = vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();

  if(has_index){
    numb = get_index();
    if(numb<0)
      console->execerror(0,"illegal index");
  }else{
    numb = 0;
  }

  obj = pop_obj(); acc_add(obj);
  if(obj == 0 || obj->TypeofOBJ() != SL_OBJ::STRING_O)
    console->execerror(0,"failed to evaluate filename");
  pt.SetIndex(0,0); pt.SetDim(1);
  tmp=((String_Buffer*)obj->GetBufferPointer())->GetString(pt);
#ifdef HAVE_STRNCPY
  strncpy(fname,tmp,256);
#else
  strcpy(fname,tmp);
#endif
  fname[255]='\0';

  // read file
  if(has_index){
    dat = ::LoadData(fname, numb, &head);
  }else{
    dat = _ReadFile(fname, &head);
  }
  if(dat==0)
    console->execerror(0,"not exist file");
  buf = new Series_Buffer();
  buf->SetIndexInfo(head.dim, head.index);
  try{
    buf->InitBuffer();
  }catch(bad_alloc){
    FreeData(dat);
    delete buf;
  }
  idxsiz = buf->IndexSize();
  switch(head.data_size){
  case 8:
    memcpy(buf->GetDataPointer(),dat, idxsiz *sizeof(double));
    break;
  case 4:
    for(i=0;i<idxsiz;i++)
      ((double*)buf->GetDataPointer())[i] = (double)(((float*)dat)[i]);
    break;
  case 2:
    for(i=0;i<idxsiz;i++)
      ((double*)buf->GetDataPointer())[i]= (double)(((short*)dat)[i]);
    break;
  }
  FreeData(dat);
  try{
    obj = new_SL_Object(SL_OBJ::SERIES_O,buf);
  }catch(bad_alloc){
    delete buf;
    throw;
  }catch(execerr_exception){
    delete buf;
    throw;
  }
  push_obj(obj);
}

// store(int i) i==0 - st[-1]: obj(-1[src]),obj(-1[fname]),obj(+1[ret])
//              i==1 - st[-2]: obj(-1[src]),obj(-1[index]),obj(-1[fname]),
//                             obj(+1[ret])
void StackMachine::store(){
  char  fname[256];
  const char *tmp;
  Index pt;
  void *data;
  int   dsiz, status;
  SL_Object     *fname_obj,*data_obj;
  Series_Buffer *data_buf;
  int dim, index[MAX_INDEX];
  int numb;
  int has_index=vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();

  data_obj = pop_obj(); acc_add(data_obj);
  if(data_obj == 0 || !(data_obj->TypeofOBJ() == SL_OBJ::SERIES_O ||
                        data_obj->TypeofOBJ() == SL_OBJ::SNAPSHOT_O))
    console->execerror(0, "illegal stored type");

  if(has_index)
    numb = get_index();

  fname_obj = pop_obj(); acc_add(fname_obj);
  if(fname_obj == 0 || fname_obj->TypeofOBJ() != SL_OBJ::STRING_O)
    console->execerror(0, "failed to evaluate filename");

  tmp = ((String_Buffer*)fname_obj->GetBufferPointer())->GetString(pt);
#ifdef HAVE_STRNCPY
  strncpy(fname,tmp,256);
#else
  strcpy(fname,tmp);
#endif
  fname[255]='\0';
  
  data_buf=(Series_Buffer *)data_obj->GetBufferPointer();
  pt.SetIndex(0,0);pt.SetDim(1);
  data_buf->GetIndexInfo(&dim,index);
  dsiz = sl2_get_data_size();
  /* convert buffer size to data size */
  data = inv_cast((double*)data_buf->GetDataPointer(), dsiz,
                  data_buf->IndexSize());
  if(has_index){
    status = StoreData(fname, numb, dim, index, (char*)data);
  }else{
    status = _WriteFile(fname, dim, index, (char*)data);
  }
  if(status == -1){
    if(data) del_inv_cast(data,dsiz);
    console->execerror("data file write error :", tmp);
  }
  del_inv_cast(data,dsiz);
  acc_del(data_obj);
  push_obj(data_obj);
}

// ifcode(pc,pc,pc) st[0]
void StackMachine::ifcode(){
  bool then_flag,else_flag;
  SL_Object *obj;
  // then part
  Program *thenpart=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  // else part
  Program *elsepart=vmstat.prog->Get_PC();
  if(elsepart->GetProgram()==STOP){ else_flag=false; }
  else { elsepart=elsepart->GetProgPtr(); else_flag=true;}
  vmstat.prog->Inc_PC();
  // next part
  Program *nextstmt=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program_Run(vmstat.prog->Get_PC());
  obj = pop_obj(); acc_add(obj);
  then_flag=(obj->IsTrue() != 0);
  if(then_flag){
    Program_Run(thenpart);
  }else if(else_flag){
    Program_Run(elsepart);
  }
  if(!returning){
    vmstat.prog->Set_PC(*nextstmt+1);
  }
}

// forcode(pc,pc,pc,pc) st[0]
void StackMachine::forcode(){
  int flag;
  SL_Object *obj;
  Program *condition=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *Increment=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *bodypart=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *nextstmt=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  // initialize
  Program_Run(vmstat.prog->Get_PC());
  // condition
  Program_Run(condition);

  obj = pop_obj(); acc_add(obj);
  flag=obj->IsTrue();
  while (flag){
    Program_Run(bodypart);
    continuing=false; // reset continuing flag
    if(returning || breaking)
      break;
    // increment
    Program_Run(Increment);
    // condition
    Program_Run(condition);

    obj = pop_obj(); acc_add(obj);
    flag=obj->IsTrue();
  }
  if(!returning)
    vmstat.prog->Set_PC(*nextstmt+1); // next statement
  breaking = false; // reset breaking flag
}

// forcode(pc,pc,pc) st[0]
void StackMachine::whilecode(){
  int flag;
  SL_Object *obj;
  Program *bodypart=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *nextstmt=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *condition=vmstat.prog->Get_PC();
  // condition
  Program_Run(condition);
  obj = pop_obj(); acc_add(obj);
  flag = obj->IsTrue();
  while (flag){
    Program_Run(bodypart);
    continuing = false;        // reset continuing flag
    if(returning || breaking)
      break;
    // condition
    Program_Run(condition);
    obj = pop_obj(); acc_add(obj);
    flag = obj->IsTrue();
  }
  if(!returning){
    vmstat.prog->Set_PC(*nextstmt+1); // next statement
    breaking = false;          // reset breaking flag
  }
}

// forcode(pc,pc,pc) st[0]
void StackMachine::dowhilecode(){
  int flag;
  SL_Object *obj;
  Program *condition=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *nextstmt=vmstat.prog->Get_PC()->GetProgPtr(); vmstat.prog->Inc_PC();
  Program *bodypart=vmstat.prog->Get_PC();
  do{
    // bodypart
    Program_Run(bodypart);
    continuing = false;    // reset continuing flag
    if(returning || breaking) break;
    // condition
    Program_Run(condition);
    obj = pop_obj(); acc_add(obj);
    flag=obj->IsTrue();
  }while(flag);
  if(!returning)
    vmstat.prog->Set_PC(*nextstmt+1); // next statement
  breaking = false;        // reset breaking flag
}

// call(sym,int)
void StackMachine::call(){
  Program_List *save=0;
  Program *savepc=0;
  SubProgram *sub_old=0;
  int i,narg;
  symbol_t       **sp=0;
  const char      *proc_name;
  symbol_t        *proc_sym, *tmp_sym;
  SL_Object       *proc_obj, *tmp;
  Builtin_Buffer  *proc_buf;

  proc_sym  = vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  proc_name = proc_sym->name;
  proc_obj  = symbol_get_object(proc_sym);
  proc_buf  = (Builtin_Buffer *) proc_obj->GetBufferPointer();

  if(proc_obj->TypeofOBJ() != SL_OBJ::BUILTIN_O){
    err_msg="incomplete definition: function/procedure ";
    err_msg+=proc_name;
    console->execerror(0,err_msg.c_str());
  }

  // save sub program counter
  sub_old=vmstat.sub;
  vmstat.sub=proc_buf->GetSubProgram();

  // set arguments
  narg = (int)vmstat.prog->Get_PC()->GetData();
  if(narg!=vmstat.sub->GetNarg()){
    err_msg="argument mismatch ";
    err_msg+=proc_name;
    console->execerror(0,err_msg.c_str());
  }

  // push symbol objects for recursive call
  vmstat.sub->push_syms();

  sp=vmstat.sub->GetArgSymbols();
  for(i=narg-1; i>=0; i--){
    /* call by value */
    // tmp = pop_obj(); acc_add(tmp);
    // symbol_set_object(sp[i], tmp, SYMBOL_TYPE_VAR);
    /* call by reference */
    tmp_sym = pop_sym();
    if(tmp_sym != NULL){
      symbol_set_depend(sp[i], tmp_sym);
    }else{
      // sp[i] must be undefined symbol
      if(symbol_get_type(sp[i]) != SYMBOL_TYPE_UNDEF)
        abort();
      tmp = pop_obj(); acc_add(tmp);
      if(tmp){
        symbol_set_object(sp[i], tmp, SYMBOL_TYPE_VAR);
      }
    }
  }

  // save program list
  save=vmstat.prog;
  // save program counter
  savepc=vmstat.prog->Get_PC();

  // switch program list
  vmstat.prog=vmstat.sub->GetProgList();
  // increment recursive call counter
  call_stack_depth++;
  // run sub program
  try{
    if(call_stack_depth>MAX_CALL_DEPTH){
      err_msg=proc_name;
      err_msg+=" call nested too deeply";
      console->execerror(0,err_msg.c_str());
    }
    Program_Run(vmstat.prog->Get_Top()); // start of code
  }catch(execerr_exception){
    ret();
    vmstat.prog=save;
    vmstat.prog->Set_PC(savepc);
    vmstat.sub->pop_syms();
    vmstat.sub=sub_old;
    returning = false;
    call_stack_depth--;
    throw;
  }catch(bad_alloc){
    ret();
    vmstat.prog=save;
    vmstat.prog->Set_PC(savepc);
    vmstat.sub->pop_syms();
    vmstat.sub=sub_old;
    returning = false;
    call_stack_depth--;
    throw;
  }
  // restore program list
  vmstat.prog=save;
  // restore program counter
  vmstat.prog->Set_PC(savepc);
  // pop stacked symbol objects
  vmstat.sub->pop_syms();
  // restore sub program 
  vmstat.sub=sub_old;
  // decrement recursive call counter
  call_stack_depth--;
  vmstat.prog->Inc_PC();
  // reset returning flag
  returning = false;
}

void StackMachine::procret(){
  symbol_t *sym;
  sym = vmstat.sub->my_symbol();
  if(symbol_get_type(sym)==SYMBOL_TYPE_FUNC){
    ret();
    err_msg=sym->name;
    err_msg+="(func) returns no value";
    console->execerror(0,err_msg.c_str());
  }
  ret();
}

void StackMachine::funcret(){
  symbol_t *sym;
  sym = vmstat.sub->my_symbol();
  if(symbol_get_type(sym)==SYMBOL_TYPE_PROC){
    ret();
    err_msg=sym->name;
    err_msg+="(proc) returns value";
    console->execerror(0,err_msg.c_str());
  }
  ret();
}

// external(sym) 
void StackMachine::external(){
  symbol_t  *gl_sym, *cur_sym;
  cur_sym = vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  if(symbol_get_object(cur_sym) == NULL){
    gl_sym = symbol_table_lookup(syscom->gl_symtab, cur_sym->name);
    if(gl_sym){
      symbol_set_depend(cur_sym, gl_sym);
    }else{
      /*  regist as global symbol */
      gl_sym = symbol_new(cur_sym->name, 0, SYMBOL_TYPE_UNDEF);
      symbol_table_install(syscom->gl_symtab, gl_sym);
      symbol_set_depend(cur_sym, gl_sym);
    }
  }else{
    err_msg="already used variable ";
    err_msg+=cur_sym->name;
    console->execerror(0, err_msg.c_str());
  }
}

void StackMachine::begin_inline(){
  Inline *inl;
  const char *fname;
  SL_Object *obj;
  obj = pop_obj(); acc_add(obj);
  if(obj->TypeofOBJ()!=SL_OBJ::STRING_O){
    err_msg="illegal object type ";
    console->execerror(0,err_msg.c_str());
  }
  fname=((string *)obj->GetBufferPointer()->GetDataPointer())[0].c_str();
  if(console->input_file_push(fname)==0){
    err_msg="inline() can't open file ";
    err_msg+=fname;
    console->execerror(0,err_msg.c_str());
  }
  inl=new Inline(STOP);
  inl->SetPrev(vmstat.prog);
  vmstat.prog = inl->GetProg();
  lex.inline_stack.push(inl);
}

void StackMachine::end_inline(){
  Inline *inl;
  inl=lex.inline_stack.top();
  vmstat.prog = inl->GetPrev();
  lex.inline_stack.pop();
  console->input_file_pop();
  delete inl;
}

// private Method 
SL_Object *StackMachine::increment(symbol_t *sym){
  static symbol_t *one = 0;
  static SL_Object *one_obj = 0;
  SL_Object *ret, *obj;
  int op_num = vmstat.prog->Get_PC()->GetData();
  vmstat.prog->Inc_PC();
  if(one_obj == 0){
    one = symbol_table_lookup(syscom->gl_symtab,"+1");
    if(one == 0) console->execerror("fatal", "+1 symbol not found");
    one_obj = symbol_get_object(one);
    if(one_obj == 0) console->execerror("fatal", "+1 object not found");
  }
  if(symbol_get_type(sym) != SYMBOL_TYPE_VAR){
    err_msg="incincrement/decrement to non-variable ";
    err_msg=sym->name;
    console->execerror(0,err_msg.c_str());
  }
  obj = symbol_get_object(sym);
  if(obj == 0){
    err_msg="illegal object (null) ";
    err_msg=sym->name;
    console->execerror(0,err_msg.c_str());
  }
  if(obj->TypeofOBJ()==SL_OBJ::STRING_O){
    err_msg="illegal object type ";
    err_msg=sym->name;
    console->execerror(0,err_msg.c_str());
  }
  ret = obj->Opecode(op_num, one_obj);
  return ret;
}

void StackMachine::ret(){
  returning=true;
}

// pipe line functions
void StackMachine::make_pipe(){
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::make_pipe()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  const char *bin;
  SL_Object  *d;
  d = pop_obj(); acc_add(d);
  if(d == 0)
    console->execerror(0, "null object in the pipe");
  if(d->TypeofOBJ()!=SL_OBJ::STRING_O)
    console->execerror(0,"illegal object type");
  bin = ((String_Buffer*)d->GetBufferPointer())->GetString();
  if(strlen(bin) >= ONELINE){
    char errcom[ONELINE + 30];
    strcpy(errcom, "(");
    strncat(errcom, bin, ONELINE);
    strcat(errcom, " ... )");
    err_msg=errcom;
    err_msg+=": command too long";
    console->execerror(0,err_msg.c_str());
  }
  if(!pipe_stream.makepipe(bin))
    console->execerror(0,"failed to making pipe");
  acc_del(d);
  push_obj(d);
}

void StackMachine::close_pipe(){
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::close_pipe()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  string str;
  SL_Object *obj=0;
  pipe_stream.closepipe(str);
  if(!str.empty()){
    Base_Buffer *sb;
    sb = new String_Buffer(str);
    obj = new_SL_Object(SL_OBJ::STRING_O,sb);
  }
  POP();
  push_obj(obj);
}

// external_cmd(obj): st[0]
void StackMachine::external_cmd(){
  SL_Object *obj=vmstat.prog->Get_PC()->GetObj(); vmstat.prog->Inc_PC();
  const char *cmd = ((String_Buffer*)obj->GetBufferPointer())->GetString();
  pipe_stream.external_cmd(cmd,false);
}

// internal_cmd(obj): st[0]
void StackMachine::internal_cmd(){
SL_Object *obj=vmstat.prog->Get_PC()->GetObj(); vmstat.prog->Inc_PC();
  const char *cmd = ((String_Buffer*)obj->GetBufferPointer())->GetString();
  if(pipe_stream.internal_cmd(cmd) == false){
    err_msg="unexpected error : ";
    err_msg+=cmd;
    console->execerror(0,err_msg.c_str());
  }
}

void StackMachine::system(){
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::system()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  const char *cmd;
  string str;
  SL_Object *d;
  d = pop_obj(); acc_add(d);
  if(d==0 || d->GetBufferPointer()==0)
    console->execerror(0,"null object operation");
  if(d->TypeofOBJ() != SL_OBJ::STRING_O)
    console->execerror(0,"illegal string");
  cmd = ((String_Buffer*) d->GetBufferPointer())->GetString();
  str=pipe_stream.external_cmd(cmd,true);
  list<string> split;
  {
    string::size_type a,b,len;
    b=0;
    len=str.length();
    for(a=0;a<len;a++){
      switch(str[a]){
      case '\t': case '\n': case ' ' : case '\r' :
        if(a==b){b=a+1; continue;}
        split.push_back(str.substr(b,a-b));
        b=a+1;
      }
    }
    if(a!=b)split.push_back(str.substr(b,a-b));
  }
  Base_Buffer *buf;
  try{ buf=new String_Buffer(split); }catch(std::exception){ buf=0; }
  SL_Object *oacc;
  oacc = new_SL_Object(SL_OBJ::STRING_O,buf);
  push_obj(oacc);
}

typedef struct _satcom_install_symtab_for_arg_t {
  symbol_table_t *symtab;
  tty_console    *console;
} satcom_install_symtab_for_arg_t;

static void satcom_install_symtab_for(void *void_cmd, void *void_arg){
  module_command_t *cmd;
  satcom_install_symtab_for_arg_t *arg;
  SL_Object      *obj;
  Builtin_Buffer *buf;
  symbol_t       *sym;
  cmd = (module_command_t*) void_cmd;
  arg = (satcom_install_symtab_for_arg_t*) void_arg;
  if(cmd->function != NULL){
    /* it is SATCOM, ignore FUNC/PROC */
    buf = new Builtin_Buffer;
    buf->SetSatCom(cmd);
    obj = new_SL_Object(SL_OBJ::SATCOM_O, buf);
    sym = symbol_new(cmd->command_name, obj, SYMBOL_TYPE_SATCOM);
    symbol_table_install(arg->symtab, sym);
  }
  arg->console->satcom_db_add(cmd);
}

// module_install(sym)
void StackMachine::module_install(){
  satcom_install_symtab_for_arg_t arg;
  symbol_t *sym;
  module_t *mod;
  const char *scr;

  sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();

  mod = symbol_to_module(sym);
  if(mod == NULL){
    console->execerror("can't load module",sym->name);
  }

  // load module
  if(module_load(mod)!=0){
    //#ifdef WIN32
    //    if(strcmp(sym->name, "SLIO") == NULL)
    //      return;
    //    else
    //#endif
    console->execerror("can't load module",mod->err_mesg);
  }
  console->tty_printf("  * SYSTEM MODULE ");
  console->tty_set_attr(SL_TTY::ATTR_REVERSE);
  console->tty_printf(" * %-10s* ",sym->name);
  console->tty_set_attr(SL_TTY::ATTR_NORMAL);
  console->tty_printf("   : %3d External Functions ... Install Ok.\n",
                      mod->commands->nkeys);

  // register initial shell script
  scr = module_getscript(mod);
  if(scr != NULL){
    // convert string to object
    SL_Object   *obj;
    Base_Buffer *buf;
    Index pt;
    buf=new String_Buffer;
    buf->SetIndex(0,1); buf->SetDim(1); buf->InitBuffer();
    pt.SetIndex(0,0);   pt.SetDim(1);
    ((String_Buffer *)buf)->SetString(pt, scr);
    obj=new_SL_Object(SL_OBJ::STRING_O, buf);
    // push object
    push_obj(obj);
    // do inline
    begin_inline();
  }

  // install satellite commands to global symbol list
  arg.console = console;
  arg.symtab  = syscom->gl_symtab;
  hash_table_foreach(mod->commands, satcom_install_symtab_for, &arg);

}

// module_define_begin(sym)
void StackMachine::module_define_begin(){
  symbol_t *sym;
  sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  if(sym!=0){ // part of "define HOGE {"
    if(vmstat.module != NULL) module_delete(vmstat.module);
    vmstat.module = module_new(sym->name);
    if(vmstat.module == NULL) throw bad_alloc();
  }
}

// module_define_end(sym)
void StackMachine::module_define_end(){
  symbol_t       *sym;
  SL_Object      *obj;
  sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  obj = symbol_get_object(sym);
  if(obj){
    module_t *mod;
    mod = symbol_to_module(sym);
    if(mod->is_loaded == 0){
      module_delete(mod);
    }
    else{
      err_msg="module ";
      err_msg+=mod->module_name;
      err_msg+=", already loaded";
      console->execerror(0,err_msg.c_str());
    }
  }
  //  module_list.push_back(vmstat.module);
  if(obj == NULL){
    Builtin_Buffer *buf;
    buf = new Builtin_Buffer();
    buf->SetModule(vmstat.module);
    obj = new_SL_Object(SL_OBJ::MODULE_NAME_O, buf);
    /* check object refcount */
    SL_Object *tobj = symbol_get_object(sym);
    if(tobj){
      tobj->obj_unref();
      if(tobj->empty())
        acc_del(tobj);
      tobj->obj_ref();
    }
    symbol_set_object(sym, obj, SYMBOL_TYPE_MODULE);
  }
  vmstat.module = NULL;
  /* move to global symbol table */
  if(symbol_table_lookup(syscom->gl_symtab, sym->name) == NULL)
    symbol_table_move(syscom->gl_symtab, syscom->cur_symtab, sym);
}

// module_setfile(int,obj)
void StackMachine::module_setfile(){
  String_Buffer *sbuf;
  SL_Object *obj;
  int  keywd;
  sbuf = 0;
  keywd=vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();
  obj = pop_obj(); acc_add(obj);
  if(obj!=0 && // "module_setpart" in parse.y
     obj->TypeofOBJ()==SL_OBJ::STRING_O &&
     obj->GetBufferPointer()!=0){
    const char *name;
    sbuf=(String_Buffer*)obj->GetBufferPointer();
    name=((string*)sbuf->GetDataPointer())[0].c_str();
    switch(keywd){
    case VMCODE_MODULE_DEFINE_DLL: // part of 'set Module_Dll "hoge.so"'
      if(Access(name,SL_FATTR_ROK) != 0){
        err_msg="module ";
        err_msg+=vmstat.module->module_name;
        err_msg+=", can not read file ";
        err_msg+=name;
        console->execerror(0,err_msg.c_str());
      }
      if(module_set_dll(vmstat.module,name) != 0){ 
        throw bad_alloc(); 
      }
      break;
    case VMCODE_MODULE_DEFINE_INI: // part of 'set Module_Ini "hoge.ini"'
      if(Access(name,SL_FATTR_ROK) != 0){
        err_msg="module ";
        err_msg+=vmstat.module->module_name;
        err_msg+=", can not read file ";
        err_msg+=name;
        console->execerror(0,err_msg.c_str());
      }
      if(module_set_ini(vmstat.module,name) != 0){ 
        throw bad_alloc(); 
      }
      break;
    case VMCODE_MODULE_DEFINE_SCR: // part of 'set Module_Scr "hoge.sl"'
      if(Access(name,SL_FATTR_ROK) != 0){
        err_msg="module ";
        err_msg+=vmstat.module->module_name;
        err_msg+=", can not read file ";
        err_msg+=name;
        console->execerror(0,err_msg.c_str());
      }
      if(module_set_scr(vmstat.module,name) != 0){ 
        throw bad_alloc(); 
      }
      break;
    }
  }
}

// module_exec(obj,int n) st[-n+1]: (obj|sym)(-1)*n,obj(+1)
void StackMachine::module_exec(){
  int i=0,len=0,narg=0,argc=0,err=0;
  module_command_t *cinfo;
  char *mod=0,*cmd=0,*work=0,*arg=0;
  SL_Object **argm=0,*ret=0, *tmp_obj;
  symbol_t  **sym_args=0,*tmp_sym, *com_sym;
  String_Buffer *buf=0;
  com_sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  narg=(int)vmstat.prog->Get_PC()->GetData(); vmstat.prog->Inc_PC();

  cinfo = symbol_to_satcom(com_sym);
  argc = Max(narg, cinfo->argc);
  if(argc!=0){
    argm = new SL_Object*[argc];
    sym_args = new symbol_t*[argc];
  }
  for(i=0; i<argc; i++){
    argm[i] = 0;
    sym_args[i] = 0;
  }
  for(i=0; i<narg; i++){
    // narg = 1  argc = 2  argm[0] pop0
    // narg = 2  argc = 2  argm[0] pop1 argm[1] pop0
    tmp_sym = pop_sym();
    sym_args[narg-i-1] = tmp_sym;
    if(tmp_sym){
      tmp_obj = symbol_get_single_object(tmp_sym);
      argm[narg-i-1] = tmp_obj;
    }else{
      argm[narg-i-1] = pop_obj(); acc_add(argm[narg-i-1]);
    }
  }
  // checking compat2 mode...
  for(i=0; i<narg; i++){
    tmp_sym = sym_args[i];
    if(tmp_sym){
      if(symbol_get_type(tmp_sym) == SYMBOL_TYPE_UNDEF){
        if(is_compat2x_mode() == 1){
          argm[i] = symbol_to_quoted_object(tmp_sym);
          sym_args[i] = 0;
          acc_add(argm[i]);
        }else{
          delete [] argm;
          delete [] sym_args;
          err_msg="undefined variable "; err_msg+=tmp_sym->name;
          console->execerror(0, err_msg.c_str());
        }
      }
    }
  }

  syscom->ComCallInit(narg,argm,sym_args);
  // execute
  try{
    err = cinfo->function();
    /* do symbol update event */
    for(i=0; i<narg; i++)
      if(sym_args[i])
        symbol_do_event(sym_args[i], SYMBOL_EVENT_UPDATE);
  }catch(command_exception e){
    err = e.errnum();
  }catch(bad_alloc){
    if(argc!=0){
      delete [] argm;
      delete [] sym_args;
    }
    delete [] work;
    throw;
  }catch(buffer_exception){
    if(argc!=0){
      delete [] argm;
      delete [] sym_args;
    }
    delete [] work;
    throw;
  }catch(execerr_exception){
    if(argc!=0){
      delete [] argm;
      delete [] sym_args;
    }
    delete [] work;
    throw;
  }
  // error check
  if(err!=0){
    if(argc!=0){ delete [] argm; delete [] sym_args; }
    delete [] work;
    /* format error message */
    char err1[1024];
    const char *err2;
    module_message_t *mes;
    mes = (module_message_t*)hash_table_lookup(cinfo->module->errors,&err);
    if(mes == NULL) err2 = "unknown error";
    else err2 = mes->message;
    snprintf(err1,1024,"Error [<%s:%s> No.%d]",cinfo->module->module_name, 
             cinfo->command_name, err);
    console->execerror(err1,err2);
  }
  ret=syscom->ComCallQuit();
  if(argc!=0){ delete [] argm; delete [] sym_args; }
  delete [] work;
  push_obj(ret);
}

// module_print(sym) st[0]:
void StackMachine::module_print(){
  symbol_t *sym;
  const char *str;
  module_t *mod;
  sym=vmstat.prog->Get_PC()->GetSymbol(); vmstat.prog->Inc_PC();
  if(sym==0 && sym->name==0)
    console->execerror(0,"null symbol");
  str=sym->name;
  mod = symbol_to_module(sym);
  if(mod==NULL)
    console->execerror(0,"module not found");

  console->tty_printf("define %s {\n",mod->module_name);
  if(mod->dll_file!=0 && mod->dll_file[0]!='\0')
    console->tty_printf("\t set Module_Dll \"%s\"\n",mod->dll_file);
  if(mod->ini_file!=0 && mod->ini_file[0]!='\0')
    console->tty_printf("\t set Module_Ini \"%s\"\n",mod->ini_file);
  if(mod->scr_file!=0 && mod->scr_file[0]!='\0')
    console->tty_printf("\t set Module_Scr \"%s\"\n",mod->scr_file);
  console->tty_printf("}\n");
}

void StackMachine::yyerror(char *s){  // called for yacc syntax error
#if defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
  static char *where="StackMachine::yyerror()";
#endif // defined(_DEBUG) && (defined(_DEBUG_CODE) || defined(_DEBUG_ERRPOS))
#if defined(_DEBUG) && defined(_DEBUG_CODE)
  console->tty_printf("%s\n",where);
#endif // defined(_DEBUG) && defined(_DEBUG_CODE)
  try{
    console->execerror(s,0);
  }catch(execerr_exception){}
  recovery();
  Program_Init();
}
