/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: pcond.cpp,v 1.2 2004/04/07 14:17:39 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "SL_header.h"

#include <libsatellite.h>

#define  __IMPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#include "SL_Index.h"
#include "Base_Buffer.h"
#include "Series_Buffer.h"
#include "Snapshot_Buffer.h"
#include "String_Buffer.h"
#include "Scalar_Buffer.h"
#include "SL_Tool.h"
#include "SL_Object.h"
#include "SymbolList.h"
#include "SystemCommon.h"
#undef   __IMPORTSYMBOL__
#include "Program.h"
#include "Datum.h"
#include "Inline.h"
#include "CommandAlias.h"
#include "LexicalAnalizer.h"
#include "pipe.h"
#include "Builtin.h"
#include "Builtin_Buffer.h"
#include "vmstat.h"
#include "StackMachine.h"
#include "pcond.h"

#include <stdio.h>
#include <stdlib.h>

using namespace std;

typedef struct _parser_condition_t {
  char            *errmes;
  SubProgram      *sub;
  symbol_table_t  *pre_symtab;
  vm_stat_t       *vmstat;
  Program_List    *pre_prog;
} parser_condition_t;

static parser_condition_t *pcond = NULL;

static int _emesgfmt(const char *s1, const char *s2, const char *s3);

int parser_condition_recovery(){
  if(pcond->errmes) free(pcond->errmes);
  if(pcond->sub) delete pcond->sub;
  if(pcond->pre_symtab) syscom->cur_symtab = pcond->pre_symtab;
  if(pcond->pre_prog)   pcond->vmstat->prog = pcond->pre_prog;
  pcond->errmes     = NULL;
  pcond->sub        = NULL;
  pcond->pre_symtab = NULL;
  pcond->pre_prog   = NULL;
  return 0;
}

int parser_condition_init(vm_stat_t *vmstat){
  pcond = (parser_condition_t*)malloc(sizeof(parser_condition_t));
  if(pcond == NULL) return -1;
  memset(pcond,0,sizeof(parser_condition_t));
  pcond->vmstat = vmstat;
  parser_condition_recovery();
  return 0;
}

void parser_condition_free(){
  parser_condition_recovery();
  free(pcond);
  pcond = NULL;
}

int parser_condition_sub_indef(){
  if(pcond->sub == NULL) return 0;
  return 1;
}

void parser_condition_sub_begin(symbol_t *sym, int type){
  if(pcond->sub != NULL){
    const char *mes;
    parser_condition_recovery();
    if(type == SYMBOL_TYPE_FUNC) mes="define func ";
    if(type == SYMBOL_TYPE_PROC) mes="define proc ";
    _emesgfmt(mes, sym->name, " in other func/procs statement");
    syscom->console->execerror(0,pcond->errmes);
  }
  if(symbol_get_type(sym) != SYMBOL_TYPE_FUNC && 
     symbol_get_type(sym) != SYMBOL_TYPE_PROC &&
     symbol_get_type(sym) != SYMBOL_TYPE_UNDEF){
    _emesgfmt("specified name is already defined as object ",sym->name,0);
    syscom->console->execerror(0,pcond->errmes);
  }
  sigintblock(true); // ignore ctrl-C
  pcond->sub = new SubProgram(sym);
  /* swap symbol table */
  pcond->pre_symtab  = syscom->cur_symtab;
  syscom->cur_symtab = pcond->sub->lo_symtab;
  /* swap Program List */
  pcond->pre_prog     = pcond->vmstat->prog;
  pcond->vmstat->prog = pcond->sub->GetProgList();
  switch(symbol_get_type(sym)){
  case SYMBOL_TYPE_UNDEF:
  case SYMBOL_TYPE_FUNC:
  case SYMBOL_TYPE_PROC:
    symbol_set_object(sym, NULL, type);
    break;
  default:
    parser_condition_recovery();
    _emesgfmt("specified name is already defined as object ",sym->name,0);
    syscom->console->execerror(0,pcond->errmes);
  }
}

int parser_condition_sub_pickup_args(symbol_t *sym){
  return pcond->sub->pickup_args(sym);
}

void parser_condition_sub_define(){
  symbol_t *sym;
  Builtin_Buffer *buf=0;
  SL_Object *obj=0;
  /* regist SubProgram to symbol */
  buf=new Builtin_Buffer;
  buf->SetSubProgram(pcond->sub);
  try{
    obj=new_SL_Object(SL_OBJ::BUILTIN_O, buf);
  }catch(bad_alloc){
    delete buf;
    parser_condition_recovery();
    throw;
  }
  sym = pcond->sub->my_symbol();
  symbol_set_object(sym, obj, symbol_get_type(sym));
  /* regist global symbol table */
  if(symbol_table_lookup(pcond->pre_symtab, sym->name) == sym)
    symbol_table_move(syscom->gl_symtab, pcond->pre_symtab, sym);
  parser_condition_sub_end();
}

void parser_condition_sub_end(){
  pcond->vmstat->prog = pcond->pre_prog;
  syscom->cur_symtab  = pcond->pre_symtab;
  pcond->pre_prog     = NULL;
  pcond->pre_symtab   = NULL;
  pcond->sub          = NULL;
  if(pcond->errmes) free(pcond->errmes);
  pcond->errmes      = NULL;
  sigintblock(false); // Ctrl-C interupt OK
}

void parser_condition_module_print_setinfo(symbol_t *sym){
  char mod_dir[512], rc_dir[512];
  if(syscom->console->is_file()) return;
  if(GetModuleDirectory(mod_dir,512) == NULL)
    mod_dir[0] = '\0';
  if(GetSystemResourceDirectory(rc_dir,512) == NULL)
    rc_dir[0] = '\0';
  syscom->console->tty_printf("\n");
  syscom->console->tty_printf("\t+--------------------+ Information ");
  syscom->console->tty_printf(" +----------------------+\n");
  syscom->console->tty_printf("\t + set Module_Dll \"%s/%s.%s\"\n", 
                              mod_dir, sym->name,(IsWindows())?"dll":"so");
  syscom->console->tty_printf("\t + set Module_Ini \"%s/%s.ini\"\n", 
                              rc_dir, sym->name);
  syscom->console->tty_printf("\t+------------------------------------");
  syscom->console->tty_printf("---------------------+\n");
}

static int _emesgfmt(const char *s1, const char *s2, const char *s3){
  size_t s1len,s2len,s3len;
  if(s1==NULL) s1len = 0; else s1len=strlen(s1);
  if(s2==NULL) s2len = 0; else s2len=strlen(s2);
  if(s3==NULL) s3len = 0; else s3len=strlen(s3);
  if(s1len+s2len+s3len == 0) return -1;
  if(pcond->errmes){
    char *tmp;
    tmp = (char*)realloc(pcond->errmes,s1len+s2len+s3len+1);
    if(tmp == NULL) return -1;
    pcond->errmes = tmp;
  }else{
    pcond->errmes = (char*)malloc(s1len+s2len+s3len+1);
    if(pcond->errmes == NULL) return -1;
  }
  pcond->errmes[0]='\0';
  if(s1len != 0){ strcat(pcond->errmes, s1); }
  if(s2len != 0){ strcat(pcond->errmes, s2); }
  if(s3len != 0){ strcat(pcond->errmes, s3); }
  return 0;
}

