/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: pipe.cpp,v 1.2 2005/01/24 15:58:57 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif

#include "SL_header.h"

#include <libsatellite.h>

#define  __IMPORTSYMBOL__
#include "SL_exception.h"
#include "history.h"
#include "module.h"
#include "tty_console.h"
#undef   __IMPORTSYMBOL__
#define  __EXPORTSYMBOL___
#include "pipe.h"
#undef  __EXPORTSYMBOL___


#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_DIRECT_H
# include <direct.h>
#endif

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH  MAXPATHLEN
# else
#  define _MAX_PATH  1024
# endif
#endif

#define OUT_BUFF_SIZE 1024

using namespace std;

sl4_fd_t sl4_io_invalid(){
  sl4_fd_t fd;
#ifdef HAVE_WINDOWS_H
  fd.handle = INVALID_HANDLE_VALUE;
#else
  fd.fd = -1;
#endif
  return fd;
}


/* IO */
#ifdef HAVE_WINDOWS_H
static HANDLE hStdin;
static HANDLE hStdout;
static HANDLE hStderr;
#endif

#ifdef HAVE_WINDOWS_H
void sl4_io_init(HANDLE i, HANDLE o, HANDLE e){
  hStdin  = i;
  hStdout = o;
  hStderr = e;
}
#endif

int sl4_io_pipe(sl4_fd_t *fds){
#ifdef HAVE_WINDOWS_H
  if(CreatePipe(&fds[0].handle, &fds[1].handle, NULL, 0) == FALSE)
    return -1;
#else
  int pfds[2];
  if(pipe(pfds) == -1) return -1;
  fds[0].fd = pfds[0];
  fds[1].fd = pfds[1];
#endif
  return 0;
}

int sl4_io_close(sl4_fd_t fd){
#ifdef HAVE_WINDOWS_H
  if(CloseHandle(fd.handle) == TRUE){
    if(fd.handle == hStdin)  hStdin  = INVALID_HANDLE_VALUE;
    if(fd.handle == hStdout) hStdout = INVALID_HANDLE_VALUE;
    if(fd.handle == hStderr) hStderr = INVALID_HANDLE_VALUE;
    return 0;
  }
  return -1;
#else
  return close(fd.fd);
#endif
}

sl4_fd_t sl4_io_dup(sl4_fd_t fd){
  sl4_fd_t dfd;
#ifdef HAVE_WINDOWS_H
  HANDLE hProc;
  hProc = GetCurrentProcess();
  if(DuplicateHandle(hProc, fd.handle, hProc, &dfd.handle,
                     0, TRUE, DUPLICATE_SAME_ACCESS) == FALSE){
    dfd.handle = INVALID_HANDLE_VALUE;
  }else{
    if(hStdin = INVALID_HANDLE_VALUE){
      SetStdHandle( STD_INPUT_HANDLE, dfd.handle);
      hStdin  = dfd.handle;
    }else if(hStdout = INVALID_HANDLE_VALUE){
      SetStdHandle( STD_OUTPUT_HANDLE, dfd.handle );
      hStdout = dfd.handle;
    }else if(hStderr = INVALID_HANDLE_VALUE){
      SetStdHandle( STD_ERROR_HANDLE, dfd.handle );
      hStderr = dfd.handle;
    }
  }
#else
  dfd.fd = dup(fd.fd);
#endif
  return dfd;
}

int sl4_io_read(sl4_fd_t fd, char *buf, int size){
#ifdef HAVE_WINDOWS_H
  DWORD nread;
  if(ReadFile(fd.handle,buf,size,&nread,NULL) == FALSE) return -1;
  return (int)nread;
#else
  return read(fd.fd,buf,size);
#endif
}

PipeStream::PipeStream(tty_console *con):
  in_pipe(false),open_pipe(false),console(con), pid(-1){
#ifdef HAVE_WINDOWS_H
  sl4_io_init(console->ifd.handle, console->ofd.handle, console->efd.handle);
#endif
  ttyout = sl4_io_invalid();
  ttyerr = sl4_io_invalid();
}

PipeStream::~PipeStream(){
  if(console->pipe_status()){string a; closepipe(a);}
}

// makepipe  ex. cat << ls
// stdin -> parent -> child -> stdout
// parent : stdin = stdin       ,  stdout = "pipe write"
// child  : stdin = "pipe read" ,  stdout = stdout
bool PipeStream::makepipe(const char *bin){
  sl4_fd_t ifd,ofd,efd;
  sl4_fd_t pofd[2];
  if(bin==0 || *bin=='\0') return false;
  if(sl4_io_pipe(pofd)== -1){
    return false;
  }
  if(!console->pipe_status()){
    if(sl4_io_pipe(out)==-1){
      sl4_io_close(pofd[0]);
      sl4_io_close(pofd[1]);
      return false; 
    }
    ofd = console->ofd;
    ttyout = sl4_io_dup(ofd); sl4_io_close(ofd); sl4_io_dup(out[1]);
    // ttyerr = dup(2); close(2); dup(out[1]);
  }
  // create child process
  ifd = pofd[0];
  ofd = console->ofd;
  efd = console->efd;
  pid = create_child_process(bin, ifd, ofd, efd, NULL);
  sl4_io_close(pofd[0]); /* close pipe read fd on parent process */
  sl4_io_close(out[1]);  /* close pipe write fd on parent process */
  if(pid == -1){
    sl4_io_close(pofd[1]); sl4_io_close(out[0]);
    ofd = console->ofd;
    sl4_io_close(ofd);  sl4_io_dup(ttyout);
    sl4_io_close(ttyout); ttyout = sl4_io_invalid();
    return false; 
  }
  ofd = console->ofd;
  sl4_io_close(ofd); sl4_io_dup(pofd[1]); sl4_io_close(pofd[1]);
  // close(2); dup(pfd[1]);
  console->pipe_status_set(true);
  return true;
}

// makepipe  : child(in -> pfd[0] out->out[1]), parent(out->pfd[1])
// print     : parent(out->pfd[1])
// closepipe : out[0]->console;
void PipeStream::closepipe(string &ret){
  sl4_fd_t tmpfd;
  // close handle of pfd[1]
  tmpfd = console->ofd;
  sl4_io_close(tmpfd);  sl4_io_dup(ttyout);
  sl4_io_close(ttyout); ttyout = sl4_io_invalid();
  // close(2); dup(ttyerr); close(ttyerr); ttyerr=-1;
  if(pid!=-1){
    int stat=0;
    size_t size;
    char buf[OUT_BUFF_SIZE];
    while((size=sl4_io_read(out[0],buf,OUT_BUFF_SIZE-1)),size>0){
      buf[size]='\0';
      ret+=buf;
    }
    wait_child_process(pid,&stat);
  }
  console->pipe_status_set(false);
  sl4_io_close(out[0]);
}

string PipeStream::external_cmd(const char *cmd, bool usepipe){
  sl4_fd_t ifd,ofd,efd;
  string ret;
  int apid(-1);
  //  bool istty;
  if(cmd==0 || *cmd=='\0') return ret;
  // usepipe:true   cin -> co-proc -pipe-> shell
  //        :false  cin -> co-proc -> cout
  if(usepipe)
    console->term_flush(console->ofd);
  //  istty=console->is_tty();
  //if(istty) console->term_start_prog();
  if(usepipe){
    int size;
    char buf[OUT_BUFF_SIZE];
    sl4_fd_t pfd[2];
    if(sl4_io_pipe(pfd) == -1) return ret;
    ifd = console->ifd;
    ofd = pfd[1];
    efd = console->efd;
    apid = create_child_process(cmd, ifd, ofd, efd, NULL);
    sl4_io_close(pfd[1]);
    if(apid==-1){
      console->tty_printf("command not found : %s\n",cmd);
      sl4_io_close(pfd[0]);
      return ret;
    }
    while(size=sl4_io_read(pfd[0],buf,OUT_BUFF_SIZE-1),size>0){
      buf[size]='\0';
      ret+=buf;
    }
    sl4_io_close(pfd[0]);
  }else{
    ifd = console->ifd;
    ofd = console->ofd;
    efd = console->efd;
    apid = create_child_process(cmd, ifd, ofd, efd, NULL);
    if(apid==-1){
      console->tty_printf("command not found : %s\n",cmd);
    }
  }
  if(apid != -1){
    int stat;
    wait_child_process(apid,&stat);
  }
  console->term_flush(console->ofd);
  return ret;
}

void PipeStream::internal_ls(const char *args){
  char *comspec, *command;
  const char *args2;
#ifdef WIN32
  const char *opts="/c dir ";
  args2 = args;
  args = opts;
  comspec = getenv("COMSPEC");
#else
  const char *dummy = "-F";
  if(args == NULL) args = dummy;
  comspec = "/bin/sh -c \"ls";
  args2 = "\"";
#endif
  if(comspec != NULL){
    command = (char*)malloc(sizeof(char)*(strlen(comspec) + strlen(args) + strlen(args2) + 2));
    if(command != NULL){
      strcpy(command, comspec);
      strcat(command, " ");
      strcat(command, args);
      strcat(command, args2);
      external_cmd(command,false);
      free(command);
    }
  }
}

// void PipeStream::internal_cd(const char *args){
//   char home[_MAX_PATH];
//   const char *p;
//   string fix;
//   bool esc(false);
//   if(GetHomeDirectory(home,_MAX_PATH)==NULL){
//     strcpy(home,"/");
//   }
//   if(args == 0 || *args == '\0'){
//     fix = home;
//   }else{
//     for(p=args;*p!=0;p++){
//       if(*p=='"'){ esc = !esc; continue; }
//       if(!esc && *p==' ') break;
//       if(p==args && *p == '~')
//         fix += home;
//       else
//         fix += *p;
//     }
//     if(esc){ console->tty_printf("unmatched: \".\n"); return; }
//     if(fix.empty()){ console->tty_printf("usage: cd directory\n"); return; }
//   }

//   if(chdir(fix.c_str())==-1) // execute chdir command
//     console->tty_printf("%s : No such file or directory.\n",fix.c_str());
//   else{
//     // chdir() command success
//     char buf[512];
//     p=getenv("PWD");
//     if(p!=0){
//       if(getcwd(buf,512)!=0){
// #ifdef HAVE_SETENV
//         setenv("PWD",buf,1);
// #elif defined(HAVE_PUTENV) || defined(WIN32)
//         string tmp("PWD=");
//         tmp+=buf;
//         putenv(tmp.c_str());
// #endif
//       }
//     }
//     console->tty_printf(args);
//     console->tty_printf("\n");
//   }
// }

void PipeStream::internal_cd(const char *args){
  if(sl4_chdir(args) != 0){
    console->tty_printf("%s : No such file or directory.\n", args);
  }
}

bool PipeStream::internal_cmd(const char *bin){
  string cmd,com,args;
  string::size_type i;
  const char *p,*tmp;
  int len;
  // strip white char in 'bin'
  for(p=bin;*p==' ';p++);
  cmd = p;
  for(tmp=p+strlen(p)-1,len=0;*tmp==' ';tmp--,len++){
    if(tmp == p) break;
  }
  if(len != 0) cmd.erase(tmp-p+1,len);

  i=cmd.find(' ',0);
  if(i!=string::npos){
    com  = cmd.substr(0,i);
    args = cmd.substr(i+1, cmd.length()-i);
  }else{
    com = cmd;
  }
  if(!com.empty()){
    // "dir" command
    if(com == "dir"){
      internal_ls(args.c_str());
      return true;
    }
    // "ls" command .. Windows Only
    if(com == "ls"){
      internal_ls(args.c_str());
      return true;
    }
    // "chdir" command
    if(com == "chdir"){
      internal_cd(args.c_str());
      return true;
    }
    // "cd" command
    if(com == "cd"){
      internal_cd(args.c_str());
      return true;
    }
  }
  return false;
}
