/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/filehndl.cpp,v 1.8 2005/02/22 07:39:28 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libbps.h"

/*************************************************
 *                                               *
 *       Back Propergation Simurator(BPS)        *
 *             subroutine package                *
 *               Version                         *
 *         coded            in Aug.25 1989       *
 *         coded by         Y.Okamura            *
 *          last modified in Nov.15 1990         *
 *          modified by   K.Kuroda               *
 *                                               *
 *************************************************
 *                                               *
 *       filename fhndl.c                        *
 *          history file handler                 *
 *                                               *
 ************************************************/

#ifdef __cplusplus
extern "C" {
#endif


/************************************************
  set NumOfLink
  ************************************************/
void SetNumOfLink()
{
  int  laynum;

  bps_cont.NumOfLink = 0;
  for (laynum = 0; laynum < bps_cont.NumOfLayer-1; laynum++) {
    if (CheckBias(laynum+1))
      bps_cont.NumOfLink += 
        (bps_cont.NumOfCell[laynum]+1)*bps_cont.NumOfCell[laynum+1];
    else
      bps_cont.NumOfLink +=
        bps_cont.NumOfCell[laynum]*bps_cont.NumOfCell[laynum+1];
  }
}


/*********************************************************
  get record number.
  input         : filename
  *********************************************************/
int GetNumOfRecord(const char *filename)
{

  Header          head;

  if(LoadHeader(filename, &head) != -1)
    return (head.index[0]);
  else 
    return 0;
}


/***********************************************
  adjust last block of weight history
  inputs:
  filename : weight history file name
  return:
  last history number
  ***********************************************/
int LastWgtHistory(const char *filename)
{
  Header  head;

  if (LoadHeader(filename, &head) != -1)
    return (head.index[0]-1);
  else
    return 0;
}


/***********************************************
  adjust last block of weight history
  inputs:
  filename : weight history file name
  return:
  next history number
  ***********************************************/
int NextWgtHistory(const char *filename)
{
  Header   head;

  if(LoadHeader(filename, &head) != -1)
    return (head.index[0]);
  else 
    return 0;
}


/***********************************************
  adjust last block of weight history
  inputs:
  filename : error history file name
  return:
  next history number
  ***********************************************/
int NextErrHistory(const char *filename)
{
  Header   head;

  if(LoadHeader(filename, &head) != -1)
    return (head.index[0] + 1);
  else
    return 0;
}


/************************************************
  read test data
  ************************************************/
int ReadData(const char *filename, float **data, int first_ptrn, 
              int last_ptrn, int num_cell)
{
  int      ptrn, pt_no, unit;
  float   *tmparea;
  Header   header;

  for (ptrn = first_ptrn; ptrn <= last_ptrn; ptrn++) {
    tmparea = (float*)LoadData(filename, ptrn - 1, &header);
    if (tmparea == (float*)NULL)
			return 13; /* Data File Read Error */
    if (header.index[1] != num_cell)
      return 100; /* Data Point Mismatch */

    pt_no = ptrn - first_ptrn;
    for (unit = 0; unit < num_cell; unit++) {
      data[pt_no][unit] = tmparea[unit];
    }
    FreeData(tmparea);
  }

	return 0;
}


/************************************************
  write weight history
  inputs:
  filename : weight history file name
  hist_no  : weight history number( form 1 )
  data     : weight data
  ************************************************/
void WriteWeight2(const char *filename, int hist_no, double *data)
{
  int     index[5], i;
  float  *buf;

  buf = (float *)emalloc(sizeof(float)*bps_cont.NumOfLink);

  for (i = 0; i < bps_cont.NumOfLink; i++)
    buf[i] = (float) data[i];

  index[0] = bps_cont.NumOfLink;
  StoreData(filename, hist_no , 1, index, (char*)buf);

	efree(buf);
}


/************************************************
  get and write weight history
  inputs:
  filename : weight history file name
  wgt_stor_cnt : count of history
  ************************************************/
int StoreWeight2(const char *filename, int wgt_stor_cnt)
{
  int      laynum, unit_to, unit_from, position_cnt;
  bps_ilin_t  *in_link;
  double  *WgtBuff;

  WgtBuff = (double*)emalloc(sizeof(double)*bps_cont.NumOfLink);

  /* WRITE WEIGHT DATA */
  position_cnt = 0;
  for (laynum = 1; laynum < bps_cont.NumOfLayer; laynum++) {
    for (unit_to = 1; unit_to <= bps_cont.NumOfCell[laynum]; unit_to++) {
      if (CheckBias(laynum)) {
        in_link = inter_ilin(bps_cont.BPNet[laynum][0].CellNode,
                             bps_cont.BPNet[laynum][unit_to].CellNode);
	      if (in_link == NULL)
		      return 120; /* Can't Find Link */
        WgtBuff[position_cnt++] = in_link->Weight;
      }
      for (unit_from = 1;
           unit_from <= bps_cont.NumOfCell[laynum-1]; 
           unit_from++) {
        in_link = inter_ilin(bps_cont.BPNet[laynum-1][unit_from].CellNode,
                             bps_cont.BPNet[laynum][unit_to].CellNode);
	      if (in_link == NULL)
		      return 120; /* Can't Find Link */
        WgtBuff[position_cnt++] = in_link->Weight;
      }
    }
  }
  WriteWeight2(filename, wgt_stor_cnt, WgtBuff);

  /* WRITE DELTA WEIGHT DATA */
  position_cnt = 0;
  for (laynum = 1; laynum < bps_cont.NumOfLayer; laynum++) {
    for (unit_to = 1; unit_to <= bps_cont.NumOfCell[laynum]; unit_to++) {
      if (CheckBias(laynum)) {
        in_link = inter_ilin(bps_cont.BPNet[laynum][0].CellNode,
                             bps_cont.BPNet[laynum][unit_to].CellNode);
	      if (in_link == NULL)
		      return 120; /* Can't Find Link */
        WgtBuff[position_cnt++] = in_link->AdjWgt;

      }
      for (unit_from = 1;
           unit_from <= bps_cont.NumOfCell[laynum-1];
           unit_from++) {
        in_link = inter_ilin(bps_cont.BPNet[laynum-1][unit_from].CellNode,
                             bps_cont.BPNet[laynum][unit_to].CellNode);
	      if (in_link == NULL)
		      return 120; /* Can't Find Link */
        WgtBuff[position_cnt++] = in_link->AdjWgt;
      }
    }
  }
  WriteWeight2(filename, wgt_stor_cnt+1, WgtBuff);

  efree(WgtBuff);
	return 0;
}


/************************************************
  write error history(direction is data point)
  inputs:
  filename : error history file name
  hist_no  : error history number( from 1 )
  position : store position( from 0 )
  data     : error data
  ************************************************/
void WriteErrDataPoint(const char *filename, int hist_no, int position, 
                       double data)
{
  Header  head;

  bps_cont.ErrBuffer[position] = (float)data;
  head.index[0]       = 1;
  head.index[1]       = position+1;
  _WriteFile(filename, 2, head.index, (char*)bps_cont.ErrBuffer);
}


/************************************************
  write error history( direction is record )
  inputs:
  hist_no : Error history number.
  ************************************************/
void WriteErrRecord(const char *filename, int hist_no)
{
  int      index[10];

  index[0] = bps_cont.NumOfCell[bps_cont.OutputLayer]+1;

  StoreData(filename, hist_no-1, 1, index, (char*)bps_cont.ErrBuffer);
}



/************************************************
  get and sotre error history( for dir1 )
  inputs:
  filename : error history file name
  err_stor_cnt : history count
  ************************************************/
void StoreErrRecord(const char *filename, int err_stor_cnt)
{
  int     unit, position_cnt;
  float  *dum;
  int     index[10];

  position_cnt = 0;
  dum = (float*)emalloc((bps_cont.NumOfCell[bps_cont.NumOfLayer-1]+1)* 
                       sizeof(float));
	/* TODO bad alloc error check */
	memset(dum, 1, (bps_cont.NumOfCell[bps_cont.NumOfLayer-1]+1)*
		     sizeof(float));

  for (unit = 0; unit < bps_cont.NumOfCell[bps_cont.NumOfLayer-1]; unit++)
    dum[position_cnt++] = (float)bps_cont.OutCellErr[unit];

  dum[position_cnt++] = (float)bps_cont.SumOfErr;
    
  index[0] = position_cnt;
  StoreData(filename, err_stor_cnt-1, 1, index, (char*)dum);

  efree(dum);
}


/************************************************
  write activation history
  inputs:
  hist_no  : activation history number( from 0 )
  ************************************************/
void WriteActivation(int hist_no)
{
  int      index[10];

  index[0] = bps_cont.NumOfCell[bps_cont.OutputLayer];
  StoreData(bps_cont.ActFile, hist_no, 1, index, (char*)bps_cont.act_buf);
}


/************************************************
  read and store old history err
  inputs:
  filename : error history file name
  hist_no         : last history number
  return:
  number of store error( last record )
  ************************************************/
int StoreOldError(const char *filename, int hist_no)
{
  int     i;
  float  *dum;
  Header  head;
  int     num_data = 0;

  if (( dum = (float*)_ReadFile(filename, &head)) == NULL )
    num_data = 0;
  else {
    if ( head.index[0] != 1 || head.index[1] <= 0 )
      num_data = 0;
    else
      num_data = head.index[1];
    if ( num_data > bps_cont.MaxLearnCount )
      num_data = 0; /* maybe don't reach here */

    for ( i = 0; i < num_data; i++ )
      bps_cont.ErrBuffer[i] = dum[i];

    efree(dum);
  }
  return (num_data);
}


/*************************************************
  create file
  inputs:
  filename : file name
  temp          :
  *************************************************/
int CreateFile1(const char *filename, const char *temp)
{
  char    answer = 'n', rem[100];
  Header  head;

  if (access(filename, 0) == -1) {
    InitHeader(&head);
    StoreHeader(filename, &head);
    printf("\n*** File [ %s ] is created. ***\n", filename);
  } else {
    printf("\n*** File [ %s ] is already exist. ***\n", filename);
    printf("    Overwrite ? ( y/n ) : ");
    if (fgets(rem, 100, stdin) != NULL)
      sscanf(rem, "%c", &answer);
    answer = tolower(answer);
    if (answer == 'y') {
      printf("    File [ %s ] is overwrited.\n", filename);
      InitHeader(&head);
      StoreHeader(filename, &head);
    } else
      return 101; /* Set Snother File Name */
  }

	return 0;
}


/**************************************************
  create file
  inputs:
  filename  : file name
  temp      :
  stor_mode : store mode
  **************************************************/
int CreateFile2(const char *filename, const char *temp, int stor_mode)
{
  char  answer = 'n', rem[100];

  if (access(filename, 0) == -1) {
    printf("\n*** File [ %s ] is created. ***\n", filename);
  } else {
    printf("\n*** File [ %s ] is already exist. ***\n", filename);
    if (stor_mode == BPS_STOREMODE_OVERWRITE) {
      printf("    Overwrite ? ( y/n ) : ");
      if (fgets(rem, 100, stdin) != NULL)
        sscanf(rem, "%c", &answer);
      answer = tolower(answer);
      if (answer == 'y') {
        printf("    File [ %s ] is overwrited.\n", filename);
      } else
        return 101; /* Set Snother File Name */
    } else {
      printf("    Append ? ( y/n ) : ");
      if (fgets(rem, 100, stdin) != NULL)
        sscanf(rem, "%c", &answer);
      answer = tolower(answer);
      if (answer != 'y') {
        printf("    File [ %s ] is overwrited.\n", filename);
      }
    }
  }

	return 0;
}


/**************************************************
  create file
  inputs:
  filename  : file name
  temp      :
  stor_mode : store mode
  Not  stop
  **************************************************/
void CreateFile3(const char *filename, const char *temp, int stor_mode)
{
  if (access(filename, 0) == -1) {
    printf("\n*** File [ %s ] is created. ***\n", filename);
  } else {
    printf("\n*** File [ %s ] is already exist. ***\n", filename);
    if (stor_mode == BPS_STOREMODE_OVERWRITE) {
      printf("    File [ %s ] is overwrited.\n", filename);
    } else {
      printf("    File [ %s ] is appended.\n", filename);
    }
  }
}

#ifdef __cplusplus
}
#endif
