/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/getparam.cpp,v 1.5 2005/02/22 07:39:49 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libbps.h"

/*************************************************
 *                                               *
 *       Back Propergation Simurator(BPS)        *
 *              subroutine package               *
 *                             Version 4.0       *
 *     coded    in May.17 1989                   *
 *     coded by  Y.Okamura                       *
 *    last modified in Nov.15 1990               *
 *    modified by  K.Kuroda                      *
 *                                               *
 *************************************************
 *                                               *
 *  filename getparam.c                          *
 *    parameter file handling functions          *
 *                                               *
 ************************************************/

#ifdef __cplusplus
extern "C" {
#endif


/************************************************
  get structure of network parameters
  ************************************************/
void GetStructureParameters()
{
  int    lay, i;

  rebps();
    
  bps_cont.NumOfLayer = bps_cont.bps_sp.NumOfLayer;
  for (lay=0; lay<bps_cont.NumOfLayer; lay++) {
    for (i=0; i<2; i++)
      bps_cont.FuncBias[lay][i] = bps_cont.bps_sp.FuncBias[lay][i];
    bps_cont.NumOfCell[lay] = bps_cont.bps_sp.NumOfCell[lay];
  }
}


/************************************************
  print parameter of structure of network
  ************************************************/
int PrintStructureParameters()
{
  int    lay;

  printf("\n\n<<  structure of network  >>\n");
  printf("### Number of layer =%d\n", bps_cont.NumOfLayer);
  printf("    layer     num of cells        condition\n");
  for (lay=0; lay<bps_cont.NumOfLayer; lay++) {
    printf("       %2d        %5d", lay, bps_cont.NumOfCell[lay]);

    switch (bps_cont.FuncBias[lay][0]) {
    case BPS_BIAS_SIGMOID:  printf("         sigmoid,"); break;
    case BPS_BIAS_LINEAR:   printf("         linear,"); break;
    default:       printf("\n"); return 110; /* Unknown Bias Flag */
    }
    switch (bps_cont.FuncBias[lay][1]) {
    case 'A':      printf(" append bias\n");    break;
    case 'N':      printf(" non bias\n");       break;
    default:       printf("\n"); return 111; /* Unknown Bias Flag */
    } 
  }

	return 0;
}


/************************************************
  get initialize parameters
  ************************************************/
void GetInitializeParameters()
{
  rebps();
    
  bps_cont.InitAlgo = bps_cont.bps_sp.InitAlgo;
  strcpy(bps_cont.InitWgtFile, bps_cont.bps_sp.InitWgtFile);
  bps_cont.Seed = bps_cont.bps_sp.Seed;
  bps_cont.MaxInitWgt = bps_cont.bps_sp.MaxInitWgt;
  bps_cont.MinInitWgt = bps_cont.bps_sp.MinInitWgt;
}


/************************************************
  print initialize parameters
  ************************************************/
int PrintInitializeParameters()
{
  printf("\n\n<<  Initialize condition  >>\n");
  printf("### Init algorithm      = ");
  switch (bps_cont.InitAlgo) {
  case BPS_INITALGO_RANDUM: printf("random\n");          break;
  case BPS_INITALGO_JIA:    printf("Jia's algorithm\n"); break;
  default:     printf("\n");return 210; /* Unknown Algorithm */
  }
  printf("### Initialize data file name = %s\n", bps_cont.InitWgtFile);
  printf("    Seed     Max Weight     Min Weighdt\n");
  printf("  %6d         %6g         %6g\n", bps_cont.Seed, 
         bps_cont.MaxInitWgt, bps_cont.MinInitWgt);
	return 0;
}


/************************************************
  get learning parameters
  ************************************************/
int GetLearningParameters()
{
  rebps();

  strcpy(bps_cont.LrnInitWgtFile, bps_cont.bps_sp.LrnInitWgtFile);
  strcpy(bps_cont.WgtHistoryFile, bps_cont.bps_sp.WgtHistoryFile);
  bps_cont.WgtStorInterval = bps_cont.bps_sp.WgtStorInterval;
  switch (bps_cont.bps_sp.WgtStorMode) {
  case BPS_STOREMODE_APPEND:
    bps_cont.WgtStorMode = BPS_STOREMODE_APPEND;    break;
  case BPS_STOREMODE_OVERWRITE:
    bps_cont.WgtStorMode = BPS_STOREMODE_OVERWRITE; break;
  default:        return 202; /* Unknown Store Mode */
  }

  strcpy(bps_cont.ErrHistoryFile, bps_cont.bps_sp.ErrHistoryFile);
  bps_cont.ErrStorInterval = bps_cont.bps_sp.ErrStorInterval;
  switch (bps_cont.bps_sp.ErrStorDirection) {
  case BPS_STOREDIR_RECORD:
    bps_cont.ErrStorDirection = BPS_STOREDIR_RECORD;
    switch (bps_cont.bps_sp.ErrStorMode) {
    case BPS_STOREMODE_APPEND:
      bps_cont.ErrStorMode = BPS_STOREMODE_APPEND;    break;
    case BPS_STOREMODE_OVERWRITE:
      bps_cont.ErrStorMode = BPS_STOREMODE_OVERWRITE; break;
    default:         return 202; /* Unknown Store Mode */
    }
    break;
  case BPS_STOREDIR_DATAPOINT:
    bps_cont.ErrStorDirection = BPS_STOREDIR_DATAPOINT;
    switch (bps_cont.bps_sp.ErrStorMode) {
    case BPS_STOREMODE_APPEND:
      bps_cont.ErrStorMode = BPS_STOREMODE_APPEND;    break;
    case BPS_STOREMODE_OVERWRITE:
      bps_cont.ErrStorMode = BPS_STOREMODE_OVERWRITE; break;
    default:         return 202; /* Unknown Store Mode */
    }
    break;
    /* ErrStorMode      = BPS_STOREMODE_APPEND; */
  default:
    return 201; /* Unknown Store Direction */
  }

  strcpy(bps_cont.InputFile, bps_cont.bps_sp.InputFile);
  strcpy(bps_cont.TeachFile, bps_cont.bps_sp.TeachFile);

  bps_cont.LrnFirstPtrn = bps_cont.bps_sp.LrnFirstPtrn;
  bps_cont.LrnLastPtrn = bps_cont.bps_sp.LrnLastPtrn;
  if (bps_cont.LrnFirstPtrn > bps_cont.LrnLastPtrn)
		return 205; /* First Pattern Number Is Larger Than Last Pattern */

  switch (bps_cont.bps_sp.LearnMode) {
  case BPS_LEARNMODE_SET:
    bps_cont.LearnMode = BPS_LEARNMODE_SET;     break;
  case BPS_LEARNMODE_PATTERN:
    bps_cont.LearnMode = BPS_LEARNMODE_PATTERN; break;
  default:             return 202; /* Unknown Store Mode */
  }

  bps_cont.LearnAlgo = bps_cont.bps_sp.LearnAlgo;
  if ((bps_cont.LearnAlgo >= BPS_LEARNALGO_STEEP) &&
      (bps_cont.LearnAlgo <= BPS_LEARNALGO_OCHI))
    bps_cont.LearnRate = bps_cont.bps_sp.LearnRate;

  if ((bps_cont.LearnAlgo >= BPS_LEARNALGO_MOMENTUM) &&
      (bps_cont.LearnAlgo <= BPS_LEARNALGO_OCHI))
    bps_cont.Momentum = bps_cont.bps_sp.Momentum;

  if (bps_cont.LearnAlgo == BPS_LEARNALGO_VOGL ||
      bps_cont.LearnAlgo == BPS_LEARNALGO_JACOB ||
      bps_cont.LearnAlgo == BPS_LEARNALGO_OCHI) {
    bps_cont.IncreaseFact = bps_cont.bps_sp.IncreaseFact;
    bps_cont.ReductFact = bps_cont.bps_sp.ReductFact;
    bps_cont.InitMoment = bps_cont.Momentum;
  }
  if (bps_cont.LearnAlgo == BPS_LEARNALGO_VOGL)
    bps_cont.VoglThresh = bps_cont.bps_sp.VoglThresh;    
  if (bps_cont.LearnAlgo == BPS_LEARNALGO_OCHI)
    bps_cont.Theta = bps_cont.bps_sp.Theta;
    
  bps_cont.MinError = bps_cont.bps_sp.MinError;
  bps_cont.MaxLearnCount = bps_cont.bps_sp.MaxLearnCount;
        
  bps_cont.DisplayInterval = bps_cont.bps_sp.DisplayInterval;
  strcpy(bps_cont.Comment, bps_cont.bps_sp.Comment);

	return 0;
}


/************************************************
  print learning parameters
  ************************************************/
int PrintLearningParameters()
{
  printf("\n\n<<  Learning condition  >>\n");
  printf("### Initialize data file name = %s\n", bps_cont.LrnInitWgtFile);
  printf("### Weight history file name  = %s\n", bps_cont.WgtHistoryFile);
  printf("    Store   Interval     Mode\n");
  printf("             %5d      ", bps_cont.WgtStorInterval);

  switch (bps_cont.WgtStorMode) {
  case BPS_STOREMODE_APPEND:    printf("append\n");    break;
  case BPS_STOREMODE_OVERWRITE: printf("overwrite\n"); break;
  default:        printf("\n"); return 202; /* Unknown Store Mode */
  }

  printf("### Error history file name = %s\n", bps_cont.ErrHistoryFile);
  printf("    Store   interval   direction");
  if (bps_cont.ErrStorDirection == BPS_STOREDIR_RECORD) printf("     mode\n");
  else printf("\n");
  printf("             %5d     ", bps_cont.ErrStorInterval);
  switch (bps_cont.ErrStorDirection) {
  case BPS_STOREDIR_RECORD:
    printf(" record      ");
    switch (bps_cont.ErrStorMode) {
    case BPS_STOREMODE_APPEND:     printf("append\n");     break;
    case BPS_STOREMODE_OVERWRITE:  printf("ovarwrite\n");  break;
    default:         printf("\n"); return 202; /* Unknown Store Mode */
    }
    break;
  case BPS_STOREDIR_DATAPOINT:
    printf("data point\n");
    break;
  default:
    printf("\n"); return 201; /* Unknown Store Direction */
  }

  printf("### Input data file name = %s\n", bps_cont.InputFile);
  printf("### Teach data file name = %s\n", bps_cont.TeachFile);

  if (bps_cont.LrnFirstPtrn > bps_cont.LrnLastPtrn)
		return 205; /* First Pattern Number Is Larger Than Last Pattern */

  printf("    Input pattern number = ");

  if (bps_cont.LrnFirstPtrn == 0)
    printf("all\n");
  else
    printf("%d -> %d\n", bps_cont.LrnFirstPtrn, bps_cont.LrnLastPtrn);

  printf("### Learning mode     = ");
  switch (bps_cont.LearnMode) {
  case BPS_LEARNMODE_SET:     printf("set learning\n");     break;
  case BPS_LEARNMODE_PATTERN: printf("pattern learning\n"); break;
  default:            printf("\n"); return 202; /* Unknown Store Mode */
  }    

  printf("### Learning algorithm  = ");
  switch (bps_cont.LearnAlgo) {
  case BPS_LEARNALGO_STEEP:     printf("steep\n");    break;
  case BPS_LEARNALGO_MOMENTUM:  printf("momentum\n"); break;
  case BPS_LEARNALGO_VOGL:      printf("Vogl\n");     break;
  case BPS_LEARNALGO_JACOB:     printf("Jacob's\n");  break;
  case BPS_LEARNALGO_MOMENTUM2: printf("momentum Vogl's coefficient\n"); break;
  case BPS_LEARNALGO_OCHI:      printf("Ochiai's method\n"); break;
  default:        printf("\n"); return 210; /* Unknown Algorithm */
  }

  if ((bps_cont.LearnAlgo >= BPS_LEARNALGO_STEEP) &&
      (bps_cont.LearnAlgo <= BPS_LEARNALGO_OCHI))
    printf("    Learning rate = %g\n", bps_cont.LearnRate);

  if ((bps_cont.LearnAlgo >= BPS_LEARNALGO_MOMENTUM) &&
      (bps_cont.LearnAlgo <= BPS_LEARNALGO_OCHI))
    printf("    Momentum      = %g\n", bps_cont.Momentum);

  if (bps_cont.LearnAlgo == BPS_LEARNALGO_VOGL  ||
      bps_cont.LearnAlgo == BPS_LEARNALGO_JACOB ||
      bps_cont.LearnAlgo == BPS_LEARNALGO_OCHI) {
    printf("    Increasing factor rate = %g\n", bps_cont.IncreaseFact);
    printf("    Reduction factor rate  = %g\n", bps_cont.ReductFact);
  }
  if (bps_cont.LearnAlgo == BPS_LEARNALGO_VOGL)
    printf("    Threshold for Vogl method = %g\n", bps_cont.VoglThresh);

  if (bps_cont.LearnAlgo == BPS_LEARNALGO_OCHI)
    printf("    Theta for Ochiai's method = %g\n", bps_cont.Theta);

  printf("### Minimum error for end     = %g\n", bps_cont.MinError);
  printf("### Maximum iteration for end = %d\n", bps_cont.MaxLearnCount);
        
  printf("### Display interval = %d\n", bps_cont.DisplayInterval);
  printf("### Comment = %s\n", bps_cont.Comment);

	return 0;
}


/************************************************
  get testing parameters
  ************************************************/
int GetTestingParameters()
{
  rebps();

  strcpy(bps_cont.TestWgtFile, bps_cont.bps_sp.TestWgtFile);
  strcpy(bps_cont.TestFile, bps_cont.bps_sp.TestFile);
  bps_cont.WgtHistoryNo = bps_cont.bps_sp.WgtHistoryNo;

  bps_cont.TestFirstPtrn = bps_cont.bps_sp.TestFirstPtrn;
  bps_cont.TestLastPtrn = bps_cont.bps_sp.TestLastPtrn;

  if (bps_cont.TestFirstPtrn > bps_cont.TestLastPtrn)
		return 205; /* First Pattern Number Is Larger Than Last Pattern */

  bps_cont.InputLayer = bps_cont.bps_sp.InputLayer;
  bps_cont.OutputLayer = bps_cont.bps_sp.OutputLayer;

  strcpy(bps_cont.ActFile, bps_cont.bps_sp.ActFile);

	return 0;
}


/************************************************
  print testing parameters
  ************************************************/
int PrintTestingParameters()
{
  printf("\n\n<<  Testing condition  >>\n");
  printf("### Weight file for test = %s\n", bps_cont.TestWgtFile);
  printf("### Weight history number = %d\n", bps_cont.WgtHistoryNo);

  printf("### Input data file for test = %s\n", bps_cont.TestFile);
  if (bps_cont.TestFirstPtrn > bps_cont.TestLastPtrn)
		return 205; /* First Pattern Number Is Larger Than Last Pattern */

  printf("    Input pattern number     = ");

  if (bps_cont.TestFirstPtrn == 0) {
    printf("all\n");
  } else
    printf("%d -> %d\n", bps_cont.TestFirstPtrn, bps_cont.TestLastPtrn);

  printf("### Input layer  = %d\n", bps_cont.InputLayer);
  printf("### Output layer = %d\n", bps_cont.OutputLayer);

  printf("### Result file for test     = %s\n", bps_cont.ActFile);

	return 0;
}

#ifdef __cplusplus
}
#endif
