/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/nethndl.cpp,v 1.7 2005/02/22 07:40:05 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libbps.h"

/************************************************
 *                                              *
 *        Back Propergation Simurator(BPS)      *
 *              subroutine package              *
 *                 Version 4.0                  *
 *        coded        in Aug.25 1989           *
 *      coded by     Y.Okamura                  *
 *      last modified in Nov.15 1990            *
 *      modified by    K.Kuroda                 *
 *                                              *
 ************************************************
 *                                              *
 *       filename nethndl.c                     *
 *               network hander                 *
 *                                              *
 ************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/************************************************
  get link between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  return:
  element of input link list
  ***********************************************/
bps_ilin_t *inter_ilin(bps_cel_t *from_cell, bps_cel_t *to_cell)
{
  bps_ilin_t  *link_pt;

  link_pt = Getintoplist(to_cell);
  for (;;) {
    if (link_pt == NULL)
			return NULL;
    if (from_cell == link_pt->InputCell) break;
    link_pt = Getinfwdlist(link_pt);
  }
  return(link_pt);
}
  

/************************************************
  get link between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  return:
  element of output link list
  ***********************************************/
bps_olin_t *inter_olin(bps_cel_t *from_cell, bps_cel_t *to_cell)
{
  bps_olin_t  *link_pt;
  bps_ilin_t  *out_ilink;
    
  link_pt = Getouttoplist(from_cell);
  for (;;) {
    if (link_pt == NULL)
			return NULL;
    out_ilink = Getouttolist(link_pt);
    if (to_cell == out_ilink->NodeCell) break;
    link_pt = Getoutfwdlist(link_pt);
  }
  return(link_pt);
}


/************************************************
  set weight between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  weight      : weight
  ***********************************************/
int SetWeight(bps_cel_t *from_cell, bps_cel_t *to_cell, double weight)
{
  bps_ilin_t  *link_pt;
    
  link_pt = inter_ilin(from_cell , to_cell);
	if (link_pt == NULL)
		return 120; /* Can't Find Link */

  link_pt->Weight = weight;
	return 0;
}


/************************************************
  get weight between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  ***********************************************/
//double GetWeight(bps_cel_t *from_cell, bps_cel_t *to_cell)
//{
//  bps_ilin_t  *link_pt;
//    
//  link_pt = inter_ilin(from_cell , to_cell);
//	if (link_pt == NULL)
//		return 120; /* Can't Find Link */
//  return(link_pt->Weight);
//}


/************************************************
  set deltaw between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  deltaw    : delta weight
  ***********************************************/
int SetDeltaWgt(bps_cel_t *from_cell, bps_cel_t *to_cell, double deltaw)
{
  bps_ilin_t  *link_pt;
    
  link_pt = inter_ilin(from_cell , to_cell);
	if (link_pt == NULL)
		return 120; /* Can't Find Link */

  link_pt->AdjWgt = deltaw;
	return 0;
}


/************************************************
  make cell node
  return:
  cell node pointer
  **********************************************/
bps_cel_t *make_cell()
{
  bps_cel_t  *cellpoint;
    
  cellpoint = (bps_cel_t*)emalloc(sizeof(bps_cel_t));
  if (cellpoint == NULL)
		return NULL;

  cellpoint->InTopList  = NULL;
  cellpoint->OutTopList = NULL;
  cellpoint->Active        = 0.0;
  cellpoint->CharFunc     = EOS;
  cellpoint->Delta         = 0.0;
  cellpoint->Net        = 0.0;
  cellpoint->SelFlg       = 0;

  return(cellpoint);
}


/************************************************
  connect between two cells
  inputs:
  from_cell : link from
  to_cell   : link to
  **********************************************/
int Connect(bps_cel_t *from_cell, bps_cel_t *to_cell)
{
  bps_ilin_t  *cur_ilin , *nxt_ilin;
  bps_olin_t  *cur_olin , *nxt_olin;

  nxt_ilin = Getintoplist(to_cell);
  if (nxt_ilin == NULL) {
    to_cell->InTopList = (bps_ilin_t*)emalloc(sizeof(bps_ilin_t));
    if (to_cell->InTopList == NULL)
			return 122; /* Can't Allocate To \"struct i_lin\" */
    nxt_ilin = Getintoplist(to_cell);
    cur_ilin = NULL;
  } else {
    while (nxt_ilin != NULL) {
      cur_ilin = nxt_ilin;
      nxt_ilin = Getinfwdlist(nxt_ilin);
    }
    cur_ilin->InFwdList = (bps_ilin_t*)emalloc(sizeof(bps_ilin_t));
    if (cur_ilin->InFwdList == NULL)
			return 122; /* Can't Allocate To \"struct i_lin\" */
    nxt_ilin = Getinfwdlist(cur_ilin);
  }
    
  nxt_ilin->InFwdList  = NULL;
  nxt_ilin->InBwdList  = cur_ilin;
  nxt_ilin->InputCell  = from_cell;
  nxt_ilin->NodeCell   = to_cell;
  nxt_ilin->Weight     = 0.0;
  nxt_ilin->AdjWgt     = 0.0;
  nxt_ilin->WgtWork    = 0.0;
  nxt_ilin->CoefLearn  = 0.0;
  nxt_ilin->dltold     = 0.0;
  nxt_ilin->dltwgtold  = 0.0;
  nxt_ilin->wgtworkold = 0.0;
  nxt_ilin->dltwork    = 0.0;
  nxt_ilin->dltworkold = 0.0;

  nxt_olin = Getouttoplist(from_cell);
  if (nxt_olin == NULL) {
    from_cell->OutTopList = (bps_olin_t*)emalloc(sizeof(bps_olin_t));
    if (from_cell->OutTopList == NULL)
			return 123; /* Can't Allocate To \"struct o_lin\" */
    nxt_olin = Getouttoplist(from_cell);
    cur_olin = NULL;
  } else {
    while (nxt_olin != NULL) {
      cur_olin = nxt_olin;
      nxt_olin = Getoutfwdlist(nxt_olin);
    }
    cur_olin->OutFwdList = (bps_olin_t*)emalloc(sizeof(bps_olin_t));
    if (cur_olin->OutFwdList == NULL)
			return 123; /* Can't Allocate To \"struct o_lin\" */
    nxt_olin = Getoutfwdlist(cur_olin);
  }
  nxt_olin->OutFwdList = NULL;
  nxt_olin->OutBwdList = cur_olin;
  nxt_olin->OutToList  = nxt_ilin;

	return 0;
}


/************************************************
  make normal BP network
  **********************************************/
int MakeNetwork()
{
  int    lay, unit, unit_to, unit_from, cell_cnt;
  int    num;

  cell_cnt = 0;
    
  for (lay=0; lay<bps_cont.NumOfLayer; lay++) {

    num = (bps_cont.NumOfCell[lay]+1) * sizeof(bps_lay_t);
    
    bps_cont.BPNet[lay] = (bps_lay_t*)emalloc(num);
    if (bps_cont.BPNet[lay] == NULL)
			return 124; /* Can't Allocate To \"BPNet\" */
    
    for (unit=0; unit<=bps_cont.NumOfCell[lay]; unit++) {
      bps_cont.BPNet[lay][unit].CellNode = make_cell();

			if(bps_cont.BPNet[lay][unit].CellNode == NULL)
				return 121; /* Can't Allocate To \"struct cel\" */

      bps_cont.BPNet[lay][unit].CellNo = cell_cnt++;
    }
  }
  
  for (unit=0; unit<=bps_cont.NumOfCell[0]; unit++)
    (bps_cont.BPNet[0][unit].CellNode)->CharFunc = bps_cont.FuncBias[0][0]; 
  
  for (lay=1; lay<bps_cont.NumOfLayer; lay++) {
    for (unit_to=1; unit_to<=bps_cont.NumOfCell[lay]; unit_to++) {
      if (CheckBias(lay)) {
        Connect(bps_cont.BPNet[lay][0].CellNode,
                bps_cont.BPNet[lay][unit_to].CellNode);
        (bps_cont.BPNet[lay][0].CellNode)->Active = 1.0;
      }
      for (unit_from=1; unit_from<=bps_cont.NumOfCell[lay-1]; unit_from++) 
        Connect(bps_cont.BPNet[lay-1][unit_from].CellNode,
                bps_cont.BPNet[lay][unit_to].CellNode);
    }
    for (unit=0; unit<=bps_cont.NumOfCell[lay]; unit++)
      (bps_cont.BPNet[lay][unit].CellNode)->CharFunc =
        bps_cont.FuncBias[lay][0]; 
  }

	return 0;
}


/************************************************
  break normal BP network
  **********************************************/
void BreakNetwork()
{
  int          lay, unit;
  bps_ilin_t  *cur_ilin, *nxt_ilin;
  bps_olin_t  *cur_olin, *nxt_olin;

  for (lay=0; lay<bps_cont.NumOfLayer; lay++) {
    for (unit=0; unit<=bps_cont.NumOfCell[lay]; unit++) {
      nxt_ilin = Getintoplist(bps_cont.BPNet[lay][unit].CellNode);
      while (nxt_ilin != NULL) {
        cur_ilin = nxt_ilin;
        nxt_ilin = Getinfwdlist(cur_ilin);
        efree(cur_ilin);
      }
      
      nxt_olin = Getouttoplist(bps_cont.BPNet[lay][unit].CellNode);
      while (nxt_olin != NULL) {
        cur_olin = nxt_olin;
        nxt_olin = Getoutfwdlist(cur_olin);
        efree(cur_olin);
      }
      efree(bps_cont.BPNet[lay][unit].CellNode);
    }
    efree(bps_cont.BPNet[lay]);
  }
}


/***********************************************
  read and set weight history
  inputs:
  filename   : file name
  hist_no : weight history number
  **********************************************/
int ReadWeight2(const char *filename, int hist_no)
{
  int     last_hist_no, lay, unit_to, unit_from, link_cnt;
  float    *wgt_data;
  /*  Header  *head; replaced by HAGI 1993.10.17 */
  Header  head;

  if (access(filename, 0) == -1)
		return 125; /* Weight File Isn't Exist */

  last_hist_no = LastWgtHistory(filename);

  if ((hist_no < 0) || (hist_no > last_hist_no))
		return 126; /* Weight History Number Error */
  if (hist_no == 0) hist_no = last_hist_no;

  wgt_data = (float*)LoadData(filename, hist_no - 1 , &head); 
  hist_no++;
  
  /* SET WEIGHT TO BP NETWORK */
  link_cnt = 0;
  for (lay=1; lay<bps_cont.NumOfLayer; lay++) {
    for (unit_to=1; unit_to<=bps_cont.NumOfCell[lay]; unit_to++) {
      if (CheckBias(lay))
        SetWeight(bps_cont.BPNet[lay][0].CellNode,
                  bps_cont.BPNet[lay][unit_to].CellNode,
                  wgt_data[link_cnt++]);
      for (unit_from=1; unit_from<=bps_cont.NumOfCell[lay-1]; unit_from++)
        SetWeight(bps_cont.BPNet[lay-1][unit_from].CellNode,
                  bps_cont.BPNet[lay][unit_to].CellNode,
                  wgt_data[link_cnt++]);
    }
  }
  FreeData(wgt_data);

  wgt_data = (float*)LoadData(filename, hist_no - 1, &head);
  hist_no++;
    
  /* SET DELTA WEIGHT TO BP NETWORK */
  link_cnt = 0;
  for (lay=1; lay<bps_cont.NumOfLayer; lay++) {
    for (unit_to=1; unit_to<=bps_cont.NumOfCell[lay]; unit_to++) {
      if (CheckBias(lay))
        SetDeltaWgt(bps_cont.BPNet[lay][0].CellNode,
                    bps_cont.BPNet[lay][unit_to].CellNode,
                    wgt_data[link_cnt++]);
      for (unit_from=1; unit_from<=bps_cont.NumOfCell[lay-1]; unit_from++)
        SetDeltaWgt(bps_cont.BPNet[lay-1][unit_from].CellNode,
                    bps_cont.BPNet[lay][unit_to].CellNode,
                    wgt_data[link_cnt++]);
    }
  }
  FreeData(wgt_data);
	return 0;
}


/************************************************
  display network information
  ***********************************************/
void netdump()
{
  int       lay , unit;
  bps_cel_t   *cellpoint;
  bps_ilin_t  *ilin_pt;
  bps_olin_t  *olin_pt;

  printf("\n\n\n**** cell dump utility ****\n");
    
  for (lay=0; lay<bps_cont.NumOfLayer; lay++) {
    printf("###   layer No.%d   ###\n", lay);
    for (unit=0; unit<=bps_cont.NumOfCell[lay]; unit++) {
      printf("<<< cell No.%d >>>\n", unit);
      cellpoint = bps_cont.BPNet[lay][unit].CellNode;
      printf("   address = %lx\n", (size_t)cellpoint);
      printf("   activation = %12e  CharFunc = %d",
             cellpoint->Active, cellpoint->CharFunc);
      printf("  delta = %12e   net_pj = %12e\n",
             cellpoint->Delta, cellpoint->Net);
      ilin_pt = Getintoplist(cellpoint);
      while (ilin_pt != NULL) {
        printf("   link_from =%lx   node is =%lx\n",
               (size_t)ilin_pt->InputCell,
               (size_t)ilin_pt->NodeCell);
        printf("   weight = %12e   adj_wgt = %12e",
               ilin_pt->Weight, ilin_pt->AdjWgt);
        printf("   wgt_work = %12e   coe_learn = %12e\n",
               ilin_pt->WgtWork, ilin_pt->CoefLearn);

        ilin_pt = Getinfwdlist(ilin_pt);
      }
      olin_pt = Getouttoplist(cellpoint);
      while (olin_pt != NULL) {
        ilin_pt = Getouttolist(olin_pt);
        printf("   link to =%lx \n", (size_t)ilin_pt->NodeCell);
        olin_pt = Getoutfwdlist(olin_pt);
      }
    }
  }
}


/************************************************
  cut between to cells
  inputs:
  from_cell : link from
  to_cell   : link to
  ***********************************************/
//int cut(bps_cel_t *from_cell, bps_cel_t *to_cell)
//{
//  bps_ilin_t  *fwd_ilin , *cur_ilin , *bwd_ilin;
//  bps_olin_t  *fwd_olin , *cur_olin , *bwd_olin;
//  
//  cur_ilin = inter_ilin(from_cell, to_cell);
//  fwd_ilin = Getinfwdlist(cur_ilin);
//  bwd_ilin = Getinbwdlist(cur_ilin);
//  fwd_ilin->InBwdList = bwd_ilin;
//  if (bwd_ilin == NULL) to_cell->InTopList = fwd_ilin;
//  else bwd_ilin->InFwdList = fwd_ilin;
//  efree(cur_ilin);
//  
//  cur_olin = inter_olin(from_cell , to_cell);
//	if(cur_olin == NULL)
//		return 120;
//  fwd_olin = Getoutfwdlist(cur_olin);
//  bwd_olin = Getoutbwdlist(cur_olin);
//  fwd_olin->OutBwdList = bwd_olin;
//  if (bwd_olin == NULL) from_cell->OutTopList = fwd_olin;
//  else bwd_olin->OutFwdList = fwd_olin;
//  efree(cur_olin);
//
//	return 0;
//}

#ifdef __cplusplus
}
#endif
