/*
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/bps/command/winit.cpp,v 1.7 2005/02/22 04:32:19 ninja Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libbps.h"

/************************************************
 *                                              *
 *       Back Propergation Simurator(BPS)       *
 *             subroutine package               *
 *                 Version                      * 
 *           coded        in Aug.25 1989        *
 *         coded by         Y.Okamura           *
 *         last modified in Nov.15 1990         *
 *         modified by         K.Kuroda         *
 *                                              *
 ************************************************
 *                                              *
 *       filename inimain.c                     *
 *        BPS WINIT command                     *
 *                                              *
 ************************************************/

#ifdef __cplusplus
extern "C" {
#endif

#ifndef  HAVE_LRAND48
static long int  lrand48();
static void      srand48(long int seedval);
#endif

#define        AREA_COEF_A     2.0   /* for jia's */
#define        AREA_COEF_B     3.0   /* for jia's */


/************************************************
  generate randum number
  inputs:
  low  : minimum of randum number
  high : maximum of randum number
  ************************************************/
static double randm(double low, double high)
{
  double norm, data;

  norm = 2147483646.0;
  data = low + (high - low) * (double)lrand48()/norm;
  return data;
}


/************************************************
  make initialize file with random data
  ************************************************/
static int initialize_random()
{
  int      i;
  Buffer  *WgtBuff;

  srand48(bps_cont.Seed);

  WgtBuff = AllocBuffer(bps_cont.NumOfLink);
  if (WgtBuff == NULL)
		return 1; /* Command Not Found */

  for (i = 0; i < bps_cont.NumOfLink; i++)
    WgtBuff[i] = randm(bps_cont.MinInitWgt, bps_cont.MaxInitWgt);
  WriteWeight2(bps_cont.InitWgtFile, 0, WgtBuff);

  for (i = 0; i < bps_cont.NumOfLink; i++)
    WgtBuff[i] = 0.0;
  WriteWeight2(bps_cont.InitWgtFile, 1, WgtBuff);

  efree(WgtBuff);

	return 0;
}


/************************************************
  make initialize file with JIA's algorithm
  ***********************************************/
int
initialize_jia()
{
  int      lay, unit_to, unit_from, i, writepoint;
  double   wgt_temp, active_area, cube_size, coeff, *weight;
  double  *WgtBuff;
  int      dpt;
  
  writepoint = 0;

  srand48(bps_cont.Seed);
  
  WgtBuff = AllocBuffer(bps_cont.NumOfLink);
  if (WgtBuff == NULL)
		return 1; /* Command Not Found */
  
  dpt = 0; /* by take */
  for (lay = 1; lay < bps_cont.NumOfLayer; lay++) {
    if (CheckBias(lay)) {
      for (unit_to = 1; unit_to <= bps_cont.NumOfCell[lay]; unit_to++) {
        weight = (double *)emalloc((bps_cont.NumOfCell[lay-1]+1)*
                                   sizeof(double));
        if (weight == NULL)
					return 30; /* Can't Allocate To Initialize Weight Array */
        
        /* ( 1 ) SET RANDOM DATA */
        for (unit_from = 1; unit_from <= bps_cont.NumOfCell[lay-1]; 
             unit_from++)
          weight[unit_from] = randm(bps_cont.MinInitWgt, bps_cont.MaxInitWgt);
        weight[0] = 0.0;
        
        /* ( 2 ) SET THRESHOLD */
        wgt_temp = 0.0;
        for (unit_from = 1; unit_from <= bps_cont.NumOfCell[lay-1];
             unit_from++) {
          weight[0] -= 0.5 * weight[unit_from];

          wgt_temp += weight[unit_from] * weight[unit_from];
        }

        /* ( 3 ) SET WIDTHS OF ACTIVE AREA */
        active_area = 4.4 / sqrt(wgt_temp);

        /* ( 4 ) ADJUST ACTIVE AREA TO CUBE SIZE */
        cube_size = sqrt((double) bps_cont.NumOfCell[lay]);
        coeff = active_area/(cube_size * randm(AREA_COEF_A, AREA_COEF_B));

        for (unit_from = 0; unit_from <= bps_cont.NumOfCell[lay-1];
             unit_from++) {
          /* weight[unit_from]  *= coeff; by take */
          WgtBuff[dpt++]      = coeff * weight[unit_from]; /* by take */
        }
        efree(weight);
      }
    } else
      return 31; /* Append Bias To Layer */
  }
  WriteWeight2(bps_cont.InitWgtFile, 0, WgtBuff);
  
  for (i = 0; i < bps_cont.NumOfLink; i++)
    WgtBuff[i] = 0.0;
  WriteWeight2(bps_cont.InitWgtFile, 1, WgtBuff);
  efree(WgtBuff);

	return 0;
}


/************************************************
  initialize data maker
  main routine
  ************************************************/
DLLEXPORT int mod_bps_winit()
{
  rebps();

  GetStructureParameters();
  PrintStructureParameters();
  GetInitializeParameters();
  PrintInitializeParameters();

  CreateFile1(bps_cont.InitWgtFile, "initialize weight data"); 

  /* WgtMaxDataPoint = 0; */
  /* CleanHistArray(WgtOutData); nouse */

  SetNumOfLink();
  /* WgtBlockSize = NumOfLink / HBUFSIZE;
     if ((NumOfLink % HBUFSIZE) != 0)
     WgtBlockSize++; */

  switch (bps_cont.InitAlgo) {
    /* INITIALIZE DATA MAKER WITH RANDOM DATA */
  case BPS_INITALGO_RANDUM:
    initialize_random();
    break;

    /* INITIALIZE DATA MAKER WITH JIA'S ALGORITHM */
  case BPS_INITALGO_JIA:
    initialize_jia();
    break;
  }
  
  return 0;
}

#ifndef HAVE_LRAND48
/*        @(#)drand48.c        2.2        */
/*
 *        drand48, etc. pseudo-random number generator
 *        This implementation assumes unsigned short integers of at least
 *        16 bits, long integers of at least 32 bits, and ignores
 *        overflows on adding or multiplying two unsigned integers.
 *        Two's-complement representation is assumed in a few places.
 *        Some extra masking is done if unsigneds are exactly 16 bits
 *        or longs are exactly 32 bits, but so what?
 *        An assembly-language implementation would run significantly faster.
 */

#define N                16
#define MASK             ((unsigned)(1 << (N - 1)) + (1 << (N - 1)) - 1)
#define LOW(x)           ((unsigned)(x) & MASK)
#define HIGH(x)          LOW((x) >> N)
#define MUL(x, y, z)     { long l = (long)(x) * (long)(y); \
                           (z)[0] = LOW(l); (z)[1] = HIGH(l); }
#define CARRY(x, y)      ((long)(x) + (long)(y) > MASK)
#define ADDEQU(x, y, z)  (z = CARRY(x, (y)), x = LOW(x + (y)))

#define X0   0x330E
#define X1   0xABCD
#define X2   0x1234
#define A0   0xE66D
#define A1   0xDEEC
#define A2   0x5
#define C    0xB

#define SET3(x, x0, x1, x2) ((x)[0] = (x0), (x)[1] = (x1), (x)[2] = (x2))
#define SEED(x0, x1, x2)    (SET3(x, x0, x1, x2), SET3(a, A0, A1, A2), c = C)

static unsigned x[3] = { X0, X1, X2 }, a[3] = { A0, A1, A2 }, c = C;
static void next();

static long int lrand48()
{
  next();
  return (((long)x[2] << (N - 1)) + (x[1] >> 1));
}

static void srand48(long int seedval)
{
  SEED(X0, LOW(seedval), HIGH(seedval));
}

static void next()
{
  unsigned p[2], q[2], r[2], carry0, carry1;

  MUL(a[0], x[0], p);
  ADDEQU(p[0], c, carry0);
  ADDEQU(p[1], carry0, carry1);
  MUL(a[0], x[1], q);
  ADDEQU(p[1], q[0], carry0);
  MUL(a[1], x[0], r);
  x[2] = LOW(carry0 + carry1 + CARRY(p[1], r[0]) + q[1] + r[1] +
             a[0] * x[2] + a[1] * x[1] + a[2] * x[0]);
  x[1] = LOW(p[1] + r[0]);
  x[0] = LOW(p[0]);
}
#endif

#ifdef __cplusplus
}
#endif
