/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 2000, Toshihiro Aoyama All rights reserved.
 * 
 * $Id: abf2satellite.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_WINDOWS_H
# include <windows.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifndef WIN32
# include <sys/mman.h>
#endif

#include "SL_macro.h"
#include "SL_cmd.h"
#undef fprintf
#include "abf2satellite.h"

#ifdef WIN32
#define MAPPING_NAME "MMAP_FILE"
#endif

#ifdef __cplusplus
extern "C" {
#endif

static char *
rename_filename(char *in, char *out, char *suffix)
{
  int             i, j;
  
  /* generate save file name */
  i = 0;
  while (in[i] != '.') {
    out[i] = in[i];
    i++;
  }
  out[i] = '.';
  for (j = 0; j < 3; j++) {
    out[i + j + 1] = suffix[j];
  }
  out[i + j + 1] = '\0';
  
  return out;
}
  
DLLEXPORT int mod_dcm_abf2satellite(int argc, char **argv)
{
#ifdef WIN32
  HANDLE          data_fd;
#else
  struct stat     stat_buf;
  int             data_fd;
#endif
  int             n, i, j = 0;
  char           *text;
  char           *p;
  FILE           *head_fp;
  char            *out_filename;
  char            *cnf_filename;
  char            *in_filename;
  int             sl_index[2];
  Buffer         *sl_buffer;
  int             k = 0;
  
  int             NumSamplesPerEpisode;
  int             EpisodesPerRun;

  /* header structure */
  FileID_Size_Info *id_size_info;
  File_Structure *file_structure;
  Display_Parameters *disp_param;
  Trial_Hierarchy_Info *tr_info;
  Environmental_Info *env_info;
  Multi_Channel_Info *mc_info;
  DAC_Output_File *DAC_out_file;
  Hardware_Info  *hw_info;
  
  /* initialize for SATELLITE */
  in_filename=GetString(0);
  if(in_filename==NULL){
    return (1);
  }
  
  /* initialize filename */
  out_filename = (char *) malloc(sizeof(char)*ABF_FILENAMELEN);
  cnf_filename = (char *) malloc(sizeof(char)*ABF_FILENAMELEN);
  rename_filename(in_filename, out_filename, "dat");
  rename_filename(in_filename, cnf_filename, "cnf");
  
  /* open header output file */
  if ((head_fp = fopen(cnf_filename, "w")) == NULL) {
    fprintf(stderr, "%s file open failed\n", cnf_filename);
    return (1);
  }
  /* open ABF file */
#ifdef WIN32
  data_fd=CreateFile(in_filename,GENERIC_READ|GENERIC_WRITE,0,
                     NULL,OPEN_EXISTING,0,NULL);
  if(data_fd==NULL){
    printf("%s file open failed\n",in_filename);
    return (1);
  }
  n=(int)GetFileSize(data_fd,NULL);
  HANDLE hMapFile=CreateFileMapping(data_fd,NULL,PAGE_READWRITE,0,n,MAPPING_NAME);
  if(hMapFile==NULL){
    printf("%s file memory mapping failed\n",in_filename);
    CloseHandle(data_fd);
    return (1);
  }
  text=(char*)MapViewOfFile(hMapFile,FILE_MAP_ALL_ACCESS,0,0,n);
#else
  data_fd = open(in_filename, O_RDONLY);
  if (data_fd == -1) {
    fprintf(stderr, "%s file open failed\n", in_filename);
    return (1);
  }
  if (fstat(data_fd, &stat_buf) == -1) {
    fprintf(stderr, "stat failed\n");
    return (1);
  }
  n = stat_buf.st_size;
  text = (char *) mmap(NULL, n + 1, PROT_READ | PROT_WRITE, MAP_PRIVATE, data_fd, 0);
  if ((int) text == -1) {
    fprintf(stderr, "mmap failed\n");
    return (1);
  }
#endif
  text[n] = 0;
  /* header structure initialization */
  id_size_info = (FileID_Size_Info *) malloc(sizeof(FileID_Size_Info));
  file_structure = (File_Structure *) malloc(sizeof(File_Structure));
  disp_param = (Display_Parameters *) malloc(sizeof(Display_Parameters));
  hw_info = (Hardware_Info *) malloc(sizeof(Hardware_Info));
  tr_info = (Trial_Hierarchy_Info *) malloc(sizeof(Trial_Hierarchy_Info));
  env_info = (Environmental_Info *) malloc(sizeof(Environmental_Info));
  mc_info = (Multi_Channel_Info *) malloc(sizeof(Multi_Channel_Info));
  DAC_out_file = (DAC_Output_File *) malloc(sizeof(DAC_Output_File));
  
  /* read File ID and Size information */
  Read_FileID_Size_Info(text, id_size_info);
  Print_FileID_Size_Info(head_fp, id_size_info);
  
  /* read File Structure */
  Read_File_Structure(text, file_structure);
  Print_File_Structure(head_fp, file_structure);
  
  /* read Trial_Hierarchy_Info */
  Read_Trial_Hierarchy_Info(text, tr_info);
  Print_Trial_Hierarchy_Info(head_fp, tr_info);
  
  /* read Hulti_Channel_Info */
  Read_Multi_Channel_Info(text, mc_info);
  Print_Multi_Channel_Info(head_fp, mc_info);
  
  /* read Display Parameters */
  Read_Display_Parameters(text, disp_param);
  Print_Display_Parameters(head_fp, disp_param);
  
  /* read Hardware Information */
  Read_Hardware_Info(text, hw_info);
  Print_Hardware_Info(head_fp, hw_info);
  
  /* read Environmental_Information */
  Read_Environmental_Info(text, env_info);
  Print_Environmental_Info(head_fp, env_info);
  
  /* read DAC Output File */
  Read_DAC_Output_File(text, DAC_out_file);
  Print_DAC_Output_File(head_fp, DAC_out_file);
  
  fclose(head_fp);
  
  NumSamplesPerEpisode = tr_info->NumSamplesPerEpisode;
  /* EpisodesPerRun=tr_info->EpisodesPerRun */
  EpisodesPerRun = id_size_info->ActualEpisodes;
  
  /* allocate SATELLITE buffer */
  sl_buffer =
    (Buffer *) malloc(sizeof(Buffer) * NumSamplesPerEpisode * EpisodesPerRun);
  if (sl_buffer == NULL) {
    printf("out of memory\n");
    return (1);
  }
  /* Read DATA */
  p = text + ABF_BLOCKSIZE * file_structure->DataSectionPtr;
  for (j = 0; j < EpisodesPerRun; j++) {
    for (i = 0; i < NumSamplesPerEpisode; i++) {
      switch (file_structure->DataFormat) {
      case ABF_DATAFORMAT_INT:
        sl_buffer[k] =
          *(short *) (p + (i + j * NumSamplesPerEpisode) * 2)
          * (hw_info->ADCRange)
          / (hw_info->ADCResolution)
          / mc_info->InstrumentScaleFactor[0];
        break;
      case ABF_DATAFORMAT_FLOAT:
        sl_buffer[k] =
          *(float *) (p + (i + j * NumSamplesPerEpisode) * 4);
        break;
      default:
        return (1);
      }
      k++;
    }
  }
#ifdef WIN32
  UnmapViewOfFile((void*)text);
  CloseHandle(hMapFile);
  CloseHandle(data_fd);
#else
  munmap((void*)text, n+1);
  close(data_fd);
#endif
  /* save satellite buffer file */
  sl_index[0] = EpisodesPerRun;
  sl_index[1] = NumSamplesPerEpisode;
  ChangeDataSize(8);
  _WriteFile(out_filename,2,sl_index,(char*)sl_buffer);

  free(out_filename);
  free(cnf_filename);
  free(id_size_info);
  free(file_structure);
  free(disp_param);
  free(hw_info);
  free(tr_info);
  free(env_info);
  free(mc_info);
  free(DAC_out_file);
  free(sl_buffer);

  return 0;
}

#ifdef __cplusplus
}
#endif
