/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Copyright (c) 2000, Toshihiro Aoyama All rights reserved.
 * 
 * $Id: abf2satellite.h,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $
 */

/* from abfheader.h */
#define ABF_ADCCOUNT 16 /* number of ADC channels supported. */
#define ABF_DACCOUNT 4 /* number of DAC channels supported. */
#define ABF_EPOCHCOUNT 10 /* number of DAC channels supported. */
#define ABF_BELLCOUNT 2 /* Number of auditory signals supported. */
#define ABF_ADCUNITLEN 8 /* length of ADC units strings */
#define ABF_ADCNAMELEN 10 /* length of ADC channel name strings */
#define ABF_DACUNITLEN 8 /* length of DAC units strings */
#define ABF_DACNAMELEN 10 /* length of DAC channel name strings */
#define ABF_VARPARAMLISTLEN 80 /* length of conditioning string */
#define ABF_FILECOMMENTLEN 56 /* length of file comment string */
#define ABF_CREATORINFOLEN 16 /* length of file creator info string */
#define ABF_DACFILENAMELEN 12 /* length of the DACFile name string */
#define ABF_DACFILEPATHLEN 60 /* length of Drive & directory for DACFile */
#define ABF_ARITHMETICOPLEN 2 /* length of the Arithmetic operator field */
#define ABF_ARITHMETICUNITSLEN 8 /* length of arithmetic units string */
#define ABF_TAGCOMMENTLEN 56 /* length of tag comment string */
#define ABF_LONGDESCRIPTIONLEN 56 /* length of long description entry */
#define ABF_NOTENAMELEN 10 /* length of the name component of a note */
#define ABF_NOTEVALUELEN 8 /* length of the value component of a note */
#define ABF_NOTEUNITSLEN 8 /* length of the units component of a note */
#define ABF_BLOCKSIZE 512 /* Size of block allignment in ABF files. */
#define ABF_CURRENTVERSION 1.3F /* File format version number */
#define ABF_HEADERSIZE 2048 /* Size of a Version 1.x header */
#define ABF_NATIVESIGNATURE 0x20464241 /* PC="ABF ", MAC=" FBA" */
#define ABF_REVERSESIGNATURE 0x41424620 /* PC=" FBA", MAC="ABF " */
#define ABF_FACESIZE 32

#define ABF_DATAFORMAT_INT 0
#define ABF_DATAFORMAT_FLOAT 1

#define ABF_FILENAMELEN 128

typedef struct FileID_Size_Info {
	char            FileSignature[5];
	float           FileVersionNumber;
	short           OperationMode;
	long            ActualAcqLength;
	short           NumPointsIgnored;
	long            ActualEpisodes;
	long            FileStartDate;
	long            FileStartTime;
	long            StopwatchTime;
	float           HeaderVersionNumber;
	short           FileType;
	short           MSBinFormat;
}               FileID_Size_Info;

typedef struct File_Structure {
	long            DataSectionPtr;
	long            TagSectionPtr;
	long            NumTagEntries;
	long            ScopeConfigPtr;
	long            NumScopes;
	long            DACFilePtr;
	long            DACFileNumEpisodes;
	long            DeltaArrayPtr;
	long            NumDeltas;
	long            SynchArrayPtr;
	long            SynchArraySize;
	short           DataFormat;
}               File_Structure;

typedef struct Trial_Hierarchy_Info {
	short           ADCNumChannels;
	float           ADCSampleInterval;
	float           ADCSecondSampleInterval;
	float           SynchTimeUnit;
	float           SecondsPerRun;
	long            NumSamplesPerEpisode;
	long            PreTriggerSamples;
	long            EpisodesPerRun;
	long            RunsPerTrial;
	long            NumberOfTrials;
	short           AveragingMode;
	short           UndoRunCount;
	short           FirstEpisodeInRun;
	float           TriggerThreshold;
	short           TriggerSource;
	short           TriggerAction;
	short           TriggerPolarity;
	float           ScopeOutputInterval;
	float           EpisodeStartToStart;
	float           RunStartToStart;
	float           TrialStartToStart;
	long            AverageCount;
	long            ClockChange;
	short           AutoTriggerStrategy;
}               Trial_Hierarchy_Info;

typedef struct Display_Parameters {
	short           DrawingStrategy;
	short           TiledDisplay;
	short           EraseStrategy;
	short           DataDisplayMode;
	long            DisplayAverageUpdate;
	short           ChannelStatsStratgy;
	long            CalculationPeriod;
	long            SamplesPerTrace;
	long            StartDisplayNum;
	long            FinishDisplayNum;
	short           MultiColor;
	short           ShowPNRawData;
	float           StatisticsPeriod;
}               Display_Parameters;

typedef struct Hardware_Info {
	float           ADCRange;
	float           DACRange;
	long            ADCResolution;
	long            DACResolution;
}               Hardware_Info;

typedef struct Environmental_Info {
	short           ExperimentType;
	short           AutosampleEnable;
	short           AutosampleADCNum;
	short           AutosampleInstrument;
	float           AutosampleAdditGain;
	float           AutosampleFilter;
	float           AutosampleMembraneCap;
	short           ManualInfoStrategy;
	float           CellID1;
	float           CellID2;
	float           CellID3;
	char            CreatorInfo[ABF_CREATORINFOLEN];
	char            FileComment[ABF_FILECOMMENTLEN];
}               Environmental_Info;

typedef struct Multi_Channel_Info {
	short           ADCPtoLChannelMap[ABF_ADCCOUNT];
	short           ADCSamplingSeq[ABF_ADCCOUNT];
	char            ADCChannelName[ABF_ADCCOUNT][ABF_ADCNAMELEN];
	char            ADCUnits[ABF_ADCCOUNT][ABF_ADCUNITLEN];
	float           ADCProgrammableGain[ABF_ADCCOUNT];
	float           ADCDisplayAmplification[ABF_ADCCOUNT];
	float           ADCDisplayOffset[ABF_ADCCOUNT];
	float           InstrumentScaleFactor[ABF_ADCCOUNT];
	float           InstrumentOffset[ABF_ADCCOUNT];
	float           SignalGain[ABF_ADCCOUNT];
	float           SignalOffset[ABF_ADCCOUNT];
	float           SignalLowpassFilter[ABF_ADCCOUNT];
	float           SignalHighpassFilter[ABF_ADCCOUNT];
	char            DACChannelName[ABF_DACCOUNT][ABF_DACNAMELEN];
	char            DACChannelUnits[ABF_DACCOUNT][ABF_DACUNITLEN];
	float           DACScaleFactor[ABF_DACCOUNT];
	float           DACHoldingLevel[ABF_DACCOUNT];
	short           SignalType;
}               Multi_Channel_Info;

typedef struct Synchronous_Timer_Output {
	short           OUTEnable;
	short           SampleNumberOUT1;
	short           SampleNumberOUT2;
	short           FirstEpisodeOUT;
	short           LastEpisodeOUT;
	short           PulseSamplesOUT1;
	short           PulseSamplesOUT2;
}               Synchronous_Timer_Output;

typedef struct Epoch_Waveform_Pulse {
	short           DigitalEnable;
	short           WaveformSource;
	short           ActiveDACChannel;
	short           InterEpisodeLeve;
	short           EpochInitLevel[ABF_EPOCHCOUNT];
	short           EpochLevelInc[ABF_EPOCHCOUNT];
	short           EpochInitDuration[ABF_EPOCHCOUNT];
	short           EpochDurationInc[ABF_EPOCHCOUNT];
	short           DigitalHolding;
	short           DigitalInterEpisode;
	short           DigitalValue[ABF_EPOCHCOUNT];
}               Epoch_Waveform_Pulse;

typedef struct DAC_Output_File {
	float           DACFileScale;
	float           DACFileOffset;
	short           DACFileEpisodeNum;
	short           DACFileADCNum;
	char            DACFileName[ABF_DACFILENAMELEN];
	char            DACFilePath[ABF_DACFILEPATHLEN];
}               DAC_Output_File;

void            Read_FileID_Size_Info(char *text, FileID_Size_Info * p);
void            Print_FileID_Size_Info(FILE * fp, FileID_Size_Info * p);
void            Read_File_Structure(char *text, File_Structure * p);
void            Print_File_Structure(FILE * fp, File_Structure * p);
void            Read_Trial_Hierarchy_Info(char *text, Trial_Hierarchy_Info * p);
void            Print_Trial_Hierarchy_Info(FILE * fp, Trial_Hierarchy_Info * p);
void            Read_Display_Parameters(char *text, Display_Parameters * p);
void            Print_Display_Parameters(FILE * fp, Display_Parameters * p);
void            Read_Hardware_Info(char *text, Hardware_Info * p);
void            Print_Hardware_Info(FILE * fp, Hardware_Info * p);
void            Read_Environmental_Info(char *text, Environmental_Info * p);
void            Print_Environmental_Info(FILE * fp, Environmental_Info * p);
void            Read_Multi_Channel_Info(char *text, Multi_Channel_Info * p);
void            Print_Multi_Channel_Info(FILE * fp, Multi_Channel_Info * p);
void            Read_Synchronous_Timer_Output(char *text, Synchronous_Timer_Output * p);
void            Print_Synchronous_Timer_Output(FILE * fp, Synchronous_Timer_Output * p);
void            Read_Epoch_Waveform_Pulse(char *text, Epoch_Waveform_Pulse * p);
void            Print_Epoch_Waveform_Pulse(FILE * fp, Epoch_Waveform_Pulse * p);
void            Read_DAC_Output_File(char *text, DAC_Output_File * p);
void            Print_DAC_Output_File(FILE * fp, DAC_Output_File * p);


void
Read_FileID_Size_Info(char *text, FileID_Size_Info * p)
{
	/* read FileSignature 4byte */
	strncpy(p->FileSignature, (char *) text, 4);
	p->FileSignature[4] = '\0';
	/* read FileVersionNumber 4byte */
	p->FileVersionNumber = *(float *) (text + 4);
	/* read OperationMode 2byte */
	p->OperationMode = *(short *) (text + 8);
	/* read ActualAcqLength 4byte */
	p->ActualAcqLength = *(long *) (text + 10);
	/* read NumPointsIgnored 2byte */
	p->NumPointsIgnored = *(short *) (text + 14);
	/* read ActualEpisodes 4byte */
	p->ActualEpisodes = *(long *) (text + 16);
	/* read FileStartDate 4byte */
	p->FileStartDate = *(long *) (text + 20);
	/* read FileStartTime 4byte */
	p->FileStartTime = *(long *) (text + 24);
	/* read StopwatchTime 4byte */
	p->StopwatchTime = *(long *) (text + 28);
	/* read HeaderVersionNumber 4byte */
	p->HeaderVersionNumber = *(float *) (text + 32);
	/* read FileType 2byte */
	p->FileType = *(short *) (text + 36);
	/* read MSBinFormat 2byte */
	p->MSBinFormat = *(short *) (text + 38);
}

void
Print_FileID_Size_Info(FILE * fp, FileID_Size_Info * p)
{
	fprintf(fp, "File ID and Size information\n");
	fprintf(fp, "offset 0 -> 39\n");
	fprintf(fp, "\n");
	fprintf(fp, "FileSignature, %s\n", p->FileSignature);
	fprintf(fp, "FileVersionNumber %f\n", p->FileVersionNumber);
	fprintf(fp, "OperationMode %d\n", p->OperationMode);
	fprintf(fp, "ActualAcqLength %ld\n", p->ActualAcqLength);
	fprintf(fp, "NumPointsIgnored %d\n", p->NumPointsIgnored);
	fprintf(fp, "ActualEpisodes %ld\n", p->ActualEpisodes);
	fprintf(fp, "FileStartDate %ld\n", p->FileStartDate);
	fprintf(fp, "FileStartTime %ld\n", p->FileStartTime);
	fprintf(fp, "StopwatchTime %ld\n", p->StopwatchTime);
	fprintf(fp, "HeaderVersionNumber %f\n", p->HeaderVersionNumber);
	fprintf(fp, "FileType %d\n", p->FileType);
	fprintf(fp, "MSBinFormat %d\n", p->MSBinFormat);
	fprintf(fp, "\n");
}

void
Read_File_Structure(char *text, File_Structure * p)
{
	p->DataSectionPtr = *(long *) (text + 40);
	p->TagSectionPtr = *(long *) (text + 44);
	p->NumTagEntries = *(long *) (text + 48);
	p->ScopeConfigPtr = *(long *) (text + 52);
	p->NumScopes = *(long *) (text + 56);
	p->DACFilePtr = *(long *) (text + 60);
	p->DACFileNumEpisodes = *(long *) (text + 64);
	p->DeltaArrayPtr = *(long *) (text + 72);
	p->NumDeltas = *(long *) (text + 76);
	p->SynchArrayPtr = *(long *) (text + 92);
	p->SynchArraySize = *(long *) (text + 96);
	p->DataFormat = *(short *) (text + 100);
}

void
Print_File_Structure(FILE * fp, File_Structure * p)
{
	fprintf(fp, "File File Structure\n");
	fprintf(fp, "offset 39 -> 119\n");
	fprintf(fp, "\n");
	fprintf(fp, "DataSectionPtr %ld\n", p->DataSectionPtr);
	fprintf(fp, "TagSectionPtr %ld\n", p->TagSectionPtr);
	fprintf(fp, "NumTagEntries %ld\n", p->NumTagEntries);
	fprintf(fp, "ScopeConfigPtr %ld\n", p->ScopeConfigPtr);
	fprintf(fp, "NumScopes %ld\n", p->NumScopes);
	fprintf(fp, "DACFilePtr %ld\n", p->DACFilePtr);
	fprintf(fp, "DACFileNumEpisodes %ld\n", p->DACFileNumEpisodes);
	fprintf(fp, "DeltaArrayPtr %ld\n", p->DeltaArrayPtr);
	fprintf(fp, "NumDeltas %ld\n", p->NumDeltas);
	fprintf(fp, "SynchArrayPtr %ld\n", p->SynchArrayPtr);
	fprintf(fp, "SynchArraySize %ld\n", p->SynchArraySize);
	fprintf(fp, "DataFormat %d\n", p->DataFormat);
	fprintf(fp, "\n");
}

void
Read_Trial_Hierarchy_Info(char *text, Trial_Hierarchy_Info * p)
{
	p->ADCNumChannels = *(short *) (text + 120);
	p->ADCSampleInterval = *(float *) (text + 122);
	p->ADCSecondSampleInterval = *(float *) (text + 126);
	p->SynchTimeUnit = *(float *) (text + 130);
	p->SecondsPerRun = *(float *) (text + 134);
	p->NumSamplesPerEpisode = *(long *) (text + 138);
	p->PreTriggerSamples = *(long *) (text + 142);
	p->EpisodesPerRun = *(long *) (text + 146);
	p->RunsPerTrial = *(long *) (text + 150);
	p->NumberOfTrials = *(long *) (text + 154);
	p->AveragingMode = *(short *) (text + 158);
	p->UndoRunCount = *(short *) (text + 160);
	p->FirstEpisodeInRun = *(short *) (text + 162);
	p->TriggerThreshold = *(float *) (text + 164);
	p->TriggerSource = *(short *) (text + 168);
	p->TriggerAction = *(short *) (text + 170);
	p->TriggerPolarity = *(short *) (text + 172);
	p->ScopeOutputInterval = *(float *) (text + 174);
	p->EpisodeStartToStart = *(float *) (text + 178);
	p->RunStartToStart = *(float *) (text + 182);
	p->TrialStartToStart = *(float *) (text + 186);
	p->AverageCount = *(long *) (text + 190);
	p->ClockChange = *(long *) (text + 194);
	p->AutoTriggerStrategy = *(short *) (text + 198);
}

void
Print_Trial_Hierarchy_Info(FILE * fp, Trial_Hierarchy_Info * p)
{
	fprintf(fp, "Trial Hierarchy Information\n");
	fprintf(fp, "offset 120 -> 199\n");
	fprintf(fp, "\n");
	fprintf(fp, "ADCNumChannels %d\n", p->ADCNumChannels);
	fprintf(fp, "ADCSampleInterval%f\n", p->ADCSampleInterval);
	fprintf(fp, "ADCSecondSampleInterval %f\n", p->ADCSecondSampleInterval);
	fprintf(fp, "SynchTimeUnit %f\n", p->SynchTimeUnit);
	fprintf(fp, "SecondsPerRun %f\n", p->SecondsPerRun);
	fprintf(fp, "NumSamplesPerEpisode %ld\n", p->NumSamplesPerEpisode);
	fprintf(fp, "PreTriggerSamples %ld\n", p->PreTriggerSamples);
	fprintf(fp, "EpisodesPerRun %ld\n", p->EpisodesPerRun);
	fprintf(fp, "RunsPerTrial %ld\n", p->RunsPerTrial);
	fprintf(fp, "NumberOfTrials %ld\n", p->NumberOfTrials);
	fprintf(fp, "AveragingMode %d\n", p->AveragingMode);
	fprintf(fp, "UndoRunCount %d\n", p->UndoRunCount);
	fprintf(fp, "FirstEpisodeInRun %d\n", p->FirstEpisodeInRun);
	fprintf(fp, "TriggerThreshold %f\n", p->TriggerThreshold);
	fprintf(fp, "TriggerSource %d\n", p->TriggerSource);
	fprintf(fp, "TriggerAction %d\n", p->TriggerAction);
	fprintf(fp, "TriggerPolarity %d\n", p->TriggerPolarity);
	fprintf(fp, "ScopeOutputInterval %f\n", p->ScopeOutputInterval);
	fprintf(fp, "EpisodeStartToStart %f\n", p->EpisodeStartToStart);
	fprintf(fp, "RunStartToStart %f\n", p->RunStartToStart);
	fprintf(fp, "TrialStartToStart%f\n", p->TrialStartToStart);
	fprintf(fp, "AverageCount %ld\n", p->AverageCount);
	fprintf(fp, "ClockChange %ld\n", p->ClockChange);
	fprintf(fp, "AutoTriggerStrategy %d\n", p->AutoTriggerStrategy);
	fprintf(fp, "\n");
}

void
Read_Display_Parameters(char *text, Display_Parameters * p)
{
	p->DrawingStrategy = *(short *) (text + 200);
	p->TiledDisplay = *(short *) (text + 202);
	p->EraseStrategy = *(short *) (text + 204);
	p->DataDisplayMode = *(short *) (text + 206);
	p->DisplayAverageUpdate = *(long *) (text + 208);
	p->ChannelStatsStratgy = *(short *) (text + 212);
	p->CalculationPeriod = *(long *) (text + 214);
	p->SamplesPerTrace = *(long *) (text + 218);
	p->StartDisplayNum = *(long *) (text + 222);
	p->FinishDisplayNum = *(long *) (text + 226);
	p->MultiColor = *(short *) (text + 230);
	p->ShowPNRawData = *(short *) (text + 232);
	p->StatisticsPeriod = *(float *) (text + 234);
}

void
Print_Display_Parameters(FILE * fp, Display_Parameters * p)
{
	fprintf(fp, "Trial Display Parameters\n");
	fprintf(fp, "offset 200 -> 243\n");
	fprintf(fp, "\n");
	fprintf(fp, "DrawingStrategy %d\n", p->DrawingStrategy);
	fprintf(fp, "TiledDisplay %d\n", p->TiledDisplay);
	fprintf(fp, "EraseStrategy %d\n", p->EraseStrategy);
	fprintf(fp, "DataDisplayMode %d\n", p->DataDisplayMode);
	fprintf(fp, "DisplayAverageUpdate %ld\n", p->DisplayAverageUpdate);
	fprintf(fp, "ChannelStatsStratgy %d\n", p->ChannelStatsStratgy);
	fprintf(fp, "CalculationPeriod %ld\n", p->CalculationPeriod);
	fprintf(fp, "SamplesPerTrace %ld\n", p->SamplesPerTrace);
	fprintf(fp, "StartDisplayNum %ld\n", p->StartDisplayNum);
	fprintf(fp, "FinishDisplayNum %ld\n", p->FinishDisplayNum);
	fprintf(fp, "MultiColor %d\n", p->MultiColor);
	fprintf(fp, "ShowPNRawData %d\n", p->ShowPNRawData);
	fprintf(fp, "StatisticsPeriod %f\n", p->StatisticsPeriod);
	fprintf(fp, "\n");
}

void
Read_Hardware_Info(char *text, Hardware_Info * p)
{
	p->ADCRange = *(float *) (text + 244);
	p->DACRange = *(float *) (text + 248);
	p->ADCResolution = *(long *) (text + 252);
	p->DACResolution = *(long *) (text + 256);
}

void
Print_Hardware_Info(FILE * fp, Hardware_Info * p)
{
	fprintf(fp, "Trial Hardware Information\n");
	fprintf(fp, "offset 244 -> 259\n");
	fprintf(fp, "\n");
	fprintf(fp, "ADCRange %f\n", p->ADCRange);
	fprintf(fp, "DACRange %f\n", p->DACRange);
	fprintf(fp, "ADCResolution %ld\n", p->ADCResolution);
	fprintf(fp, "DACResolution %ld\n", p->DACResolution);
	fprintf(fp, "\n");
}

void
Read_Environmental_Info(char *text, Environmental_Info * p)
{
	p->ExperimentType = *(short *) (text + 260);
	p->AutosampleEnable = *(short *) (text + 262);
	p->AutosampleADCNum = *(short *) (text + 264);
	p->AutosampleInstrument = *(short *) (text + 266);
	p->AutosampleAdditGain = *(float *) (text + 268);
	p->AutosampleFilter = *(float *) (text + 272);
	p->AutosampleMembraneCap = *(float *) (text + 276);
	p->ManualInfoStrategy = *(short *) (text + 280);
	p->CellID1 = *(float *) (text + 282);
	p->CellID2 = *(float *) (text + 286);
	p->CellID3 = *(float *) (text + 290);
	strncpy(p->CreatorInfo, (char *) (text + 294), 16);
	strncpy(p->FileComment, (char *) (text + 310), 56);
}

void
Print_Environmental_Info(FILE * fp, Environmental_Info * p)
{
	fprintf(fp, "Environmental Information\n");
	fprintf(fp, "offset 260 -> 377\n");
	fprintf(fp, "\n");
	fprintf(fp, "ExperimentType %d\n", p->ExperimentType);
	fprintf(fp, "AutosampleEnable %d\n", p->AutosampleEnable);
	fprintf(fp, "AutosampleADCNum %d\n", p->AutosampleADCNum);
	fprintf(fp, "AutosampleInstrument %d\n", p->AutosampleInstrument);
	fprintf(fp, "AutosampleAdditGain %f\n", p->AutosampleAdditGain);
	fprintf(fp, "AutosampleFilter %f\n", p->AutosampleFilter);
	fprintf(fp, "AutosampleMembraneCap %f\n", p->AutosampleMembraneCap);
	fprintf(fp, "ManualInfoStrategy %d\n", p->ManualInfoStrategy);
	fprintf(fp, "CellID1 %f\n", p->CellID1);
	fprintf(fp, "CellID2 %f\n", p->CellID2);
	fprintf(fp, "CellID3 %f\n", p->CellID3);
	fprintf(fp, "CreatorInfo %s\n", p->CreatorInfo);
	fprintf(fp, "FileComment %s\n", p->FileComment);
	fprintf(fp, "\n");
}

void
Read_Multi_Channel_Info(char *text, Multi_Channel_Info * p)
{
	int             i;
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		p->ADCPtoLChannelMap[i] = *(short *) (text + 378 + i * 2);
		p->ADCSamplingSeq[i] = *(short *) (text + 410 + i * 2);
		strncpy(p->ADCChannelName[i], (char *) (text + 442 + (i * ABF_ADCNAMELEN)), ABF_ADCNAMELEN);
		p->ADCChannelName[i][ABF_ADCNAMELEN - 1] = '\0';
		strncpy(p->ADCUnits[i], (char *) (text + 602 + (i * ABF_ADCUNITLEN)), ABF_ADCUNITLEN);
		p->ADCUnits[i][ABF_ADCUNITLEN - 1] = '\0';
		p->ADCProgrammableGain[i] = *(float *) (text + 730 + i * 4);
		p->ADCDisplayAmplification[i] = *(float *) (text + 794 + i * 4);
		p->ADCDisplayOffset[i] = *(float *) (text + 858 + i * 4);
		p->InstrumentScaleFactor[i] = *(float *) (text + 922 + i * 4);
		p->InstrumentOffset[i] = *(float *) (text + 986 + i * 4);
		p->SignalGain[i] = *(float *) (text + 1050 + i * 4);
		p->SignalOffset[i] = *(float *) (text + 1114 + i * 4);
		p->SignalLowpassFilter[i] = *(float *) (text + 1178 + i * 4);
		p->SignalHighpassFilter[i] = *(float *) (text + 1242 + i * 4);
	}
	for (i = 0; i < ABF_DACCOUNT; i++) {
		strncpy(p->DACChannelName[i], (char *) (text + 1306 + i * ABF_DACNAMELEN), ABF_DACNAMELEN);
		p->DACChannelName[i][ABF_DACNAMELEN - 1] = '\0';
		strncpy(p->DACChannelUnits[i], (char *) (text + 1346 + i * ABF_DACUNITLEN), ABF_DACUNITLEN);
		p->DACChannelUnits[i][ABF_DACUNITLEN - 1] = '\0';
		p->DACScaleFactor[i] = *(float *) (text + 1378 + i * 4);
		p->DACHoldingLevel[i] = *(float *) (text + 1394 + i * 4);
	}
	p->SignalType = *(short *) (text + 1410);
}

void
Print_Multi_Channel_Info(FILE * fp, Multi_Channel_Info * p)
{
	int             i;
	fprintf(fp, "ADC channel information:\n");

	fprintf(fp, "No.  ADCChannelName ADCUnits\n");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%.2d %15s %12s\n",
			i,
			p->ADCChannelName[i], p->ADCUnits[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "No. InstrumentOffset  ADCDisplayOffset  SignalOffset\n");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%.2d %15.4f %15.4f %15.4f\n",
			i,
			p->InstrumentOffset[i],
			p->ADCDisplayOffset[i],
			p->SignalOffset[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "   InstrumentScaleFactor       ADCProgrammableGain\n");
	fprintf(fp, "No.         ADCDisplayAmplification           SignalGain\n");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%.2d %12.4f %12.4f %12.4f %12.4f\n",
			i,
			p->InstrumentScaleFactor[i],
			p->ADCDisplayAmplification[i],
			p->ADCProgrammableGain[i],
			p->SignalGain[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "No.  fSignalLowpassFilter  fSignalHighpassFilter\n");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%.2d %18.4f %18.4f\n", i,
			p->SignalLowpassFilter[i],
			p->SignalHighpassFilter[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "ADCPtoLChannelMap");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%3d", p->ADCPtoLChannelMap[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "ADCSamplingSeq");
	for (i = 0; i < ABF_ADCCOUNT; i++) {
		fprintf(fp, "%3d", p->ADCSamplingSeq[i]);
	}
	fprintf(fp, "\n");

	fprintf(fp, "DAC channel information:\n");
	fprintf(fp, "No. sDACChannelName  sDACChannelUnits  fDACScaleFactor  fDACHoldingLevel\n");
	for (i = 0; i < ABF_DACCOUNT; i++) {
		fprintf(fp, "%d %15s %14s %20.4f %15.4f\n", i,
			p->DACChannelName[i],
			p->DACChannelUnits[i],
			p->DACScaleFactor[i],
			p->DACHoldingLevel[i]);
	}
	fprintf(fp, "\n");
	fprintf(fp, "SignalType %d\n", p->SignalType);
	fprintf(fp, "\n");
}

void
Read_Synchronous_Timer_Output(char *text, Synchronous_Timer_Output * p)
{
}

void
Print_Synchronous_Timer_Output(FILE * fp, Synchronous_Timer_Output * p)
{
}

void
Read_Epoch_Waveform_Pulse(char *text, Epoch_Waveform_Pulse * p)
{
}

void
Print_Epoch_Waveform_Pulse(FILE * fp, Epoch_Waveform_Pulse * p)
{
}

void
Read_DAC_Output_File(char *text, DAC_Output_File * p)
{
	p->DACFileScale = *(float *) (text + 1620);
	p->DACFileOffset = *(float *) (text + 1624);
	p->DACFileEpisodeNum = *(short *) (text + 1630);
	p->DACFileADCNum = *(short *) (text + 1632);

	strncpy(p->DACFileName, (char *) (text + 1634), 12);
	strncpy(p->DACFilePath, (char *) (text + 1646), 60);
}

void
Print_DAC_Output_File(FILE * fp, DAC_Output_File * p)
{
	fprintf(fp, "DAC Output Filenvironmental Information\n");
	fprintf(fp, "offset 1620 -> 1717\n");
	fprintf(fp, "\n");
	fprintf(fp, "DACFileScale %f\n", p->DACFileScale);
	fprintf(fp, "DACFileOffset %f\n", p->DACFileOffset);
	fprintf(fp, "DACFileEpisodeNum %d\n", p->DACFileEpisodeNum);
	fprintf(fp, "DACFileADCNum %d\n", p->DACFileADCNum);
	fprintf(fp, "DACFileName %s\n", p->DACFileName);
	fprintf(fp, "DACFilePath %s\n", p->DACFilePath);
	fprintf(fp, "\n");
}
