/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: matlab2satellite.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef WIN32
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#include "matfile.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct {
  int type;    /* Computer Type */
  int rows;    /* Number of Data rows */
  int cols;    /* Number of Data columns */
  int imgf;    /* Flag Complex ( 0 = REAL , 1 = COMPLEX ) */
  int namelen; /* Length of Name for Data */
} MAT_Header ; /* Matlab Version 4 format */

typedef struct {
  char fname[64];
  int dim,index[10],length;
  Buffer *data;
} Satellite_Data;


int isBigEndian(int type)
{
  if (type > 9999 || type < 0)
    return 1;
  else return 0;
}

void swap_data( char *data, int nbyte)
{
  int i;
  char temp;
  for(i=0;i<nbyte/2;i++){
    temp=data[i];
    data[i]=data[nbyte-i-1];
    data[nbyte-i-1]=temp;
  }
}

DLLEXPORT int mod_dcm_matlab2satellite()
{
  char *fname;
  FILE *fp;
  unsigned short     ver[2];
  char matinfo[128];
  unsigned int  item[2];
  char name[128],pad[128];
  Satellite_Data sd;

  /* Get File Name */
  fname=GetString(0);
  if(fname==NULL)
    return (1);

  /* Input Matlab File Open */
  fp=fopen(fname,"rb");
  if(fp==NULL)
    return (2);

  // read header infomation
  fread(&matinfo, matVERSION_INFO_OFFSET, 1, fp);
  fread(&ver, sizeof(ver), 1, fp);
  // Matrix structute declare
  fread(&item, sizeof(item), 1, fp);
  // arrayFlags
  fread(&item, sizeof(item), 1, fp);
  fread(&item, sizeof(item), 1, fp);
  printf("array[0]:%x array[1]:%x\n",item[0],item[1]);
  // read dimension
  fread(&item, sizeof(item), 1, fp);
  fread(&item, sizeof(item), 1, fp);
  printf("row:%x column:%x\n",item[0],item[1]);
  sd.dim = item[0];
  sd.index[0] = item[1];
  sd.index[1] = item[0];
  // read name
  int nAlignBytes,i,nBytes;
  char ch;

  fread(&item, sizeof(item), 1, fp);
  nBytes = item[0] >> 16;
  if(nBytes) {
	  for(i = 0; i < nBytes; i++) {
		  ch = item[1] & 0xff;
		  name[i] = ch;
		  item[1] = item[1] >> 8;
	  }
	  name[i] = '\0';
  }
  else {
	  fread(&name, sizeof(char), item[1], fp);
	  nAlignBytes = matINT64_ALIGN(item[1]) - item[1];
	  if(nAlignBytes)
		fread(&name, sizeof(char), nAlignBytes, fp);
  }
  strcpy(sd.fname, name);
  // read data
  fread(&item, sizeof(item), 1, fp);
  sd.length = sd.index[0] * sd.index[1];
  sd.data=AllocBuffer(sd.length);
  fread(sd.data, sizeof(double), sd.length, fp);
  fclose(fp);

  /* Satellite File Object Data Output */
  ChangeDataSize(8);
  printf("Output Satellite File Object : \"%s\"\n",sd.fname);
  _WriteFile(sd.fname, sd.dim, sd.index, (char *) sd.data);
  FreeBuffer(sd.data);

  return 0;
}
#ifdef __cplusplus
}
#endif

#else		/*#ifdef WIN32*/

#include <stdio.h>
#include <string.h>
#include "SL_macro.h"
#include "SL_cmd.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef struct {
  int type;    /* Computer Type */
  int rows;    /* Number of Data rows */
  int cols;    /* Number of Data columns */
  int imgf;    /* Flag Complex ( 0 = REAL , 1 = COMPLEX ) */
  int namelen; /* Length of Name for Data */
} MAT_Header ; /* Matlab Version 4 format */

typedef struct {
  char fname[64];
  int dim,index[10],length;
  Buffer *data;
} Satellite_Data;


int isBigEndian(int type)
{
  if (type > 9999 || type < 0)
    return 1;
  else return 0;
}

void swap_data( char *data, int nbyte)
{
  int i;
  char temp;
  for(i=0;i<nbyte/2;i++){
    temp=data[i];
    data[i]=data[nbyte-i-1];
    data[nbyte-i-1]=temp;
  }
}

int read_mat_file_header(FILE *fp,MAT_Header *mat,int *swap)
{
  *swap=0;
  fread(&(mat->type),4,1,fp);
  fread(&(mat->rows),4,1,fp);
  fread(&(mat->cols),4,1,fp);
  fread(&(mat->imgf),4,1,fp);
  fread(&(mat->namelen),4,1,fp);
  if(feof(fp))  return 1;
  if ( isBigEndian(mat->type) && mat->type != 0 ){
    *swap=1;
    swap_data((char *) &(mat->type),4);
    swap_data((char *) &(mat->rows),4);
    swap_data((char *) &(mat->cols),4);
    swap_data((char *) &(mat->imgf),4);
    swap_data((char *) &(mat->namelen),4);
  }
#ifdef _DEBUG_DCM
  printf("type : %d\n",mat->type);
  printf("rows : %d\n",mat->rows);
  printf("cols : %d\n",mat->cols);
  printf("imgf : %d\n",mat->imgf);
  printf("len  : %d\n",mat->namelen);
#endif
  return 0;
}

DLLEXPORT int mod_dcm_matlab2satellite()
{
  register int i,j,k;
  int swap,type,prec,order,mach,n,m;
  char *fname;
  FILE *fp;
  char tmp[64];
  union {
    double d;
    int  i;
    float  f;
    short  sh;
    unsigned short  us;
    unsigned char c;
  } temp ;

  MAT_Header mat;
  Satellite_Data sd;

  /* Get File Name */
  fname=GetString(0);
  if(fname==NULL)
    return (1);

  /* Input Matlab File Open */
  fp=fopen(fname,"r");
  if(fp==NULL)
    return (2);

  while(!read_mat_file_header(fp,&mat,&swap)){
    type = mat.type % 10;  /* Full, sparse, etc */
                           /* 1=string */
    mat.type /= 10;
    prec = mat.type % 10;  /* double, float, int, short, ushort, uchar */
    mat.type /= 10;
    order = mat.type % 10; /* Row or column major ordering */
    mat.type /= 10;
    mach = mat.type % 10;  /* Endian type */
                           /* 0=little, 1=big, 2=vax_d, 3=vax_g, 4=cray */
#ifdef _DEBUG_DCM
  printf("data : %d\n",type);
  printf("prec : %d\n",prec);
  printf("order : %d\n",order);
  printf("mach : %d\n",mach);
#endif
    if ( prec == 5 )
      return (4); /* No Supported String Data */
    if ( type!=0 && type!=1 )
      return (4); /* can't read sparse matrices */
    if ( mach!=0 && mach!=1 )
      return (4); /* No Supported Endian type */
    if ( order == 0 ){
      m=mat.cols;
      n=mat.rows;
    } else {
      n=mat.cols;
      m=mat.rows;
    }

    fread(tmp,mat.namelen,1,fp);

    sd.length=mat.rows*mat.cols;
    sd.dim=2;
    sd.index[0]=mat.rows;
    sd.index[1]=mat.cols;

    for(k=0;k<mat.imgf+1;k++){
      sd.data=AllocBuffer(sd.length);
      if(mat.imgf){ /* If Complex Data */
	if(k==0){
	  sprintf(sd.fname,"mat_%s_r.dat",tmp);
	} else {
	  sprintf(sd.fname,"mat_%s_i.dat",tmp);
	}
      } else {
	sprintf(sd.fname,"mat_%s.dat",tmp);
      }
      for(i=0;i<n;i++){
	for(j=0;j<m;j++){
	  switch(prec){
	  case 0: /* double 8 */
	    fread(&temp.d,sizeof(double),1,fp);
	    if(swap)swap_data((char *)&temp.d,8);
	    sd.data[i*m+j]=temp.d;
	    break;
	  case 1: /* float 4 */
	    fread(&temp.f,sizeof(float),1,fp);
	    if(swap)swap_data((char *)&temp.f,4);
	    sd.data[i*m+j]=temp.f;
	    break;
	  case 2: /* int 4 */
	    fread(&temp.i,sizeof(int),1,fp);
	    if(swap)swap_data((char *)&temp.i,4);
	    sd.data[i*m+j]=temp.i;
	    break;
	  case 3: /* short 2 */
	    fread(&temp.sh,sizeof(short),1,fp);
	    if(swap)swap_data((char *)&temp.sh,2);
	    sd.data[i*m+j]=temp.sh;
	    break;
	  case 4: /* unsigned short 2 */
	    fread(&temp.us,sizeof(unsigned short),1,fp);
	    if(swap)swap_data((char *)&temp.us,2);
	    sd.data[i*m+j]=temp.us;
	    break;
	  }
#ifdef _DEBUG_DCM
	  printf("sd.data[i*m+j]=%f\n",sd.data[i*m+j]);
#endif
	}
      }
      /* Satellite File Object Data Output */
      ChangeDataSize(8);
      printf("Output Satellite File Object : \"%s\"\n",sd.fname);
      _WriteFile(sd.fname, sd.dim, sd.index, (char *) sd.data);
      FreeBuffer(sd.data);
    }
#ifdef _DEBUG_DCM
  printf("OK!\n");
#endif
  }
  fclose(fp);
  return 0;
}
#ifdef __cplusplus
}
#endif
#endif
