/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: teac2satellite.cpp,v 1.3 2004/08/16 05:03:46 orrisroot Exp $ */
/*
  <><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>
<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>
<><><>                                                                  <><><>
<><>     DR-M2a Format to SATELLITE-Lang FileObject Datafile converter    <><>
<><>     ( Original source by Mr.Suzuki (DR-M2a -> Canopus command)        <><>
<><>                                                                       <><>
<><><>                               Modified by okumura  1999.12.21    <><><>
<><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>
  <><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><><>
*/
#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include "SL_macro.h"
#include "SL_cmd.h"

#define PATHLENGTH  80
#define LINELENGTH  255
#define CH_NUM    8       /* Max channel size */

#ifdef __cplusplus
extern "C" {
#endif

/*
===============================================================================
  Header proto type definition
===============================================================================
*/
typedef struct
{
  char     fname_base[PATHLENGTH];
  int      num_series;
  double   slope[CH_NUM];
  double   y_offset[CH_NUM];
  long     num_samps;
  char     name_ch[CH_NUM][PATHLENGTH];
  double   samp_freq;
  int      input_range;
} DR_hdr;

/*
===============================================================================
   for LittleEndian to BigEndian conversion
===============================================================================
*/
typedef union
{
  short  sval;
  char  cval[2];
} char_short;

/*
===============================================================================
   read from header file functions
    - each value is set to 'DR_hdr' header struct
===============================================================================

===============================================================================
   NUM_SERIES: record channel number
===============================================================================
*/
static void RH_num_series( DR_hdr *hdr, char *buf)
{
  char      str1[LINELENGTH];
  int       num;
  
  sscanf( buf, "%s %d", str1, &num );
  hdr->num_series = num;
}

/* 
===============================================================================
   SLOPE
===============================================================================
*/
static void RH_slope( DR_hdr *hdr, char *buf )
{
  char      str1[LINELENGTH];
  double    slope;
  int       i;
  
  sscanf( buf, "%s %lf", str1, &slope );
  hdr->slope[0] = slope;
  for ( i = 1; i < hdr->num_series; i++ )
  {
    sscanf( buf, ",%lf", &slope );
    hdr->slope[i] = slope;
  }
}

/* 
===============================================================================
   Y_OFFSET
===============================================================================
*/
static void RH_yoffset( DR_hdr *hdr, char *buf )
{
  char  str1[LINELENGTH];
  double    y_offset;
  int     i;
  
  sscanf( buf, "%s %lf", str1, &y_offset );
  hdr->y_offset[0] = y_offset;
  for ( i = 1; i < hdr->num_series; i++ )
  {
    sscanf( buf, ",%lf", &y_offset );
    hdr->y_offset[i] = y_offset;
  }
}

/* 
===============================================================================
   NUM_SAMPS
===============================================================================
*/
static void RH_samps( DR_hdr *hdr, char *buf )
{
  char  str1[LINELENGTH];
  long  samps;

  sscanf( buf, "%s %ld", str1, &samps );
  hdr->num_samps = samps;
}

/* 
===============================================================================
   CH-? part of signal name (max 16 chars)
===============================================================================
*/
static void RH_name_ch( DR_hdr *hdr, char *buf, int ch_num )
{
    char      str1[LINELENGTH];
    char    name_ch[LINELENGTH];

    sscanf( buf, "%s %s", str1, name_ch );
    strcpy( hdr->name_ch[ch_num - 1], name_ch );
}

/* 
===============================================================================
   sampling frequency : 1Hz - 100kHz  (step 1, 2, 5)
===============================================================================
*/
static void RH_freq( DR_hdr *hdr, char *buf )
{
  char      str1[LINELENGTH];
  char      str2[LINELENGTH];
  double    freq;
  
  sscanf( buf, "%s %s", str1, str2 );
  if ( strstr( str2, "MHz" ) != NULL )
  {
    str2[strlen( str2 ) - strlen( "MHz" )] = '\0';
    sscanf( str2, "%lf\n", &freq );
    hdr->samp_freq = freq * 1000000.0;
  }
  else if ( strstr( str2, "KHz" ) != NULL )
  {
    str2[strlen( str2 ) - strlen( "KHz" )] = '\0';
    sscanf( str2, "%lf\n", &freq );
    hdr->samp_freq = freq * 1000.0;
  }
  else if ( strstr( str2, "Hz" ) != NULL )
  {
    str2[strlen( str2 ) - strlen( "Hz" )] = '\0';
    sscanf( str2, "%lf\n", &freq );
    hdr->samp_freq = freq;
  }
}

/* 
===============================================================================
   voltage range of analog input ( 1, 2, 5, 10 V)
===============================================================================
*/
static void RH_range( DR_hdr *hdr, char *buf )
{
  char  str1[LINELENGTH];
  char  str2[LINELENGTH];
  int   range;

  sscanf( buf, "%s %s", str1, str2 );
  str2[strlen( str2 ) - strlen( "V" )] = '\0';
  sscanf( str2, "%d", &range );
  hdr->input_range = range;
}

/*
===============================================================================
  read header and  assignment to header struct
   must be call this function, before set base of filename to "hdr.fname_base"
===============================================================================
*/
static int hdr_ReadHeader( DR_hdr *hdr )
{
  FILE  *fp;
  char   filename[PATHLENGTH];
  char   buf[LINELENGTH + 1];
  char   key[LINELENGTH + 1];

  strcpy( filename, hdr->fname_base ); /* create file name */
  strcat( filename, ".hdr" );

  if ( ( fp = fopen( filename, "r" ) ) == NULL ) /* file open */
  {
    return ( 2 );
  }

  while ( fgets( buf, LINELENGTH, fp ) != NULL )
  {
    sscanf( buf, "%s", key );

    if ( strcmp( key, "NUM_SERIES" ) == 0 )
    {
      RH_num_series( hdr, buf );
      continue;
    }
    if ( strcmp( key, "SLOPE" ) == 0 )
    {
      RH_slope( hdr, buf );
      continue;
    }
    if ( strcmp( key, "Y_OFFSET" ) == 0 )
    {
      RH_yoffset( hdr, buf );
      continue;
    }
    if ( strcmp( key, "NUM_SAMPS" ) == 0 )
    {
      RH_samps( hdr, buf );
      continue;
    }
    if ( strcmp( key, "NAME_CH1" ) == 0 )
    {
      RH_name_ch( hdr, buf, 1 );
      continue;
    }
    if ( strcmp( key, "NAME_CH2" ) == 0 )
    {
      RH_name_ch( hdr, buf, 2 );
      continue;
    }
    if ( strcmp( key, "NAME_CH3" ) == 0 )
    {
      RH_name_ch( hdr, buf, 3 );
      continue;
    }
    if ( strcmp( key, "NAME_CH4" ) == 0 )
    {
      RH_name_ch( hdr, buf, 4 );
      continue;
    }
    if ( strcmp( key, "NAME_CH5" ) == 0 )
    {
      RH_name_ch( hdr, buf, 5 );
      continue;
    }
    if ( strcmp( key, "NAME_CH6" ) == 0 )
    {
      RH_name_ch( hdr, buf, 6 );
      continue;
    }
    if ( strcmp( key, "NAME_CH7" ) == 0 )
    {
      RH_name_ch( hdr, buf, 7 );
      continue;
    }
    if ( strcmp( key, "NAME_CH8" ) == 0 )
    {
      RH_name_ch( hdr, buf, 8 );
      continue;
    }
    if ( strcmp( key, "SAMPLE_FREQUENCY" ) == 0 )
    {
      RH_freq( hdr, buf );
      continue;
    }
    if ( strcmp( key, "INPUT_RANGE" ) == 0 )
    {
      RH_range( hdr, buf );
      continue;
    }
  }
  fclose( fp );
  return 0;
}

/*
===============================================================================
   function of Endian converter (for BigEndian CPU architecture, ex. sparc)
===============================================================================
*/
#ifdef WORDS_BIGENDIAN
static short chg_endian( short sval )
{
  char_short  le, be;

  le.sval = sval;

  be.cval[0] = le.cval[1];
  be.cval[1] = le.cval[0];
  
  return be.sval;
}
#endif

/*
===============================================================================
  read data
===============================================================================
*/
static float *hdr_ReadData_f( DR_hdr *hdr, int *ret )
{
  long  i;
  int    j;
  char  filename[PATHLENGTH];
  FILE  *fp;
  float  *buf;
  short      buf2[CH_NUM];
  
  if ( ( buf = (float*)malloc( hdr->num_series * hdr->num_samps * 
                               sizeof( float ) ) ) == NULL ){
    *ret = 8 ; /* memory allocate */
  }
  
  strcpy( filename, hdr->fname_base );
  strcat( filename, ".dat" ); /* create file name */
  
  if ( ( fp = fopen( filename,"rb" ) ) == NULL ){
      *ret = 2 ; /* file open */
  }

  for( i = 0L; i < hdr->num_samps; i++ )
    {
      fread( buf2, sizeof( short ), hdr->num_series, fp );
      for ( j = 0; j < hdr->num_series; j++ )
        {
#ifdef WORDS_BIGENDIAN
          buf[i + hdr->num_samps * j] = (float)(chg_endian( buf2[j] ) * 
            hdr->slope[j] + hdr->y_offset[j] * hdr->input_range * 
            hdr->input_range);
#else
          buf[i + hdr->num_samps * j] = (float)(buf2[j] * hdr->slope[j]
            + hdr->y_offset[j] * hdr->input_range * hdr->input_range);
#endif
        }
    }
  fclose( fp ); /* file close */
  *ret = 0;
  return buf; /* return all cannels */
}

/*
===============================================================================
  main function
===============================================================================
*/
DLLEXPORT int mod_dcm_teac2satellite( void )
{
  char   *filename;
  char    distname[PATHLENGTH];
  DR_hdr  hdr;
  int     i;
  float  *data;
  char   *direction;
  int     dflag = 1;
  int     index[MAX_INDEX];
  int     stat;

  if ( ( filename = GetString( 0 ) ) == NULL )
    {
      return ( 2 );
    }
  direction = (char *)GetString( 1 );
  if( direction != NULL ){
    switch( direction[0] ){
    case 'T':
      dflag = 0;
      break;
    case 'D':
      dflag = 1;
      break;
    default:
      dflag = 1;
      break;
    }
  }
  
  strcpy( hdr.fname_base, filename );
  stat = hdr_ReadHeader( &hdr ); /* read header info */
  if(stat != 0) return stat;
  data = hdr_ReadData_f( &hdr, &stat ); /* read data */
  if(stat != 0) return stat;
  for ( i = 0; i < hdr.num_series; i++ )
    { /* save each channels */
      strcpy( distname, filename );
      strcat( distname, "_" );            /* create file name */
      strcat( distname, hdr.name_ch[i] );
      strcat( distname, ".dat" );
      if( dflag == 0 ){
        _WriteFile( distname, 1, (int *)&hdr.num_samps, 
                    (char *)&data[hdr.num_samps * i] );
      }else{
        index[0] = 1;
        index[1] = hdr.num_samps;
        _WriteFile( distname, 2, index,
                    (char *)&data[hdr.num_samps * i] );
      }
    }
  free(data);

  return 0;
}
#ifdef __cplusplus
}
#endif
