/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: gpm2file.cpp,v 1.2 2004/05/18 09:14:53 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif
#ifdef HAVE_DIRECT_H
# include <direct.h>
#endif

#include <gtk/gtk.h>

#include "SL_macro.h"
#include "SL_cmd.h"
#include "GPMdef.h"
#include "GPMwin.h"
#include "GpmWindow.h"
#include "GpmThread.h"
#include "gpm2file.h"

#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH MAXPATHLEN
# else
#  define _MAX_PATH 1024
# endif
#endif

#define PS_STACK_MAX 100
#define DVI_BLUE	0
#define DVI_GREEN	1
#define	DVI_RED		2
#define	AREA_SCALE	1.2		/*1.2*/

extern	char	gpm_head[];	/* defined in gpmdvi.cpp */
static void wmf_plot(FILE *fd, double x, double y, int mode);
int		old_mode;
static unsigned int times = 0;
static unsigned int maxpath = 0;
extern int psmode;
FILE *ps_fd = NULL;
/* defined in psfunc.cpp for rainbow level */
extern int	ps_rainbow_flag;

int GpmWindow::GPM_gpm2file()
{
	int		winno;		//window no
	char	*fname;		//output name
	char	*ftype;		//file type
	char	*ctype;		//color type
	int		errcode;
	char	dvifile[FILE_LENGTH];
	char	dvino[16];

	winno = (int)GetScalar(0);
	if(winno < 1 || winno > 10)
		return 35;
	//Check GPMDVIFILE
	sprintf(dvino, "GPMDVIFILE%d",winno);
	getcwd(dvifile, FILE_LENGTH);
#ifdef WIN32
	strcat( dvifile, "\\" );
#else
	strcat( dvifile, "/" );
#endif
	strcat(dvifile, dvino);
	
	if(Access(dvifile,SL_FATTR_FOK) != 0)
		return 33;
	fname = GetString(1);
	if(!fname)
		return 2;
	ftype = GetString(2);
	// ftype="ps",ftype="wmf"
	if(strcmp(ftype,"ps") != 0 && strcmp(ftype,"wmf") != 0)
		return 2;

	ctype = GetString(3);
	// ctype="color",ctype="gray",ctype="bw"
	if(strcmp(ctype,"color") == 0)
		gpm_colorps_mode = 1;
	else if(strcmp(ctype,"gray") == 0)
		gpm_colorps_mode = 2;
	else if(strcmp(ctype,"bw") == 0)
		gpm_colorps_mode = 0;	// Black & White mode
	else
		return 2;

	// gpm_label initialized
	gpm_label.ichar = NULL;

#ifdef WIN32
	if(_stricmp(ftype, "ps") == 0) {
#else
	if(strcmp(ftype, "ps") == 0) {
#endif
		if(errcode = create_gpm2ps(winno, fname)) {
			if(gpm_label.ichar)
				free(gpm_label.ichar);
			return(errcode);
		}
	}
	else {
		if(errcode = create_gpm2wmf(winno, fname)) {
			if(gpm_label.ichar)
				free(gpm_label.ichar);
			return(errcode);
		}
	}

	if(gpm_label.ichar)
		free(gpm_label.ichar);
	return 0;
}

int GpmWindow::create_gpm2ps(int winno, char* fname)
{
	int     num;
	int     result;
	int     div     = RAINBOW_DIVNUM;
	int     basecol = RAINBOW_BASECOL;
	int     rollflag = 0;
	int     lwidth, ltype;
	int     pageNumber = 1, mode;
	int     paperSize, orientation, device;
	int     trayflag = 0;
	char	argv0[16];
	int		lncnt;
        int     stat;

  /* it exist dvifile */
  mode = 0;

  strcpy(argv0, "gpm2ps");
  opendvi_for_read();
  if ( readdvihead() == -1 ) {
    //fprintf( stderr, "%s: %s is not GPM dvi file.\n\7", argv[0], gpm_dvifile );
    return 1;
  }
  
  lseek( dvi_fd, strlen(gpm_head) + 5L, 0 );


  paperSize   = mdev.number  = GpmCont.paper;
  orientation = mdev.irotate = GpmCont.orientation;
  device      = GpmCont.device;

  getlength(mdev.number, &mgraph.xmax, &mgraph.ymax, mdev.irotate);
  
  mgraph.xmin = 0.0;
  mgraph.ymin = 0.0;
  
  // create ps
  ps_fd = fopen(fname, "w");
  if(ps_fd == NULL)
	  return(44);

  ps_initAdobe((int)paperSize, (int)orientation, argv0, GpmCont.dvifile/*gpm_dvifile*/ );
  if(ps_init())
	  return(45);
  if (trayflag) {
    ps_setTray((int)paperSize, (int)orientation);
  }
  ps_page(pageNumber);
  ps_open();
  ps_rainbow_mode(mode);
  
  do {
    
    result = readgpm();
    
    if ( rollflag == 1 && result > 0 ) {
      pageNumber++;
      ps_page(pageNumber);
    }

    if (result != 0) {
      
      ps_ginit();
      ps_factor((double)gpm_param.factor);
      ps_origin((double)gpm_param.xorg, (double)gpm_param.yorg);

lncnt = 0;
      do {
	num = readdvi();
	
	if (num > 0) {
	  
	  switch (num) {
	    
	  case NPLOT:
	    if (gpm_plot.rainbow <= 0)
	      ps_color((int) gpm_plot.color);
	    else
	      ps_rainbow((double)(gpm_plot.rainbow)/(div*basecol));
	    
	    lwidth = (int) gpm_plot.lwidth;
	    ltype  = (int) gpm_plot.ltype;
	    
	    ps_lwidth(lwidth);
	    ps_ltype(ltype);

lncnt++;
	    ps_plot(gpm_plot.x, gpm_plot.y, (int)gpm_plot.ipen_mode);
	    rollflag = 0;
	    break;

	  case NBOX:
	    lwidth = (int) gpm_box.lwidth;
	    ltype = (int) gpm_box.ltype;
	    
	    ps_lwidth(lwidth);
	    ps_ltype(ltype);
	    
	    ps_box(gpm_box.x1,gpm_box.y1,gpm_box.x2,gpm_box.y2,gpm_box.mode,
		   gpm_box.level,(int)gpm_box.color);
	    rollflag = 0;
	    break;

	  case NCSYMBOL:
	    ps_color((int) gpm_csymbol.color);
	    ps_lwidth((int) gpm_csymbol.lwidth);
	    ps_csymbol(gpm_csymbol.x, gpm_csymbol.y, (int)gpm_csymbol.cdata,
		       gpm_csymbol.height);
	    rollflag = 0;
	    break;

	  case NLABEL:
	    ps_color((int) gpm_label.color);
	    ps_string(gpm_label.x,gpm_label.y,gpm_label.ichar,gpm_label.height,
		      gpm_label.theta, gpm_label.mode, gpm_label.font);
	    rollflag = 0;
	    break;

	  case NROLL:
	    ps_roll();
	    rollflag = 1;
	    break;
	  }
	}
      } while (num > 0);
    }
  } while (result > 0);
  
  stat = closedvi();
  if(stat != 0) return stat;
  
  if (rollflag == 0)
    ps_roll();
  
  ps_pages(pageNumber);
  ps_close();
  fclose(ps_fd);
  ps_fd = NULL;
	return 0;
}

int GpmWindow::create_gpm2wmf(int winno, char* fname)
{
	int     paperSize, orientation, device;
	int     num;
	int     result;
	int     rollflag = 0;
	int     lwidth, ltype;
	int     div     = RAINBOW_DIVNUM;
	int     basecol = RAINBOW_BASECOL;
	int     pageNumber = 1;
	FILE	*wmf_fd;
	double	bxmin,bymin,bxmax,bymax,xorg,yorg;
	int flag;
	int	npcnt=0,nbcnt=0,nccnt=0,nlcnt=0;
	double wsize,xoffset,yoffset;
	int     stat;
	char	metafile[_MAX_PATH];
	double	area_xmin,area_ymin,area_xmax,area_ymax;

	// initialize
	old_mode = 0;

	// read dvifile and create drawing list
	opendvi_for_read();
	if ( readdvihead() == -1 ) {
          return 1;
	}

	lseek( dvi_fd, strlen(gpm_head) + 5L, 0 );

	paperSize   = mdev.number  = GpmCont.paper;
	orientation = mdev.irotate = GpmCont.orientation;
	device      = GpmCont.device;

	getlength(mdev.number, &mgraph.xmax, &mgraph.ymax, mdev.irotate);

	mgraph.xmin = 0.0;
	mgraph.ymin = 0.0;

	// create tempolary
	sprintf(metafile,"%s/metafile.dat", get_tmpdir());
	wmf_fd = fopen(metafile, "w");
	if(wmf_fd == NULL) {
		// failed to open metafile.dat
		return(44);
	}

	/* initialzed boundary infomation */
	xorg = gpm_param.xorg;
	yorg = gpm_param.yorg;
	bxmin = bymin = 1e6;
	bxmax = bymax = 0;
	flag = 0;
	ps_rainbow_mode(0);

	// analized dvifile
	do {

	result = readgpm();

	if ( rollflag == 1 && result > 0 ) {
	  pageNumber++;
	  wmf_page(wmf_fd, pageNumber);
	  /* initialzed boundary infomation */
	  xorg = gpm_param.xorg;
	  yorg = gpm_param.yorg;
	  bxmin = bymin = 1e6;
	  bxmax = bymax = 0;
	}

	if (result != 0) {
  
	  wmf_ginit(wmf_fd);
	  wmf_factor(wmf_fd, (double)gpm_param.factor);
	  wmf_origin(wmf_fd,(double)gpm_param.xorg, (double)gpm_param.yorg);
	  // changed origin
	  xorg = gpm_param.xorg;
	  yorg = gpm_param.yorg;

	  do {
		num = readdvi();

		if (num > 0) {
  			switch (num) {
		
			case NPLOT:
				// plot function
			if (gpm_plot.rainbow <= 0)
				wmf_color(wmf_fd, (int) gpm_plot.color);
			else
				wmf_rainbow(wmf_fd, (double)(gpm_plot.rainbow)/(div*basecol));
			
			lwidth = (int) gpm_plot.lwidth;
			ltype  = (int) gpm_plot.ltype;
			wmf_lwidth(wmf_fd, lwidth);
			wmf_ltype(wmf_fd, ltype);

			wmf_plot(wmf_fd, gpm_plot.x, gpm_plot.y, (int)gpm_plot.ipen_mode);
			rollflag = 0;
			/* boundary information */
			if(bxmin > (xorg+gpm_plot.x))
				bxmin = (xorg+gpm_plot.x);
				if(bxmin < 0)
					flag = 1;
			if(bymin > (yorg+gpm_plot.y))
				bymin = (yorg+gpm_plot.y);
				if(bymin < 0)
					flag = 1;
			if(bxmax < (xorg+gpm_plot.x))
				bxmax = (xorg+gpm_plot.x);
			if(bymax < (yorg+gpm_plot.y))
				bymax = (yorg+gpm_plot.y);
npcnt++;
			break;

			case NBOX:
			lwidth = (int) gpm_box.lwidth;
			ltype = (int) gpm_box.ltype;
			wmf_lwidth(wmf_fd, lwidth);
			wmf_ltype(wmf_fd, ltype);
			wmf_box(wmf_fd, gpm_box.x1,gpm_box.y1,gpm_box.x2,gpm_box.y2,gpm_box.mode,
				gpm_box.level,(int)gpm_box.color);
			/* boundary information */
			if(bxmin > (xorg+gpm_box.x1))
				bxmin = (xorg+gpm_box.x1);
				if(bxmin < 0)
					flag = 1;
			if(bymin > (yorg+gpm_box.y1))
				bymin = (yorg+gpm_box.y1);
				if(bymin < 0)
					flag = 1;
			if(bxmax < (xorg+gpm_box.x1))
				bxmax = (xorg+gpm_box.x1);
			if(bymax < (yorg+gpm_box.y1))
				bymax = (yorg+gpm_box.y1);
			if(bxmin > (xorg+gpm_box.x2))
				bxmin = (xorg+gpm_box.x2);
				if(bxmin < 0)
					flag = 1;
			if(bymin > (yorg+gpm_box.y2))
				bymin = (yorg+gpm_box.y2);
				if(bymin < 0)
					flag = 1;
			if(bxmax < (xorg+gpm_box.x2))
				bxmax = (xorg+gpm_box.x2);
			if(bymax < (yorg+gpm_box.y2))
				bymax = (yorg+gpm_box.y2);
nbcnt++;
			break;

			case NCSYMBOL:
			wmf_color(wmf_fd, (int) gpm_csymbol.color);
			wmf_lwidth(wmf_fd, (int) gpm_csymbol.lwidth);
			wmf_csymbol(wmf_fd, gpm_csymbol.x, gpm_csymbol.y, (int)gpm_csymbol.cdata,
				gpm_csymbol.height);
			rollflag = 0;
			if(bxmin > (xorg+gpm_csymbol.x))
				bxmin = (xorg+gpm_csymbol.x);
				if(bxmin < 0)
					flag = 1;
			if(bymin > (yorg+gpm_csymbol.y))
				bymin = (yorg+gpm_csymbol.y);
				if(bymin < 0)
					flag = 1;
			if(bxmax < (xorg+gpm_csymbol.x))
				bxmax = (xorg+gpm_csymbol.x);
			if(bymax < (yorg+gpm_csymbol.y))
				bymax = (yorg+gpm_csymbol.y);
nccnt++;
			break;

			case NLABEL:
			wmf_color(wmf_fd, (int) gpm_label.color);
			if(gpm_label.ilen) {
				wsize = getstrwidth(gpm_label.ichar,gpm_label.height, msymbol.font);
				wmf_string(wmf_fd,gpm_label.x,gpm_label.y,gpm_label.ichar,gpm_label.height,
					  gpm_label.theta, gpm_label.mode, (int)gpm_label.font, wsize);
				rollflag = 0;
				yoffset=wsize*sin(2*3.1415926*gpm_label.theta/360);
				xoffset=wsize*cos(2*3.1415926*gpm_label.theta/360);
				if(bxmin > (xorg+gpm_label.x+xoffset))
					bxmin = (xorg+gpm_label.x+xoffset);
					if(bxmin < 0)
						flag = 1;
				if(bymin > (yorg+gpm_label.y+yoffset))
					bymin = (yorg+gpm_label.y+yoffset);
					if(bymin < 0)
						flag = 1;
				if(bxmax < (xorg+gpm_label.x+xoffset))
					bxmax = (xorg+gpm_label.x+xoffset);
				if(bymax < (yorg+gpm_label.y+yoffset))
					bymax = (yorg+gpm_label.y+yoffset);
			}
nlcnt++;
			break;

			case NROLL:
			wmf_area(wmf_fd, bxmin, bymin, bxmax, bymax);
			/* ps_roll() ;*/
			wmf_roll(wmf_fd);
			rollflag = 1;
			break;
			}
		}
	} while (num > 0);
	}	//if (result != 0)
	} while (result > 0);

	stat = closedvi();
        if(stat != 0) return stat;

	if (rollflag == 0) {
          wmf_area(wmf_fd, bxmin, bymin, bxmax, bymax);
          /*ps_roll();*/
          wmf_roll(wmf_fd);
	}
	fclose(wmf_fd);

	//
	// create wmf from drawing list
	//

	// Unix enviroment
	if(MakeAMetaFile()) {
		MakeAPlaceableMetaFile( fname );
	}
	return 0;
}

/***************************
 *        wmf_area           *
 ***************************/

void GpmWindow:: wmf_area(FILE *fd, double xmin, double ymin, 
						  double xmax, double ymax)
{
	fprintf(fd,"area %f %f %f %f\n",xmin,ymin,xmax,ymax);
}


/***************************
 *        wmf_plot           *
 ***************************/

void GpmWindow:: wmf_plot(FILE *fd, double x, double y, int mode)
{
  static double x1, y1, x0=0.0, y0=0.0;
  
  x1 = x;
  y1 = y;

  if ( ( psmode == -1 && mode != 0 ) || mode < 0 ) {
    times++; maxpath++; psmode = 0;
    x0 = mgraph.xpre;
    y0 = mgraph.ypre;
	fprintf(fd,"moveto %f %f %d \n",x0,y0,mode);
  } 

  if ( mode >= 0 ) {
    mgraph.xpre = x;
    mgraph.ypre = y;
  }
  switch(mode) {
  case -1: case 1:
    if ( maxpath < PS_STACK_MAX ) {
      if ( fabs( x0 - x1 ) > 0.02  || fabs( y0 - y1 ) > 0.02 )  {
		fprintf(fd,"lineto %f %f %d\n",x,y,mode);
		times   ++; maxpath ++;
		psmode = 1;
      }
      break; /* this line must be this position */
    } 
  case -2: case 2:
	fprintf(fd,"lineto %f %f %d\n",x,y,mode);
    maxpath = 0; times = 0; psmode = -1;
    break;
  case 3:
	fprintf(fd,"lineto %f %f %d\n",x,y,mode);
    maxpath = 0; times = 0; psmode = -1;
    break;
  case 0:
  default:
	fprintf(fd,"moveto %f %f %d \n",x1,y1,mode);
    times   ++; maxpath ++;
    psmode = 0;
    break;
  }
  if ( (times+1) % 5 == 0 ) { times = 0; /*puts("")*/; }

  if ( mode < 0 ) {
	fprintf(fd,"moveto %f %f %d \n",x0,y0,mode);
    psmode = 0; times ++; maxpath++;
    if ( (times+1) % 5 == 0 ) { times = 0; /*puts("")*/; }
  }
}

/***************************
 *        wmf_lwidth       *
 ***************************/
void GpmWindow::wmf_lwidth(FILE *fd, int width)
{
	int lwidth;
	//lwidth = width-1;
	lwidth = width;
	if(lwidth < 0)
		lwidth = 0;
	fprintf(fd,"setlinewidth %f\n",(double)lwidth);
}


/***************************
 *        wmf_ltype        *
 ***************************/

void GpmWindow::wmf_ltype(FILE *fd, int type)
{
	fprintf(fd,"setdash %d\n",type);
}

/***************************
 *        wmf_page         *
 ***************************/

void GpmWindow::wmf_page(FILE *fd, int pageNumber)
{
	fprintf(fd, "page %d\n", pageNumber);
}

/***************************
 *        wmf_ginit        *
 ***************************/

void GpmWindow::wmf_ginit(FILE *fd)
{
	fprintf(fd, "ginit\n");
}

/***************************
 *        ps_factor         *
 ***************************/

void GpmWindow::wmf_factor(FILE *fd, double factor)
{
	fprintf(fd, "scale %f\n", factor);
}

/***************************
 *        ps_origin         *
 ***************************/

void GpmWindow::wmf_origin(FILE *fd, double x, double y)
{
	fprintf(fd, "origin %f %f\n",x,y);
}

/***************************
 *        ps_color          *
 ***************************/

void GpmWindow::get_color(int color, double rgb[])
{
  static double  gray[16] = {1.0, 0.8, 0.6, 0.3, 0.0, 0.0, 0.0, 0.0,
			     0.3, 0.6, 0.8, 1.0, 1.0, 1.0, 1.0, 1.0};
  static double   red[8]  = {1.0, 0.0, 1.0, 1.0, 0.0, 0.0, 1.0, 0.0};
  static double green[8]  = {1.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 0.0};
  static double  blue[8]  = {1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 0.0, 0.0};
  int index;
  
  //if (color != gpm_dev.color || gpm_dev.level != -1.0 ) {

  if ( color <= 0 ) {
	  rgb[0] = 0.0; rgb[1] = 0.0; rgb[2] = 0.0;
	  return;
  }

    if ( gpm_colorps_mode == 1 && color <= 8 ) {
		switch(color) {
		case	1:	// black
			index = 7;	break;
		case	2:	// blue
			index = 1;	break;
		case	3:	// red
			index = 2;	break;
		case	4:	// magenta
			index = 3;	break;
		case	5:	// green
			index = 4;	break;
		case	6:	// cyan
			index = 5;	break;
		case	7:	// yellow
			index = 6;	break;
		case	8:	// white
			index = 0;	break;
		}
		rgb[DVI_BLUE] = blue[index];
		rgb[DVI_GREEN] = green[index];
		rgb[DVI_RED] = red[index];
    } else {
		index = color % 16;
		rgb[DVI_BLUE] = gray[index];
		rgb[DVI_GREEN] = gray[index];
		rgb[DVI_RED] = gray[index];
	}
  //}
}

// create setrgbcolor command
void GpmWindow::wmf_color(FILE *fd, int color)
{
  double rgb[3];
  get_color(color, rgb);
  fprintf(fd,"setrgbcolor %d %d %d\n",
	(int)(255*rgb[DVI_BLUE]),
	(int)(255*rgb[DVI_GREEN]),
	(int)(255*rgb[DVI_RED]));
}

/***************************
 *        ps_rainbow        *
 ***************************/

GdkColor GpmWindow::get_rainbow(double level)
{
  int rainbow;
  int div     = RAINBOW_DIVNUM;
  int basecol = RAINBOW_BASECOL;
  int  RainbowMax = RAINBOW_DIVNUM * RAINBOW_BASECOL+2;
  GdkColor c;
  
  if ((level >= 0.0) && (level <= 1.0)) {
    rainbow = (int) (level*(double)(div * basecol));
  } else
    rainbow = (level < 0.0) ? 0 : div*basecol;
 
  if ( rainbow >= 0 && rainbow < RainbowMax ) {
    c = gpm_rainbow[rainbow];
  } else {
	c = ( rainbow < 0 ) ? gpm_rainbow[0] : gpm_rainbow[RainbowMax-1];
  }
  return c;
}

void GpmWindow::wmf_rainbow(FILE *fd, double level)
{
  GdkColor c;
  
  c = get_rainbow(level);
  fprintf(fd,"setrgbcolor %d %d %d\n",
	  c.blue,c.green,c.red);
}

/***************************
 *        ps_box           *
 ***************************/

void GpmWindow::wmf_box(FILE *fd, double x1, double y1, double x2, double y2,
	   int mode, double level, int color)
{
  double rgb[3];
  GdkColor cl;

  if (mode == 3) {
	get_color( color, rgb );
	fprintf(fd,"box %f %f %f %f %d %d %d %d\n",
			x1,y1,x2,y2,mode,
			(int)(255*rgb[DVI_RED]),
			(int)(255*rgb[DVI_GREEN]),
			(int)(255*rgb[DVI_BLUE]));
  }
  else if (mode == 2) {
	if      (level < 0.0) level = 0.0;
	else if (level > 1.0) level = 1.0;

	if (!gpm_colorps_mode && ps_rainbow_flag )
		level = 1.0 - level;
	cl = get_rainbow(level);
	fprintf(fd,"box %f %f %f %f %d %d %d %d\n",
			x1,y1,x2,y2,mode, cl.blue, cl.green, cl.red);
  } else {
	wmf_color(fd, color);

	fprintf(fd,"box %f %f %f %f %d %f %d\n",
			x1,y1,x2,y2,mode,level,color);
  }
}

#define FONT_MAX 12
void GpmWindow::wmf_string(FILE *fd, double x, double y, char *str, double height, double theta,
		  int centermode, int font, double cx)
{
    if (gpm_colorps_mode != 1)
      wmf_color(fd, 7);		//
	fprintf(fd,"label %f %f %s %f %f %d %d %f\n",
		x,y,"####",height,theta,centermode,font,cx);
	fprintf(fd,"%s\n",str);
}

/***************************
 *                          *
 *        ps_symbl          *
 *                          *
 ***************************/
void GpmWindow::wmf_csymbol(FILE *fd, double x, double y, int cdata, double height)
{
	fprintf(fd,"csymbol %f %f %d %f\n",x,y,cdata,height);
}

/***************************
 *                          *
 *        ps_roll           *
 *                          *
 ***************************/
void GpmWindow::wmf_roll(FILE *fd)
{
	fprintf(fd,"roll\n");
}

//
// Create Meatafile in Unix
//
#define	SCALE 22.7272727272
#define	PEN_SCALE	8.5
unsigned long	color_rgb;	//refered from other functions
unsigned long	file_size;
unsigned long	max_record;
double wmf_width,wmf_height;
short	max_hndl,save_hndl;


int MakeAPlaceableMetaFile(char *filename)
{
	METAHEADER	mt;
	METAFILEHEADER	mh;
	int *wp;
	FILE *fp;
	int	i,length;
	int sum;
	short *buffer;
	char targetfile[_MAX_PATH];

	mh.key = 0x9AC6CDD7;
	mh.hmf = 0x0000;
	mh.left = 0;
	mh.top = 0;
	mh.right = (short)((wmf_width*AREA_SCALE) * SCALE);
	mh.bottom = (short)((wmf_height*AREA_SCALE) * SCALE);
	mh.inch = 576;
	mh.reserved = 0x0;

	// checksum
	sprintf(targetfile, "%s/target.met", get_tmpdir());
	fp = fopen(targetfile,"rb");
	//length = fread(&mt, sizeof(char), sizeof(mt), fp);
	fread(&mt.mtType, sizeof(unsigned short), 1, fp);
	fread(&mt.mtHeaderSize, sizeof(unsigned short), 1, fp);
	fread(&mt.mtVersion, sizeof(unsigned short), 1, fp);
	fread(&mt.mtSize, sizeof(unsigned long), 1, fp);
	fread(&mt.mtNoObjects, sizeof(unsigned short), 1, fp);
	fread(&mt.mtMaxRecord, sizeof(unsigned long), 1, fp);
	fread(&mt.mtNoParameters, sizeof(unsigned short), 1, fp);
	wp = (int *)(&mt);
	sum = 0;
	for(i = 0; i < 5; i++) {
		sum ^= *wp;
		wp++;
	}
	mh.checksum = sum;
	fclose(fp);

	length = sizeof(mh);
	// read
	buffer = (short *)malloc(mt.mtSize*sizeof(short));
	sprintf(targetfile, "%s/target.met", get_tmpdir());
	fp = fopen(targetfile,"rb");
	length = fread(buffer, sizeof(char), mt.mtSize*2, fp);
	fclose(fp);

	// write
	fp = fopen(filename/*"metafile.wmf"*/,"wb");
	//fp = fopen("c:\\work\\projects\\debug\\Sample\\metafile.wmf","wb");
	length = fwrite(&mh, sizeof(char), 22, fp);
	length = fwrite(buffer, sizeof(char), mt.mtSize * 2, fp);
	fclose(fp);
	free(buffer);

	return TRUE;
}

int MetaCreatePen(FILE* ofp, unsigned short type, unsigned short width, unsigned long color)
{
	unsigned long size;
	unsigned short code,tmp;

	size = 9;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_CREATEPENINDIRECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&type, sizeof(unsigned short), 1, ofp);
	fwrite(&width, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	fwrite(&color, sizeof(unsigned long), 1, ofp);
	tmp = 0x22;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	if(size > max_record)
		max_record = size;
	file_size += size;
	//Handle
	return 0;
}

int MetaSelectObject(FILE* ofp, unsigned short object)
{
	unsigned long size;
	unsigned short code,tmp;

	size = 4;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_SELECTOBJECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&object, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
	//Handle
	tmp = save_hndl;
	save_hndl = object;
	return tmp;
}
int MetaCreateSolidBrush(FILE* ofp, unsigned long color)
{
	unsigned long size;
	unsigned short code,tmp;

	size = 7;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_CREATEBRUSHINDIRECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	tmp = 0;	//style
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	fwrite(&color, sizeof(unsigned long), 1, ofp);
	tmp = 0;	//kind
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
	//Handler
	return 0;
}

void MetaFillRect(FILE* ofp, unsigned long op, unsigned short height,
				   unsigned short width, unsigned short y, unsigned short x)
{
	unsigned long size;
	unsigned short code;

	size = 9;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_PATBLT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&op, sizeof(unsigned long), 1, ofp);
	fwrite(&height, sizeof(unsigned short), 1, ofp);
	fwrite(&width, sizeof(unsigned short), 1, ofp);
	fwrite(&y, sizeof(unsigned short), 1, ofp);
	fwrite(&x, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaEllipse(FILE* ofp, unsigned short left, unsigned short top,
				   unsigned short right, unsigned short bottom)
{
	unsigned long size;
	unsigned short code;

	size = 7;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_ELLIPSE;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&bottom, sizeof(unsigned short), 1, ofp);
	fwrite(&right, sizeof(unsigned short), 1, ofp);
	fwrite(&top, sizeof(unsigned short), 1, ofp);
	fwrite(&left, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaArc(FILE* ofp, unsigned short left, unsigned short top,
				   unsigned short right, unsigned short bottom,
				   unsigned short xs, unsigned short ys,
				   unsigned short xe, unsigned short ye)
{
	unsigned long size;
	unsigned short code;

	size = 11;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_ARC;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&ye, sizeof(unsigned short), 1, ofp);
	fwrite(&xe, sizeof(unsigned short), 1, ofp);
	fwrite(&ys, sizeof(unsigned short), 1, ofp);
	fwrite(&xs, sizeof(unsigned short), 1, ofp);
	fwrite(&bottom, sizeof(unsigned short), 1, ofp);
	fwrite(&right, sizeof(unsigned short), 1, ofp);
	fwrite(&top, sizeof(unsigned short), 1, ofp);
	fwrite(&left, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaPolygon(FILE* ofp, int ptx[], int pty[], unsigned short count)
{
	unsigned long size;
	unsigned short i,code,x,y;

	size = 3 + 1 + 2*count;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_POLYGON;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&count, sizeof(unsigned short), 1, ofp);
	for(i = 0; i < count; i++) {
		x = ptx[i];
		fwrite(&x, sizeof(unsigned short), 1, ofp);
		y = pty[i];
		fwrite(&y, sizeof(unsigned short), 1, ofp);
	}

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaPolyline(FILE* ofp, int ptx[], int pty[], unsigned short count)
{
	unsigned long size;
	unsigned short i,code,x,y;

	size = 3 + 1 + 2*count;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_POLYLINE;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&count, sizeof(unsigned short), 1, ofp);
	for(i = 0; i < count; i++) {
		x = ptx[i];
		fwrite(&x, sizeof(unsigned short), 1, ofp);
		y = pty[i];
		fwrite(&y, sizeof(unsigned short), 1, ofp);
	}

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaDeleteObject(FILE* ofp, unsigned short object)
{
	unsigned long size;
	unsigned short code;

	size = 4;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_DELETEOBJECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&object, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaMoveToEx(FILE* ofp, unsigned short y, unsigned short x)
{
	unsigned long size;
	unsigned short code;

	size = 5;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_MOVETO;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&y, sizeof(unsigned short), 1, ofp);
	fwrite(&x, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}
void MetaLineTo(FILE* ofp, unsigned short y, unsigned short x)
{
	unsigned long size;
	unsigned short code;

	size = 5;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_LINETO;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&y, sizeof(unsigned short), 1, ofp);
	fwrite(&x, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaSetTextColor(FILE* ofp, unsigned long color)
{
	unsigned long size;
	unsigned short code;

	size = 5;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_SETTEXTCOLOR;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&color, sizeof(unsigned long), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

#define	FONT_CHANGE		0
int MetaCreateFont(FILE* ofp,int height,int width,int escape,int orient,int weight, int font)
{
	unsigned long size;
	unsigned short code,tmp,Weight;
	static char    *fontname[3] =
		{"Times", "Helvetica", "Courier"};
	unsigned char Italic,Underline,StrikeOut,CharSet;
	unsigned char OutPrecision,ClipPrecision,Quality,PitchAndFamily;
	int i,len,index;
	char ch;

	/*Weight*/
	if((font % 4) >= 2)
		Weight = FW_BOLD;
	else
		Weight = FW_NORMAL;
	/*italic*/
	if((font % 2) == 1)
		Italic = 1;
	else
		Italic = 0;
	/*underline*/
	Underline = 0;
	/*StrikeOut*/
	StrikeOut = 0;
	/*CharSet*/
	CharSet = 0;
	/*OutPrecision*/
	OutPrecision = 0;
	/*ClipPrecision*/
	ClipPrecision = 0;
	/*quality*/
	Quality = 0;
	/*PitchAndFamily*/
	PitchAndFamily = 0;

	/* calculate size */
	size = 12;
	index = font / 4;
	len = strlen(fontname[index]);
	len++;	/* '\0' */
	if((len % 2) == 1) {
			len++;
	}
	else {
			len += 2;
	}
	size = size + len / 2;

	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_CREATEFONTINDIRECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	tmp = (unsigned short)height;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = (unsigned short)width;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = (unsigned short)escape;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = (unsigned short)orient;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = Weight;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	fwrite(&Italic, 1, 1, ofp);
	fwrite(&Underline, 1, 1, ofp);
	fwrite(&StrikeOut, 1, 1, ofp);
	fwrite(&CharSet, 1, 1, ofp);
	fwrite(&OutPrecision, 1, 1, ofp);
	fwrite(&ClipPrecision, 1, 1, ofp);
	fwrite(&Quality, 1, 1, ofp);
	fwrite(&PitchAndFamily, 1, 1, ofp);

	index = font / 4;
	len = strlen(fontname[index]);
	for(i = 0; i < len; i++) {
		ch = fontname[index][i];
		fwrite(&ch, 1, 1, ofp);
	}
	ch = 0x0;	/*NULL*/
	fwrite(&ch, 1, 1, ofp);
	len++;
	if((len % 2) == 1) {
		ch = 0x1;
		fwrite(&ch, 1, 1, ofp);
		len++;
	}
	else {
		tmp = 0x101;
		fwrite(&tmp, sizeof(unsigned short), 1, ofp);
		len += 2;
	}

	if(size > max_record)
		max_record = size;
	file_size += size;
	//Handler
	return 0;
}
//size:13
//CreateFontIndirect 2fb
//0 0 0 0 0 0 8000 0 100 8000
void MetaCreateFontAdd(FILE* ofp)
{
	unsigned long size;
	unsigned short code,tmp;

	size = 13;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_CREATEFONTINDIRECT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
		//0 8000 0 100
		tmp = 0x0;
		fwrite(&tmp, sizeof(unsigned short), 1, ofp);
		tmp = 0x8000;
		fwrite(&tmp, sizeof(unsigned short), 1, ofp);
		tmp = 0x0;
		fwrite(&tmp, sizeof(unsigned short), 1, ofp);
		tmp = 0x100;
		fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	//8000
	tmp = 0x8000;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

//03 00 00 00 00 00
void MetaDummyData(FILE* ofp)
{
	unsigned long size;
	unsigned short tmp;

	size = 3;
	tmp = 0x3;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0x0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);
	tmp = 0x0;
	fwrite(&tmp, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}

void MetaTextOut(FILE* ofp, unsigned short len, char* label,
				 unsigned short yorg, unsigned short xorg)
{
	unsigned long size;
	unsigned short code;
	unsigned short tlen;
	unsigned char* text;
	int i;
	unsigned char ch;

	tlen = len;
	if(tlen % 2)
		tlen += 1;
	tlen /= 2;
	size = 6 + tlen;
	fwrite(&size, sizeof(unsigned long), 1, ofp);
	code = META_TEXTOUT;
	fwrite(&code, sizeof(unsigned short), 1, ofp);
	fwrite(&len, sizeof(unsigned short), 1, ofp);
	text = (unsigned char*)malloc(tlen*2);
	memset(text, 0, tlen*2);
	for(i = 0; i < len; i++)
		text[i] = label[i];
	for(i = 0; i < tlen*2; i++) {
		ch = text[i];
		fwrite(&ch, sizeof(unsigned char), 1, ofp);
	}
	fwrite(&yorg, sizeof(unsigned short), 1, ofp);
	fwrite(&xorg, sizeof(unsigned short), 1, ofp);

	if(size > max_record)
		max_record = size;
	file_size += size;
}


//
// Create target.met
//
int gMetaPen, gMetaFont, gMetaBrush, gMetaSymBrush;
int bMetaPen, bMetaFont, bMetaBrush, bMetaSymBrush;

//int gMetaFontTmp;

int MetaHandle()
{
	int ahndl,hndl;
	ahndl = gMetaPen + gMetaFont + gMetaBrush + gMetaSymBrush;
	if(ahndl == -4)
		hndl = 0;
	else if(ahndl == -3)
		hndl = 1;
	else if(ahndl == -1)
		hndl = 2;
	else if(ahndl == 2)
		hndl = 3;
	else
		hndl = 4;
	if(max_hndl < hndl)
		max_hndl = hndl;
	return hndl;
}

int MakeAMetaFile()
{
	METAHEADER mh;
	unsigned short hOldMetaBrush,hOldMetaPen;
	FILE	*fp;
	unsigned short	xs,ys,xe,ye,cx,cy,left,right,top,bottom;
	unsigned short	mode,xorg=0,yorg=0;
	double	fxs,fys,fxe,fye,fheight,ftheta,flength;
	char	linebuf[256],cmd[64],str[128];
	unsigned short	font,center,type,color,lwidth,dash,dash_code,line_style;
	unsigned short	color_r,color_g,color_b;
	double	lwidth_code;
	unsigned long	color_rgb;
	FILE	*ofp;
	double	wxs,wys,wxe,wye,width,height,wsize;
	int box_mode;
	char metafile[_MAX_PATH],targetfile[_MAX_PATH];

	//initialize
	file_size = 0x9;
	max_record = 0x0;
	max_hndl = 0;
	save_hndl = 0;
	//Traget.met
	sprintf(targetfile, "%s/target.met", get_tmpdir());
	ofp = fopen(targetfile,"wb");
	if(ofp == NULL)
		return FALSE;
	// deteced area parameter from metafile.dat
	sprintf(metafile, "%s/metafile.dat", get_tmpdir());
	fp = fopen(metafile, "r");
	if(fp == NULL) {
		return FALSE;
	}
	while(!feof(fp)) {
		if(fgets(linebuf, 256, fp) == NULL)
			break;
		sscanf(linebuf,"%s",cmd);	// drawing command
		if(strcmp(cmd, "area") == 0) { //area
			sscanf(linebuf,"%s %lg %lg %lg %lg",cmd,&wxs,&wys,&wxe,&wye);
			// set boundary
			width = wxe - wxs;
			wmf_width = width;
			height = wye - wys;
			wmf_height = height;
		}
	}
	fclose(fp);

	//METAHEADER
	memset(&mh, 0, sizeof(mh));
	fwrite(&mh.mtType, sizeof(unsigned short), 1, ofp);
	fwrite(&mh.mtHeaderSize, sizeof(unsigned short), 1, ofp);
	fwrite(&mh.mtVersion, sizeof(unsigned short), 1, ofp);
	fwrite(&mh.mtSize, sizeof(unsigned long), 1, ofp);
	fwrite(&mh.mtNoObjects, sizeof(unsigned short), 1, ofp);
	fwrite(&mh.mtMaxRecord, sizeof(unsigned long), 1, ofp);
	fwrite(&mh.mtNoParameters, sizeof(unsigned short), 1, ofp);
	//fwrite(&mh, sizeof(char), sizeof(mh), ofp);

	// set device
	//gMetaFontTmp = -1;
	gMetaFont = gMetaPen = gMetaBrush = gMetaSymBrush = -1;
	bMetaFont = bMetaPen = bMetaBrush = bMetaSymBrush = 0;
	dash = 0;
	line_style = PS_SOLID;
	lwidth = (unsigned short)(PEN_SCALE);
	color = 0;
	color_rgb = (unsigned long) 0x00000000;
	//CreatePen
	MetaCreatePen(ofp, PS_SOLID, lwidth, color_rgb);
	if(gMetaPen == -1)
		gMetaPen = MetaHandle();
	//SelectObject
	hOldMetaPen = MetaSelectObject(ofp, gMetaPen);
	//CreateSolidBrush
	MetaCreateSolidBrush( ofp, 0x00FFFFFF );
	if(gMetaBrush == -1) {
		gMetaBrush = MetaHandle();
		bMetaBrush = 1;
	}
	//SelectObject
	hOldMetaBrush = MetaSelectObject(ofp, gMetaBrush);
	cx = (unsigned short)(width*AREA_SCALE*SCALE);
	cy = (unsigned short)(height*AREA_SCALE*SCALE);
	MetaFillRect(ofp, PATCOPY, cy, cx, 0, 0);
	MetaSelectObject(ofp, hOldMetaBrush);
	MetaDeleteObject(ofp, gMetaBrush);
	gMetaBrush = -1;

	//metafile.datɤE	sprintf(metafile, "%s/metafile.dat", get_tmpdir());
	fp = fopen(metafile, "r");
	if(fp == NULL) {
		return FALSE;
	}
	while(!feof(fp)) {
		if(fgets(linebuf, 256, fp) == NULL)
			break;
		sscanf(linebuf,"%s",cmd);	//drawing command
		if(strcmp(cmd, "origin") == 0) { //origin
			sscanf(linebuf,"%s %lg %lg",cmd,&fxs,&fys);
			xorg = (unsigned short)((fxs-(wxs-width*0.1)) * SCALE);
			yorg = (unsigned short)(height*AREA_SCALE*SCALE)-(unsigned short)((fys-(wys-height*0.1)) * SCALE);
			//yorg = 100;
		}
		else if(strcmp(cmd, "moveto") == 0) {	//moveto
			sscanf(linebuf,"%s %lg %lg %d",cmd,&fxs,&fys,&mode);
			xs = xorg + (unsigned short)(fxs*SCALE);
			ys = yorg - (unsigned short)(fys*SCALE);
			MetaMoveToEx(ofp, ys, xs);
		}
		else if(strcmp(cmd, "lineto") == 0) { //lineto
			sscanf(linebuf,"%s %lg %lg %d",cmd,&fxe,&fye,&mode);
			xe = xorg + (unsigned short)(fxe*SCALE);
			ye = yorg - (unsigned short)(fye*SCALE);
			MetaLineTo(ofp, ye, xe);
		}
		else if(strcmp(cmd, "box") == 0) { //box
			sscanf(linebuf,"%s %lg %lg %lg %lg %d",cmd,&fxs,&fys,&fxe,&fye,&box_mode);
			if(box_mode != 2 && box_mode != 3) {
				left = xorg + (unsigned short)(fxs*SCALE);
				right = xorg + (unsigned short)(fxe*SCALE);
				if((yorg-(unsigned short)(fys*SCALE)) > (yorg-(unsigned short)(fye*SCALE))) {
					top = yorg - (unsigned short)(fye*SCALE);
					bottom = yorg - (unsigned short)(fys*SCALE);
				}
				else {
					top = yorg - (unsigned short)(fys*SCALE);
					bottom = yorg - (unsigned short)(fye*SCALE);
				}
				MetaMoveToEx(ofp, top, left);
				MetaLineTo(ofp, top, right);
				MetaLineTo(ofp, bottom, right);
				MetaLineTo(ofp, bottom, left);
				MetaLineTo(ofp, top, left);
			}
			else {
			//retry
			sscanf(linebuf,"%s %lg %lg %lg %lg %d %d %d %d",cmd,&fxs,&fys,&fxe,&fye,&box_mode,
							&color_b, &color_g, &color_r);
	color_rgb = ((color_r & 0xff) << 16);
	color_rgb += ((color_g & 0xff) << 8);
	color_rgb += (color_b & 0xff);
	MetaCreateSolidBrush( ofp, color_rgb );
	if(gMetaBrush == -1) {
		gMetaBrush = MetaHandle();
		bMetaBrush = 1;
	}
	//SelectObject
	hOldMetaBrush = MetaSelectObject(ofp, gMetaBrush);
	if((yorg-(unsigned short)(fys*SCALE)) > (yorg-(unsigned short)(fye*SCALE))) {
		top = yorg - (unsigned short)(fye*SCALE);
		bottom = yorg - (unsigned short)(fys*SCALE);
		ys = top;
		ye = bottom - top;
	}
	else {
		top = yorg - (unsigned short)(fys*SCALE);
		bottom = yorg - (unsigned short)(fye*SCALE);
		ys = top;
		ye = top - bottom;
	}
	xs = xorg + (unsigned short)(SCALE * fxs);
	xe = (unsigned short)(SCALE * fabs(fxe - fxs));
	MetaFillRect(ofp, PATCOPY, ye, xe, ys, xs);
	MetaSelectObject(ofp, hOldMetaBrush);
	MetaDeleteObject(ofp, gMetaBrush);
	gMetaBrush = -1;
			}
		}
		else if(strcmp(cmd, "label") == 0) { //label
			sscanf(linebuf,"%s %lg %lg %s %lg %lg %d %d %lg",
				cmd,&fxs,&fys,str,&fheight,&ftheta,&center,&font, &wsize);
			fgets(str, 128, fp);	//label̾
			str[strlen(str)-1] = '\0';
			if(gMetaFont == -1) {
			LabelPrint(ofp,xorg+(int)(fxs*SCALE),yorg-(int)(fys*SCALE),
				str,fheight,ftheta,center,font,wsize);
			}
			LabelPrint(ofp,xorg+(int)(fxs*SCALE),yorg-(int)(fys*SCALE),
				str,fheight,ftheta,center,font,wsize);
		}
		else if(strcmp(cmd, "csymbol") == 0) { //label
			sscanf(linebuf,"%s %lg %lg %d %lg",
				cmd,&fxs,&fys,&type,&flength);
			SymbolPrint(ofp,xorg+(int)(fxs*SCALE),yorg-(int)(fys*SCALE),
				type, (int)(flength*SCALE));
		}
		else if(strcmp(cmd, "setrgbcolor") == 0) { //label
			sscanf(linebuf,"%s %d %d %d",
				cmd, &color_r, &color_g, &color_b);
			if(gMetaPen != -1) {
				MetaSelectObject(ofp, hOldMetaPen);
				MetaDeleteObject(ofp, gMetaPen);
				gMetaPen = -1;
			}
			color_rgb = 0x00000000;
			color_rgb += (color_r << 16);
			color_rgb += (color_g << 8);
			color_rgb += color_b;
			MetaSetTextColor(ofp, color_rgb);
			MetaCreatePen(ofp, line_style, lwidth, color_rgb);
			if(gMetaPen == -1) {
				gMetaPen = MetaHandle();
				bMetaPen = 1;
			}
			hOldMetaPen = MetaSelectObject(ofp, gMetaPen);
		}
		else if(strcmp(cmd, "setlinewidth") == 0) { //setlinewidth
			sscanf(linebuf,"%s %lg",
				cmd, &lwidth_code);
			if(lwidth_code < 1.0)
				lwidth_code = 1.0;
			if(lwidth != (unsigned short)(lwidth_code*PEN_SCALE)) {
				lwidth = (unsigned short)(lwidth_code*PEN_SCALE);
				if(gMetaPen != -1) {
					MetaSelectObject(ofp, hOldMetaPen);
					MetaDeleteObject(ofp, gMetaPen);
					gMetaPen = -1;
				}
				MetaCreatePen(ofp, line_style, lwidth, color_rgb);
				if(gMetaPen == -1) {
					gMetaPen = MetaHandle();
					bMetaPen = 1;
				}
				hOldMetaPen = MetaSelectObject(ofp, gMetaPen);
			}
		}
		else if(strcmp(cmd, "setdash") == 0) { //setlinewidth
			sscanf(linebuf,"%s %d",
				cmd, &dash_code);
			if(dash != dash_code) {
				dash = dash_code;
				switch(dash) {
				case	0:
					line_style = PS_SOLID;		break;
				case	1:
					line_style = PS_DASH;		break;
				case	2:
					line_style = PS_DASHDOT;	break;
				case	3:
					line_style = PS_DASHDOTDOT;	break;
				case	4:
					line_style = PS_DOT;		break;
				default:
					line_style = PS_SOLID;
					break;
				}
				if(gMetaPen != -1) {
					MetaSelectObject(ofp, hOldMetaPen);
					MetaDeleteObject(ofp, gMetaPen);
					gMetaPen = -1;
				}
				MetaCreatePen(ofp, line_style, lwidth, color_rgb);
				if(gMetaPen == -1) {
					gMetaPen = MetaHandle();
					bMetaPen = 1;
				}
				hOldMetaPen = MetaSelectObject(ofp, gMetaPen);
			}
		}
	}
	fclose(fp);
	if(gMetaPen != -1) {
		MetaDeleteObject(ofp, gMetaPen);
		gMetaPen = -1;
	}
	if(gMetaBrush != -1) {
		MetaDeleteObject(ofp, gMetaBrush);
		gMetaBrush = -1;
	}
	if(gMetaFont != -1) {
		MetaDeleteObject(ofp, gMetaFont);
		gMetaFont = -1;
	}
	if(gMetaSymBrush != -1) {
		MetaDeleteObject(ofp, gMetaSymBrush);
		gMetaSymBrush = -1;
	}

	//SelectObject(hDCMeta, hOldBrush);
	//MetaSelectObject(ofp, hOldBrush);
	//DeleteObject(hMetaVioletBrush);
	//MetaDeleteObject(ofp, hMetaVioletBrush);
	//SelectObject(hDCMeta, hOldPen);
	// dummy data
	MetaDummyData( ofp );
	// close metafile
	//hMeta = CloseMetaFile(hDCMeta);
	// chaned METAHEADER
	fseek(ofp, 0L, SEEK_SET);
	mh.mtType = 1;
	fwrite(&mh.mtType, sizeof(unsigned short), 1, ofp);
	mh.mtHeaderSize = 9;
	fwrite(&mh.mtHeaderSize, sizeof(unsigned short), 1, ofp);
	mh.mtVersion = 0x300;
	fwrite(&mh.mtVersion, sizeof(unsigned short), 1, ofp);
	mh.mtSize = file_size;
	fwrite(&mh.mtSize, sizeof(unsigned long), 1, ofp);
	mh.mtNoObjects = bMetaPen + bMetaFont + bMetaBrush + bMetaSymBrush + 1;
	fwrite(&mh.mtNoObjects, sizeof(unsigned short), 1, ofp);
	mh.mtMaxRecord = max_record;
	fwrite(&mh.mtMaxRecord, sizeof(unsigned long), 1, ofp);
	mh.mtNoParameters = 0x0;
	fwrite(&mh.mtNoParameters, sizeof(unsigned short), 1, ofp);
	//fseek(ofp, 0L, SEEK_SET);
	//fwrite(&mh, sizeof(mh), 1, ofp);
	
	fclose(ofp);

	return TRUE;
}

// No Windows API
void LabelPrint(FILE* ofp, int xs, int ys, char *label, double height,
				double theta, int center, int font, double wsize)
{
	static short	hFont = -1;
	static int		hFontAdd = -1;
	unsigned short	hOldMetaFont;
	int				cx,cy;
	int				kakudo;
	double			rad,rcos,rsin,xoff,yoff;
	unsigned short	xorg,yorg,len;
	int	flag = 0;

	if(gMetaFont == -1)
		flag = 1;
	if ( strlen(label) > 0 && height > 0.01 ) {
		//change font
		if(theta > 0.0) {
			kakudo = (int)(theta*10+0.5);
			//CreateFont
			MetaCreateFont(ofp, (int)(height*SCALE),0,kakudo,kakudo,FW_NORMAL,font);
			if(gMetaFont == -1) {
				gMetaFont = MetaHandle();
				bMetaFont = 1;
			}

		}
		else {
			kakudo = 0;
			//CreateFont
			MetaCreateFont(ofp, (int)(height*SCALE),0,3600,3600,FW_NORMAL,font);
			if(gMetaFont == -1) {
				gMetaFont = MetaHandle();
				bMetaFont = 1;
			}
		}

		//SelectObject
		hOldMetaFont = MetaSelectObject(ofp, gMetaFont);
		//GetTextExtent32
		cx = (int)(wsize*SCALE);
		cy = (int)(height*SCALE);

		rad  = theta * 3.1415926 / 180.0;
		rcos = (double)cos((double)rad);
		rsin = (double)sin((double)rad);
		switch(center) {
		case	0:
			//xoff = siz.cy * rsin + siz.cx/2 * rcos;
			//yoff = siz.cy * rcos - siz.cx/2 * rsin;
			xoff = cy * rsin;
			yoff = cy * rcos;
			break;
		case	1:
			xoff = cy * rsin + cx * rcos;
			yoff = cy * rcos - cx * rsin;
			break;
		case	2:
			xoff = cy * rsin + cx/2 * rcos;
			yoff = cy * rcos - cx/2 * rsin;
			break;
		default:
			xoff = cy * rsin + cx/2 * rcos;
			yoff = cy * rcos - cx/2 * rsin;
		}
		xorg = (unsigned short)(xs-xoff);
		yorg = (unsigned short)(ys-yoff);
		len = strlen(label);
		//TextOut
		MetaTextOut(ofp, len, label, yorg, xorg); 
		//SelectObject
		hOldMetaFont = MetaSelectObject(ofp, gMetaFont/*gMetaFont*/);
		MetaDeleteObject(ofp, gMetaFont);
		gMetaFont = -1;
	}
}

void SymbolPrint(FILE* ofp, int xs, int ys, int type, int length)
{
	unsigned short hOldBrush;
	int	length2,left,top,right,bottom;
	int  ptx[10],pty[10];
	length2 = (int)((double)length / 2);
	//polygon coordinates
	if(type == 0 || type == 6 || type == 12) {
		// square
		if(type == 6) {
			MetaCreateSolidBrush( ofp, color_rgb );
			if(gMetaSymBrush == -1) {
				gMetaSymBrush = MetaHandle();
				bMetaSymBrush = 1;
			}
			hOldBrush = MetaSelectObject( ofp , gMetaSymBrush);
		}
		ptx[0] = xs-length2; pty[0] = ys-length2;
		ptx[1] = xs+length2; pty[1] = ys-length2;
		ptx[2] = xs+length2; pty[2] = ys+length2;
		ptx[3] = xs-length2; pty[3] = ys+length2;
		ptx[4] = xs-length2; pty[4] = ys-length2;
		if(type == 6)
			MetaPolygon(ofp, ptx, pty, 4);
		else
			MetaPolyline(ofp, ptx, pty, 5);
		if(type == 6) {
			MetaSelectObject(ofp, hOldBrush);
			MetaDeleteObject(ofp, gMetaSymBrush);
			gMetaSymBrush = -1;
		}
	}
	if(type == 2 || type == 8 || type == 14) {
		// triangle
		if(type == 8) {
			MetaCreateSolidBrush( ofp, color_rgb );
			if(gMetaSymBrush == -1) {
				gMetaSymBrush = MetaHandle();
				bMetaSymBrush = 1;
			}
			hOldBrush = MetaSelectObject( ofp , gMetaSymBrush);
		}
		ptx[0] = xs; pty[0] = ys-length2;
		ptx[1] = xs+length2; pty[1] = ys+length2;
		ptx[2] = xs-length2; pty[2] = ys+length2;
		ptx[3] = xs; pty[3] = ys-length2;
		if(type == 8)
			MetaPolygon(ofp, ptx, pty, 3);
		else
			MetaPolyline(ofp, ptx, pty, 4);
		if(type == 8) {
			MetaSelectObject(ofp, hOldBrush);
			MetaDeleteObject(ofp, gMetaSymBrush);
			gMetaSymBrush = -1;
		}
	}
	if(type == 5 || type == 11 || type == 15) {
		// diamond
		if(type == 11) {
			MetaCreateSolidBrush( ofp, color_rgb );
			if(gMetaSymBrush == -1) {
				gMetaSymBrush = MetaHandle();
				bMetaSymBrush = 1;
			}
			hOldBrush = MetaSelectObject( ofp , gMetaSymBrush);

		}
		pty[0] = xs; pty[0] = ys-length2;
		pty[1] = xs+length2; pty[1] = ys;
		pty[2] = xs; pty[2] = ys+length2;
		pty[3] = xs-length2; pty[3] = ys;
		pty[4] = xs; pty[4] = ys-length2;
		if(type == 11)
			//Polygon(hdc, pts, 4);
			MetaPolygon(ofp, ptx, pty, 4);
		else
			//Polyline(hdc, pts, 5);
			MetaPolyline(ofp, ptx, pty, 5);
		if(type == 11) {
			MetaSelectObject(ofp, hOldBrush);
			MetaDeleteObject(ofp, gMetaSymBrush);
			gMetaSymBrush = -1;
		}
	}
	if(type == 1 || type == 7 || type == 13) {
		if(type == 7) {
			MetaCreateSolidBrush( ofp, color_rgb );
			if(gMetaSymBrush == -1) {
				gMetaSymBrush = MetaHandle();
				bMetaSymBrush = 1;
			}
			hOldBrush = MetaSelectObject( ofp , gMetaSymBrush);
		}
		left = xs - length2;
		top = ys - length2;
		right = xs + length2;
		bottom = ys + length2;
		if(type == 7)
			MetaEllipse(ofp, left, top, right, bottom);
		else
			//Arc(hdc, left, top, right, bottom,
			//	xs+length2,ys,xs+length2,ys);
			MetaArc(ofp, left, top, right, bottom,
				xs+length2,ys,xs+length2,ys);
		if(type == 7) {
			MetaSelectObject(ofp, hOldBrush);
			MetaDeleteObject(ofp, gMetaSymBrush);
			gMetaSymBrush = -1;
		}
	}
	if(type == 3) {
		//+
		//-
		MetaMoveToEx(ofp, ys, xs-length2);
		MetaLineTo(ofp, ys, xs+length2);
		//|
		MetaMoveToEx(ofp, ys-length2, xs);
		MetaLineTo(ofp, ys+length2, xs);
	}
	if(type == 4) {
		//*
		//-
		MetaMoveToEx(ofp, ys-length2, xs-length2);
		MetaLineTo(ofp, ys+length2, xs+length2);
		//|
		MetaMoveToEx(ofp, ys-length2, xs+length2);
		MetaLineTo(ofp, ys+length2, xs-length2);
	}

}
