/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: xbm2data.c,v 1.4 2004/05/18 08:28:58 orrisroot Exp $ */

/* $XConsortium: XRdBitF.c,v 1.15 91/02/01 16:34:46 gildea Exp $ */
/* Copyright, 1987, Massachusetts Institute of Technology */

/*
Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in advertising or
publicity pertaining to distribution of the software without specific,
written prior permission.  M.I.T. makes no representations about the
suitability of this software for any purpose.  It is provided "as is"
without express or implied warranty.
*/

/*
 *	Code to read bitmaps from disk files. Interprets 
 *	data from X10 and X11 bitmap files and creates
 *	Pixmap representations of files. Returns Pixmap
 *	ID and specifics about image.
 *
 *	Modified for speedup by Jim Becker, changed image
 *	data parsing logic (removed some fscanf()s). 
 *	Aug 5, 1988
 *
 * Note that this file and ../Xmu/RdBitF.c look very similar....  Keep them
 * that way (but don't use common source code so that people can have one 
 * without the other).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "xbm2data.h"

#define MAX_SIZE 255

#ifdef __cplusplus
extern "C" {
#endif

/* shared data for the image read/parse logic */
static short hexTable[256];   /* conversion value */
static int   initialized = 0; /* easier to fill in at run time */

/* private function */
static char *_rindex(const char *str, int c);
static void   initHexTable();
static int    NextInt(FILE *fp);

static char *_rindex(const char *str, int c){
  register char *s=NULL;
  while(1){
    if(*str == c)
      s = (char*)str;
    if(*str == '\0')
      return s;
    str++;
  }
  return NULL; /* not-reached */
}

/*
 *  Table index for the hex values. Initialized once, first time.
 *  Used for translation value or delimiter significance lookup.
 */
static void initHexTable(){
  /*
   * We build the table at run time for several reasons:
   *
   *     1.  portable to non-ASCII machines.
   *     2.  still reentrant since we set the init flag after setting table.
   *     3.  easier to extend.
   *     4.  less prone to bugs.
   */
  hexTable['0'] = 0;  hexTable['1'] = 1;
  hexTable['2'] = 2;  hexTable['3'] = 3;
  hexTable['4'] = 4;  hexTable['5'] = 5;
  hexTable['6'] = 6;  hexTable['7'] = 7;
  hexTable['8'] = 8;  hexTable['9'] = 9;
  hexTable['A'] = 10;  hexTable['B'] = 11;
  hexTable['C'] = 12;  hexTable['D'] = 13;
  hexTable['E'] = 14;  hexTable['F'] = 15;
  hexTable['a'] = 10;  hexTable['b'] = 11;
  hexTable['c'] = 12;  hexTable['d'] = 13;
  hexTable['e'] = 14;  hexTable['f'] = 15;
  
  /* delimiters of significance are flagged w/ negative value */
  hexTable[' '] = -1;  hexTable[','] = -1;
  hexTable['}'] = -1;  hexTable['\n'] = -1;
  hexTable['\t'] = -1;
    
  initialized = 1;
}
  
/*
 *  read next hex value in the input stream, return -1 if EOF
 */
static int NextInt(FILE *fp){
  int ch;
  int value = 0;
  int gotone = 0;
  int done = 0;
  
  /* loop, accumulate hex value until find delimiter  */
  /* skip any initial delimiters found in read stream */

  while(!done){
    ch = getc(fp);
    if(ch == EOF){
      value = -1;
      done++;
    }else{
      /* trim high bits, check type and accumulate */
      ch &= 0xff;
      if(isascii(ch) && isxdigit(ch)){
        value = (value << 4) + hexTable[ch];
        gotone++;
      }else if((hexTable[ch]) < 0 && gotone)
        done++;
    }
  }
  return value;
}

int sl4_xbm2data(const char *filename, char **xbmdata, unsigned int *width, 
                 unsigned int *height, int *x_hot, int *y_hot){
  FILE *fp;                      /* handle on file  */
  unsigned char *data = NULL;    /* working variable */
  char  line[MAX_SIZE];          /* input line from file */
  int   size;                    /* number of bytes of data */
  char  name_and_type[MAX_SIZE]; /* an input line */
  char *type;                    /* for parsing */
  int   value;                   /* from an input line */
  int   version10p;              /* boolean, old format */
  int   padding;                 /* to handle alignment */
  int   bytes_per_line;          /* per scanline of data */
  unsigned int ww = 0;           /* width */
  unsigned int hh = 0;           /* height */
  int hx = -1;                   /* x hotspot */
  int hy = -1;                   /* y hotspot */
  
  /* first time initialization */
  if(initialized == 0) initHexTable();
  
  if((fp = fopen(filename, "r")) == NULL){
    return XBM2DATA_OPEN_FAILED;
  }
  
  /* error cleanup and return macro  */
#define  FAIL_RETURN(code) { if(data) free(data); fclose(fp); return code; }
  while(fgets(line, MAX_SIZE, fp)){
    if(strlen(line) == MAX_SIZE-1){
      FAIL_RETURN(XBM2DATA_INVALID_FILE); /* invalid xbm */
    }
    if(sscanf(line,"#define %s %d",name_and_type,&value) == 2){
      if(!(type = _rindex(name_and_type, '_')))
        type = name_and_type;
      else
        type++;
      if(!strcmp("width", type))
        ww = (unsigned int)value;
      if(!strcmp("height", type))
        hh = (unsigned int) value;
      if(!strcmp("hot", type)){
        if(type-- == name_and_type || type-- == name_and_type)
          continue; /* ignore */
        if(!strcmp("x_hot", type))
          hx = value;
        if(!strcmp("y_hot", type))
          hy = value;
      }
      continue;
    }
    if(sscanf(line, "static short %s = {", name_and_type) == 1)
      version10p = 1;
    else if(sscanf(line,"static unsigned char %s = {",name_and_type) == 1)
      version10p = 0;
    else if(sscanf(line, "static char %s = {", name_and_type) == 1)
      version10p = 0;
    else
      continue;
      
    if(!(type = _rindex(name_and_type, '_')))
      type = name_and_type;
    else
      type++;
    
    if(strcmp("bits[]", type))
      continue;
    
    if(!ww || !hh){
      FAIL_RETURN(XBM2DATA_INVALID_FILE); /* invalid xbm */
    }
    
    if((ww % 16) && ((ww % 16) < 9) && version10p)
      padding = 1;
    else
      padding = 0;
    
    bytes_per_line = (ww+7)/8 + padding;
    
    size = bytes_per_line * hh;
    if(data) free(data); /* for illegal xbm format */
    data = (unsigned char *)malloc((unsigned int)size);
    if(!data)
      FAIL_RETURN(XBM2DATA_BAD_ALLOC); /* out of memory */
    if(version10p){
      unsigned char *ptr;
      int bytes;
      for(bytes=0, ptr=data; bytes<size; (bytes += 2)){
        if((value = NextInt(fp)) < 0)
          FAIL_RETURN(XBM2DATA_INVALID_FILE); /* invalid xbm */
        *(ptr++) = value;
        if(!padding || ((bytes+2) % bytes_per_line))
          *(ptr++) = value >> 8;
      }
    }else{
      unsigned char *ptr;
      int bytes;
      for(bytes=0, ptr=data; bytes<size; bytes++, ptr++){
        if((value = NextInt(fp)) < 0)
          FAIL_RETURN(XBM2DATA_INVALID_FILE); /* invalid xbm */
        *ptr=value;
      }
    }
  } /* end while */
  
  if(data == NULL){
    FAIL_RETURN(XBM2DATA_INVALID_FILE); /* invalid xbm */
  }
  
  fclose(fp);
  *xbmdata = (char*)data;
  *width = ww;
  *height = hh;
  if(x_hot) *x_hot = hx;
  if(y_hot) *y_hot = hy;

  return XBM2DATA_SUCCESS; /* success */
}
  
#ifdef __cplusplus
}
#endif
