/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/command/graph_sub.cpp,v 1.8 2005/04/06 08:42:15 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libgpm.h"
#include "graph_sub.h"

#ifdef __cplusplus
extern "C" {
#endif

static int  _gpmcmd_convert_to_log(float **data, int n, int m, int is);
static void _gpmcmd_scale(float **data, float *amin, float *amax,
                          int n, int m);
static void _gpmcmd_scalg(float **data, float *amin, float *amax,
                          int n, int m, int is);

int gpmcmd_open_window(gpm_cont_t *gpm_cont){
  if(gopen(gpm_cont, gpm_cont->paper, gpm_cont->orientation, gpm_cont->device, 
           gpm_cont->paperWidth, gpm_cont->paperHeight) != 0)
    return -1;
  if(checktodvi(gpm_cont) != 0)
    libgpm_dvi_write_param(&gpmdev_cont.dvi, gpm_cont->factor, 
                           gpm_cont->xOrigin, gpm_cont->yOrigin);
  gnewpen(gpm_cont->gColor);
  gpen(gpm_cont->gLineWidth, gpm_cont->gLineType, 0);
  gorigin(gpm_cont->xOrigin, gpm_cont->yOrigin);
  gfsize(gpm_cont->xSize, gpm_cont->ySize);
  gfactor(gpm_cont->factor);
  glimit(0.0, 0.0, gpm_cont->xSize, gpm_cont->ySize);
  return 0;
}

void gpmcmd_close_window()
{
  glimit(0.0, 0.0, 0.0, 0.0);
  gflush();
  gclose();
}

void gpmcmd_btof(int dim, int *index, Buffer *buf, float **data){
  register int i, j, k;
  int          index2[MAX_INDEX];
  int          siz = IndexSize(dim, index);
  
  for(i = 0; i < siz; i++){
    rIndex(i, index2, dim, index);
    k = Index(SubIndex(index2), dim - 1, SubIndex(index));
    j = index2[0];
    data[k][j] = (float)buf[i];
  }
}

void gpmcmd_set_xscale(gpm_cont_t *gpm_cont,
                       float **xdata, int n, int m, int is){
  /*********** Set X-Scale ***********/
  if(gpm_cont->axisType == GPM_AXIS_TYPE_RR || 
     gpm_cont->axisType == GPM_AXIS_TYPE_RI){
    if(gpm_cont->xMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scale(xdata, &gpm_cont->xMin, &gpm_cont->xMax, n, m);
  }else{ /* axis_type 1 or 3 */
    _gpmcmd_convert_to_log(xdata, n, m, is);
    if(gpm_cont->xMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scalg(xdata, &gpm_cont->xMin, &gpm_cont->xMax, n, m, is);
  }
}

void gpmcmd_set_yscale(gpm_cont_t *gpm_cont,
                       float **ydata, int n, int m, int is){
  /*********** Set Y-Scale ***********/
  if(gpm_cont->axisType == GPM_AXIS_TYPE_RR || 
     gpm_cont->axisType == GPM_AXIS_TYPE_RI){
    if(gpm_cont->yMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scale(ydata, &gpm_cont->yMin, &gpm_cont->yMax, n, m);
  }else{
    _gpmcmd_convert_to_log(ydata, n, m, is);
    if(gpm_cont->yMode == GPM_SCALE_MODE_AUTO)
      _gpmcmd_scalg(ydata, &gpm_cont->yMin, &gpm_cont->yMax, n, m, is);
  }
}

void gpmcmd_set_scale(gpm_cont_t *gpm_cont){
  if(gpm_cont->axisType == GPM_AXIS_TYPE_IR || 
     gpm_cont->axisType == GPM_AXIS_TYPE_II){
    if(gpm_cont->xMin <= 0.0) gpm_cont->xMin = (float)1.0e-37;
    if(gpm_cont->xMax <= 0.0) gpm_cont->xMax = (float)1.0e-37;
  }
  if(gpm_cont->axisType == GPM_AXIS_TYPE_RI || 
     gpm_cont->axisType == GPM_AXIS_TYPE_II){
      if(gpm_cont->yMin <= 0.0) gpm_cont->yMin = (float)1.0e-37;
    if(gpm_cont->yMax <= 0.0) gpm_cont->yMax = (float)1.0e-37;
  }

  switch(gpm_cont->axisType){
  case GPM_AXIS_TYPE_RR:
    gscale(gpm_cont->xMin, gpm_cont->yMin, gpm_cont->xMax, gpm_cont->yMax);
    break;
  case GPM_AXIS_TYPE_IR:
    gscale(log10((double)fabs(gpm_cont->xMin)), gpm_cont->yMin,
           log10((double)fabs(gpm_cont->xMax)), gpm_cont->yMax);
    break;
  case GPM_AXIS_TYPE_RI:
    gscale(gpm_cont->xMin, log10(fabs((double)gpm_cont->yMin)),
           gpm_cont->xMax, log10(fabs((double)gpm_cont->yMax)));
    break;
  case GPM_AXIS_TYPE_II:
    gscale(log10(fabs((double)gpm_cont->xMin)), 
           log10(fabs((double)gpm_cont->yMin)),
           log10(fabs((double)gpm_cont->xMax)),
           log10(fabs((double)gpm_cont->yMax)));
    break;
  }
}

void gpmcmd_get_draw_scale(gpm_cont_t *gpm_cont,
                           float *xMin, float *xMax, float *yMin, float *yMax){
  switch(gpm_cont->axisType){
  case GPM_AXIS_TYPE_RR:
  case GPM_AXIS_TYPE_RI:
    *xMin = gpm_cont->xMin;
    *xMax = gpm_cont->xMax;
    break;
  case GPM_AXIS_TYPE_IR:
  case GPM_AXIS_TYPE_II:
    *xMin = (float)log10((double)fabs(gpm_cont->xMin));
    *xMax = (float)log10((double)fabs(gpm_cont->xMax));
    break;
  }
  switch(gpm_cont->axisType){
  case GPM_AXIS_TYPE_RR:
  case GPM_AXIS_TYPE_IR:
    *yMin = gpm_cont->yMin;
    *yMax = gpm_cont->yMax;
    break;
  case GPM_AXIS_TYPE_RI:
  case GPM_AXIS_TYPE_II:
    *yMin = (float)log10(fabs((double)gpm_cont->yMin));
    *yMax = (float)log10(fabs((double)gpm_cont->yMax));
    break;
  }
}


void gpmcmd_set_data(int type, int dim, int *index, Buffer *buf){
  register int   i;
  double         p, delta;
  int            index2[MAX_INDEX];
  int            n = index[0];
  int            siz = IndexSize(dim, index);
  
  switch(type){
  case 'T':
    printf("(Time scale)");
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = 1000.0 / (double)get_sampling() * p;
    }
    break;
  case 'F':
    printf("(Frequency scale)");
    delta = (double)get_sampling() / (double)((n - 1) * 2.0);
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = delta * p;
    }
    break;
  case 'D':
    printf("(Data point scale)");
    for(i = 0; i < siz; i++){
      rIndex(i, index2, dim, index);
      p = (double)index2[0];
      buf[i] = p;
    }
    break;
  default:
    break;
  }
}

/* zero at 0 data point : return value 0 or 1 */
int gpmcmd_zerop(int n_series, float **data){
  int          is = 0;
  register int i;
  for(i = 0; i < n_series; i++){
    if(data[i][0] == 0.0){
      is = 1;
      break;
    }
  }
  return is;
}

void gpmcmd_disp_scale_mode(int mode1, int mode2){
  /***** Scale Auto or Manual *****/
  int mode = mode1+2*mode2;
  switch(mode){
  case 1:
    printf(" Fixed Scale[ x ] ... ");
    break;
  case 2:
    printf(" Fixed Scale[ y ] ... ");
    break;
  case 3:
    printf(" Fixed Scale[ x & y ] ... ");
    break;
  case 0:
    printf(" Auto Scaling ... ");
    break;
  }
}

static int _gpmcmd_convert_to_log(float **data, int n, int m, int is){
  int i, j, erflag = 0;

  for(i = 0; i < n; i++)
    for(j = is; j < m; j++){
      if(data[i][j] > 0.0){
        data[i][j] = (float)log10((double)data[i][j]);
      }else{
        data[i][j] = -38.0;
        erflag = -1;
      }
    }
  if(erflag != 0){
    printf("graph: Warning ... used negative values in log scale mode\n");
  }
  return erflag;
}


static void _gpmcmd_scale(float **data, float *amin, float *amax, 
                          int n, int m){
  register int i, j;
  double dmin, dmax;
  
  dmax = dmin = data[0][0];
  for(i = 0; i < n; i++){
    for(j = 0; j < m; j++){
      if(data[i][j] < dmin)
        dmin = data[i][j];
      else if(data[i][j] > dmax)
        dmax = data[i][j];
    }
  }
  if(dmax == dmin){
    if(dmax == 0.0){
      dmax = 1.0;
      dmin = -1.0;
    }else{
      dmax += 0.05 * fabs(dmax);
      dmin -= 0.05 * fabs(dmin);
    }
  }else{
    dmax += 0.05 * fabs(dmax);
  }
  *amin = (float)dmin;
  *amax = (float)dmax;
}


static void _gpmcmd_scalg(float **data, float *amin, float *amax, 
                          int n, int m, int is){
  register int i, j;
  double       dmin, dmax;
  
  dmax = dmin = data[0][is];
  for(i = 0; i < n; i++)
    for(j = is; j < m; j++){
      if(data[i][j] > dmax)
        dmax = data[i][j];
      else if(data[i][j] < dmin)
        dmin = data[i][j];
    }

  if(dmin < -37.0)
    *amin = (float)1.0e-37;
  else if(dmin >  38.0)
    *amin = (float)1.0e38;
  else
    *amin = (float)pow(10.0,dmin);

  if(dmax < -37.0)
    *amax = (float)1.0e-37;
  else if(dmax >  38.0)
    *amax = (float)1.0e38;
  else
    *amax = (float)pow(10.0,dmax);
}

#ifdef __cplusplus
}
#endif
