/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $SATELLITE: satellite4/modules/gpm/gpm2ps/gpm2ps.c,v 1.12 2005/03/03 07:57:35 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libgpm.h"

#ifdef __cplusplus
extern "C" {
#endif

static void print_usage(const char *bin){
  fprintf(stderr, "\n");
  fprintf(stderr, "%s : Create PS file from GPM dvifile. \n", bin);
  fprintf(stderr, "\n");
  fprintf(stderr, "Usage: %s [arguments] [file]\n", bin);
  fprintf(stderr, "\n");
  fprintf(stderr, "Arguments:\n");
  fprintf(stderr, "  -rv         Use reverse colors mode\n");
  fprintf(stderr, "  -cps        Output color postscript\n");
  fprintf(stderr, "  -tray       Append tray set information\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Copyright (c) 2003-2005 ");
  fprintf(stderr, "RIKEN Japan, ");
  fprintf(stderr, "All rights reserved.\n");
}

static int paramCheck(int argc, char *argv[], char *name, 
                      int *mode, int *trayflag, int *cpsflag){
  int i, ret;
  ret = 0;
  *cpsflag  = 0;
  *trayflag = 0;
  *name     = '\0';
  *mode     = 0;
  for(i=1; i < argc; i++){
    if(strcmp(argv[i],"-rv") == 0){
      *mode = 1;
    }else if(strcmp(argv[i],"-cps") == 0){
      *cpsflag = 1;
    }else if(strcmp(argv[i],"-tray") == 0){
      *trayflag = 1;
    }else if(strcmp(argv[i],"--help") == 0){
      print_usage(argv[0]);
      exit(65); /* usage */
    }else if(*name == '\0'){
      strncpy(name, argv[i], GPM_ENV_MAX_FILEPATH);
      name[GPM_ENV_MAX_FILEPATH-1]='\0';
      ret = 1;
    }
  }
  return ret;
}

int main(int argc, char *argv[]){
  char  gpm_dvifile[GPM_ENV_MAX_FILEPATH];
  int   num;
  int   result;
  int   div     = RAINBOW_DIVNUM;
  int   basecol = RAINBOW_BASECOL;
  int   rollflag = 0;
  int   lwidth, ltype;
  int   pageNumber = 1, mode;
  char  paperSize, orientation, device;
  int   trayflag = 0;
  int   cpsflag  = 0;

  gpmdvi_context_t dc;
  libgpm_dvi_context_init(&dc);
  if(paramCheck(argc, argv, gpm_dvifile, &mode, &trayflag, &cpsflag) == 0){
    strncpy(gpm_dvifile, "GPMDVIFILE1", GPM_ENV_MAX_FILEPATH);
    gpm_dvifile[GPM_ENV_MAX_FILEPATH-1] = '\0';
  }

  libgpm_dvi_open(&dc, gpm_dvifile, 1);
  if(libgpm_dvi_read_head(&dc, &paperSize, &orientation, &device) == -1){
    fprintf(stderr, "%s: %s is not GPM dvi file.\n", argv[0], gpm_dvifile);
    exit(1);
  }

  ps_context_init(stdout, paperSize, orientation, cpsflag);
  
  ps_initAdobe(argv[0], gpm_dvifile);
  ps_init();
  if(trayflag){
    ps_setTray();
  }
  ps_page(pageNumber);
  ps_open(dc.version);
  ps_rainbow_mode(mode);
  
  do{
    result = libgpm_dvi_read_param(&dc);
    if(rollflag == 1 && result > 0){
      pageNumber++;
      ps_page(pageNumber);
    }
    if(result != 0){
      ps_ginit();
      ps_factor((double)dc.param.factor);
      ps_origin((double)dc.param.xorg,(double)dc.param.yorg);
      do{
	num = libgpm_dvi_read_element(&dc);
	if(num > 0){
	  switch(num){
	  case GPM_DVIFILE_ELEMENT_PLOT:
	    if(dc.plot.rainbow <= 0)
	      ps_color((int)dc.plot.color);
	    else
	      ps_rainbow((double)(dc.plot.rainbow)/(div*basecol));
	    lwidth = (int)dc.plot.lwidth;
	    ltype  = (int)dc.plot.ltype;
	    ps_lwidth(lwidth);
	    ps_ltype(ltype);
	    ps_plot(dc.plot.x, dc.plot.y, 
                    (int)dc.plot.ipen_mode);
	    rollflag = 0;
	    break;
	  case GPM_DVIFILE_ELEMENT_BOX:
	    lwidth = (int)dc.box.lwidth;
	    ltype  = (int)dc.box.ltype;
	    ps_lwidth(lwidth);
	    ps_ltype(ltype);
	    ps_box(dc.box.x1, dc.box.y1, dc.box.x2, dc.box.y2, 
                   dc.box.mode, dc.box.level, (int)dc.box.color);
	    rollflag = 0;
	    break;
	  case GPM_DVIFILE_ELEMENT_CSYMBOL:
	    ps_color((int)dc.csymbol.color);
	    ps_lwidth((int)dc.csymbol.lwidth);
	    ps_csymbol(dc.csymbol.x, dc.csymbol.y, (int)dc.csymbol.cdata,
                       dc.csymbol.height);
	    rollflag = 0;
	    break;
	  case GPM_DVIFILE_ELEMENT_LABEL:
	    ps_color((int)dc.label.color);
	    ps_string(dc.label.x, dc.label.y, dc.label.ichar, dc.label.height,
                      dc.label.theta, dc.label.mode, dc.label.font);
	    rollflag = 0;
	    break;
	  case GPM_DVIFILE_ELEMENT_ROLL:
	    ps_roll();
	    rollflag = 1;
	    break;
	  }
	}
      }while(num > 0);
    }
  }while(result > 0);
  libgpm_dvi_close(&dc);
  if(rollflag == 0)
    ps_roll();
  ps_pages(pageNumber);
  ps_close();
  libgpm_dvi_context_clean(&dc);
  return 0;
}

#ifdef __cplusplus
}
#endif
