/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: win32.c,v 1.29 2005/10/28 00:14:33 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32

#pragma warning(disable:4312)
// warning C4312: 'type cast' : conversion from 'LONG' to 'HINSTANCE' of greater size

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "res\gpmwin_res.h"
#include "libgpm.h"
#include "gpmwinpriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define PENFUNC_XOR    0
#define PENFUNC_CLEAR  1
#define PENFUNC_COPY   2

#ifndef M_PI
# define M_PI 3.14159265358979323846
#endif

#define COORD_WIDTH  110
#define COORD_HEIGHT  16

extern int    paper, orient;
extern double height, width;

typedef struct _gpm_win32_t {
  CRITICAL_SECTION cs;
  BOOL       IsWinNT;
  BOOL       bShowCoord;
  WNDCLASSEX klass;
  HINSTANCE  hInst;
  HWND       hMainWnd;
  int        x, y;   /* window size */
  int        mx, my; /* mouse position */
  COLORREF   cc; /* current color */
  COLORREF   fgcolor;
  COLORREF   bgcolor;
  COLORREF   color[GPM_COLOR_SIZE];
  COLORREF   rainbow[GPM_RAINBOW_BASECOL * GPM_RAINBOW_DIVNUM + 2];
  COLORREF   gradation[7][GPM_RAINBOW_BASECOL * GPM_RAINBOW_DIVNUM + 2];
  int        pen_dash_type;
  int        pen_width;
  int        pen_func;
  HBITMAP    hBitmap;
  HDC        hBuffer;
  HBITMAP    hBitmapCoord;
  HDC        hBufferCoord;
  HFONT      hMousePosFont;
  int        InLoop;
} gpm_win32_t;

static void color_allocation();
static void color_alloc_rainbow();
static void color_alloc_gradation();
static void color_alloc_setcols_rainbow(int n, double r, double g, double b);
static void color_alloc_setcols_gradation(int n, double x, int c);
static int  copy_coord_to_clipboard(HWND hwnd);
static int  write_clipboard(HWND hwnd, const char *str);
static void draw_coord(RECT *rc);
static void create_bitmap_buffer(HDC hdc);
static HPEN create_pen();

LRESULT CALLBACK gpmwin_proc(HWND hwnd, unsigned int message, 
                             WPARAM wParam, LPARAM lParam);
DWORD WINAPI     gpmwin_comm_thread(LPVOID lParameter);

static char       gpmwin_className[] = "SATELLITE4_GPM_WINDOW";
static HANDLE     gpmwin_hThread;
static gpm_win32_t w32cont;

static int gpmwin_regist_class(WNDCLASSEX *klass, WNDPROC winProc, 
                               const char *className, HINSTANCE hInst){
  /* initialize window class */
  klass->cbSize = sizeof(WNDCLASSEX);
  klass->style = CS_HREDRAW|CS_VREDRAW;
  klass->lpfnWndProc = winProc; /* window procedure: mandatory */
  klass->cbClsExtra = 0;
  klass->cbWndExtra = 0;
  klass->hInstance = hInst; /* owner of the class: mandatory */
  klass->hIcon = LoadIcon(hInst, MAKEINTRESOURCE(IDI_MAIN_ICON)); /* icon resources */
  klass->hIconSm = NULL;
  klass->hCursor = LoadCursor(0, IDC_ARROW); /* optional */
  klass->hbrBackground = GetStockObject(GRAY_BRUSH); /* optional */
  klass->lpszMenuName = NULL;
  klass->lpszClassName = className; /* mandatory */
  /* register window class */
  if(RegisterClassEx(klass) == 0)
    return -1;
  return 0;
}

int gpmwin_init(int argc, char *argv[]){
  OSVERSIONINFO osvi;
  osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
  GetVersionEx(&osvi);
  InitializeCriticalSection(&w32cont.cs);
  w32cont.IsWinNT = (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT);
  w32cont.hInst = (HINSTANCE)argv; //GetModuleHandle(NULL);
  w32cont.pen_dash_type = GPM_LINE_TYPE_SOLID;
  w32cont.pen_func = PENFUNC_COPY;
  w32cont.hMousePosFont = CreateFont(
    14,                        // nHeight
    0,                         // nWidth
    0,                         // nEscapement
    0,                         // nOrientation
    FW_NORMAL,                 // nWeight
    FALSE,                     // bItalic
    FALSE,                     // bUnderline
    0,                         // cStrikeOut
    ANSI_CHARSET,              // nCharSet
    OUT_DEFAULT_PRECIS,        // nOutPrecision
    CLIP_DEFAULT_PRECIS,       // nClipPrecision
    DEFAULT_QUALITY,           // nQuality
    FIXED_PITCH | FF_MODERN,   // nPitchAndFamily
    NULL);                     // lpszFacename
  if(w32cont.hMousePosFont == NULL)
    return -1;
  w32cont.hBitmap = NULL;
  w32cont.hBuffer = NULL;
  w32cont.hBitmapCoord = NULL;
  w32cont.hBufferCoord = NULL;
  w32cont.bShowCoord = TRUE;
  return gpmwin_regist_class(&w32cont.klass, gpmwin_proc, 
                             gpmwin_className, w32cont.hInst);
}

int gpmwin_create(int x, int y, const char *title){
  DWORD dwStyle = WS_OVERLAPPED | WS_VISIBLE | WS_CAPTION | 
                  WS_SYSMENU | WS_THICKFRAME;
  RECT  rc;
  w32cont.x = x;
  w32cont.y = y;
  rc.top = 0; rc.left = 0; rc.bottom = y; rc.right = x;
  /* color allocation */
  color_allocation();
  /* window creation */
  AdjustWindowRect(&rc, dwStyle, FALSE);
  w32cont.hMainWnd = CreateWindow(
    gpmwin_className,    /* name of registered window class */
    title,               /* window caption */
    dwStyle,             /* window style */
    CW_USEDEFAULT,       /* x position */
    CW_USEDEFAULT,       /* y position */
    rc.right - rc.left,  /* width */
    rc.bottom - rc.top,  /* height */
    NULL,                /* handle to parent window */
    NULL,                /* handle to menu */
    w32cont.hInst,       /* application instance */
    NULL);               /* window creation data */
  ShowWindow(w32cont.hMainWnd, SW_SHOWDEFAULT);
  UpdateWindow(w32cont.hMainWnd);
  return 0;
}

static void color_allocation(){
  w32cont.color[GPM_COLOR_BLACK]   = RGB(  0,   0,   0);
  w32cont.color[GPM_COLOR_BLUE]    = RGB(  0,   0, 255);
  w32cont.color[GPM_COLOR_RED]     = RGB(255,   0,   0);
  w32cont.color[GPM_COLOR_MAGENTA] = RGB(255,   0, 255);
  w32cont.color[GPM_COLOR_GREEN]   = RGB(  0, 255,   0);
  w32cont.color[GPM_COLOR_CYAN]    = RGB(  0, 255, 255);
  w32cont.color[GPM_COLOR_YELLOW]  = RGB(255, 255,   0);
  w32cont.color[GPM_COLOR_WHITE]   = RGB(255, 255, 255);
  w32cont.color[GPM_COLOR_GRAY1]   = RGB(204, 204, 204);
  w32cont.color[GPM_COLOR_GRAY2]   = RGB(153, 153, 153);
  w32cont.color[GPM_COLOR_GRAY3]   = RGB( 77,  77,  77);
  w32cont.fgcolor = w32cont.color[GPM_COLOR_BLACK];
  w32cont.bgcolor = w32cont.color[GPM_COLOR_WHITE];
  w32cont.cc = w32cont.color[GPM_COLOR_BLACK];
  color_alloc_rainbow();
  color_alloc_gradation();
}

static void color_alloc_rainbow(){
  double  r, g, b, level;
  int     i, n, maxcols;
  int     div = GPM_RAINBOW_DIVNUM;
  int     basecol = GPM_RAINBOW_BASECOL;
  double  rbasecol = 1.0 / basecol;
  maxcols = div * basecol;
  n = 1;
  for(i=0; i < div; i++, n++){
    level = (double)n / (double)maxcols;
    r = 0.0;  g = 0.0; b = level/rbasecol;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }
  for(i=0; i < div; i++, n++){
    level = (double)n / (double)maxcols;
    r = 0.0;  g = (level-rbasecol)/rbasecol; b = 1.0;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }
  for(i=0; i < div; i++, n++){
    level = (double)n / (double)maxcols;
    r = 0.0; g = 1.0; b = 1.0 - (level-rbasecol*2.0)/rbasecol;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }
  for(i=0; i < div; i++, n++){
    level = (double)n / (double)maxcols;
    r = (level-rbasecol*3.0)/rbasecol; g = 1.0;  b = 0.0;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }
  for(i=0; i <= div; i++, n++){
    level = (double)n / (double)maxcols;
    r = 1.0;  g = 1.0 - (level-rbasecol*4.0)/rbasecol;  b = 0.0;
    color_alloc_setcols_rainbow(n-1, r, g, b);
  }
  r = g = b = 1.0;
  color_alloc_setcols_rainbow(n-1, r, g, b);
}

static void color_alloc_gradation(){
  int    c, i, n, maxcols;
  double level;
  int    div = GPM_RAINBOW_DIVNUM;
  int    basecol  = GPM_RAINBOW_BASECOL;
  for(c = 0; c < 7; c++){
    maxcols = div * basecol;
    n = 1;
    for(i = 0; i < maxcols; i++, n++){
      level = (double)n / (double)maxcols;
      color_alloc_setcols_gradation(n-1, level, c);
    }
    color_alloc_setcols_gradation(n-1, 1.0, c);
  }
}

static void color_alloc_setcols_rainbow(int n, double r, double g, double b){
  int red, green, blue;
  red   = ( r > 0.0 ) ? (int)(255 * sqrt(r)) : 0;
  green = ( g > 0.0 ) ? (int)(255 * sqrt(g)) : 0;
  blue  = ( b > 0.0 ) ? (int)(255 * sqrt(b)) : 0;
  w32cont.rainbow[n] = RGB(red, green, blue);
}

static void color_alloc_setcols_gradation(int n, double x, int c){
  int  col, red, green, blue;
  static int t[3][7] = {
    { 1, 0, 1, 0, 1, 0, 1 },
    { 0, 1, 1, 0, 0, 1, 1 },
    { 0, 0, 0, 1, 1, 1, 1 }};
  col = (x > 0.0) ?  (int)(255 * sqrt(x)) : 0;
  blue  = t[0][c] ? col : 0 ;
  red   = t[1][c] ? col : 0 ;
  green = t[2][c] ? col : 0 ;
  w32cont.gradation[c][n] = RGB(red, green, blue);
}
 
DWORD WINAPI gpmwin_comm_thread(LPVOID lParameter){
  while(gpmwin_comm_command() != 0);
  EnterCriticalSection(&w32cont.cs);
  if(w32cont.InLoop){
    SendMessage(w32cont.hMainWnd, WM_CLOSE, 0, 0);
  }
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_mainloop(){
  MSG msg;
  int status;
  DWORD threadId;
  w32cont.InLoop = 1;
  gpmwin_hThread = CreateThread(NULL, 0, gpmwin_comm_thread,
                                NULL, 0, &threadId);
  if(gpmwin_hThread == NULL){
    return -1;
  }
  while((status = GetMessage(&msg, NULL, 0, 0)) != 0){
    if(status == -1) return -1;
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
  EnterCriticalSection(&w32cont.cs);
  w32cont.InLoop = 0;
  LeaveCriticalSection(&w32cont.cs);
  if(gpmwin_hThread != INVALID_HANDLE_VALUE){
    CloseHandle(GetStdHandle(STD_INPUT_HANDLE));
    CloseHandle(GetStdHandle(STD_OUTPUT_HANDLE));
    WaitForSingleObject(gpmwin_hThread, INFINITE);
    CloseHandle(gpmwin_hThread);
  }
  DeleteDC(w32cont.hBufferCoord);
  DeleteObject(w32cont.hBitmapCoord);
  DeleteDC(w32cont.hBuffer);
  DeleteObject(w32cont.hBitmap);
  DeleteObject(w32cont.hMousePosFont);
  DeleteCriticalSection(&w32cont.cs);
  return (int)msg.wParam;
}

LRESULT CALLBACK gpmwin_proc(HWND hwnd, unsigned int message, 
                             WPARAM wParam, LPARAM lParam){
  HDC hdc;
  PAINTSTRUCT ps;
  POINT pt;
  HMENU hmenu, hSubmenu;
  LPMINMAXINFO lpmmi;
  RECT rc;
  HINSTANCE hinst;
  switch(message){
  case WM_COMMAND:
    switch (LOWORD(wParam)){
    case IDM_SHOW_HIDE_COORDLABEL:
      w32cont.bShowCoord =! w32cont.bShowCoord;
      draw_coord(&rc);
      InvalidateRect(w32cont.hMainWnd, &rc, FALSE);
      break;
    default:
      return DefWindowProc(hwnd, message, wParam, lParam);
    }
    break;
  case WM_CHAR:
    switch((UINT)wParam){
    case 0x11: /* Cntl-Q */
      gpmwin_quit();
      break;
    case 0x17: /* Cntl-W */
      gpmwin_erase();
      break;
    default:
      return DefWindowProc(hwnd, message, wParam, lParam);
    }
    return 0;
  case WM_LBUTTONUP:
    copy_coord_to_clipboard(hwnd);
    return 0;
  case WM_RBUTTONDOWN:
    pt.x = LOWORD(lParam);
    pt.y = HIWORD(lParam);
    hinst = (HINSTANCE)GetWindowLong(hwnd, GWL_HINSTANCE);
    hmenu = LoadMenu(hinst, MAKEINTRESOURCE(IDR_POPUPMENU));
    if(w32cont.bShowCoord){
      hSubmenu = GetSubMenu(hmenu, 0);
    }else{
      hSubmenu = GetSubMenu(hmenu, 1);
    }
    ClientToScreen(hwnd, &pt);
    TrackPopupMenu(hSubmenu, TPM_LEFTALIGN, pt.x, pt.y, 0, hwnd, NULL);
    DestroyMenu(hmenu);
    break;
  case WM_GETMINMAXINFO:
    /* limit max and min window size */
    lpmmi = (LPMINMAXINFO)lParam;
    lpmmi->ptMinTrackSize.x = 10;
    lpmmi->ptMinTrackSize.y = 10;
    return 0;
  case WM_CREATE:
    EnterCriticalSection(&w32cont.cs);
    hdc = GetDC(hwnd);
    create_bitmap_buffer(hdc);
    ReleaseDC(hwnd, hdc);
    w32cont.mx = 0; w32cont.my = w32cont.y;
    LeaveCriticalSection(&w32cont.cs);
    draw_coord(NULL);
    return 0;
  case WM_MOUSEMOVE:
    w32cont.mx = LOWORD(lParam);
    w32cont.my = HIWORD(lParam);
    draw_coord(&rc);
    InvalidateRect(w32cont.hMainWnd, &rc, FALSE);
    return 0;
  case WM_PAINT:
    EnterCriticalSection(&w32cont.cs);
    hdc = BeginPaint(hwnd, &ps);
/*     { */
/*       char buf[1024]; */
/*       snprintf(buf, 1024, "rect(top-%d,left-%d,right-%d,bottom-%d)",  */
/*                ps.rcPaint.top, ps.rcPaint.left, */
/*                ps.rcPaint.right, ps.rcPaint.bottom); */
/*       SetWindowText(hwnd, buf); */
/*     } */
    BitBlt(hdc, ps.rcPaint.left, ps.rcPaint.top,
           ps.rcPaint.right - ps.rcPaint.left, 
           ps.rcPaint.bottom - ps.rcPaint.top, w32cont.hBuffer, 
           ps.rcPaint.left, ps.rcPaint.top, SRCCOPY);
    if(w32cont.bShowCoord){
      BitBlt(hdc, 0, 0, COORD_WIDTH, COORD_HEIGHT, w32cont.hBufferCoord, 
             0, 0, SRCCOPY);
    }
    EndPaint(hwnd, &ps);
    LeaveCriticalSection(&w32cont.cs);
    return 0;
  case WM_CLOSE:
    DestroyWindow(hwnd);
  case WM_DESTROY:
    /* quit application */
    PostQuitMessage(0);
    return 0;
  default:
    return DefWindowProc(hwnd, message, wParam, lParam);
  }
  return 0;
}

static HPEN create_pen(){
  HPEN hPen;
  LOGBRUSH logBrush;
  static DWORD  dot1[] = { 4, 4 };
  static DWORD  dot2[] = { 2, 2, 6, 2 };
  static DWORD  dot3[] = { 2, 2, 2, 2, 6, 2 };
  static DWORD  dot4[] = { 2, 2 };
  static DWORD  dot5[] = { 1, 3 };
  static DWORD  dot6[] = { 1, 7 };
  static DWORD  dot7[] = { 1, 1 };
  static DWORD *dashtbl[] = { dot1, dot2, dot3, dot4, dot5, dot6, dot7 }; 
  static int    dashlen[] = { 2, 4, 6, 2, 2, 2, 2};
  logBrush.lbColor = w32cont.cc;
  logBrush.lbStyle = BS_SOLID; 
  logBrush.lbHatch = 0;
  if(w32cont.IsWinNT){
    DWORD  dwStyle;
    DWORD  dwNumDash;
    DWORD *pdwDashPat;
    if(w32cont.pen_dash_type == GPM_LINE_TYPE_SOLID){
      dwStyle    = PS_GEOMETRIC | PS_SOLID | PS_ENDCAP_SQUARE | PS_JOIN_MITER;
      dwNumDash  = 0;
      pdwDashPat = NULL;
    }else{
      dwStyle    = PS_GEOMETRIC | PS_USERSTYLE | PS_ENDCAP_SQUARE | PS_JOIN_MITER; 
      dwNumDash  = dashlen[w32cont.pen_dash_type-1];
      pdwDashPat = dashtbl[w32cont.pen_dash_type-1];
    }
    hPen = ExtCreatePen(dwStyle, w32cont.pen_width, &logBrush,
                        dwNumDash, pdwDashPat);
  }else{
    hPen = CreatePen(PS_SOLID, w32cont.pen_width, w32cont.cc);
  }
  return hPen;
}

static void create_bitmap_buffer(HDC hdc){
  /* buffer for drawing area */
  if(w32cont.hBuffer != NULL){
    DeleteDC(w32cont.hBuffer);
  }
  if(w32cont.hBitmap != NULL){
    DeleteObject(w32cont.hBitmap);
  }
  w32cont.hBitmap = CreateCompatibleBitmap(hdc, w32cont.x, w32cont.y);
  w32cont.hBuffer = CreateCompatibleDC(hdc);
  SelectObject(w32cont.hBuffer , w32cont.hBitmap);
  SelectObject(w32cont.hBuffer , GetStockObject(NULL_PEN));
  PatBlt(w32cont.hBuffer, 0, 0, w32cont.x, w32cont.y, WHITENESS);
  /* buffer for coordination */
  if(w32cont.hBufferCoord != NULL){
    DeleteDC(w32cont.hBufferCoord);
  }
  if(w32cont.hBitmapCoord != NULL){
    DeleteObject(w32cont.hBitmapCoord);
  }
  w32cont.hBitmapCoord = CreateCompatibleBitmap(hdc, COORD_WIDTH, COORD_HEIGHT);
  w32cont.hBufferCoord = CreateCompatibleDC(hdc);
  SelectObject(w32cont.hBufferCoord, w32cont.hBitmapCoord);
  SelectObject(w32cont.hBufferCoord, GetStockObject(NULL_PEN));
  PatBlt(w32cont.hBufferCoord, 0, 0, COORD_WIDTH, COORD_HEIGHT, WHITENESS);
}

static void  draw_coord(RECT *rc){
  char buf[32];
  double posx, posy;
  int  x,y;
  RECT   rect;
  HPEN  hPen, hOldPen;
  HFONT hOldFont;
  EnterCriticalSection(&w32cont.cs);
  rect.top = 0; rect.left = 0; rect.right = COORD_WIDTH; rect.bottom = COORD_HEIGHT;
  /* draw background rectangle */
  hPen    = CreatePen(PS_SOLID, 1, w32cont.bgcolor);
  hOldPen = SelectObject(w32cont.hBufferCoord, hPen);
  Rectangle(w32cont.hBufferCoord, rect.top, rect.left, rect.right, rect.bottom);
  SelectObject(w32cont.hBufferCoord, hOldPen);
  DeleteObject(hPen);
  if(w32cont.bShowCoord){
    x = w32cont.mx;
    y = w32cont.my;
    libgpm_pixel2mm(paper, &posx, &posy, x, y);
    posy = height - posy;
    if(posy < 0.0) posy = 0.0;
    snprintf(buf, 32, "(%5.1f, %5.1f)", posx, posy);

    /* draw string */
    hPen     = CreatePen(PS_SOLID, 1, w32cont.fgcolor);
    hOldPen  = SelectObject(w32cont.hBufferCoord, hPen);
    hOldFont = SelectObject(w32cont.hBufferCoord, w32cont.hMousePosFont);
    TextOut(w32cont.hBufferCoord, 0, 0, buf, lstrlen(buf));
    SelectObject(w32cont.hBufferCoord, hOldPen);
    SelectObject(w32cont.hBufferCoord, hOldFont);
    DeleteObject(hPen);
  }
  if(rc != NULL){
    *rc = rect;
  }
  LeaveCriticalSection(&w32cont.cs);
}

static int copy_coord_to_clipboard(HWND hwnd){
  char buf[32];
  double posx, posy;
  EnterCriticalSection(&w32cont.cs);
  libgpm_pixel2mm(paper, &posx, &posy, w32cont.mx, w32cont.my);
  posy = height - posy;
  if(posy < 0.0) posy = 0.0;
  snprintf(buf, 32, "(%5.1f, %5.1f)", posx, posy);
  LeaveCriticalSection(&w32cont.cs);
  return write_clipboard(hwnd, buf);
}

static int write_clipboard(HWND hwnd, const char *str){
  HGLOBAL hGlobal;
  int iLength;
  LPSTR lpStr;
  iLength = lstrlen(str);
  if(iLength > 1024)
    iLength = 1024;
  hGlobal = GlobalAlloc(GHND, iLength + 1);
  if(hGlobal == NULL)
    return -1;
  lpStr = (LPSTR)GlobalLock(hGlobal);
  memset(lpStr, 0, iLength + 1);
  strcpy(lpStr, str); /* safe */
  GlobalUnlock(hGlobal);
  if(OpenClipboard(hwnd) == 0){
    GlobalFree(hGlobal);
    return -1;
  }
  EmptyClipboard();
  SetClipboardData(CF_TEXT, hGlobal);
  CloseClipboard();
  return 0;
}

int gpmwin_quit(){ 
  SendMessage(w32cont.hMainWnd, WM_CLOSE, 0, 0);
  return 0;
}

int gpmwin_erase(){
  RECT rc;
  EnterCriticalSection(&w32cont.cs);
  PatBlt(w32cont.hBuffer, 0, 0, w32cont.x, w32cont.y, WHITENESS);
  LeaveCriticalSection(&w32cont.cs);
  draw_coord(&rc);
  InvalidateRect(w32cont.hMainWnd, NULL, FALSE);
  return 0;
}

int gpmwin_getwattr(int *w, int *h){
  EnterCriticalSection(&w32cont.cs);
  *w = w32cont.x;
  *h = w32cont.y;
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_flush(){
  InvalidateRect(w32cont.hMainWnd, NULL, FALSE);
  return 0;
}

int gpmwin_resize(const char *title, int w, int h){
  HDC hdc;
  EnterCriticalSection(&w32cont.cs);
  w32cont.x = w;  w32cont.y = h;
  hdc = GetDC(w32cont.hMainWnd);
  create_bitmap_buffer(hdc);
  ReleaseDC(w32cont.hMainWnd, hdc);
  w32cont.mx = 0; w32cont.my = w32cont.y;
  SetWindowText(w32cont.hMainWnd, title);
  LeaveCriticalSection(&w32cont.cs);
  draw_coord(NULL);
  InvalidateRect(w32cont.hMainWnd, NULL, TRUE);
  return 0;
}

int gpmwin_setcolor(int color, int rainbow){
  COLORREF c;
  static   int  RainbowMax = RAINBOW_DIVNUM * RAINBOW_BASECOL+2;
  if(color < 0 || rainbow >= 0){
    if(rainbow >= 0 && rainbow < RainbowMax){
      c = w32cont.rainbow[rainbow];
    }else{
      c = (rainbow < 0) ? w32cont.rainbow[0] : w32cont.rainbow[RainbowMax-1];
    }
  }else{
    c = w32cont.color[color%MAX_COLOR];
  }
  EnterCriticalSection(&w32cont.cs);
  w32cont.cc = c;
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_setdash(int itype, int isize, int imode){
  EnterCriticalSection(&w32cont.cs);
  /* dash type */
  if(itype > GPM_LINE_TYPE_SOLID && itype < GPM_LINE_TYPE_SIZE){
    w32cont.pen_dash_type    = itype;
  }else{
    w32cont.pen_dash_type = GPM_LINE_TYPE_SOLID;
  }

  /* line width */
  w32cont.pen_width = isize * 2;

  /* draw mode - currently no effect */
  switch(imode){
  case 1:
    w32cont.pen_func = PENFUNC_XOR;
    break;
  case 2:
    w32cont.pen_func = PENFUNC_CLEAR;
    break;
  case 0:
  default:
    w32cont.pen_func = PENFUNC_COPY;
    break;
  }
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_drawrect(int x, int y, int w, int h){
  HPEN   hPen, hOldPen;
  HBRUSH hOldBrush;
  EnterCriticalSection(&w32cont.cs);
  hPen = create_pen();
  hOldPen   = SelectObject(w32cont.hBuffer, hPen);
  hOldBrush = SelectObject(w32cont.hBuffer, GetStockObject(NULL_BRUSH));
  Rectangle(w32cont.hBuffer, x, y, x+w+1, y+h+1);
  SelectObject(w32cont.hBuffer, hOldPen);
  SelectObject(w32cont.hBuffer, hOldBrush);
  DeleteObject(hPen);
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_fillrect(int x, int y, int w, int h){
  HPEN   hPen, hOldPen;
  HBRUSH hBrush, hOldBrush;
  LOGBRUSH logBrush;
  EnterCriticalSection(&w32cont.cs);
  logBrush.lbStyle = BS_SOLID;
  logBrush.lbColor = w32cont.cc;
  logBrush.lbHatch = 0;
  hPen = create_pen();
  hBrush = CreateBrushIndirect(&logBrush);
  hOldPen   = SelectObject(w32cont.hBuffer, hPen);
  hOldBrush = SelectObject(w32cont.hBuffer, hBrush);
  Rectangle(w32cont.hBuffer, x, y, x+w+1, y+h+1);
  SelectObject(w32cont.hBuffer, hOldPen);
  SelectObject(w32cont.hBuffer, hOldBrush);
  DeleteObject(hPen);
  DeleteObject(hBrush);
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_drawpoint(gpm_xpoint_t *ps, int npts){
  int i;
  EnterCriticalSection(&w32cont.cs);
  for(i=0; i<npts; i++)
    SetPixel(w32cont.hBuffer, ps[i].x, ps[i].y, w32cont.cc);
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_drawline(gpm_xpoint_t *pts, int npts){
  int i;
  HPEN hPen, hOldPen;
  POINT points[GPM_ENV_MAX_PLOTSTACK];
  if(npts>1){
    EnterCriticalSection(&w32cont.cs);
    for(i=0; i < npts; i++){
      points[i].x = pts[i].x;
      points[i].y = pts[i].y;
    }
    hPen = create_pen();
    hOldPen = SelectObject(w32cont.hBuffer, hPen);
    Polyline(w32cont.hBuffer, points, npts);
    SelectObject(w32cont.hBuffer, hOldPen);
    DeleteObject(hPen);
    LeaveCriticalSection(&w32cont.cs);
  }
  return 0;
}

int gpmwin_fillpoly(gpm_xpoint_t *pts, int npts){
  int i;
  HPEN hPen, hOldPen;
  HBRUSH hBrush, hOldBrush;
  LOGBRUSH logBrush;
  POINT points[GPM_ENV_MAX_PLOTSTACK];
  if(npts>1){
    EnterCriticalSection(&w32cont.cs);
    for(i=0; i < npts; i++){
      points[i].x = pts[i].x;
      points[i].y = pts[i].y;
    }
    logBrush.lbStyle = BS_SOLID;
    logBrush.lbColor = w32cont.cc;
    logBrush.lbHatch = 0;
    hPen = create_pen();
    hBrush = CreateBrushIndirect(&logBrush);
    hOldPen   = SelectObject(w32cont.hBuffer, hPen);
    hOldBrush = SelectObject(w32cont.hBuffer, hBrush);
    Polygon(w32cont.hBuffer, points, npts);
    SelectObject(w32cont.hBuffer, hOldPen);
    SelectObject(w32cont.hBuffer, hOldBrush);
    DeleteObject(hPen);
    DeleteObject(hBrush);
    LeaveCriticalSection(&w32cont.cs);
  }
  return 0;
}
int gpmwin_drawarc_sub(int x, int y, int w, int h, int a1, int a2, int fill){
  HPEN   hPen, hOldPen;
  HBRUSH hBrush, hOldBrush;
  LOGBRUSH logBrush;
  int nXStartArc;
  int nYStartArc;
  int nXEndArc;
  int nYEndArc;
  if(a2 >= 360*64){
    nXStartArc = 0; nXEndArc = 0;
    nYStartArc = 0; nYEndArc = 0;
  }else if(a2 > 0){
    nXStartArc = (int)(x + w/2 + w * cos(a1/64.0*2.0*M_PI/360.0));
    nYStartArc = (int)(y + h/2 - h * sin(a1/64.0*2.0*M_PI/360.0));
    nXEndArc = (int)(x + w/2 + w * cos((a1+a2)/64.0*2.0*M_PI/360.0));
    nYEndArc = (int)(y + h/2 - h * sin((a1+a2)/64.0*2.0*M_PI/360.0));
  }else{
    nXStartArc = (int)(x + w/2 + w * cos((a1+a2)/64.0*2.0*M_PI/360.0));
    nYStartArc = (int)(y + h/2 - h * sin((a1+a2)/64.0*2.0*M_PI/360.0));
    nXEndArc = (int)(x + w/2 + w * cos(a1/64.0*2.0*M_PI/360.0));
    nYEndArc = (int)(y + h/2 - h * sin(a1/64.0*2.0*M_PI/360.0));
  }
  EnterCriticalSection(&w32cont.cs);
  logBrush.lbStyle = BS_SOLID;
  logBrush.lbColor = w32cont.cc;
  logBrush.lbHatch = 0;
  hPen = create_pen();
  hOldPen   = SelectObject(w32cont.hBuffer, hPen);
  if(fill){
    hBrush = CreateBrushIndirect(&logBrush);
    hOldBrush = SelectObject(w32cont.hBuffer, hBrush);
  }
  if(fill){
    Pie(w32cont.hBuffer, x, y, x+w, y+h, nXStartArc, nYStartArc,
        nXEndArc, nYEndArc);
  }else{
    Arc(w32cont.hBuffer, x, y, x+w, y+h, nXStartArc, nYStartArc,
        nXEndArc, nYEndArc);
  }
  SelectObject(w32cont.hBuffer, hOldPen);
  DeleteObject(hPen);
  if(fill){
    SelectObject(w32cont.hBuffer, hOldBrush);
    DeleteObject(hBrush);
  }
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

int gpmwin_drawarc(int x, int y, int w, int h, int a1, int a2){
  gpmwin_drawarc_sub(x, y, w, h, a1, a2, 0);
  return 0;
}

int gpmwin_fillarc(int x, int y, int w, int h, int a1, int a2){
  gpmwin_drawarc_sub(x, y, w, h, a1, a2, 1);
  return 0;
}

int gpmwin_gradation(int color, int n, double d){
  EnterCriticalSection(&w32cont.cs);
  w32cont.cc = w32cont.gradation[color][n];
  LeaveCriticalSection(&w32cont.cs);
  return 0;
}

#ifdef __cplusplus
}
#endif

#endif /* WIN32 */
