/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: dvifile.c,v 1.21 2005/10/30 09:55:30 orrisroot Exp $ */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef WIN32
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_IO_H
# include <io.h>
#endif

#include "libgpm.h"
#include "libgpmpriv.h"

#ifndef O_BINARY
# define O_BINARY 0
#endif

#ifdef __cplusplus
extern "C" {
#endif

static char gpmdvi_head[GPM_DVIFILE_HEADER_SIZE]  = GPM_DVIFILE_HEADER3;
static char gpmdvi_head2[GPM_DVIFILE_HEADER_SIZE] = GPM_DVIFILE_HEADER2;

static void  libgpm_dvi_read_plot(gpmdvi_context_t *cont);
static void  libgpm_dvi_read_box(gpmdvi_context_t *cont);
static void  libgpm_dvi_read_csymbol(gpmdvi_context_t *cont);
static void  libgpm_dvi_read_label(gpmdvi_context_t *cont);
static void  libgpm_dvi_read_roll(gpmdvi_context_t *cont);

static int   cpu_type();
static void  reverse(void *s, int siz);

int libgpm_dvi_context_init(gpmdvi_context_t *cont){
  cont->fd      = -1;
  cont->version =  3;
  cont->endian  = cpu_type();
  /* gpmdvi_plot_t */
  cont->plot.x         = 0.0;
  cont->plot.y         = 0.0;
  cont->plot.ipen_mode = 0;
  cont->plot.color     = GPM_COLOR_BLACK;
  cont->plot.rainbow   = -1;
  cont->plot.ltype     = GPM_LINE_TYPE_SOLID;
  cont->plot.lwidth    = 1;
  /* gpmdvi_box_t */
  cont->box.x1         = 0.0;
  cont->box.y1         = 0.0;
  cont->box.x2         = 0.0;
  cont->box.y2         = 0.0;
  cont->box.level      = 0.0;
  cont->box.mode       = 0;
  cont->box.color      = GPM_COLOR_BLACK;
  cont->box.ltype      = GPM_LINE_TYPE_SOLID;
  cont->box.lwidth     = 1;
  /* gpmdvi_csymbol_t */
  cont->csymbol.x      = 0.0;
  cont->csymbol.y      = 0.0;
  cont->csymbol.cdata  = 0x20;
  cont->csymbol.height = 3.0;
  cont->csymbol.theta  = 0.0;
  /* gpmdvi_label_t */
  cont->label.x        = 0.0;
  cont->label.y        = 0.0;
  cont->label.height   = 3.0;
  cont->label.theta    = 0.0;
  cont->label.ilen     = 0;
  cont->label.ichar    = NULL;
  cont->label.mode     = 0;
  cont->label.color    = GPM_COLOR_BLACK;
  cont->label.font     = GPM_FONT_TYPE_TR;
  /* gpmdvi_param_t */
  cont->param.endmark  = GPM_DVIFILE_ENDMARK;
  cont->param.factor   = 1.0;
  cont->param.xorg     = 0.0;
  cont->param.yorg     = 0.0;
  return 0;
}

int libgpm_dvi_context_clean(gpmdvi_context_t *cont){
  if(cont->fd != -1)
    close(cont->fd);
  if(cont->label.ichar != NULL)
    free(cont->label.ichar);
  libgpm_dvi_context_init(cont);
  return 0;
}

int libgpm_dvi_create(const char *fname, int paper, int orient, int device){
  int  fd;
  char cpaper  = (char)paper;
  char corient = (char)orient;
  char cdevice = (char)device;
  fd = open(fname, O_CREAT|O_TRUNC|O_WRONLY|O_BINARY, 0644);
  if(fd == -1) return -1;
  write(fd, gpmdvi_head, GPM_DVIFILE_HEADER_SIZE);
  write(fd, &cpaper,  sizeof(char));
  write(fd, &corient, sizeof(char));
  write(fd, &cdevice, sizeof(char));
  close(fd);
  return 0;
}

int libgpm_dvi_open(gpmdvi_context_t *cont, const char *fname, 
                    int is_readonly){
  if(is_readonly != 0){
    cont->fd = open(fname, O_RDONLY | O_BINARY);
  }else{
    cont->fd = open(fname, O_RDWR | O_BINARY, 0644);
  }
  if(cont->fd == -1)
    return -1;
  lseek(cont->fd, (long) 0, SEEK_END);
  /*  cont->endian = cpu_type(); */
  return 0;
}

int libgpm_dvi_close(gpmdvi_context_t *cont){
  if(cont->fd == -1)
    return -1;
  if(close(cont->fd) != 0)
    return -1;
  cont->fd = -1;
  return 0;
}

int libgpm_dvi_read_head(gpmdvi_context_t *cont, char *paper, char *orient,
                         char *device){
  int  result;
  char buf[GPM_DVIFILE_HEADER_SIZE];
  if(cont == NULL || paper == NULL || orient == NULL)
    return -1;
  if(cont->fd == -1)
    return -1;
  result = lseek(cont->fd, 0L, SEEK_SET);
  result = read(cont->fd, buf, GPM_DVIFILE_HEADER_SIZE);
  cont->version = 3;
  if(strcmp(buf, gpmdvi_head) != 0){
    if(strcmp(buf, gpmdvi_head2) != 0){
      return -1;
    }
    cont->version = 2;
  }
  result = read(cont->fd, paper,  sizeof(char));
  result = read(cont->fd, orient, sizeof(char));
  result = read(cont->fd, device, sizeof(char));
  lseek(cont->fd, 1L, SEEK_CUR); /* skip endmark */

  /*  cont->endian        = cpu_type();  */
  return 0;
}


int libgpm_dvi_write_param(gpmdvi_context_t *cont, float factor, 
                           float xorg, float yorg){
  unsigned char comnum;
  gpmdvi_param_t temp;
  if(cont->fd == -1)
    return -1;
  comnum = GPM_DVIFILE_ENDMARK;
  write(cont->fd, &comnum, sizeof(unsigned char));
  cont->param.endmark = temp.endmark = GPM_DVIFILE_ENDMARK;
  cont->param.factor  = temp.factor  = factor;
  cont->param.xorg    = temp.xorg    = xorg;
  cont->param.yorg    = temp.yorg    = yorg;
  if(cont->endian != 0){
    reverse(&temp.factor, sizeof(float));
    reverse(&temp.xorg,   sizeof(float));
    reverse(&temp.yorg,   sizeof(float));
  }
  write(cont->fd, &temp, sizeof(gpmdvi_param_t));
  return 0;
}

int libgpm_dvi_write_plot(gpmdvi_context_t *cont, double x, double y,
                          int ipen_mode, gpmdev_pen_t *pen){
  char num = GPM_DVIFILE_ELEMENT_PLOT;
  gpmdvi_plot_t temp;
  if(cont->fd == -1)
    return -1;
  write(cont->fd, &num, sizeof(char));
  cont->plot.x         = temp.x         = (float)x;
  cont->plot.y         = temp.y         = (float)y;
  cont->plot.ipen_mode = temp.ipen_mode = (char)ipen_mode;
  cont->plot.color     = temp.color     = pen->color;
  cont->plot.rainbow   = temp.rainbow   = pen->rainbow;
  cont->plot.lwidth    = temp.lwidth    = pen->size;
  cont->plot.ltype     = temp.ltype     = pen->type;
  if(cont->endian != 0){
    reverse(&temp.x, sizeof(float));
    reverse(&temp.y, sizeof(float));
  }
  write(cont->fd, (char *) &temp, sizeof(gpmdvi_plot_t));
  return 0;
}

int libgpm_dvi_write_box(gpmdvi_context_t *cont, double x1, double y1,
                         double x2, double y2, int mode, double level,
                         gpmdev_pen_t *pen){
  char num = GPM_DVIFILE_ELEMENT_BOX;
  gpmdvi_box_t temp;
  if(cont->fd == -1)
    return -1;
  write(cont->fd, &num, sizeof(char));
  cont->box.x1     = temp.x1     = (float)x1;
  cont->box.y1     = temp.y1     = (float)y1;
  cont->box.x2     = temp.x2     = (float)x2;
  cont->box.y2     = temp.y2     = (float)y2;
  cont->box.level  = temp.level  = (float)level;
  cont->box.mode   = temp.mode   = (char)mode;
  cont->box.color  = temp.color  = pen->color;
  cont->box.ltype  = temp.ltype  = pen->type;
  cont->box.lwidth = temp.lwidth = pen->size;
  if(cont->endian != 0){
    reverse(&temp.x1,    sizeof(float));
    reverse(&temp.y1,    sizeof(float));
    reverse(&temp.x2,    sizeof(float));
    reverse(&temp.y2,    sizeof(float));
    reverse(&temp.level, sizeof(float));
  }
  write(cont->fd, &temp, sizeof(gpmdvi_box_t));
  return 0;
}

int libgpm_dvi_write_csymbol(gpmdvi_context_t *cont, double x, double y,
                             gpmdev_pen_t *pen, gpmdev_symbol_t *symbol){
  char num = GPM_DVIFILE_ELEMENT_CSYMBOL;
  gpmdvi_csymbol_t temp;
  if(cont->fd == -1)
    return -1;
  write(cont->fd, &num, sizeof(char));
  cont->csymbol.x      = temp.x      = (float)x;
  cont->csymbol.y      = temp.y      = (float)y;
  cont->csymbol.cdata  = temp.cdata  = symbol->cdata;
  cont->csymbol.height = temp.height = symbol->height;
  cont->csymbol.theta  = temp.theta  = symbol->theta;
  cont->csymbol.color  = temp.color  = pen->color;
  cont->csymbol.lwidth = temp.lwidth = pen->size;
  if(cont->endian != 0){
    reverse(&temp.x,      sizeof(float));
    reverse(&temp.y,      sizeof(float));
    reverse(&temp.height, sizeof(float));
    reverse(&temp.theta,  sizeof(float));
  }
  write(cont->fd, &temp, sizeof(gpmdvi_csymbol_t));
  return 0;
}


int libgpm_dvi_write_label(gpmdvi_context_t *cont, double x, double y,
                           const char *ichar, double height,
                           double theta, int imode, gpmdev_pen_t *pen,
                           gpmdev_symbol_t *symbol){
  char num = GPM_DVIFILE_ELEMENT_LABEL;
  gpmdvi_label_t temp;
  if(cont->fd == -1)
    return -1;
  write(cont->fd, &num, sizeof(char));
  cont->label.x      = temp.x      = (float)x;
  cont->label.y      = temp.y      = (float)y;
  cont->label.ilen   = temp.ilen   = (unsigned char)strlen(ichar);
  if(cont->label.ichar != NULL) free(cont->label.ichar);
  cont->label.ichar  = temp.ichar  = (char*)strdup(ichar);
  cont->label.height = temp.height = (float)height;
  cont->label.theta  = temp.theta  = (float)theta;
  cont->label.mode   = temp.mode   = (char)imode;
  cont->label.color  = temp.color  = pen->color;
  cont->label.font   = temp.font   = symbol->font;
  if(cont->endian != 0){
    reverse(&temp.x,      sizeof(float));
    reverse(&temp.y,      sizeof(float));
    reverse(&temp.height, sizeof(float));
    reverse(&temp.theta,  sizeof(float));
  }
  write(cont->fd, &temp.x,      sizeof(float));
  write(cont->fd, &temp.y,      sizeof(float));
  write(cont->fd, &temp.ilen,   sizeof(unsigned char));
  write(cont->fd, temp.ichar,   (int)cont->label.ilen+1);
  write(cont->fd, &temp.height, sizeof(float));
  write(cont->fd, &temp.theta,  sizeof(float));
  write(cont->fd, &temp.mode,   sizeof(char));
  write(cont->fd, &temp.color,  sizeof(char));
  write(cont->fd, &temp.font,   sizeof(char));
  return 0;
}


int libgpm_dvi_write_roll(gpmdvi_context_t *cont){
  char num = GPM_DVIFILE_ELEMENT_ROLL;
  if(cont->fd == -1)
    return -1;
  write(cont->fd, &num, sizeof(char));
  return 0;
}


int libgpm_dvi_read_param(gpmdvi_context_t *cont){
  int result;
  if(cont->fd == -1)
    return -1;
  result = read(cont->fd, &cont->param, sizeof(gpmdvi_param_t));
  if(cont->endian != 0){
    reverse(&cont->param.factor, sizeof(float));
    reverse(&cont->param.xorg,   sizeof(float));
    reverse(&cont->param.yorg,   sizeof(float));
  }
  return result;
}


int libgpm_dvi_read_element(gpmdvi_context_t *cont){
  char num;
  int  result;
  if(cont->fd == -1)
    return -1;
  result = read(cont->fd, &num, sizeof(char));
  if(result > 0){
    switch(num){
    case GPM_DVIFILE_ELEMENT_PLOT:    libgpm_dvi_read_plot(cont);    break;
    case GPM_DVIFILE_ELEMENT_BOX:     libgpm_dvi_read_box(cont);     break;
    case GPM_DVIFILE_ELEMENT_CSYMBOL: libgpm_dvi_read_csymbol(cont); break;
    case GPM_DVIFILE_ELEMENT_LABEL:   libgpm_dvi_read_label(cont);   break;
    case GPM_DVIFILE_ELEMENT_ROLL:    libgpm_dvi_read_roll(cont);    break;
    }
    result = (int)num;
  }else{
    result -= 1;
  }
  return result;
}

static void libgpm_dvi_read_plot(gpmdvi_context_t *cont){
  read(cont->fd, &cont->plot, sizeof(gpmdvi_plot_t));
  if(cont->endian != 0){
    reverse(&cont->plot.x, sizeof(float));
    reverse(&cont->plot.y, sizeof(float));
  }
}


static void libgpm_dvi_read_box(gpmdvi_context_t *cont){
  read(cont->fd, &cont->box, sizeof(gpmdvi_box_t));
  if(cont->endian != 0){
    reverse(&cont->box.x1,    sizeof(float));
    reverse(&cont->box.y1,    sizeof(float));
    reverse(&cont->box.x2,    sizeof(float));
    reverse(&cont->box.y2,    sizeof(float));
    reverse(&cont->box.level, sizeof(float));
  }
}


static void libgpm_dvi_read_csymbol(gpmdvi_context_t *cont){
  read(cont->fd, &cont->csymbol, sizeof(gpmdvi_csymbol_t));
  if(cont->endian != 0){
    reverse(&cont->csymbol.x,      sizeof(float));
    reverse(&cont->csymbol.y,      sizeof(float));
    reverse(&cont->csymbol.height, sizeof(float));
    reverse(&cont->csymbol.theta,  sizeof(float));
  }
}


static void libgpm_dvi_read_label(gpmdvi_context_t *cont){
  /* 
     TODO: check of memory allocation and release for cont->label.ichar
           it is assigned memory pointer only in writelabel().
           but in this function, this pointer is released..
           i think, segmentesion fault error will be occured.
  */
  read(cont->fd, &cont->label.x,    sizeof(float));
  read(cont->fd, &cont->label.y,    sizeof(float));
  read(cont->fd, &cont->label.ilen, sizeof(char));
  if(cont->label.ichar != NULL)
    free(cont->label.ichar);
  cont->label.ichar = (char*)calloc((size_t)cont->label.ilen+1, sizeof(char));
  read(cont->fd, cont->label.ichar, (int)cont->label.ilen+1);
  read(cont->fd, &cont->label.height, sizeof(float));
  read(cont->fd, &cont->label.theta,  sizeof(float));
  read(cont->fd, &cont->label.mode,   sizeof(char));
  read(cont->fd, &cont->label.color,  sizeof(char));
  read(cont->fd, &cont->label.font,   sizeof(char));
  if(cont->endian != 0){
    reverse(&cont->label.x,      sizeof(float));
    reverse(&cont->label.y,      sizeof(float));
    reverse(&cont->label.height, sizeof(float));
    reverse(&cont->label.theta,  sizeof(float));
  }
}

static void libgpm_dvi_read_roll(gpmdvi_context_t *cont){
}

static int cpu_type(){
  static const short i = 0x0001;
  static const char *c = (char *)&i;
  return ((int)c[0] == 1);
}

static void reverse(void *b, int siz){
  int  i, j, half;
  char c;
  char *s = (char*)b;
  half = siz/2;
  j = siz-1;
  for(i=0; i<half; i++){
    c = s[i];
    s[i] = s[j-i];
    s[j-i] = c;
  }
}

#ifdef __cplusplus
}
#endif
