/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: libgpm.h,v 1.36 2005/10/30 09:55:30 orrisroot Exp $ */

#ifndef _LIBGPM_H_
#define _LIBGPM_H_

#ifdef LIBGPM_EXTERN
# undef LIBGPM_EXTERN
#endif

#ifdef  _LIBGPM_GLOBAL_VARS_
# define LIBGPM_EXTERN
#else
# define LIBGPM_EXTERN extern
#endif

#define GPM_CONTEXT_FILE_GPM  "GPM_CONTEXT"
#define GPM_CONTEXT_FILE_WIN  "GPMWIN"

#define COLOR_YES        1
#define COLOR_NO         0

#define MAX_COLOR        11
#define RAINBOW_BASECOL  5
#define RAINBOW_DIVNUM   8

/* global settings */
#define GPM_PAPER_A4_WIDTH      210.0
#define GPM_PAPER_A4_HEIGHT     294.0
#define GPM_PAPER_B4_WIDTH      257.0
#define GPM_PAPER_B4_HEIGHT     364.0
#define GPM_PAPER_FREE_DEFAULT  150.0

/* environments */
#define GPM_ENV_MAX_WIN          10  /* max number of window            */
#define GPM_ENV_MAX_TITLE         3  /* max number of title             */
#define GPM_ENV_MAX_FILEPATH    512  /* max length of file name         */
#define GPM_ENV_MAX_TITLELEN    128  /* max length of title             */
#define GPM_ENV_MAX_PLOTSTACK   256  /* max number of plot stack        */


#define GPM_MAX_CHARBUF    1024  /* max number of char array buffer */
#define GPM_TEMPDIR_ENV        "SL4_GPM_TEMPDIR"
                /* environment variable of temporary directory name for GPM */

#define GPM_RAINBOW_BASECOL       5  /* base color number of rainbow colors */
#define GPM_RAINBOW_DIVNUM        8  /* division number of rainbow colors */

/* gpm dvifile */
#define GPM_DVIFILE_BASENAME        "GPMDVIFILE"     /* dvifile base name */
#define GPM_DVIFILE_HEADER2         "gpm dvi file 2" /* dvifile header 2.9x  */
#define GPM_DVIFILE_HEADER3         "gpm dvi file 3" /* dvifile header 4.x */
#define GPM_DVIFILE_HEADER_SIZE      15  /* strlen("gpm dvi file 2") + '\0' */
#define GPM_DVIFILE_ENDMARK          0x80
/* gpm dvifile drawing elements */
#define GPM_DVIFILE_ELEMENT_PLOT      1
#define GPM_DVIFILE_ELEMENT_BOX       2
#define GPM_DVIFILE_ELEMENT_CSYMBOL   3
#define GPM_DVIFILE_ELEMENT_LABEL     4
#define GPM_DVIFILE_ELEMENT_ROLL      5


/* device parameters */
#define GPM_PAPER_A4              0  /* paper size : A4   */
#define GPM_PAPER_B4              1  /* paper size : B4   */
#define GPM_PAPER_FREE            2  /* paper size : Free */
#define GPM_ORIENT_PORTRAIT       0  /* paper oriantation : portrait  */
#define GPM_ORIENT_LANDSCAPE      1  /* paper oriantation : landscape */
#define GPM_DEVICE_NONE          -1  /* oparation device : <none>           */
#define GPM_DEVICE_DISPLAY        0  /* oparation device : display          */
#define GPM_DEVICE_DISPFILE       1  /* oparation device : display and file */
#define GPM_DEVICE_FILE           2  /* oparation device : file             */

/* common parameters */
#define GPM_FONT_TYPE_TR          0  /* font type : times roman           */
#define GPM_FONT_TYPE_TI          1  /* font type : times italic          */
#define GPM_FONT_TYPE_TB          2  /* font type : times bold            */
#define GPM_FONT_TYPE_TBI         3  /* font type : times bold italic     */
#define GPM_FONT_TYPE_HR          4  /* font type : helvetica roman       */
#define GPM_FONT_TYPE_HI          5  /* font type : helvetica italic      */
#define GPM_FONT_TYPE_HB          6  /* font type : helvetica bold        */
#define GPM_FONT_TYPE_HBI         7  /* font type : helvetica bold italic */
#define GPM_FONT_TYPE_CR          8  /* font type : courie roman          */
#define GPM_FONT_TYPE_CI          9  /* font type : courie italic         */
#define GPM_FONT_TYPE_CB         10  /* font type : courie bold           */
#define GPM_FONT_TYPE_CBI        11  /* font type : courie bold italic    */
#define GPM_LINE_TYPE_SOLID       0  /* line dash type : solid           */
#define GPM_LINE_TYPE_DASH        1  /* line dash type : dash            */
#define GPM_LINE_TYPE_DOTDASH     2  /* line dash type : dot-dash        */
#define GPM_LINE_TYPE_DDOTDASH    3  /* line dash type : dot-dot-dash    */
#define GPM_LINE_TYPE_LDASH       6  /* line dash type : long dash       */
#define GPM_LINE_TYPE_SDASH       4  /* line dash type : short dash      */
#define GPM_LINE_TYPE_WSDASH      5  /* line dash type : wide short dash */
#define GPM_LINE_TYPE_DOT         7  /* line dash type : dot             */
#define GPM_LINE_TYPE_SIZE        8  /* line dash type size              */
#define GPM_COLOR_BLACK           0  /* color : black   */
#define GPM_COLOR_BLUE            1  /* color : bule    */
#define GPM_COLOR_RED             2  /* color : red     */
#define GPM_COLOR_MAGENTA         3  /* color : magenta */
#define GPM_COLOR_GREEN           4  /* color : green   */
#define GPM_COLOR_CYAN            5  /* color : cyan    */
#define GPM_COLOR_YELLOW          6  /* color : yellow  */
#define GPM_COLOR_WHITE           7  /* color : white   */
#define GPM_COLOR_GRAY1           8  /* color : gray1   */
#define GPM_COLOR_GRAY2           9  /* color : gray2   */
#define GPM_COLOR_GRAY3          10  /* color : gray3   */
#define GPM_COLOR_SIZE           11  /* color size      */
#define GPM_AXIS_TYPE_RR          0  /* axis type : real and real       */
#define GPM_AXIS_TYPE_IR          1  /* axis type : integer and real    */
#define GPM_AXIS_TYPE_RI          2  /* axis type : real and integer    */
#define GPM_AXIS_TYPE_II          3  /* axis type : integer and integer */
#define GPM_AXIS_TYPE_SIZE        4  /* axis type size */
#define GPM_AXIS_DRAW_LB          1  /* draw axis : left and bottom  */
#define GPM_AXIS_DRAW_LT          2  /* draw axis : left and top     */
#define GPM_AXIS_DRAW_RB          3  /* draw axis : right and bottom */
#define GPM_AXIS_DRAW_RT          4  /* draw axis : right and top    */
#define GPM_SCALE_TYPE_LINEAR     0  /* axis scaling function : linear */
#define GPM_SCALE_TYPE_LOG10      1  /* axis scaling function : log10  */
#define GPM_SCALE_MODE_AUTO       0  /* axis scaling mode     : auto   */
#define GPM_SCALE_MODE_FIXED      1  /* axis scaling mode     : fixed  */

/* GPM Window communication protocol */
#define GPM_PROTO_CLOSE_WINDOW           0x00  /* close gpm window      */
#define GPM_PROTO_GET_WINDOW_ATTRIBUTES  0x01  /* get window attributes */
#define GPM_PROTO_FLUSH_WINDOW           0x02  /* flush window          */
#define GPM_PROTO_CLEAR_WINDOW           0x03  /* clear window          */
#define GPM_PROTO_SET_COLOR              0x04  /* set color             */
#define GPM_PROTO_SET_GRADATION          0x05  /* set gradation         */
#define GPM_PROTO_SET_DASH_WIDTH_DMODE   0x06  /* set dash width dmode  */
#define GPM_PROTO_DRAW_POINTS            0x07  /* draw points           */
#define GPM_PROTO_DRAW_LINES             0x08  /* draw lines            */
#define GPM_PROTO_DRAW_RECTANGLE         0x09  /* draw rectangle        */
#define GPM_PROTO_FILL_RECTANGLE         0x0a  /* fill rectangle        */
#define GPM_PROTO_FILL_POLYGON           0x0b  /* draw polygon          */
#define GPM_PROTO_DRAW_ARC               0x0c  /* draw arc              */
#define GPM_PROTO_FILL_ARC               0x0d  /* fill arc              */
#define GPM_PROTO_RESIZE_WINDOW          0x0e  /* resize window         */


#ifdef __cplusplus
extern "C" {
#endif

typedef struct _gpm_cont_t {
  /* device parameters */
  char    winNum;                   /* window number (0--MAX_WIN-1)     */
  char    device;                   /* 0: window, 1: win and file, 2: file */
  char    dvifile[GPM_ENV_MAX_FILEPATH]; /* dvi file path               */
  char    orientation;              /* 0: portrait, 1: landscape        */
  char    paper;                    /* 0: a4, 1: b4, 2: free            */
  float   paperWidth;               /* paper width  [mm]                */
  float   paperHeight;              /* paper height [mm]                */
  /* common parameters */
  float   factor;        /* scale factor > 0.0               */
  float   xOrigin;       /* position of x origin [mm]        */
  float   yOrigin;       /* position of y origin [mm]        */
  float   xSize;         /* drawing area width  [mm]         */
  float   ySize;         /* drawing area height [mm]         */
  float   fontHeight;    /* font height [mm]                 */
  char    fontType;      /* font type ( 1--12 )              */
  char    fLineWidth;    /* frame line width [mm] (0: 0.1mm) */
  char    gLineWidth;    /* graph line width [mm] (0: 0.1mm) */
  char    fLineType;     /* frame dash type (1--7)           */
  char    gLineType;     /* graph dash type (1--7)           */
  char    fColor;        /* frame color (0--7)               */
  char    gColor;        /* graph color (0--7)               */
  /* for draw graph */
  char    axisType;      /* axis type (0: LnrLnr, 1: LogLnr, *
			  *            2: LnrLog, 3: LogLog) */
  char    axisDraw;      /* axis draw (0: L_L, 1: L_U,       *
			  *            2: R_L, 3: R_U )      */
  char    xType;         /* scaling function of axis-x       */
  char    yType;         /* scaling function of axis-y       */
  char    zType;         /* scaling function of axis-z       *
			  *      0: Normal(Linear), 1: Log10 */
  char    xMode;         /* scaling mode of axis-x           */
  char    yMode;         /* scaling mode of axis-y           */
  char    zMode;         /* scaling mode of axis-z           *
			  *                0: Auto, 1: Fixed */
  float   xMin;          /* x minimum (convert Normal mode ) */
  float   yMin;          /* y minimum (convert Normal mode ) */
  float   zMin;          /* z minimum (convert Normal mode ) */
  float   xMax;          /* x maximum (convert Normal mode ) */
  float   yMax;          /* y maximum (convert Normal mode ) */
  float   zMax;          /* z maximum (convert Normal mode ) */
  /* title buffer */
  char	  title_x[GPM_ENV_MAX_TITLE][GPM_ENV_MAX_TITLELEN];
  char	  title_y[GPM_ENV_MAX_TITLE][GPM_ENV_MAX_TITLELEN];
  char	  title_z[GPM_ENV_MAX_TITLE][GPM_ENV_MAX_TITLELEN];
} gpm_cont_t;


/* GPMDVIFILE context  */
typedef struct _gpmdvi_plot_t {
  float   x;
  float   y;
  char    ipen_mode;
  char    color;
  char    rainbow;
  char    ltype;
  char    lwidth;
} gpmdvi_plot_t;

typedef struct _gpmdvi_box_t {
  float   x1;
  float   y1;
  float   x2;
  float   y2;
  float   level;
  char    mode;
  char    color;
  char    ltype;
  char    lwidth;
} gpmdvi_box_t;

typedef struct _gpmdvi_csymbol_t {
  float   x;
  float   y;
  char    cdata;
  float   height;
  float   theta;
  char    color;
  char    lwidth;
} gpmdvi_csymbol_t;

typedef struct _gpmdvi_label_t {
  float          x;
  float          y;
  float          height;
  float          theta;
  unsigned char  ilen;
  char          *ichar;
  char           mode;
  char           color;
  char           font;
} gpmdvi_label_t;

typedef struct _gpmdvi_param_t {
  unsigned char  endmark;
  float          factor;
  float          xorg;
  float          yorg;
} gpmdvi_param_t;

typedef struct _gpmdvi_context_t {
  int              fd;
  int              version;
  int              endian;
  gpmdvi_plot_t    plot;
  gpmdvi_box_t     box;
  gpmdvi_csymbol_t csymbol;
  gpmdvi_label_t   label;
  gpmdvi_param_t   param;
} gpmdvi_context_t;


/* gpm device context */
typedef struct _gpmdev_common_t {
  char         paper;
  char         orientation;
  char         device;
  float        xleng;
  float        yleng;
} gpmdev_common_t;

typedef struct _gpmdev_graph_t {
  float   xmin;
  float   ymin;
  float   xmax;
  float   ymax;
  float   xorg;
  float   yorg;
  float   xaxis;
  float   yaxis;
  float   factor;
  float   xpre;
  float   ypre;
  int     cline;
} gpmdev_graph_t;

typedef struct _gpmdev_pen_t {
  char    color;
  char    size;
  char    type;
  char    mode;
  char    rainbow;
} gpmdev_pen_t;

typedef struct _gpmdev_scale_t {
  float   xdat_min;
  float   ydat_min;
  float   xdat_max;
  float   ydat_max;
} gpmdev_scale_t;

typedef struct _gpmdev_symbol_t {
  char    cdata;
  char    font;
  float   height;
  float   theta;
} gpmdev_symbol_t;

typedef struct _gpmdev_win_t {
  unsigned int ixleng;
  unsigned int iyleng;
  float        ffx;
  float        ffy;
} gpmdev_win_t;

typedef struct _gpmdev_context_t {
  gpmdev_common_t  common;
  gpmdev_graph_t   graph;   /* cont.c gsolm_sub.c */
  gpmdev_pen_t     pen;
  gpmdev_scale_t   scale;
  gpmdev_symbol_t  symbol;  /* prev.c */
  gpmdev_win_t     win;
  gpmdvi_context_t dvi;
} gpmdev_context_t;

/* for window drawing */
typedef struct _gpm_xpoint_t {
  int x;
  int y;
} gpm_xpoint_t;

/* global variable */
LIBGPM_EXTERN gpmdev_context_t gpmdev_cont; /* device context */

/* comm-unix.c or comm-win32.c */
int  libgpm_comm_send_int(int i);
int  libgpm_comm_send_float(double d);
int  libgpm_comm_send_str(const char *s, size_t len);
int  libgpm_comm_recive_int(int *i);
int  libgpm_comm_recive_float(double *d);
int  libgpm_comm_recive_str(char *s, size_t blen, size_t *rlen);

/* rwgpm.c */
int regpm(gpm_cont_t *cont);
int wrgpm(gpm_cont_t *cont);
int chgpm(gpm_cont_t *cont, int num);
int libgpm_context_init(const char *src_tmpdir);
int libgpm_context_clean();
int libgpm_context_settempdir(const char *src_tmpdir);

/* gpmwin.c */
int libgpm_window_create(gpm_cont_t *cont, int wnum, 
                         int paper, int orient, double width, double height);
int libgpm_window_close(gpm_cont_t *cont, int wnum);

/* dvifile.c */
int libgpm_dvi_context_init(gpmdvi_context_t *cont);
int libgpm_dvi_context_clean(gpmdvi_context_t *cont);

int libgpm_dvi_create(const char *fname, int paper, int orient, int device);
int libgpm_dvi_open(gpmdvi_context_t *cont, const char *fname, 
                    int is_readonly);
int libgpm_dvi_close(gpmdvi_context_t *cont);
int libgpm_dvi_read_head(gpmdvi_context_t *cont, char *paper, char *orient,
                         char *device);
int libgpm_dvi_read_param(gpmdvi_context_t *cont);
int libgpm_dvi_read_element(gpmdvi_context_t *cont);
int libgpm_dvi_write_param(gpmdvi_context_t *cont, float factor,
                           float xorg, float yorg);
int libgpm_dvi_write_plot(gpmdvi_context_t *cont, double x, double y,
                          int ipen_mode, gpmdev_pen_t *pen);
int libgpm_dvi_write_box(gpmdvi_context_t *cont, double x1, double y1,
                         double x2, double y2, int mode, double level,
                         gpmdev_pen_t *pen);
int libgpm_dvi_write_csymbol(gpmdvi_context_t *cont, double x, double y,
                             gpmdev_pen_t *pen, gpmdev_symbol_t *symbol);
int libgpm_dvi_write_label(gpmdvi_context_t *cont, double x, double y,
                           const char *ichar, double height,
                           double theta, int imode, gpmdev_pen_t *pen,
                           gpmdev_symbol_t *symbol);
int libgpm_dvi_write_roll(gpmdvi_context_t *cont);

/* fontsize.c */
void getlength(int number, float *x, float *y, int rotate);

/* for postscript */
/* psfunc.c */
void ps_context_init(FILE *fp, char paper, char orientation, int is_cps);
void ps_initAdobe(const char *creator, const char *infilename);
void ps_init();
void ps_setTray();
void ps_ginit();
void ps_page(int pageNumber);
void ps_pages(int pageNumber);
void ps_open(int version);
void ps_rainbow_mode(int mode);
void ps_factor(double factor);
void ps_origin(double x, double y);
void ps_color(int color);
void ps_rainbow(double level);
void ps_lwidth(int width);
void ps_ltype(int type);
void ps_plot(double x, double y, int mode);
void ps_box(double x1, double y1, double x2, double y2, int mode, 
            double level, int color);
void ps_csymbol(double x, double y, int cdata, double height);
void ps_string(double x, double y, const char *str, double height, 
               double theta, int centermode, char font);
void ps_roll();
void ps_close();

/* gpm subroutines */
/* gpmcmd.c */
int  checktodvi(gpm_cont_t *cont);
int  gopen(gpm_cont_t *cont, char pa, char ori, 
           char dev, double pw, double ph);
void gclose();
void gflush();
void gnewpen(int color);
void gcsize(double size);
void gfactor(double factor);
void gfsize(double x, double y);
void glimit(double xmin, double ymin, double xmax, double ymax);
void gorigin(double x, double y);
void gpen(int ip_size, int ip_type, int ip_mode);
void gscale(double mwin, double ymin, double xmax, double ymax);
void grainbow(double level);
void gplot(double x, double y, int ipen_mode);
void gbox(double x1, double y1, double x2, double y2, int mode, double level);
void gerase(gpm_cont_t *cont);
void gfont(int type);
void glabel(double x, double y, const char *ichar, double height, 
            double theta, int imode, const char *specialfont);
void ggraph(float *xdata, float *ydata, int idat_number, int inc, 
            int imode, int symbl, float symbl_size);
void gframe();
void gaxis(double x, double y, double aleng, double theta,
           double dmin, double dmax, double ainc, int iformat,
           int imode, char *ctitle, double size, int igrid, double gleng);
void gaxisl(double x, double y, double aleng, double theta, 
            double dmin, double dmax, int iformat, int imode, 
            char *ctitle, double  size, int igrid, double gleng);
void ggradation(int c, double level);
void groll();
void gsymbol(double x, double y, char symbl);

void libgpm_mm2pixel(int paper, int *dst_px, int *dst_py,
                     double src_mx, double src_my);
void libgpm_pixel2mm(int paper, double *dst_mx, double *dst_my, 
                     int src_px, int src_py);

#ifdef __cplusplus
}
#endif

#endif /* _LIBGPM_H_ */

