/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: interp2.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"
  
  
#define sabs(x) fabs((double)(x))

/*****************************************************
*                                                    *
*   interpolate on 2-dimension                       *
*                                                    *
*****************************************************/
#ifdef __cplusplus
extern "C" {
#endif

static void  sfcfit (int iu, int lx, int ly, double **z, int mx, 
		     int my, int nu, int nv, double **w);
  
DLLEXPORT int mod_ispp_interp2(){
  double  **rdata, **rdata2;
  Buffer  *work1, *work2;

  int     dim, idx[MAX_INDEX];
  int     n1, n2, n;
  int     nn, j, i, ii, iii;
  int     ipara3, ipara4, inx1, iny1, inx2, iny2;
  int     tmp1x, tmp1y, tmp2x, tmp2y;
  int     ipara[10];
  
  /*-------------  load system parameter  ---------------------*/

  work1    = GetSeries( 0, &dim, idx );
  ipara[3] = (int)GetScalar( 1 );
  ipara[2] = (int)GetScalar( 2 );

  if ( dim != 2 ) return (16);

  if ( work1 == NULL ) return (4);
  if ( ipara[2] < 0 || ipara[3] < 0 ) return (2);

  tmp1x = idx[1];
  tmp1y = idx[0];
  tmp2x = tmp1x + ipara[2] * (tmp1x-1) ;
  tmp2y = tmp1y + ipara[3] * (tmp1y-1) ;
  
  rdata = (double **) emalloc(sizeof(double *) * tmp1y);
  rdata2 = (double **) emalloc(sizeof(double *) * tmp2y);
  for (i = 0; i < tmp1y; i++)
    rdata[i] = (double *) emalloc(sizeof(double) * tmp1x);
  for (i = 0; i < tmp2y; i++)
    rdata2[i] = (double *) emalloc(sizeof(double) * tmp2x);

  n1 = 1;
  n2 = idx[0];
  
  n = n2 - n1 + 1;
  
  for (iii = 1, i = n1; i <= n2; i++)
    for (j = 1, ii = 1; ii <= tmp1x; ii++, iii++, j++)
      rdata[i - 1][j - 1] = (double)work1[iii - 1];

  ipara3 = ipara[2]+1;
  ipara4 = ipara[3]+1;
  nn = n * ipara3 * idx[1] * ipara4;
  
  inx1 = idx[1];
  iny1 = idx[0];
  inx2 = (inx1-1) * ipara3+1;
  iny2 = (iny1-1) * ipara4+1;
  
  printf("(%d,%d) -> (%d,%d)\n",iny1,inx1,iny2,inx2);

  sfcfit(7, inx1, iny1, rdata, ipara3, ipara4, inx2, iny2, rdata2);
  
  printf("done.\n");

  FreeBuffer(work1);
  for (i = 0; i < tmp1y; i++)
    efree( rdata[i] );
  efree( rdata );

  if (( work2 = AllocBuffer(inx2*iny2))== NULL ) return (8);

  for (iii = 1, i = 1; i <= iny2; i++)
    for (j = 1, ii = 1; ii <= inx2; ii++, iii++, j++ )
      work2[iii - 1] = (double)rdata2[i - 1][j - 1];

  dim = 2;
  idx[0] = iny2;
  idx[1] = inx2;

  ReturnSeries( work2, dim, idx );

  return 0;
}

static void sfcfit(int iu,int lx,int ly,double **z,int mx,int my,int nu,
		   int nv, double **w){
  
  /*
   * this subroutine fits a smooth surface of a single-valued bivariate
   * function z = z(x,y) to a set of input data points given at input
   * grid points in an x-y plane.  it generates a set of output grid
   * points by equally dividing the x and y coordinates in each
   * interval between a pair of input grid points, interpolates the z
   * value for the x and y values of each output grid point, and
   * generates a set of output points consisting of input data points
   * and the interpolated points. the method is based on a piece-wise
   * function composed of a set of bicubic polynomials in x and y.
   * each polynomial is applicable to a rectangle of the input grid in
   * the x-y plane.  each polynomial is determind locally.
   * 
   * the input parameters are
   * 
   * iu  = logacal unit number of standard output unit lx  = number of
   * input grid points in the x coordinate (must be 2 or greater) ly  =
   * number of input grid points in the y coordinate (must be 2 or
   * greater) z   = double-dimensioned array of dimension (lx,ly)
   * storing the values of the function at input grid points mx  =
   * number of subintervals between each pair of input grid points in
   * the x coordinate (must be 2 or greater) my  = number of
   * subintervals between each pair of input grid points in the y
   * coordinate (must be 2 or greater) nu  = number of output grid
   * points in the x coordinate = (lx-1)*mx+1 nv  = number of output
   * grid points in the y coordinate = (ly-1)*my+1
   * 
   * the output parameters are
   * 
   * w   = doubly-dimensioned array of dimension (nu,nv) where the z
   * coordinates of output points are to be displayed
   * 
   * some variables internally used are
   * 
   * za  = divided difference of z with respect to x zb  = divided
   * difference of z with respect to y zab = second order divided
   * difference of z with respect to x and y zx  = partial derivative
   * of z with respect to x zy  = partial derivative of z with respect
   * to y zxy = second order partial derivative of z with respect to x
   * and y
   * 
   * declaration statements
   */
  
  double  za[2][4], zb[5], zab[3][2], zx[2], zy[2], zxy[2];
  double  z3a2 = 0.0, z3a3 = 0.0, z3a4 = 0.0, z3a5 = 0.0;
  double  z4a2 = 0.0, z4a3 = 0.0, z4a4 = 0.0, z4a5 = 0.0;
  double  z4b1 = 0.0, z4b2 = 0.0, z4b3 = 0.0, z4b4 = 0.0, z4b5 = 0.0;
  double  za3b2 = 0.0, za4b2 = 0.0;
  double  za3b3 = 0.0, za4b3 = 0.0;
  double  za3b4 = 0.0, za4b4 = 0.0;
  double  zx43 = 0.0, zx44 = 0.0, zy43 = 0.0, zy44 = 0.0;
  double  zxy43 = 0.0, zxy44 = 0.0;
  double  p00 = 0.0, p01 = 0.0, p10 = 0.0, p11 = 0.0;
  double  z33 = 0.0, zy33 = 0.0, zx33 = 0.0, zxy33 = 0.0;
  int     ixm1, ixml, jx, jy;
  double  du = 0.0, dv = 0.0, dx = 0.0, dy = 0.0;
  double  fmx = 0.0, rmx = 0.0, fmy = 0.0, rmy = 0.0, sw = 0.0, e = 0.0;
  double  w2 = 0.0, wy2 = 0.0, a = 0.0, q0 = 0.0, w3 = 0.0, wy3 = 0.0;
  double  b = 0.0, q1 = 0.0, wx2 = 0.0, c = 0.0, q2 = 0.0;
  double  wx3 = 0.0, d = 0.0, q3 = 0.0, p02 = 0.0, p03 = 0.0;
  double  p12 = 0.0, p13 = 0.0, p20 = 0.0, p21 = 0.0, p22 = 0.0, p23 = 0.0;
  int     iu0, lx0, lxm1, lxm2, ly0, lym1, lym2;
  int     mx0, mxp1, mxm1, my0, myp1, mym1, nu0, nv0;
  int     jymx, kv0, iy, iym2, iym3, iyml, iyml1, ix6;
  int     jxmx, ku0, ix, kv, ku;

  double  y3 = 0.0, y4 = 0.0, b3 = 0.0, b3sq = 0.0, b2 = 0.0, b1 = 0.0;
  double  b4 = 0.0, b5 = 0.0, x3 = 0.0, x4 = 0.0;
  double  z43 = 0.0, z3b3 = 0.0, a3 = 0.0, a4 = 0.0, a3sq = 0.0;
  double  x5 = 0.0, z53 = 0.0;
  double  z5b1 = 0.0, z5b2 = 0.0, z5b3 = 0.0, z5b4 = 0.0, z5b5 = 0.0;
  double  a5 = 0.0, za5b2 = 0.0, za5b3 = 0.0, za5b4 = 0.0;
  double  x6 = 0.0, z63 = 0.0, z54 = 0.0, z64 = 0.0;
  double  z6b1 = 0.0, z6b2 = 0.0, z6b3 = 0.0, z6b4 = 0.0, z6b5 = 0.0;
  double  z62 = 0.0, z65 = 0.0;
  double  zx34 = 0.0, zy34 = 0.0, zxy34 = 0.0;
  double  zx3b3 = 0.0, zx4b3 = 0.0, zy3a3 = 0.0, zy4a3 = 0.0;
  double  p30 = 0.0, p31 = 0.0, p32 = 0.0, p33 = 0.0;
  
  /* dimension z(lx,ly),w(nu,nv) */
  
  
  iu0 = iu;
  lx0 = lx;
  lxm1 = lx0 - 1;
  lxm2 = lxm1 - 1;
  ly0 = ly;
  lym1 = ly0 - 1;
  lym2 = lym1 - 1;
  mx0 = mx;
  mxp1 = mx0 + 1;
  mxm1 = mx0 - 1;
  my0 = my;
  myp1 = my0 + 1;
  mym1 = my0 - 1;
  nu0 = nu;
  nv0 = nv;
  
  /* error check */
  
  if (lxm2 < 0) {
    printf(" \n  ***  LX = 1 OR LESS.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  if (lym2 < 0) {
    printf(" \n  ***  LY = 1 OR LESS.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  if (mxm1 <= 0) {
    printf(" \n  ***  MX = 1 OR LESS.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  if (mym1 <= 0) {
    printf(" \n  ***  MY = 1 OR LESS.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  if (nu0 != lxm1 * mx0 + 1) {
    printf(" \n  ***  IMPROPER NU VALUE.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  if (nv0 != lym1 * my0 + 1) {
    printf(" \n  ***  IMPROPER NV VALUE.\n");
    printf("   LX =%d          MX =%d          NU =%d\n", lx0, mx0, nu0);
    printf("   LY =%d          MY =%d          NV =%d\n", ly0, my0, nv0);
    printf("   ERROR DETECTED IN ROUTINE    SFCFIT");
    return;
  }
  /* main do-loops */
  
  jymx = my0;
  kv0 = 0;
  for (iy = 2; iy <= ly0; iy++) {
    /* printf("S1 "); */
    iym2 = iy - 2;
    iym3 = iym2 - 1;
    iyml = iy - ly0;
    iyml1 = iyml + 1;
    ix6 = 0;
    if (iyml == 0)
      jymx = myp1;
    jxmx = mx0;
    ku0 = 0;
    for (ix = 1; ix <= lx0; ix++) {
      /* printf("S2 "); */
      jx = ixm1 = ix - 1;
      jy = ixml = ix - lx0;
      if (ixml == 0)
	jxmx = mxp1;
      
      /*
       * routine to pick up necessary x, y, and z values,
       * to compute the za, zb, and zab values, and to
       * estimate term when necessary
       */
      
      /* preliminary when ix.eq.1 */
      
      if (ixm1 == 0) {
	/* printf("S3 "); */
	y3 = (double) (iy - 2);
	y4 = (double) (iy - 1);
	b3 = 1.0 / (y4 - y3);
	b3sq = b3 * b3;
	if (iym2 > 0)
	  b2 = 1.0 / (y3 - (double) (iy - 3));
	if (iym3 > 0)
	  b1 = 1.0;
	if (iyml < 0)
	  b4 = 1.0 / ((double) (iy) - y4);
	if (iyml1 < 0)
	  b5 = 1.0;
      }
      /* to save the old values */
      
      else {
	/* printf("S4 "); */
	p02 = za[0][0] = z3a2 = z3a3;
	p03 = za[1][0] = z4a2 = z4a3;
	x3 = x4;
	p00 = z33 = z43;
	z3b3 = z4b3;
	a3 = a4;
	a3sq = a3 * a3;
	za[0][1] = z3a3 = z3a4;
	za[1][1] = z4a3 = z4a4;
	zab[1][0] = za3b3 = za4b3;
	p22 = zab[0][0] = za3b2 = za4b2;
	p23 = zab[2][0] = za3b4 = za4b4;
      num421:	x4 = x5;
	z43 = z53;
	p12 = zb[0] = z4b1 = z5b1;
	p13 = zb[1] = z4b2 = z5b2;
	zb[2] = z4b3 = z5b3;
	p20 = zb[3] = z4b4 = z5b4;
	p21 = zb[4] = z4b5 = z5b5;
	a4 = a5;
	za[0][2] = z3a4 = z3a5;
	za[1][2] = z4a4 = z4a5;
	zab[0][1] = za4b2 = za5b2;
	zab[1][1] = za4b3 = za5b3;
	zab[2][1] = za4b4 = za5b4;
      num422:	x5 = x6;
	z53 = z63;
	z54 = z64;
	z5b1 = z6b1;
	z5b2 = z6b2;
	z5b3 = z6b3;
	z5b4 = z6b4;
	z5b5 = z6b5;
      }
      /*
       * to compute the za, zb, and zab values and to
       * estimate the zb values when
       * (iy.le.3).or.(iy.ge.ly-1)
       */
      
      ix6 = ix6 + 1;
      if (ix6 <= lx0) {
	/* printf("S5 "); */
	x6 = (double) (ix6 - 1);
	z63 = z[iy - 2][ix6 - 1];
	z64 = z[iy - 1][ix6 - 1];
	z6b3 = (z64 - z63) * b3;
	if (lym2 != 0) {
	  if (iym2 != 0) {
	    /* printf("S6 "); */
	    z62 = z[iy - 3][ix6 - 1];
	    z6b2 = (z63 - z62) * b2;
	    if (iyml == 0) {
	      z6b4 = z6b3 + z6b3 - z6b2;
	      goto num433;
	    }
	  }
	  z65 = z[iy][ix6 - 1];
	  z6b4 = (z65 - z64) * b4;
	  if (iym2 != 0)
	    goto num433;
	  z6b2 = z6b3 + z6b3 - z6b4;
	  goto num433;
	}
	z6b2 = z6b3;
	z6b4 = z6b3;
      num433:	if (iym3 > 0) {
	/* printf("S7 "); */
	z6b1 = (z62 - z[iy - 4][ix6 - 1]) * b1;
      } else {
	/* printf("S8 "); */
	z6b1 = z6b2 + z6b2 - z6b3;
      }
	if (iyml1 < 0) {
	  /* printf("S9 "); */
	  z6b5 = (z[iy + 1][ix6 - 1] - z65) * b5;
	} else {
	  /* printf("S10 "); */
	  z6b5 = z6b4 + z6b4 - z6b3;
	}
	if (ix6 == 1)
	  goto num422;
	/* printf("S11 "); */
	a5 = 1.0 / (x6 - x5);
	za[0][3] = z3a5 = (z63 - z53) * a5;
	za[1][3] = z4a5 = (z64 - z54) * a5;
	za5b2 = (z6b2 - z5b2) * a5;
	za5b3 = (z6b3 - z5b3) * a5;
	za5b4 = (z6b4 - z5b4) * a5;
	if (ix6 == 2)
	  goto num421;
      }
      /*
       * to estimate the za and zab values when
       * (ix.ge.lx-1).and.(lx.gt.2)
       */
      
      else {
	/* printf("S12 "); */
	if (lxm2 != 0) {
	  /* printf("S13 "); */
	  za[0][3] = z3a5 = z3a4 + z3a4 - z3a3;
	  za[1][3] = z4a5 = z4a4 + z4a4 - z4a3;
	  if (ixml == 0)
	    goto num500;
	  za5b2 = za4b2 + za4b2 - za3b2;
	  za5b3 = za4b3 + za4b3 - za3b3;
	  za5b4 = za4b4 + za4b4 - za3b4;
	  goto num500;
	}
	/*
	 * to estimate the za and zab values when
	 * (ix.ge.lx-1).and.(lx.eq.2)
	 */
	
	za[0][3] = z3a5 = z3a4;
	za[1][3] = z4a5 = z4a4;
	if (ixml == 0)
	  goto num500;
	za5b2 = za4b2;
	za5b3 = za4b3;
	za5b4 = za4b4;
      }
      /*
       * to estimate the za and zab values when ix.eq.1
       */
      
      if (ixm1 == 0) {
	/* printf("S14 "); */
	za[0][1] = z3a3 = z3a4 + z3a4 - z3a5;
	p02 = za[0][0] = z3a2 = z3a3 + z3a3 - z3a4;
	za[1][1] = z4a3 = z4a4 + z4a4 - z4a5;
	za[1][0] = z4a2 = z4a3 + z4a3 - z4a4;
	p22 = zab[0][0] = za3b2 = za4b2 + za4b2 - za5b2;
	zab[1][0] = za3b3 = za4b3 + za4b3 - za5b3;
	p23 = zab[2][0] = za3b4 = za4b4 + za4b4 - za5b4;
      }
      /*
       * numerical differentiation   ---   to determine
       * partial derivatives zx, zy, and zxy as weighted
       * means of divided differences za, zb, and zab,
       * respectively to save the old values when ix.ne.1
       */
      
      else {
	/* printf("S15 "); */
      num500:	p10 = zx33 = zx43;
	zx34 = zx44;
	p01 = zy33 = zy43;
	zy34 = zy44;
	p11 = zxy33 = zxy43;
	zxy34 = zxy44;
      }
      /* new computation  */
      
      for (ixml = jy = 1; jy <= 2; jy++, ixml++) {
	/* printf("S16 "); */
	q0 = a = wy2 = w2 = sabs(za[jy - 1][3] - za[jy - 1][2]);
	q1 = b = wy3 = w3 = sabs(za[jy - 1][1] - za[jy - 1][0]);
	e = rmy = fmy = rmx = fmx = sw = w2 + w3;
	if (sw != 0.0) {
	  q2 = c = wx2 = w2 / sw;
	  q3 = d = wx3 = w3 / sw;
	} else {
	  q2 = c = wx2 = 0.5;
	  q3 = d = wx3 = 0.5;
	}
	zx[jy - 1] = wx2 * za[jy - 1][1] + wx3 * za[jy - 1][2];
	zx43 = zx[0];
	zx44 = zx[1];
	q0 = a = wy2 = w2 = sabs(zb[jy + 2] - zb[jy + 1]);
	q1 = b = wy3 = w3 = sabs(zb[jy] - zb[jy - 1]);
	fmx = rmx = fmy = rmy = e = sw = w2 + w3;
	if (sw != 0.0) {
	  q0 = a = w2 = wy2 = w2 / sw;
	  q1 = b = w3 = wy3 = w3 / sw;
	} else {
	  q0 = a = w2 = wy2 = 0.5;
	  q1 = b = w3 = wy3 = 0.5;
	}
	zy[jy - 1] = wy2 * zb[jy] + wy3 * zb[jy + 1];
	zy43 = zy[0];
	zy44 = zy[1];
	zxy[jy - 1] = wy2 * (wx2 * zab[jy - 1][0] + wx3 * zab[jy - 1][1])
	  + wy3 * (wx2 * zab[jy][0] + wx3 * zab[jy][1]);
	zxy43 = zxy[0];
	zxy44 = zxy[1];
      }
      if (ixm1 != 0) {
	
	/*
	 * determination of the coefficients of the
	 * polynomial
	 */
	
	/* printf("S17 "); */
	zx3b3 = (zx34 - zx33) * b3;
	zx4b3 = (zx44 - zx43) * b3;
	zy3a3 = (zy43 - zy33) * a3;
	zy4a3 = (zy44 - zy34) * a3;
	q0 = wy2 = w2 = a = za3b3 - zx3b3 - zy3a3 + zxy33;
	q1 = wy3 = w3 = b = zx4b3 - zx3b3 - zxy43 + zxy33;
	q2 = wx2 = c = zy4a3 - zy3a3 - zxy34 + zxy33;
	q3 = wx3 = d = zxy44 - zxy43 - zxy34 + zxy33;
	fmx = rmx = fmy = rmy = sw = e = a + a - b - c;
	z3a2 = za[0][0] = p02 = (2.0 * (z3b3 - zy33) + z3b3 - zy34) * b3;
	z4a2 = za[1][0] = p03 = (-2.0 * z3b3 + zy34 + zy33) * b3sq;
	z4b1 = zb[0] = p12 = (2.0 * (zx3b3 - zxy33) + zx3b3 - zxy34) * b3;
	z4b2 = zb[1] = p13 = (-2.0 * zx3b3 + zxy34 + zxy33) * b3sq;
	z4b4 = zb[3] = p20 = (2.0 * (z3a3 - zx33) + z3a3 - zx43) * a3;
	z4b5 = zb[4] = p21 = (2.0 * (zy3a3 - zxy33) + zy3a3 - zxy43) * a3;
	za3b2 = zab[0][0] = p22 = (3.0 * (a + e) + d) * a3 * b3;
	za3b4 = zab[2][0] = p23 = (-3.0 * e - b - d) * a3 * b3sq;
	p30 = (-2.0 * z3a3 + zx43 + zx33) * a3sq;
	p31 = (-2.0 * zy3a3 + zxy43 + zxy33) * a3sq;
	p32 = (-3.0 * e - c - d) * b3 * a3sq;
	p33 = (d + e + e) * a3sq * b3sq;
	
	/* computation of the polynomial */
	
	for (jy = 1; jy <= jymx; jy++) {
	  /* printf("S18 "); */
	  kv = kv0 + jy;
	  du = dv = dx = dy = (double) (kv - 1) / (double) (my) - y3;
	  w2 = wy2 = a = q0 = p00 + dy * (p01 + dy * (p02 + dy * p03));
	  w3 = wy3 = b = q1 = p10 + dy * (p11 + dy * (p12 + dy * p13));
	  wx2 = c = q2 = p20 + dy * (p21 + dy * (p22 + dy * p23));
	  wx3 = d = q3 = p30 + dy * (p31 + dy * (p32 + dy * p33));
	  for (jx = 1; jx <= jxmx; jx++) {
	    /*
	     * printf("S19 ixm1=%d jx=%d
	     * \n",ixm1,jx);
	     */
	    ku = ku0 + jx;
	    du = dv = dy = dx = (double) (ku - 1) / (double) (mx) - x3;
	    w[kv - 1][ku - 1] = q0 + dx * (q1 + dx * (q2 + dx * q3));
	  }
	  ixm1 = jx;
	}
	ixml = jy;
	ku0 = ku0 + mx0;
      }
    }
    kv0 = kv0 + my0;
  }
  return;
}
#ifdef __cplusplus
}
#endif
