/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: norm.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

/******************************************************
*           normalize                                 *
*---------------------------------------------------- *
*     norm ibuf1,typ4                                 *
*          ibuf1 : input data buffer                  *
*          typ4  :                                    *
*            = 1 : normalize at MAX                   *
*            = 2 : normalize at MIN                   *
*            = 3 : normalize at abusolute MAX         *
*            = 4 : normalize at abusolute MIN         *
*            = 5 : normalize at standard deviation    *
*-----------------------------------------------------*
*                   1986 , 7 , 29                     *
*                   1993 , 4 , 15                     *
*                   1996 , 4 ,  5 (added typ4 = 5)    *
******************************************************/
#ifdef __cplusplus
extern "C" {
#endif

static int    srch   (Buffer *data, int leng, int iflag);
static double absol  (double inp);


DLLEXPORT int mod_ispp_norm(){
  double xnorm, xmean, tmp;
  Buffer *data;
  int    indx, iaddr, i;
  int    dim, index[MAX_INDEX], leng;
  
  data   = GetSeries( 0, &dim, index );

  if ( data == NULL )
    return (4);

  iaddr  = (int)GetScalar( 1 );
  leng = IndexSize( dim, index );

  if ( ( iaddr < 1) || ( iaddr > 5) ) {
    printf("bud type of normalization\n");
    return (2);
  }

  if ( ( iaddr <= 4) ){
    indx = srch( data, leng, iaddr );

    xnorm = data[ indx ];
    if ( xnorm == 0.0 ) {
      printf("cannot normalize by zero\n");
      return (2);
    }
  
    for (i = 0; i < leng; i++)
      data[ i ] = data[ i ] / xnorm;

  }else{ 	/* 1996.4.5. by Dora */
    tmp = 0.0;
    for (i = 0; i < leng; i++)
      tmp = tmp + data[ i ];
    xmean = tmp / (double)leng;	
/*    printf("MEAN = %f\n", xmean); */
    
    tmp = 0.0;
    for (i = 0; i < leng; i++)
      tmp = tmp + pow( (data[ i ] - xmean), 2.0 );
    xnorm = sqrt( tmp / (double)leng );
    printf("Standard Deviation = %f\n", xnorm);

    for (i = 0; i < leng; i++)
      data[ i ] = ( data[ i ] - xmean ) / xnorm;
  }

  ReturnSeries( data, dim, index );
  FreeBuffer( data );
  return 0;
}

/*****************************************************
 *                                                    *
 *         srch ( data , iflag  )                     *
 *                                                    *
 *----------------------------------------------------*
 *    search max , min , absmax , absmin              *
 *****************************************************/

static int srch(Buffer *data,int leng,int iflag){
  double xmax, xmin, abxmax, abxmin, abx;
  int    i, indx;

  indx = 0;
  switch (iflag) {

    /* .....max search */
  case 1:
    xmax = data[0];
    for (i = 1; i < leng; i++) {
      if ( data[i] > xmax ) {
	xmax = data[i];
	indx = i;
      }
    }
    break;
    
    /* .....min search */
  case 2:
    xmin = data[0];
    for (i = 1; i < leng; i++) {
      if (data[i] < xmin ) {
	xmin = data[i];
	indx = i;
      }
    }
    break;
    
    /* .....abs max serch */
  case 3:
    abxmax = absol(data[0]);
    for (i = 1; i < leng; i++) {
      abx = absol(data[i]);
      if ( abx > abxmax) {
	abxmax = abx;
	indx = i;
      }
    }
    break;
    
    /* .....abs min search */
  case 4:
    abxmin = absol(data[0]);
    for (i = 1; i < leng; i++) {
      abx = absol(data[i]);
      if ( abx < abxmin ) {
	abxmin = abx;
	indx = i;
      }
    }
    break;
  }
  return( indx );
}

static double absol(double inp){
  if ( inp < 0 )
    inp = -inp;
  return( inp );
}
#ifdef __cplusplus
}
#endif

