/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: pole.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include "SL_macro.h"
#include "SL_cmd.h"

/*****************************************************
*                                                    *
*   POLE  PBUF , IBX , IBY                           *
*                                                    *
*         PBUF : AR PARAMETER BUFFER                 *
*         IBX  : REAL                                *
*         IBY  : IMAGINARY                           *
*                                                    *
*----------------------------------------------------*
*              CODED BY Dora       Sep 30, 1999      *
*****************************************************/

#define FMAX(a,b) ((double)(((double)(a) > (double)(b)) ? (a) : (b)))

#ifdef __cplusplus
extern "C" {
#endif

struct fcom{
  double r;
  double i;
};

typedef struct fcom fcomplex;

fcomplex Cadd    (fcomplex a, fcomplex b);
fcomplex Csub    (fcomplex a, fcomplex b);
fcomplex Cmul    (fcomplex a, fcomplex b);
fcomplex Cdiv    (fcomplex a, fcomplex b);
fcomplex Complex (double re, double im);
double   Cabs    (fcomplex z);
fcomplex Csqrt   (fcomplex z);
fcomplex RCmul   (double x, fcomplex a);
void     laguer  (fcomplex *a, int m, fcomplex *x);
void     zroots  (fcomplex *a, int m, fcomplex *roots);

DLLEXPORT int mod_ispp_pole(){
  Buffer *coef, *data1, *data2;
  int    dim, index[MAX_INDEX];
  int    ibx, iby;
  int    i, m;
  fcomplex a[101],roots[101];

  if ((coef = GetSeries( 0, &dim, index )) == NULL ) 
    return (4);

  ibx    = GetBufferID(1);
  iby    = GetBufferID(2);

  m = index[0];
  if ((data1 = AllocBuffer(m)) == NULL) return (8);
  if ((data2 = AllocBuffer(m)) == NULL) return (8);

/*
  a[0] = Complex(1.0,0.0);
  for (i=1; i<=m; i++)
    a[i] = Complex(coef[i-1],0.0);
*/

  a[m] = Complex(1.0,0.0);
  for (i=1; i<=m; i++)
    a[m-i] = Complex(coef[i-1],0.0);

  zroots(a,m,roots);

  for (i=1; i<=m; i++){
    data1[i-1] = roots[i-1].r;
    data2[i-1] = roots[i-1].i;
  }

  if ( WriteBuffer( ibx, dim, index, data1 ) == -1 ) return (3);
  if ( WriteBuffer( iby, dim, index, data2 ) == -1 ) return (3);
  FreeBuffer(data1);
  FreeBuffer(data2);
  return 0;
}

fcomplex Cadd(fcomplex a,fcomplex b){
  fcomplex c;

  c.r = a.r + b.r;
  c.i = a.i + b.i;
  return c;
}


fcomplex Csub(fcomplex a,fcomplex b){
  fcomplex c;

  c.r = a.r - b.r;
  c.i = a.i - b.i;
  return c;
}

fcomplex Cmul(fcomplex a,fcomplex b){
  fcomplex c;

  c.r = a.r*b.r - a.i*b.i;
  c.i = a.r*b.i + a.i*b.r;
  return c;
}


fcomplex Cdiv(fcomplex a,fcomplex b){
  fcomplex c;
  double r,den;

  if (fabs(b.r) >= fabs(b.i)){
    r = b.i / b.r;
    den = b.r + r*b.i;
    c.r = (a.r+r*a.i) / den;
    c.i = (a.i-r*a.r) / den;
  }else{
    r = b.r / b.i;
    den = b.i + r*b.r;
    c.r = (a.r*r+a.i) / den;
    c.i = (a.i*r-a.r) / den;
  }
  return c;
}


fcomplex Complex(double re,double im){
  fcomplex c;

  c.r = re;
  c.i = im;
  return c;
}


double Cabs(fcomplex z){
  double x, y, ans, temp;

  x = fabs(z.r);
  y = fabs(z.i);
  if (x == 0.0){
    ans = y;
  }else{
    if (y == 0.0) {
      ans = x;
    }else{
      if (x > y){
        temp = y/x;
        ans = x * sqrt(1.0+temp*temp);
      }else{
        temp = x/y;
        ans = y * sqrt(1.0+temp*temp);
      }
    }
  }
  return ans;
}


fcomplex Csqrt(fcomplex z){
  fcomplex c;
  double x,y,w,r;

  if ((z.r == 0.0) && (z.i == 0.0)) {
    c.r = 0.0;
    c.i = 0.0;
    return c;
  }else{
    x = fabs(z.r);
    y = fabs(z.i);
    if (x >= y) {
      r = y/x;
      w = sqrt(x) * sqrt(0.5*(1.0+sqrt(1.0+r*r)));
    }else{
      r = x/y;
      w = sqrt(y) * sqrt(0.5*(r+sqrt(1.0+r*r)));
    }
    if (z.r >= 0.0){
      c.r = w;
      c.i = z.i / (2.0 * w);
    }else{
      c.i = (z.i >= 0.0) ? w : -w;
      c.r = z.i / (2.0 * c.i);
    }
    return c;
  }
}


fcomplex RCmul(double x,fcomplex a){
  fcomplex c;

  c.r = x * a.r;
  c.i = x * a.i;
  return c;
}


void laguer(fcomplex a[],int m,fcomplex *x){
  int iter,j;
  double abx,abp,abm,err;
  fcomplex dx,x1,b,d,f,g,h,sq,gp,gm,g2;
  static double frac[10] = {0.0,0.5,0.25,0.75,0.13,0.38,0.62,0.88,1.0};
  
  for (iter = 1; iter <= 80; iter++){
    b = a[m];
    err = Cabs(b);
    d = f = Complex(0.0,0.0);
    abx = Cabs(*x);

    for (j=m-1; j>=0; j--){
      f = Cadd(Cmul(*x,f),d);
      d = Cadd(Cmul(*x,d),b);
      b = Cadd(Cmul(*x,b),a[j]);
      err = Cabs(b) + abx * err;
    }

    err *= 1.0e-7;
    if (Cabs(b) <= err) return;
    g = Cdiv(d,b);
    g2 = Cmul(g,g);
    h = Csub(g2,RCmul(2.0,Cdiv(f,b)));
    sq = Csqrt(RCmul((double)(m-1),Csub(RCmul((double)m,h),g2)));
    gp = Cadd(g,sq);
    gm = Csub(g,sq);
    abp = Cabs(gp);
    abm = Cabs(gm);
    if (abp < abm) gp = gm;
    dx = ((FMAX(abp,abm) > 0.0 ? Cdiv(Complex((double)m,0.0),gp)
      : RCmul(exp(log(1+abx)),Complex(cos((double)iter),sin((double)iter)))));
    x1 = Csub(*x,dx);
    if (x->r == x1.r && x->i == x1.i) return;
    if (iter % 10) *x = x1;
    else *x = Csub(*x,RCmul(frac[iter/10],dx));
  }
  printf("Too Many Iterations in Laguer Method!\n");
  return;
}


void zroots(fcomplex a[],int m,fcomplex roots[]){
  int i,j,jj;
  fcomplex x,b,c,ad[101];

  for (j=0; j<=m; j++)
    ad[j] = a[j];
  for (j=m; j>=1; j--){
    x = Complex(0.0,0.0);
    laguer(ad,j,&x);
    if (fabs(x.i) <= 4.0e-6 * fabs(x.r)) x.i = 0.0;
    roots[j-1] = x;
    b = ad[j];
    for (jj=j-1; jj>=0; jj--){
      c = ad[jj];
      ad[jj] = b;
      b = Cadd(Cmul(x,b),c);
    }
  }

  for (j=1; j<=m; j++)
    laguer(a,m,&roots[j-1]);
  for (j=2; j<=m; j++){
    x = roots[j-1];
    for (i=j-1; i>=1; i--){
      if (roots[i-1].r <= x.r) break;
      roots[i] = roots[i-1];
    }
    roots[i] = x;
  }
}
#ifdef __cplusplus
}
#endif
