/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: spectr2.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $ */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "SL_macro.h"
#include "SL_cmd.h"
#include "SLfloat.h"

/*****************************************************
*   Power Spectrum by FFT                            *
*----------------------------------------------------*
*   SPCF  IBUF1 , IBUF2 , IBUF3                      *
*       IBUF1 : input data                           *
*       IBUF2 : output data (power)                  *
*       IBUF3 : output data (phase)                  *
*----------------------------------------------------*
*                       1986, 6, 25                  *
*            C Version  1989,10,  5  T. Kobayashi    *
*                       1993, 9, 14  K. Takebe       *
*****************************************************/

#define PAI  M_PI
#define PAI2 (PAI*2.0)
#ifdef __cplusplus
extern "C" {
#endif

static int  sspec2 (int *ret, Buffer *xr, Buffer *xi,
                    Buffer *pw, Buffer *ph, int dpt);

/* in fft_sub.c */
extern int  fft1 (double *ar, double *ai, int n, int flag );

DLLEXPORT int mod_ispp_spectr2(){
  Buffer *xr, *xi, *max, *xxi;
  Buffer *Power, *Phase;
  int	 nn1, ob1, ob2, nn2;
  int    dim, idx[MAX_INDEX];
  int    stat;

  if (( xr = GetSeries( 0, &dim, idx ) ) == NULL )
    return (4);
  nn1 = IndexSize( dim, idx );
  if ( dim != 1 )
    return (2);

  if (( xi = AllocBuffer(nn1) ) == NULL )
    return (8);

  ob1 = GetBufferID(1);
  ob2 = GetBufferID(2);
  if ( ob1 <= 0 || ob2 <= 0 )
    return (2);

  Power = xr;
  Phase = xi;

  max = xi + nn1;
  for( xxi = Phase; xxi < max; xxi++ )
    (*xxi) = 0.0;
  
  fft1( xr, xi, nn1, 0 );
  
  stat = sspec2( &nn2, xr, xi, Power, Phase, nn1 );
  if(stat != 0) return stat;
  
  dim = 1;
  idx[0] = nn2;
  if ( WriteBuffer( ob1, dim, idx, Power ) == -1 ) return (3);
  if ( WriteBuffer( ob2, dim, idx, Phase ) == -1 ) return (3);
  return 0;
}


static int sspec2(int *ret, Buffer *xr, Buffer *xi,
                  Buffer *pw, Buffer *ph, int dpt){
    /*--------------------------------
      Subroutine sspec
      
      xr : Real Data 
      xi : Imagenary Data
      pw : Power
      ph : Phase
      dpt: Calc Data Point
      
      return dpt 
      ---------------------------------*/
  double dif, dph, dph1, dph2;
  double phps, phng;
  double real, imag;
  int	 m, i, i1;
  double samf;
  
  samf = (double)get_sampling();
  
  if( samf <= 0.0 )
    return ( 21 );
  
  m = dpt / 2 + 1;
  
  for( i = 0; i < m; i ++ ) {
    real = xr[ i ];
    imag = xi[ i ];
    
    pw[ i ] = ( real * real + imag * imag )/ ( (1.0/samf) * (Buffer)dpt );
    
    if( fabs(real) > DBL_EPSILON || fabs(imag) > DBL_EPSILON )
      ph[ i ] = atan2( imag, real );
    else {
      ph[ i ] = 0.0;
    }
  }
  
  phps = phng = 0.0;
  
  for( i = 1; i < m; i ++ ) {
    i1   = i - 1;
    dif  = ph[ i ] - ph[ i1 ] + phps - phng;
    dph  = fabs( dif );
    dph1 = fabs( dif + PAI2 );
    dph2 = fabs( dif - PAI2 );
    
    if( dph > dph1 ) {
      phps += PAI2;
    } else if ( dph > dph2 ) {
      phng += PAI2;
    }    
    ph[ i ] += ( phps - phng );
  }
  
  for( i = 0; i < m; i ++ ) ph[ i ] *= ( 180.0 / PAI );
  
  *ret = m;
  return 0;
}
#ifdef __cplusplus
}
#endif
