/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: nlink.cpp,v 1.6 2005/11/11 07:43:46 orrisroot Exp $ */
/*****************************************************************************
  
                    	NCS LINKER COMMAND

*****************************************************************************/

#ifdef WIN32
#include <windows.h>
#endif
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>

#ifdef WIN32
#include <process.h>
#include <direct.h>
#include <io.h>
#else
#include <unistd.h>
#endif

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libncsc.h"
#include "ncs.h"

#define CONSOLE(x) syscom->console->x
#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH  MAXPATHLEN
# else
#  define _MAX_PATH  1024
# endif
#endif

#define COMPILER_TYPE_MSVC   (0)
#define COMPILER_TYPE_BCC    (1)
#define COMPILER_TYPE_MINGW  (2)
#define COMPILER_TYPE_UNIX   (3)

#ifdef __cplusplus
extern "C" {
#endif

#define CONSOLE(x) syscom->console->x

static void print_link_error_message(const char *model);

static int create_nlink_makewrapper(const char *model, const char *flag,
                                    int cxxtype);

#ifdef  WIN32
static int get_compiler_type();
static int create_nlink_makefile(const char *model, const char *option, 
                                 const char *library,int cxxtype);
static int create_nlink_makefile_msvc(const char *model, const char *option, 
                                      const char *library);
static int create_nlink_makefile_mingw(const char *model, const char *option, 
                                        const char *library);
static char *filename_dos2unix(const char *dos);
/* static char *pathname_unix2dos(const char *unix); */
#else
static int create_nlink_makefile_unix(const char *model, const char *option, 
                                      const char *library);
#endif

DLLEXPORT int mod_ncs_nlink()
{
  int   code;
  char *option, *flag, *library;
  
  /* optimize level */
  option   = GetString(0);
  /* file output flag */
  flag     = GetString(1);
  /* depended library */
  library  = GetString(2);
  code = do_ncslink_cmd(option, flag, library);
  return code;
}

/* get compiler type */
#ifdef WIN32
static int get_compiler_type()
{
  char path[_MAX_PATH],buf[_MAX_PATH];
  FILE *fp;
  int compfg = COMPILER_TYPE_MINGW;
  if(GetSystemResourceDirectory(path, _MAX_PATH)) {
    strcat(path, "\\compiler.ini");
    if((fp = fopen(path, "r")) != NULL) {
      fgets(buf, _MAX_PATH, fp);
      buf[strlen(buf)-1] = '\0';
      fclose(fp);
      if(strcmp(buf, "VC++") == NULL)
        compfg = COMPILER_TYPE_MSVC;
      else if(strcmp(buf, "BCC") == NULL)
        compfg = COMPILER_TYPE_BCC;
    }
  }
  return compfg;
}
#endif

int do_ncslink_cmd(const char *option, const char *flag, const char *library){
  char  scf_name[NCS_FNAME_WD], scf_model[NCS_FNAME_WD];
  char  scf_lib[NCS_FNAME_WD],  scf_header[NCS_FNAME_WD];
  int  code,pid;
  char bin[NCS_FNAME_WD];
  static char *compiler_name[] =
    {"Microsoft Visual C++","Borland C++","Mingw32 GCC", NULL};
  char *cxx;
  int  compiler_type = COMPILER_TYPE_UNIX;

  if(option == NULL) option = "-g -O2";
  if(flag == NULL) flag = "";
  if(library == NULL) library = "";


  /* get model name and simulation condition files name */
  if( ModelNameSCFN( scf_name ) == NULL ){
    return(85);
  }
  if( GetSCFN( scf_model, scf_header, scf_lib ) == NULL ){
    return(85);
  }

#ifdef WIN32
  compiler_type = get_compiler_type();
  code = create_nlink_makefile(scf_name, option, library, compiler_type);
#else
  code = create_nlink_makefile_unix(scf_name, option, library);
#endif

  if(code !=0 ) return (code);
  code = create_nlink_makewrapper(scf_name, flag, compiler_type);
  if(code !=0 ) return (code);

  cxx = compiler_name[compiler_type];
  if(cxx == NULL){
    cxx = getenv("CXX");
#ifndef WIN32
    if(cxx == NULL) cxx = NCS_CXX;
#endif
  }

  printf("%s compiler selected.\n", cxx);
  printf("Linking now ...\n" );
#ifdef WIN32
  snprintf(bin, NCS_FNAME_WD, "\"%s/%s.bat\"", get_tmpdir(), scf_name);
  {
    char *p;
    for(p=bin; *p != '\0'; p++) if(*p == '/') *p = '\\';
  }
#else
  snprintf(bin, NCS_FNAME_WD, "%s/%s.sh", get_tmpdir(), scf_name);
#endif
  pid = create_child_process(bin, CONSOLE(ifd), CONSOLE(ofd), CONSOLE(efd), 
                             NULL);
  if(pid == -1){
    code = 436; /* could not create process */
  }else{
    int status;
    status = wait_child_process(pid, &code);
    if(status == -1){
      code = 450; /* process not found */
    }else if(status == 0){  /* normal exit */
      if(code == 0) printf("done.\n"); /* no error */
    }else if(status == 1){
      code = 451; /* signal caught */
    }else if(status == 2){
      code = 452; /* process is stoping */
    }else if(status == 3){
      code = 453; /* core dumped */
    }else{ 
      code = 454; /* unknown error */
    }
  }
  if(code != 0)
    print_link_error_message(scf_name);

  return code;
}
#ifdef WIN32
static int create_nlink_makefile(const char *model, const char *option, 
                                 const char *library, int cxxtype)
{
  int code = 254; /* fail to create makefile */ 
  switch(cxxtype){
  case COMPILER_TYPE_MSVC:
    code = create_nlink_makefile_msvc(model, option, library);
    break;
  case COMPILER_TYPE_MINGW:
    code = create_nlink_makefile_mingw(model, option, library);
    break;
  }
  return code;
}

static int create_nlink_makefile_msvc(const char *model, const char *option, 
                                      const char *library)
{
  /* for msvc */
  const char *tmp;
  char  *ncs_tmpdir,*p;
  FILE  *fp;
  char	 makefile[NCS_FNAME_WD];
  char   sl4drv[_MAX_DRIVE], sl4prefix[_MAX_DIR], sl4path[_MAX_PATH];
  char   opts[64];
  size_t len;

  /* get full path info */
  ::GetModuleFileName(NULL, sl4path, _MAX_PATH);
  /* split drive and directory */
  _splitpath(sl4path, sl4drv, sl4prefix, NULL, NULL);
  len = strlen(sl4prefix);
  if(sl4prefix[len-1] == '\\') sl4prefix[len-1] = '\0';

  /* parse option */
  opts[0]='\0';
  if(strstr(option,"-O4") != NULL){
    strcat(opts, "/O2");
  }else if(strstr(option,"-O3") != NULL){
    strcat(opts, "/O2");
  }else if(strstr(option,"-O2") != NULL){
    strcat(opts, "/O2");
  }else if(strstr(option,"-O") != NULL){
    strcat(opts, "/O1");
  }
  /* ignore option '-g' */

  /* get temporary directory */
  tmp =  get_tmpdir();
  ncs_tmpdir = strdup(tmp);
  if(ncs_tmpdir == NULL)
    return 213; /* memory allocation error */
  for(p=ncs_tmpdir; *p!='\0'; p++)
    if(*p == '/') *p='\\';

  /* create a makefile */
  snprintf(makefile, NCS_FNAME_WD, "%s\\%s.mak", ncs_tmpdir, model);
  fp = fopen(makefile, "w");
  if(fp == NULL)
    return(254); /* fail to create makefile. */
  
  fprintf(fp,"CXX = cl\n");
  fprintf(fp,"DFLAGS = /Wall\n");
  fprintf(fp,"CURINC = /I .");
  fprintf(fp,"SL4INC = /I \"%s%s\\include\"\n",sl4drv, sl4prefix);
  fprintf(fp,"NCSINC = /I \"%s%s\\include\\ncs\"\n",sl4drv, sl4prefix);
  fprintf(fp,"LIBSATELLITE = \"%s%s\\lib\\libsatellite.lib\"\n",
          sl4drv, sl4prefix);
  /*  fprintf(fp,"NCSLIBC = \"%s%s\\lib\\ncs\\libncsc.lib\"\n", */
  /*          sl4drv, sl4prefix); */
  fprintf(fp,"NCSLIBS = \"%s%s\\lib\\ncs\\libncss.lib\"\n",
          sl4drv, sl4prefix);
  fprintf(fp,"\n");

  fprintf(fp,"SRC    = \"%s\\%s.cpp\"\n", ncs_tmpdir, model);
  fprintf(fp,"OBJ    = \"%s\\%s.obj\"\n", ncs_tmpdir, model);
  fprintf(fp,"TARGET = \"%s\\%s.dll\"\n", ncs_tmpdir, model);
  fprintf(fp,"\n");

  fprintf(fp,"INCLUDES = $(CURINC) $(SL4INC) $(NCSINC)\n");
  
  fprintf(fp,"NCS_CXXFLAGS = /Gm /ZI /MD /D \"WIN32\"\n");
  fprintf(fp,"NCS_LDFLAGS = /link /DLL /OUT:$(TARGET)\n");
  fprintf(fp,"NCS_LDADD = \"%s\"\n", library);
  
  fprintf(fp,"OPTS = %s\n", opts);

  fprintf(fp,"CXXFLAGS = $(DFLAGS) $(OPTS) $(NCS_CXXFLAGS) $(INCLUDES)\n");
  
  fprintf(fp,"\n");

  fprintf(fp,"LDFLAGS = $(NCS_LDFLAGS)\n");
  /*  fprintf(fp,"LDADD = $(NCSLIBS) $(NCSLIBC) $(LIBSATELLITE)\n"); */
  fprintf(fp,"LDADD = $(NCSLIBS) $(NCSLIBC) $(LIBSATELLITE)\n");

  fprintf(fp,"\n");
  
  fprintf(fp,"all: $(TARGET)\n");
  fprintf(fp,"\n");

  fprintf(fp,"$(TARGET) : $(SRC)\n");
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) $(SRC) $(LDADD) $(LDFLAGS)\n");
  fprintf(fp,"\n");

  fclose(fp);
  free(ncs_tmpdir);

  return(0);
}

static int create_nlink_makefile_mingw(const char *model, const char *option, 
                                       const char *library){
  /* for mingw */
  int   retcode = 0;
  char  *ncs_tmpdir;
  FILE  *fp;
  char	 makefile[NCS_FNAME_WD];
  char   sl4drv[_MAX_DRIVE], sl4prefix_dos[_MAX_DIR], *sl4prefix, sl4path[_MAX_PATH];
  size_t len;

  ncs_tmpdir = NULL;
  sl4prefix = NULL;

  /* get full path info */
  ::GetModuleFileName(NULL, sl4path, _MAX_PATH);
  /* split drive and directory */
  _splitpath(sl4path, sl4drv, sl4prefix_dos, NULL, NULL);
  len = strlen(sl4prefix_dos);
  if(sl4prefix_dos[len-1] == '\\') sl4prefix_dos[len-1] = '\0';

  sl4prefix = filename_dos2unix(sl4prefix_dos);
  if(sl4prefix == NULL) return 213; /* memory allocation error */

  /* get temporary directory */
  ncs_tmpdir = filename_dos2unix(get_tmpdir());
  if(ncs_tmpdir == NULL){ free(sl4prefix); return 213; } /* memory allocation error */

  /* create a makefile */
  snprintf(makefile, NCS_FNAME_WD, "%s\\%s.mak", get_tmpdir(), model);
  fp = fopen(makefile, "w");
  if(fp == NULL)
    return(254); /* fail to create makefile. */
  
  fprintf(fp,"CXX = mingw32-g++\n");
  fprintf(fp,"DFLAGS = -Wall\n");

  fprintf(fp,"CURINC = -I.\n");
  fprintf(fp,"SL4INC = -I%s%s/include\n", sl4drv, sl4prefix);
  fprintf(fp,"NCSINC = -I%s%s/include/ncs\n", sl4drv, sl4prefix);
  fprintf(fp,"LIBSATELLITE = %s%s/libsatellite.dll\n", sl4drv, sl4prefix);
  /*  fprintf(fp,"NCSLIBC =  %s%s/libncsc.dll\n", sl4drv, sl4prefix); */
  fprintf(fp,"NCSLIBS =  %s%s/modules/ncs/libncss.dll\n",
          sl4drv, sl4prefix);
  fprintf(fp,"\n");

  fprintf(fp,"SRC    = %s/%s.cpp\n", ncs_tmpdir, model);
  fprintf(fp,"OBJ    = %s/%s.obj\n", ncs_tmpdir, model);
  fprintf(fp,"TARGET = %s/%s.dll\n", ncs_tmpdir, model);
  fprintf(fp,"\n");

  fprintf(fp,"INCLUDES = $(CURINC) $(SL4INC) $(NCSINC)\n");
#ifdef  DSPTEC_EXTENSION
  fprintf(fp,"NCS_CXXFLAGS = -DWIN32 -DDSPTEC_EXTENSION\n");
#else
  fprintf(fp,"NCS_CXXFLAGS = -DWIN32\n");
#endif
  fprintf(fp,"NCS_LDFLAGS = -shared -Wl,--export-all-symbols -Wl,--enable-auto-import -Wl,--no-whole-archive -o $(TARGET)\n");
  fprintf(fp,"NCS_LDADD = %s\n", library);
  
  fprintf(fp,"OPTS = %s\n", option);

  fprintf(fp,"CXXFLAGS = $(DFLAGS) $(OPTS) $(NCS_CXXFLAGS) $(INCLUDES)\n");
  
  fprintf(fp,"\n");

  fprintf(fp,"LDFLAGS = $(NCS_LDFLAGS)\n");
  /* fprintf(fp,"LDADD = $(NCSLIBS) $(NCSLIBC) $(LIBSATELLITE)\n"); */
  fprintf(fp,"LDADD = $(NCSLIBS) $(LIBSATELLITE)\n");

  fprintf(fp,"\n");
  
  fprintf(fp,"all: $(TARGET)\n");
  fprintf(fp,"\n");

  fprintf(fp,"$(TARGET) : $(SRC)\n");
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) $(SRC) $(LDADD) $(LDFLAGS)\n");
  fprintf(fp,"\n");

  fclose(fp);
  free(ncs_tmpdir);
  free(sl4prefix);

  return(0);
}
#else
static int create_nlink_makefile_unix(const char *model, const char *option, 
                                      const char *library){
  /* for UNIX */
  FILE  *fp;
  char  *gxx_tmp;
  char	 makefile[NCS_FNAME_WD];
  const char *ncs_tmpdir;

  ncs_tmpdir = get_tmpdir();

  /* create a makefile */
  snprintf(makefile, NCS_FNAME_WD, "%s/%s.mak", ncs_tmpdir, model);
  fp = fopen(makefile, "w");
  if(fp == NULL)
    return(254);
  
  gxx_tmp = getenv("CXX");
  fprintf(fp,"CXX = %s\n",(gxx_tmp!=0) ? gxx_tmp : NCS_CXX);
  fprintf(fp,"DFLAGS = -g -ggdb\n");
  fprintf(fp,"INCLUDES = -I. -I%s -I%s\n", 
          SL4_INCLUDE_DIR, NCS_INCLUDE_DIR);
  
  fprintf(fp,"NCS_CXXFLAGS = %s\n", NCS_CXXFLAGS);
  fprintf(fp,"NCS_LDFLAGS = %s\n",  NCS_LDFLAGS);
  fprintf(fp,"NCS_LDADD = %s\n",    library);
  
  fprintf(fp,"OPTS = %s\n", option);


  fprintf(fp,"CXXFLAGS = $(DFLAGS) $(OPTS) $(NCS_CXXFLAGS) $(INCLUDES)\n");
  
  fprintf(fp,"\n");
  /* fprintf(fp,"RUN_PATH = -Wl,-rpath -Wl,%s -Wl,-rpath -Wl,%s\n", 
    SL4_LIB_DIR, NCS_LIB_DIR); */
  fprintf(fp,"LIBSATELLITE = -L%s -lsatellite -lm\n", SL4_LIB_DIR);
  fprintf(fp,"NCSLIBC = -L%s -lncsc\n", NCS_LIB_DIR);
  fprintf(fp,"NCSLIBS = -L%s -lncss\n", NCS_LIB_DIR);
  fprintf(fp,"NCSLIBM = -L%s -lncsm\n", NCS_LIB_DIR);
  fprintf(fp,"TERMCAP_LIBS = %s \n",    TERMCAP_LIBS);

  fprintf(fp,"LDFLAGS = $(NCSLIBM) $(NCSLIBS) $(NCSLIBC) $(LIBSATELLITE) $(TERMCAP_LIBS) $(NCS_LDADD) $(NCS_LDFLAGS)\n");

  fprintf(fp,"\n");
  
  fprintf(fp,"PROG = %s/%s\n", ncs_tmpdir, model);
  fprintf(fp,"SRCS = %s/%s.cpp\n", ncs_tmpdir, model);
  fprintf(fp,"OBJS = $(SRCS:%%.cpp=%%.o)\n");
  fprintf(fp,"\n");
  
  fprintf(fp,"all: $(PROG)\n");
  fprintf(fp,"\n");
  fprintf(fp,"nlink-compile: $(OBJS)\n");
  fprintf(fp,"\n");
  fprintf(fp,"nlink-link: $(PROG)\n");
  fprintf(fp,"\n");

  fprintf(fp,"%s/%s.o : %s/%s.cpp\n", ncs_tmpdir, model, ncs_tmpdir, model);
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) -c %s/%s.cpp -o %s/%s.o\n",
          ncs_tmpdir, model, ncs_tmpdir, model);
  fprintf(fp,"\n");

  fprintf(fp,"$(PROG): $(OBJS)\n");
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) -o $@ $(OBJS) $(LDFLAGS)\n");
  fprintf(fp,"\n");
  
  fprintf(fp,"depend: $(SRCS)\n");
  fprintf(fp,"\t@echo \"depending all sources\"\n");
  fprintf(fp,"\t@$(RM) .depend\n");
  fprintf(fp,"\t@for i in $(SRCS); do \\\n");
  fprintf(fp,"\t  $(CXX) $(CXXFLAGS) -MM $$i >> .depend ; \\\n");
  fprintf(fp,"\tdone\n");

  fprintf(fp,"\n");
  fprintf(fp,".PHONY: all nlink-compile nlink-link depend\n");
  fclose(fp);

  return(0);
}
#endif
/* create make command wrapper shell script */
static int create_nlink_makewrapper(const char *model, const char *flag,
                                    int cxxtype){
  const char *ncs_tmpdir;
  char  tmp[NCS_FNAME_WD];
  FILE *fp;
  int   do_obj_copy, do_err_copy;
  char *make[]={"nmake","make","make","make"};

#ifdef WIN32
  const char *scrext = ".bat";
  /* temporary */
  const char *vcvars_path="C:\\Program Files\\Microsoft Visual Studio .NET2003\\Vc7\\bin";
  const char *mingw_path="tools\\bin";
  char  sl4drv[_MAX_DRIVE], sl4prefix[_MAX_DIR], sl4path[_MAX_PATH];
  /* get full path info */
  ::GetModuleFileName(NULL, sl4path, _MAX_PATH);
  /* split drive and directory */
  _splitpath(sl4path, sl4drv, sl4prefix, NULL, NULL);
#else
  const char *scrext = ".sh";
#endif

  /* parse flag */
  do_obj_copy = 0;
  do_err_copy = 0;
  if(!strcmp(flag,"E"))      do_err_copy = 1;
  else if(!strcmp(flag,"O")) do_obj_copy = 1;
  else if(!strcmp(flag,"N")){
    do_err_copy = 1;
    do_obj_copy = 1;
  }

  ncs_tmpdir = get_tmpdir();

  snprintf(tmp, NCS_FNAME_WD, "%s/%s%s", ncs_tmpdir, model, scrext);
  fp = fopen(tmp, "w");
  if(fp == NULL) return 254;
#ifdef WIN32
  {
    char *tmpdir,*p;
    tmpdir = strdup(ncs_tmpdir);
    for(p=tmpdir; *p!='\0'; p++)
      if(*p == '/') *p = '\\';
    fprintf(fp,"@echo off\n");
    switch(cxxtype){
    case COMPILER_TYPE_MSVC:
      fprintf(fp,"call \"%s\\vcvars32.bat\"\n", vcvars_path);
      break;
    case COMPILER_TYPE_MINGW:
      fprintf(fp,"SET PATH=%s%s%s;%%PATH%%\n", sl4drv, sl4prefix, mingw_path);
      break;
    }
    fprintf(fp,"del /f \"%s\\%s\" \"%s\\%s.obj\" \"%s\\%s.err\"\n",
            tmpdir, model, tmpdir, model, tmpdir, model);
    fprintf(fp,"%s -f \"%s\\%s.mak\" >  \"%s\\%s.err\" 2>&1\n",
            make[cxxtype], tmpdir, model, tmpdir, model);
    fprintf(fp,"if errorlevel 1 goto linkerr\n");
    if(do_obj_copy){
      fprintf(fp,"copy /y \"%s\\%s.dll\" . >> \"%s\\%s.err\" 2>&1\n", 
              tmpdir, model, tmpdir, model);
    }
    fprintf(fp,"exit 0\n");
    fprintf(fp,":linkerr\n");
    if(do_err_copy){
      fprintf(fp,"copy /y \"%s\\%s.err\" . >> \"%s\\%s.err\" 2>&1\n",
              tmpdir, model, tmpdir, model);
    }
    fprintf(fp,"exit 251\n");
    fclose(fp);
    free(tmpdir);
  }
#else
  fprintf(fp,"#!/bin/sh\n");
  fprintf(fp,"rm -f \"%s/%s\" \"%s/%s.obj\" \"%s/%s.err\"\n",
          ncs_tmpdir,model, ncs_tmpdir,model, ncs_tmpdir,model);
  fprintf(fp,"%s -f %s/%s.mak nlink-compile > %s/%s.err 2>&1\n",
          make[cxxtype], ncs_tmpdir, model, ncs_tmpdir, model);
  fprintf(fp,"RETCODE=$?\n");
  if(do_err_copy){
    fprintf(fp,"cp -f \"%s/%s.err\" .\n", ncs_tmpdir, model);
  }
  fprintf(fp,"if [ $RETCODE -eq 1 ]; then\n");
  fprintf(fp," exit 250\n");
  fprintf(fp,"fi\n");
  if(do_obj_copy){
    fprintf(fp,"cp -f \"%s/%s.o\" .\n", ncs_tmpdir, model);
  }
  fprintf(fp,"make -f \"%s/%s.mak\" nlink-link >> \"%s/%s.err\" 2>&1\n", 
          ncs_tmpdir, model, ncs_tmpdir, model);
  fprintf(fp,"RETCODE=$?\n");
  if(do_err_copy){
    fprintf(fp,"cp -f \"%s/%s.err\" .\n", ncs_tmpdir, model);
  }
  fprintf(fp,"if [ $RETCODE -eq 1 ]; then\n");
  fprintf(fp," exit 251\n");
  fprintf(fp,"fi\n");
  fclose(fp);
  if(chmod(tmp, 0700) == -1) return 253;
#endif
  return 0;
}

static void print_link_error_message(const char *model){
  FILE *fp;
  const char *ncs_tmpdir;
  char  buf[1024];
  char  fname[_MAX_PATH];
  ncs_tmpdir = get_tmpdir();
  snprintf(fname,_MAX_PATH,"%s/%s.err", ncs_tmpdir, model);
  fp = fopen(fname,"r");
  if(fp){
    while(feof(fp)==0){
      fgets(buf,1023,fp);
      buf[1023]='\0';
      printf("%s",buf);
    }
    fclose(fp);
  }
}

#ifdef WIN32
static char *filename_dos2unix(const char *dos){
  const char *p;
  char *unix;
#ifndef _VC6
  size_t len;
  len = strlen(dos); 
  for(p=dos; *p!='\0'; p++) if(*p==' ') len++;
  unix = (char*)malloc(sizeof(char) * (len + 1));
  if(unix == NULL) return NULL;
  len = 0;
  for(p=dos; *p!='\0'; p++){
    unix[len++] = *p;
    if(*p == '\\'){
      unix[len-1] = '/';
    }else if(*p == ' '){
      unix[len-1] = '\\';
      unix[len++] = ' ';
    }
  }
  unix[len] = '\0';
#else
  char spath[_MAX_PATH];

  if(GetShortPathName(dos, spath, _MAX_PATH) == NULL)
	  strcpy(spath, dos);
  unix = (char*)malloc(sizeof(char) * (strlen(spath) + 1));
  strcpy(unix, spath);
#endif
  return unix;
}

// static char *pathname_unix2dos(const char *unix){
//   const char *p;
//   char *dos;
//   size_t len;
//   len = strlen(unix); 
//   for(p=unix; *p!='\0'; p++) if(*p=='\\') len--;
//   dos = (char*)malloc(sizeof(char) * (len + 1));
//   if(dos == NULL) return NULL;
//   len = 0;
//   for(p=dos; *p!='\0'; p++){
//     dos[len++] = *p;
//     if(*p == '\\'){
//       len--;
//     }else if(*p == '/'){
//       dos[len-1] = '\\';
//     }
//   }
//   dos[len] = '\0';
//   return dos;
// }
#endif

#ifdef __cplusplus
}
#endif

