/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************

   	Function Library to Control NCS Simulation Condition File

        $Id: scf.c,v 1.2 2004/09/28 04:18:31 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_STDLIB_H
#  include <stdlib.h>
# endif
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#include <string.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#ifdef WIN32
# define FOPEN_RW "r+b"
# define FOPEN_W  "wb"
#else
# define FOPEN_RW "r+"
# define FOPEN_W  "w"
#endif

#ifdef __cplusplus
extern "C" {
#endif

static char *scf_work_dir = NULL;

/*
 *  Open simulation condition file(SCF)
 * 	char *fname : file name of file
 *      Return      : advance the file position indicator.
 *                    not NULL  : success.
 *                    NULL      : failed.
 */
FILE *_OpenSCF( const char *fname )
{
  FILE 		*fp;

  if( (fp = fopen( fname, FOPEN_RW )) == NULL ){
    return( NULL );
  } else {
    return( fp );
  }
}

/*
 *  Open simulation condition file(SCF)
 * 	char *fname : file name of file
 *      Return      : advance the file potion indicator.
 *                    not NULL : success.
 *                    NULL     : failed.
 */
FILE *_CreateSCF( const char *fname )
{
  FILE 		*fp;

  if( (fp = fopen( fname, FOPEN_W )) == NULL ){
    return( NULL );
  } else {
    return( fp );
  }
}


/*
 *  Close Simulation condition file(SCF)
 * 	FILE  *fp   : file pointer of file
 */
int _CloseSCF( FILE *fp )
{
  return( fclose( fp ) );
}

/*
 *  Read one entry from SCF file
 * 	FILE *fp : file pointer of SCF file
 * 	void *buf: storing at the location 
 *      int  size: read size bytes long.
 *      Return   : advance the file potion indicator.
 *                 1 : success.
 *                 0 : failed.
 */
size_t _ReadSCF( FILE *fp, void *buf, size_t size )
{
  return( fread( buf, size, 1, fp ) );
}


/*
 *  Write one entry to SCF file
 * 	FILE *fp : file pointer of SCF file
 * 	void *buf: reading at the location 
 *      int  size: read size bytes long.
 *      Return   : advance the file potion indicator.
 *                 1 : success.
 *                 0 : failed.
 */
size_t _WriteSCF( FILE *fp, void *buf, size_t size )
{
  size_t stat;
  stat = fwrite(buf, size, 1, fp);
  if(stat == SCF_SUCCESS) fflush( fp );
  return( stat );
}

/*
 *  set path of working directory for SCF files
 * 	char *path : path of working directory
 *      Return     : 1 : success.
 *                   0 : failed.
 */
int SetWorkDirSCF( const char *path )
{
  if( scf_work_dir != NULL ){
    free( scf_work_dir );
    scf_work_dir = NULL;
  }
  if( path == NULL || *path == '\0' ){
    return( NCS_FALSE );
  }
#ifdef HAVE_STRDUP
  scf_work_dir = strdup( path );
  if(scf_work_dir == NULL) return( NCS_FALSE );
#else
  scf_work_dir = (char*)malloc(strlen(path)+1);
  if(scf_work_dir == NULL) return( NCS_FALSE );
  strcpy(scf_work_dir, path);
#endif
  return( NCS_TRUE );
}

int CleanWorkDirSCF(){
  if( scf_work_dir != NULL ){
    free( scf_work_dir );
    scf_work_dir = NULL;
  }
  return NCS_TRUE;
}

/*
 *  get path of working directory for SCF files
 *      Return     : path of working directory : success.
 *                   NULL                      : failed.
 */
const char *GetWorkDirSCF()
{
  return( scf_work_dir );
}

#ifdef __cplusplus
}
#endif
