/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************
  
   	  Function Library to Control NCS Simulation Condition File 
  	       	   for information of DELAY variables

        $Id: scfdly.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <fcntl.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of output information */
static FILE *dly;

static int   DlyOpenSCF();
static int   DlyCreateSCF();
static int   DlyCloseSCF();
static int   DlyWriteSCF( DlyFile *src );
static int   DlyReadSCF( DlyFile *src );
static void  DlyTopSCF();
static void  DlyEndSCF();
static void  DlyNextSCF();
static void  DlyNthNextSCF( int n );
static void  DlyPreviousSCF();
static void  DlyNthPreviousSCF( int n );
static int   DlyListupAllSCF( FILE *disp );

/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make output information file
 *      ( ->SCF_EOF )
 *   Return      : TRUE  : success.
 *                 FALSE : failed.
 */
int DlyMakeSCF(){
  DlyFile   src;
  if( DlyCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( DlyWriteSCF( &src ) == NCS_FALSE ){
    DlyCloseSCF();
    return( NCS_FALSE );
  }
  DlyCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Add one entry to end of delay condition file
 *   OutFile *add : add informaiton
 *   Return       : TRUE  : success.
 *                  FALSE : failed.
 */
int DlyAddInfoSCF( DlyFile *add ){
  DlyFile src;
  if( DlyOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  DlyEndSCF();
    if( DlyWriteSCF( add ) == SCF_ERROR ){
    DlyCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
    if( DlyWriteSCF( &src ) == SCF_ERROR ){
    DlyCloseSCF();
    return( NCS_FALSE );
  }
  DlyCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Write one entry to n position of delay condition file 
 *   OutFile *add: add informaiton
 *   int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int DlyPutInfoSCF( DlyFile *add, int n ){
  if( DlyOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  DlyTopSCF();
  DlyNthNextSCF( n );
    if( DlyWriteSCF( add ) == SCF_ERROR ){
    DlyCloseSCF();
    return( NCS_FALSE );
  }
  DlyCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Set one entry to delay condition structure.
 *   OutFile *src    : storing at the location 
 *   int buff_id     : buffer ID to set information.
 *   char *buff_name : buffer name to set information.
 *   int buff_dim    : index of buffer to set information.
 *   char *mdl_name  : module name to set information.
 *   int comp_no     : component number to set information.
 *   int type        : type of output variable. 
 *                    ( output, input, observable )
 *   char *obs_name  : name of variable in model file.
 *   Return          : TRUE  : success.
 *                     FALSE : failed.
 */
int DlySetInfoSCF( DlyFile *src, const char *mdl_name,
                             const char *inp_name, double time, 
                             double init_out, const char *flag ){
  src->type = SCF_DLY_TYPE_NORMAL;
  if( mdl_name != NULL ){
    if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
      strncpy( src->delay.mdl_name, mdl_name, SCF_NAME_LEN-1 );
      src->delay.mdl_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: model name is too long.\n" );
    }else{
      strcpy( src->delay.mdl_name, mdl_name );
    }
  } else {
    src->delay.mdl_name[0] = '\0';
  }
  if( inp_name != NULL ){
    if( strlen( inp_name ) >= (SCF_NAME_LEN-1) ){
      strncpy( src->delay.inp_name, inp_name, SCF_NAME_LEN-1 );
      src->delay.inp_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->delay.inp_name, inp_name );
    }
  } else {
    src->delay.inp_name[0] = '\0';
  }
  if( time >= 0 ){
    src->delay.time = time;
  } else {
    src->delay.time = 0.0;
  }
  src->delay.init_flag = *flag;
  if( *flag == SCF_DLY_FLAG_FIX ){
    src->delay.init_out = init_out;
  } else {
    src->delay.init_out = 0.0;
  }
  return( NCS_TRUE );
}

/*
 *  Write one entry to delay condition file
 *   DlyFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int DlyWriteInfoSCF( DlyFile *src ){
  int   pos;
  pos = DlySearchInfoSCF( src );
  if( pos < 0 ){
    return( DlyAddInfoSCF( src ) );
  } else {
    return( DlyPutInfoSCF( src, pos ) );
  }
}


/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of delay condition file
 *   DlyFile *src : storing at the location 
 *   int  n       : position
 *   Return       : TRUE  : success.
 *                  FALSE : failed.
 */
int DlyGetInfoSCF( DlyFile *src, int n ){
  if( DlyOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  DlyTopSCF();
  DlyNthNextSCF( n );
    if( DlyReadSCF( src ) == NCS_FALSE ){
    DlyCloseSCF();
    return( NCS_FALSE );
  }else{
    DlyCloseSCF();
    return( NCS_TRUE );
  }
}

/*
 *  Search same entry to delay condition file
 *   DlyFile *cmp : compare entry
 *   Return       :  n > 0  : potion of the same entry
 *                   n < 0  : no match. n is end of entry.
 *                   0      : ERROR
 */
int DlySearchInfoSCF( DlyFile *cmp ){
  DlyFile src;
  int     n = 0;
  if( DlyOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  DlyTopSCF();
  while( DlyReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      DlyCloseSCF();  
      return( -(n+1) );
      break;
    }
    if( strcmp( cmp->delay.mdl_name , src.delay.mdl_name ) == 0 &&
        strcmp( cmp->delay.inp_name , src.delay.inp_name ) == 0 ){
      break;
    }
    
    n++;
  }
  DlyCloseSCF();  
  return( n );
}


/**************************************************************************
       Functions to delete entry of simulation condition file
 **************************************************************************/

/*
 *  Delete one entry from n potition of delay condition file
 *      int  n      : position   
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int DlyDeleteInfoSCF( int n ){
  DlyFile  tmp;
  if( DlyOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  DlyTopSCF();
  DlyNthNextSCF( n );
  do{
    DlyNextSCF();
    if( DlyReadSCF( &tmp ) == NCS_FALSE ){
      DlyCloseSCF();
      return( NCS_FALSE );
    }
    DlyNthPreviousSCF( 2 );
    if( DlyWriteSCF( &tmp ) == NCS_FALSE ){
      DlyCloseSCF();
      return( NCS_FALSE );
    }
  }while( tmp.type != SCF_EOF );
  DlyCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Delete all entry of delay condition file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int DlyDeleteAllInfoSCF(){
  DlyFile src;
  if( DlyOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  DlyTopSCF();
  while( ( DlyReadSCF( &src ) != NCS_FALSE ) &&
         ( src.type != SCF_EOF ) ){
    DlyPreviousSCF();
    src.type = SCF_EOF;
    if( DlyWriteSCF( &src ) == NCS_FALSE ){
      DlyCloseSCF();
      return( NCS_FALSE );
    }
  }
  DlyCloseSCF();
  return( NCS_TRUE );
}

/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of all entry of delay information file
 *   FILE *disp  : output file
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int DlyListupSCF( const char *tmpfile ){
  FILE *disp;
  int  row = 1;
  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;
  fprintf( disp, "\nDELAY INFORMATION\n");
  row += DlyListupAllSCF( disp );
  fclose(disp);
  return( row );
}


/**************************************************************************
         Basic functions to control simulation condition file
 **************************************************************************/

/*
 *  Open delay condition file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int DlyOpenSCF(){
  const char *path;
  char        mname[SCF_SCFN_PATH_SIZE];
  char        fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE,"%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (dly = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Create delay condition file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int DlyCreateSCF(){
  const char *path;
  char     mname[SCF_SCFN_PATH_SIZE];
  char     fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE,"%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_DLY_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (dly = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Close delay condition file
 */
static int DlyCloseSCF(){
  return( _CloseSCF( dly ) );
}

/*
 *  Write one entry to delay condition file
 *   OutFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int DlyWriteSCF( DlyFile *src ){
  if( _WriteSCF( dly, (void *)src, sizeof(DlyFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from delay condition file
 *   OutFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int DlyReadSCF( DlyFile *src ){
  if( _ReadSCF( dly, (void *)src, sizeof(DlyFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of file
 */
static void DlyTopSCF(){
  fseek( dly, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of file
 */
static void DlyEndSCF(){
  DlyFile  tmp;
  DlyTopSCF();
  while( DlyReadSCF( &tmp ) != NCS_FALSE && tmp.type != SCF_EOF );
  fseek( dly, (long)sizeof(DlyFile)*(-1), SEEK_CUR );
}

/*
 *  advance file pointer 
 */
static void DlyNextSCF(){
  fseek( dly, (long)sizeof(DlyFile), SEEK_CUR );
}

/*
 *  advance nth file pointer 
 */
static void DlyNthNextSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    DlyNextSCF();
  }
}

/*
 *  retreat file pointer 
 */
static void DlyPreviousSCF(){
  fseek( dly, (long)sizeof(DlyFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer 
 */
static void DlyNthPreviousSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    DlyPreviousSCF();
  }
}

/*
 *  Print list of outupt variables 
 *   FILE *disp  : output file
 */
static int DlyListupAllSCF( FILE *disp ){
  int  num = 0;
  int  row = 0;
  DlyFile  src;
  if( DlyOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  DlyTopSCF();
  while( DlyReadSCF( &src ) != NCS_FALSE && src.type != SCF_EOF ){
    if( num == 0 ){
      fprintf( disp, " Data No.,         Input Name,        Delay Time,    Initial Output\n");
    }
    num++;
    fprintf( disp, " %3d          %9s(%9s)      %-8g", num,
             src.delay.mdl_name, src.delay.inp_name, src.delay.time );
    switch( src.delay.init_flag ){
    case SCF_DLY_FLAG_AUTO:
      fprintf( disp, "      %8s\n", "(Auto)");
      row++;
      break;
    case SCF_DLY_FLAG_FIX:
      fprintf( disp, "      %-8g\n",src.delay.init_out);
      row++;
      break;
    default:
      break;
    }
  }
  if( num == 0 ){
    fprintf( disp, "\tNo information.\n" );
    row++;
  }
  DlyCloseSCF();
  return( row );
}

#ifdef __cplusplus
}
#endif
