/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************
  
  	Function Library to Control NCS Simulation Condition File 
  	       for information of MATHEMATICAL variables

     $Id: scfmat.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <fcntl.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of mathematical information */
static FILE *mat;

static int   MatOpenSCF();
static int   MatCreateSCF();
static int   MatCloseSCF();
static int   MatWriteSCF( MatFile *src );
static int   MatReadSCF( MatFile *src );
static void  MatTopSCF();
static void  MatEndSCF();
static void  MatNextSCF();
static void  MatNthNextSCF( int n );
static void  MatPreviousSCF();
static void  MatNthPreviousSCF( int n );

/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make mathematical information file
 *      ( SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int MatMakeSCF(){
  MatFile   src;
  memset(&src, 0, sizeof(MatFile));
  if( MatCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( MatWriteSCF( &src ) == NCS_FALSE ){
    MatCloseSCF();
    return( NCS_FALSE );
  }
  MatCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Add one entry to end of mathematical information file
 *   MatFile *add : add informaiton
 *   Return       : TRUE : success.
 *                  FALSE : failed.
 */
int MatAddInfoSCF( MatFile *add ){
  MatFile src;
  memset(&src, 0, sizeof(MatFile));
  if( MatOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  MatEndSCF();
    if( MatWriteSCF( add ) == SCF_ERROR ){
    MatCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
    if( MatWriteSCF( &src ) == SCF_ERROR ){
    MatCloseSCF();
    return( NCS_FALSE );
  }
  MatCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Write one entry to n position of mathematical information file 
 *   MatFile *add : add informaiton
 *   int n        : position
 *   Return       : TRUE : success.
 *                  FALSE : failed.
 */
int MatPutInfoSCF( MatFile *add, int n ){
  if( MatOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  MatTopSCF();
  MatNthNextSCF( n );
    if( MatWriteSCF( add ) == SCF_ERROR ){
    MatCloseSCF();
    return( NCS_FALSE );
  }
  MatCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Set one entry to mathematical information structure.
 *   MatFile *src        : storing at the location 
 *   double last_time    : Last time
 *   double calc_time    : Calculation time
 *   double store_time   : Store time
 *   double write_time   : Write time
 *   char* integ_method  : integration method
 *   int integ_maxcell   : max cell
 *   double integ_relerr : relative error
 *   Return              : TRUE  : success.
 *                         FALSE : failed.
 */
int MatSetInfoSCF( MatFile *src, double last_time, double calc_time,
                             double store_time, double write_time,
                             const char *integ_method, int integ_maxcell, 
                             double integ_relerr ){
  src->type = SCF_MAT_TYPE_NORMAL;
  src->timer.sim_timer[0] = last_time;
  src->timer.sim_timer[1] = calc_time;
  src->timer.sim_timer[2] = store_time;
  src->timer.sim_timer[3] = write_time;
  src->integral.method  = *integ_method;
  src->integral.maxcell = integ_maxcell;
  src->integral.relerr  = integ_relerr;
  return( NCS_TRUE );
}

/*
 *  Search same entry to mathematical information file
 *   ObsFile *cmp : compare entry
 *   Return       :  n > 0  : potion of the same entry
 *                   n < 0  : no match. n is end of entry.
 *                   0      : ERROR
 */
int MatSearchInfoSCF( MatFile *cmp ){
  MatFile src;
  int     n = 0;
  memset(&src, 0, sizeof(MatFile));
  if( MatOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  MatTopSCF();
  while( MatReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      MatCloseSCF();
      return( -n );
      break;
    }
    if( cmp->type == src.type ){
      break;
    }
    n++;
  }
  MatCloseSCF();  
  return( n );
}

/*
 *  Write one entry to mathematical information file
 *   char *fname  : file name
 *   OutFile *src : storing at the location 
 *   Return       : TRUE : success.
 *                  FALSE : failed.
 */
int MatWriteInfoSCF( MatFile *src ){
  return( MatPutInfoSCF( src, 0 ) );
}



/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of mathematical information file
 *   char *fname  : file name
 *   MatFile *str : storing at the location 
 *   int  n       : position
 *   Return       : TRUE  : success.
 *                  FALSE : failed.
 */
int MatGetInfoSCF( MatFile *str, int n ){
  if( MatOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  MatTopSCF();
  MatNthNextSCF( n );
    if( MatReadSCF( str ) == NCS_FALSE ){
    MatCloseSCF();
    return( NCS_FALSE );
  }else{
    MatCloseSCF();
    return( NCS_TRUE );
  }
}


/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of entry of mathematical information file
 *  Information of time of simulation
 *   FILE *disp  : output file
 *   char *fname : file name
 *   Return      : success : number of lines of lists
 *                 0       : failed
 */
int MatTimeListupSCF( const char *tmpfile ){
  FILE *disp;
  MatFile  src;
  int   row = 0;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;
  if((disp = fopen( tmpfile, "w" ) ) == NULL){ /* TODO for WIN32*/
    return( 0 );
  }
  memset(&src, 0, sizeof(MatFile));
  fprintf( disp, "\nTIMER\n" );
  if( MatOpenSCF() == NCS_FALSE ){
    fclose(disp);
    return( 0 );
  }
  MatTopSCF();
  if( MatReadSCF( &src ) != NCS_FALSE ){
    if( src.type == SCF_MAT_TYPE_NORMAL ){
      fprintf( disp,"  Last  Time = %8g\n", src.timer.sim_timer[0] );
      fprintf( disp,"  Calc. Step = %8g\n", src.timer.sim_timer[1] );
      fprintf( disp,"  Store Step = %8g\n", src.timer.sim_timer[2] );
      fprintf( disp,"  BufferStep = %8g\n", src.timer.sim_timer[3] );
      row +=4;
    } else {
      fprintf( disp,"Error: Can't find mathematical information in the condition file\n" );
      row++;
    }
    fprintf( disp, "\n");
    row++;
    MatCloseSCF();
  } else {
    fprintf( disp, "Error: Can't find the condition file of mathematical information\n" );
    row++;
  }
  fclose(disp);
  return( row );
}

/*
 *  Print list of entry of mathematical information file
 *  Information of integration
 *   FILE *disp  : output file
 *   char *fname : file name
 *   Return      : success : number of lines of lists
 *                 0       : failed
 */
int MatIntegListupSCF( const char *tmpfile ){
  FILE *disp;
  MatFile  src;
  int   row = 0;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  memset(&src, 0, sizeof(MatFile));
  fprintf( disp, "\nINTEGRAL\n" );
  if( MatGetInfoSCF( &src, 0 ) != NCS_FALSE ){
    fprintf( disp, "  Integration Method : " );
    switch( src.integral.method ){
    case SCF_MAT_INTEG_FERBERG:
      fprintf( disp, "5th Fehlberg Method (Adaptive Integration)\n" );
      row++;
      break;
    case SCF_MAT_INTEG_EULER:
      fprintf( disp, "Euler Method\n" );
      row++;
      break;
    case SCF_MAT_INTEG_RKG:
      fprintf( disp, "4th Runge-Kutta-Gill Method\n" );
      row++;
        break;
    }
    fprintf( disp, "  Number of Max Cell : %d\n", src.integral.maxcell );
    row++;
    if( src.integral.method == SCF_MAT_INTEG_FERBERG ) {
      fprintf( disp, "  Relative Error     : %g\n", 
               src.integral.relerr );
      row+=2;
    }
  } else {
    fprintf( disp,"Error: Can't find mathematical information in the condition file\n" );
    row++;
  }
  fclose(disp);
  return( row );
}


/**************************************************************************
    Basic functions to control simulation condition file
 **************************************************************************/

/*
 *  Open mathematical information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int MatOpenSCF(){
  const char *path;
  char     mname[SCF_SCFN_PATH_SIZE];
  char     fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (mat = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Create mathematical information file
 *   char *fname : file name of file
 *   Return      : TRUE  : success
 *                 FALSE : failed.
 */
static int MatCreateSCF(){
  const char *path;
  char     mname[SCF_SCFN_PATH_SIZE];
  char     fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
#else
    sprintf( fname, "%s/%s.%s", 
             path, ModelNameSCFN( mname ), SCF_MAT_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (mat = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Close mathematical information file
 */
static int MatCloseSCF(){
  return( _CloseSCF( mat ) );
}

/*
 *  Write one entry to mathematical information file
 *   MatFile *src : reading at the location 
 *   Return       : advance the file potion indicator.
 *                  TRUE  : success.
 *                  FALSE : failed.
 */
static int MatWriteSCF( MatFile *src ){
  if( _WriteSCF( mat, (void *)src, sizeof(MatFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from mathematical information file
 *   MatFile *src : storing at the location 
 *   Return       : advance the file potion indicator.
 *                  TRUE  : success.
 *                  FALSE : failed.
 */
static int MatReadSCF( MatFile *src ){
  if( _ReadSCF( mat, (void *)src, sizeof(MatFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of mathematical informatition file
 */
static void MatTopSCF(){
  fseek( mat, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of mathematical information file
 */
static void MatEndSCF(){
  fseek( mat, (long)sizeof(MatFile)*(-1), SEEK_END );
}

/*
 *  advance file pointer of mathematical information file
 */
static void MatNextSCF(){
  fseek( mat, (long)sizeof(MatFile), SEEK_CUR );
}

/*
 *  advance nth file pointer of mathematical information file
 */
static void MatNthNextSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    MatNextSCF();
  }
}

/*
 *  retreat file pointer of mathematical information file
 */
static void MatPreviousSCF(){
  fseek( mat, (long)sizeof(MatFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer of mathematical information file
 */
static void MatNthPreviousSCF( int n ){
  int  i;
  for( i = 0; i < n; i++ ){
    MatPreviousSCF();
  }
}

#ifdef __cplusplus
}
#endif
