/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************

        Function Library to Control NCS Simulation Condition File 
               for information of OBSERVABLE variables

    $Id: scfobs.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of observable information */
static FILE *obs;

static int   ObsOpenSCF();
static int   ObsCreateSCF();
static int   ObsCloseSCF();
static int   ObsReadSCF( ObsFile *src );
static int   ObsWriteSCF( ObsFile *src );
static void  ObsEndSCF();

/*
 *  Make observable information file
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int ObsMakeSCF(){
  ObsFile   src;
  memset(&src, 0, sizeof(ObsFile));
  if( ObsCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( ObsWriteSCF( &src ) == NCS_FALSE ){
    ObsCloseSCF();
    return( NCS_FALSE );
  }
  ObsCloseSCF();
  return( NCS_TRUE );
}


/*
 *  Add one entry to observable information file
 *   char *fname : file name
 *   ObsFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int ObsAddInfoSCF( ObsFile *add ){
  ObsFile src;
  memset(&src, 0, sizeof(ObsFile));
  if( ObsOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  ObsEndSCF();
  if( ObsWriteSCF( add ) == SCF_ERROR ){
    ObsCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( ObsWriteSCF( &src ) == SCF_ERROR ){
    ObsCloseSCF();
    return( NCS_FALSE );
  }
  ObsCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Read one entry from observable information file
 *   char *fname : file name
 *   ObsFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int ObsGetInfoSCF( ObsFile *src ){
  if( ObsOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  if( ObsReadSCF( src ) == NCS_FALSE ){
    ObsCloseSCF();
    return( NCS_FALSE );
  }
  ObsCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Search same entry to observable information file
 *   char *fname : name of file
 *   ObsFile *cmp: compare entry
 *      Return      :  n : potion of the same entry
 *                    -1 : no match
 */
int ObsSearchInfoSCF( ObsFile *cmp ){
  ObsFile src;
  int     n = 0;
  memset(&src, 0, sizeof(ObsFile));
  if( ObsOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  while( ObsReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      ObsCloseSCF();
      return( -(n+1) );
      break;
    }
    if( (strcmp( cmp->info.module, src.info.module ) == 0) 
        && (strcmp( cmp->info.var, src.info.var ) == 0) 
        && (cmp->info.max_comp < src.info.max_comp) ){
      break;
    }
    n++;
  }
  ObsCloseSCF();
  return( n );
}


/*
 *  Print list of all entry to observable information file
 *   FILE *disp  : output file
 *   char *fname : file name
 */
int ObsListupSCF( const char *tmpfile ){
  FILE *disp;
  ObsFile src;
  int  n = 0, line = 0;
  char  *module = NULL;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;

  memset(&src, 0, sizeof(ObsFile));
  if( ObsOpenSCF() == NCS_FALSE ){
    fprintf( stderr, "Wanning: Failed to open the SCF\n" );
    fclose(disp);
    return( 0 );
  }
  fprintf( disp, "\nOBSERVABLE VARIABLE" );
  line++;
  while( ObsReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      break;
    }
    if( ( n == 0 ) || (strcmp( src.info.module, module ) != 0 ) ){
      if( n != 0 ){
        free( module );
      }
#ifdef HAVE_STRDUP
      module = strdup( src.info.module );
#else
      module = (char*)malloc(strlen(src.info.module)+1);
      if(module != NULL) strcpy(module,src.info.module);
#endif
      fprintf( disp, "\n  Module: \"%s[%d]\"\n    ", 
               module, src.info.max_comp );
      line += 2;
    } else {
      fprintf( disp, ", " );
    }
    fprintf( disp, "%s", src.info.var );
    n++;
  }
  if( n == 0 ){
    fprintf( disp, "\n  none.\n\n" );
    line+= 2;
  }else{
    fprintf( disp, "\n\n" );
    line++;
  }
  ObsCloseSCF();
  fclose(disp);
  return( line );
}


/*
 *  Set one entry to observable information structure.
 *   OutFile *src    : storing at the location 
 *  char *mdl_name  : module name to set information.
 *  char *var_name  : name of variable in model file.
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int ObsSetInfoSCF( ObsFile *src, const char *mdl_name, 
                             int comp, const char *var_name ){
  if( mdl_name != NULL ){
    if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
      strncpy( src->info.module, mdl_name, SCF_NAME_LEN-1 );
      src->info.module[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->info.module, mdl_name );
    }
  } else {
    src->info.module[0] = '\0';
  }
  if( comp >= 0 ){
    src->info.max_comp = comp;
  } else {
    src->info.max_comp = 0;
  }
  if( var_name != NULL ){
    if( strlen( var_name ) >= ( SCF_NAME_LEN-1 ) ){
      strncpy( src->info.var, var_name, SCF_NAME_LEN-1 );
      src->info.var[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->info.var, var_name );
    }
  } else {
    src->info.var[0] = '\0';
  }
  return( NCS_TRUE );
}


/*
 *  Open observable information file
 *      Return      : TRUE  : success
 *                    FALSE : failed
 */
static int ObsOpenSCF(){
  const char *path;
  char  mname[SCF_SCFN_PATH_SIZE];
  char  fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (obs = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Open observable information file
 *   char *fname : file name of file
 *      Return      : TRUE  : sucess
 *                    FALSE : failed
 */
static int ObsCreateSCF(){
  const char *path;
  char  mname[SCF_SCFN_PATH_SIZE];
  char  fname[SCF_SCFN_PATH_SIZE];
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_OBS_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  if( (obs = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Close observable information file
 */
static int ObsCloseSCF(){
  return( _CloseSCF( obs ) );
}

/*
 *  Write one entry to observable information file
 *   ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int ObsWriteSCF( ObsFile *src ){
  if( _WriteSCF( obs, (void *)src, sizeof(ObsFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }
  else {
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from observable information file
 *   ObsFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int ObsReadSCF( ObsFile *src ){
  if( _ReadSCF( obs, (void *)src, sizeof(ObsFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }
  return( NCS_TRUE );
}

/*
 *  Set file pointer end of file
 *   FILE    *fp : file pointer of file
 */
static void ObsEndSCF(){
  fseek( obs, (long)sizeof(ObsFile)*(-1), SEEK_END );
}

#ifdef __cplusplus
}
#endif
