/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

 /*****************************************************************************
  
  	Function Library to Control NCS Simulation Condition File 
  	          for information of parameter variables

    $Id: scfprm.c,v 1.2 2004/04/15 16:57:53 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <fcntl.h>

#define MOD_NCS_LIBNCSC_EXPORTS
#include "libncsc.h"

#include "scf.h"

#ifdef __cplusplus
extern "C" {
#endif

/* File pointer for simulaiton condition file of output information */
static FILE *prm;

static int   PrmOpenSCF();
static int   PrmCreateSCF();
static int   PrmCloseSCF();
static int   PrmWriteSCF( PrmFile *src );
static int   PrmReadSCF( PrmFile *src );
static void  PrmTopSCF();
static void  PrmEndSCF();
static void  PrmNextSCF();
static void  PrmNthNextSCF( int n );
static void  PrmPreviousSCF();
static void  PrmNthPreviousSCF( int n );
static int   PrmNthGetInfoSCF( PrmFile *src, int n );

/**************************************************************************
       Functions to write simulation condition file
 **************************************************************************/

/*
 *  Make parameter information file
 *      ( ->SCF_EOF )
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int PrmMakeSCF(){
  PrmFile src;
  memset(&src, 0, sizeof(PrmFile));
  if( PrmCreateSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( PrmWriteSCF( &src ) == NCS_FALSE ){
    PrmCloseSCF();
    return( NCS_FALSE );
  }
  PrmCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Add one entry to end of parameter information file
 * 	PrmFile *add: add informaiton
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int PrmAddInfoSCF( PrmFile *add ){
  PrmFile src;
  memset(&src, 0, sizeof(PrmFile));
  if( PrmOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  PrmEndSCF();
  if( PrmWriteSCF( add ) == SCF_ERROR ){
    PrmCloseSCF();
    return( NCS_FALSE );
  }
  src.type = SCF_EOF;
  if( PrmWriteSCF( &src ) == SCF_ERROR ){
    PrmCloseSCF();
    return( NCS_FALSE );
  }
  PrmCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Write one entry to n position of parameter information file 
 * 	PrmFile *add: add informaiton
 * 	int n       : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int PrmPutInfoSCF( PrmFile *add, int n ){
  PrmFile org;
  memset(&org, 0, sizeof(PrmFile));
  if( PrmOpenSCF() == NCS_FALSE ){
    return( NCS_FALSE );
  }
  PrmTopSCF();
  PrmNthNextSCF( n );
  if( PrmReadSCF( &org ) == SCF_ERROR ){
    PrmCloseSCF();
    return( NCS_FALSE );
  } 
  org.prm_val.prm_real = add->prm_val.prm_real;
  PrmPreviousSCF();
  if( PrmWriteSCF( &org ) == SCF_ERROR ){
    PrmCloseSCF();
    return( NCS_FALSE );
  }
  PrmCloseSCF();
  return( NCS_TRUE );
}

/*
 *  Set one entry to parameter information structure.
 * 	PrmFile *src    : storing at the location 
 *	char type       : type of parameter
 *	char *mdl_name  : module name to set information.
 *	char *prm_name  : parameter name with ID to set information.
 *      double data     : value of parameter
 *      Return          : TRUE  : success.
 *                        FALSE : failed.
 */
int PrmSetInfoSCF( PrmFile *src, const char *type, 
                             const char *mdl_name, const char *prm_name,
                             double data ){
  src->type = *type;
  if( mdl_name != NULL ){
    if( strlen( mdl_name ) >= ( SCF_NAME_LEN-1 ) ){
#ifdef HAVE_STRNCPY
      strncpy( src->mdl_name, mdl_name, SCF_NAME_LEN-1 );
#else
      strcpy( src->mdl_name, mdl_name );
#endif
      src->mdl_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->mdl_name, mdl_name );
    }
  } else {
    src->mdl_name[0] = '\0';
  }
  if( prm_name != NULL ){
    if( strlen( prm_name ) >= ( SCF_NAME_LEN-1 ) ){
      strncpy( src->prm_name, prm_name, SCF_NAME_LEN-1 );
      src->prm_name[SCF_NAME_LEN-1] = '\0';
      fprintf( stderr, "Warnning: module name is too long.\n" );
    }else{
      strcpy( src->prm_name, prm_name );
    }
  } else {
    src->prm_name[0] = '\0';
  }
  switch( *type ){
  case SCF_PRM_TYPE_NORMAL:
    src->prm_val.prm_real = data;
    break;
  case SCF_PRM_TYPE_INT:
    src->prm_val.prm_real = (long)data;
    break; 
  }
  return( NCS_TRUE );
}

/*
 *  Search same entry to parameter information file
 * 	PrmFile *cmp: compare entry
 *      Return      :  n > 0  : potion of the same entry
 *                     n < 0  : no match. n is end of entry.
 *                     0      : ERROR
 */
int PrmSearchInfoSCF( PrmFile *cmp ){
  PrmFile src;
  int     n = 0;
  memset(&src, 0, sizeof(PrmFile));
  if( PrmOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  PrmTopSCF();
  while( PrmReadSCF( &src ) != SCF_ERROR ){
    if( src.type == SCF_EOF ){
      PrmCloseSCF();	
      return( -n );
      break;
    }
    if( !strcmp(src.mdl_name,cmp->mdl_name) 
        && !strcmp(&src.prm_name[3], cmp->prm_name) ){
      break;
    }
    n++;
  }
  PrmCloseSCF();	
  return( n );
}

/*
 *  Write one entry to parameter information file
 * 	PrmFile *src: storing at the location 
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int PrmWriteInfoSCF( PrmFile *src ){
  int pos;
  pos = PrmSearchInfoSCF( src );
  if( pos >= 0 ){
    return( PrmPutInfoSCF( src, pos ) );
  }
  return( NCS_FALSE );
}

/*
 *  Read one entry from n potition of output information file
 * 	PrmFile *src: search name & return value
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
int PrmGetValueSCF( PrmFile *src ){
  int pos;
  if( (pos = PrmSearchInfoSCF( src )) < 0 ){
    return( NCS_FALSE );
  }
  return(PrmNthGetInfoSCF( src, pos ));
}


/**************************************************************************
       Functions to display content of simulation condition file
 **************************************************************************/

/*
 *  Print list of all entry of output information file
 * 	FILE *disp  : output file
 * 	char *fname : file name
 *      Return      : success : number of lines of lists
 *                    0       : failed
 */
int PrmListupSCF( const char *tmpfile, const char *module ){
  FILE *disp;
  int	cmp_mdl, cmp_all;
  int     found = 0;
  char    current_mdl_name[SCF_NAME_LEN];
  PrmFile	sc;
  int	row = 0;

  disp = fopen(tmpfile, "w");
  if(disp == NULL) return 0;
  
  memset(&sc, 0, sizeof(PrmFile));
  
  fprintf( disp, "\nParameter List\n" );
  row++;
  
  if( PrmOpenSCF() != NCS_FALSE ){
    current_mdl_name[0] = '\0';
    while( PrmReadSCF(&sc) && sc.type != SCF_EOF ){
      cmp_mdl = strcmp( module, sc.mdl_name );
      cmp_all = ( ( strcmp( module, SCF_PRM_ALL_MDL ) == 0 )
                  || ( strcmp( module, SCF_PRM_ALL_MDL_OLD ) == 0 ) );
      if( ( cmp_mdl == 0 ) || cmp_all ){
        if( strcmp( current_mdl_name, sc.mdl_name ) ){
          fprintf( disp, "\nModule name : %s\n", sc.mdl_name );
          row++;
        }
        if( sc.type != SCF_PRM_TYPE_INT ){
          fprintf( disp, "  %-9s = [%8g]\n", 
                   &sc.prm_name[3], sc.prm_val.prm_real);
          row++;
        } else {
          fprintf( disp, "  %-9s = [%8ld]\n", &sc.prm_name[3], 
                   sc.prm_val.prm_intg);
          row++;
        }
        found = 1;
        strcpy( current_mdl_name, sc.mdl_name );
      }
    }
    if( found == 0 ){
      fprintf( disp, "\nThere is no parameter.\n");
      row+=2;
    }
    PrmCloseSCF();
  } else {
    fprintf( disp, "Error: Can't open the file of parameter condition!!\n");
  }
  fclose(disp);
  return( row );
}

/**************************************************************************
       Functions to read simulation condition file
 **************************************************************************/

/*
 *  Read one entry from n potition of output information file
 * 	PrmFile *src: storing at the location 
 *      int  n      : position
 *      Return      : TRUE : success.
 *                    FALSE : failed.
 */
static int PrmNthGetInfoSCF( PrmFile *src, int n ){
  if( PrmOpenSCF() == NCS_FALSE ){
    return( 0 );
  }
  PrmTopSCF();
  PrmNthNextSCF( n );
  if( PrmReadSCF( src ) == NCS_FALSE ){
    PrmCloseSCF();
    return( NCS_FALSE );
  }else{
    PrmCloseSCF();
    return( NCS_TRUE );
  }
}

/**************************************************************************
  	Basic functions to control simulation condition file
 **************************************************************************/


/*
 *  Open parameter information file to read and write
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int PrmOpenSCF(){
  const char *path;
  char mname[SCF_SCFN_PATH_SIZE];
  char fname[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s", 
              path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
    fname[SCF_SCFN_PATH_SIZE-1]='\0';
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  
  if( (prm = _OpenSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Create parameter information file
 * 	char *fname : file name of file
 *      Return      : TRUE  : success
 *                    FALSE : failed.
 */
static int PrmCreateSCF(){
  const char *path;
  char mname[SCF_SCFN_PATH_SIZE];
  char fname[SCF_SCFN_PATH_SIZE];
  
  if( (path=GetWorkDirSCF()) != NULL ){
#ifdef HAVE_SNPRINTF
    snprintf( fname, SCF_SCFN_PATH_SIZE, "%s/%s.%s",
              path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
    fname[SCF_SCFN_PATH_SIZE-1]='\0';
#else
    sprintf( fname, "%s/%s.%s", path, ModelNameSCFN( mname ), SCF_PRM_EXTN );
#endif
  } else {
    return( NCS_FALSE );
  }
  
  if( (prm = _CreateSCF( fname )) == NULL ){
    return( NCS_FALSE );
  } else {
    return( NCS_TRUE );
  }
}

/*
 *  Close parameter information file
 */
static int PrmCloseSCF(){
  return( _CloseSCF( prm ) );
}

/*
 *  Write one entry to parameter information file
 * 	PrmFile *src: reading at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE  : success.
 *                    FALSE : failed.
 */
static int PrmWriteSCF( PrmFile *src ){
  if( _WriteSCF( prm, (void *)src, sizeof(PrmFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Read one entry from parameter information file
 * 	PrmFile *src: storing at the location 
 *      Return      : advance the file potion indicator.
 *                    TRUE : success.
 *                    FALSE : failed.
 */
static int PrmReadSCF( PrmFile *src ){
  if( _ReadSCF( prm, (void *)src, sizeof(PrmFile) ) == SCF_ERROR ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 *  Set file pointer top of parameter information file
 */
static void PrmTopSCF(){
  fseek( prm, 0L, SEEK_SET );
}

/*
 *  Set file pointer end of parameter information file file
 */
static void PrmEndSCF(){
  fseek( prm, (long)sizeof(PrmFile)*(-1), SEEK_END );
}

/*
 *  advance file pointer of parameter information file file
 */
static void PrmNextSCF(){
  fseek( prm, (long)sizeof(PrmFile), SEEK_CUR );
}

/*
 *  advance nth file pointer of parameter information file file
 */
static void PrmNthNextSCF( int n ){
  int	i;
  for( i = 0; i < n; i++ ){
    PrmNextSCF();
  }
}

/*
 *  retreat file pointer of parameter information file file
 */
static void PrmPreviousSCF(){
  fseek( prm, (long)sizeof(PrmFile)*(-1), SEEK_CUR );
}

/*
 *  retreat nth file pointer of parameter information file file
 */
static void PrmNthPreviousSCF( int n ){
  int i;
  for( i = 0; i < n; i++ ){
    PrmPreviousSCF();
  }
}

#ifdef __cplusplus
}
#endif
