/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/******************************************************************************

                   NEURAL CIRCUIT SIMULATOR                 
                      SIMULATION PROGRAM                    
                         MAIN ROUTINE                       

    $Id: ncssmain.cpp,v 1.5 2005/02/23 04:40:23 orrisroot Exp $

*****************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "libsatellite.h"

#include "libncsc.h"
#include "libncss.h"
#include "ncsmodel.h"

#ifdef __cplusplus
extern "C" {
#endif

typedef void (*INTEG_METHOD)(INTEG_FUNC,double,double,int,int,double**,double*,int*);

INTEG_METHOD Integrl;

static INTEG_METHOD ncsf_integset( char *integ_method );
static void PreProcess(char *moni_vflag, int *dpoint, int *calc_end, 
                       int *str_period, double *report_end);
static void CalcInit(char *moni_vflag, int *iflag, int *index, 
                     double report_end);
static void Calculate(char *moni_vflag, int *iflag, int *index, 
                      double report_end, int calc_end, int str_period);
static void PostProcess(char *moni_vflag, int dpoint, int nstrd);

/* realtime.cpp */
#ifdef DSPTEC_EXTENSION
void realtime_calculate_mode2(char *moni_vflag, int *iflag, int *index,
							  double report_end, int calc_end, int str_period);
#endif

/******************************************************************************
*          Cell counter  Modify function                                      *
*                 : Calling from Main function                                *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/

static INTEG_METHOD ncsf_integset( char  *integ_method ){
  int             type;
  for( type = 0; type < ncsg_ntout; type++ ){
    if( ncsg_ncout[type] > ncsg_cell ){
      ncsg_ncout[type] = ncsg_cell;
    }
  }
  switch( *integ_method ){
    case 'E':
      return( EulerIntgrl );
      break;
    case 'F':
      return( AdapIntgrl );
      break;
    case 'R':
      return( RKG4Intgrl );
      break;
    default:
      break;
  }
  return( NULL );
}



/******************************************************************************
*          Calling from AdapIntegrl function                                  *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/

void cf( double t, double **ncs_y, double **ncs_dy ){
  NCS_TIME = t;

  /*
   * NETWORK() is produced by preprocessor 
   */
  NETWORK( t, ncs_y, ncs_dy );

  /* update memory contents */
  ncsf_update( ncsg_ntout, ncsg_ncout, ncsm_outptr, ncsm_subptr );
}



/******************************************************************************
*          NCS Simulation Program Main function                               *
*                                            Coded by S.Hitomi  09/05/1990    *
******************************************************************************/

void PreProcess( char *moni_vflag, int *dpoint, int *calc_end, 
                 int *str_period, double *report_end ){
  /* allocate memory for global variables of simulation program */
  ncsf_workset();

  /*
   * setting up of constants (This function is produced by
   * preprocessor.) 
   */
  ncsf_constset();

  /* preparation of exinput buffer information */
  ncsg_buff_info = plist_create( PL_ERROR_DISP );
  if(ncsg_buff_info == NULL)
    exit(213); /* memory allocation error */

  ncsg_buff_info_p = plist_go_top( ncsg_buff_info );
  exbuff_num = 0;
  /*
    printf( "ncsg_buff_info, next = %x, %x\n", 
            ncsg_buff_info, ncsg_buff_info->next );
  */

  /* reading of simulation conditions */
  if( ncsf_scfread() == NCS_ERROR ){
    printf("Simulation Condition File Reading Error !!\n");
    exit(219);
  }

  /* setting up of simulation conditions */
  ncsf_scset( dpoint, calc_end, str_period, report_end );

  /* this is for error on FreeBSD */
  /* You NOT remove this line.     */
  /*rr_end = (int)*report_end;  */

  /* Integration Information Setting    */
  Integrl = ncsf_integset( &ncsg_intg_method );

  /* Memory Allocation ... y, yp, buffer */
  ncsf_memalloc( ncsg_nstrd, *dpoint );

  /* initializing states of components */
  ncsf_initset();

  /*
   * initial value set for Integration ( This function is produced by
   * preprocessor ) 
   */

  initvalue(ncs_y);

  /* setting up of monitering condition */
  ncsf_moniset( moni_vflag );

  /* displaying of start sign */
  ncsf_startsign();

  /* indicating of calculation progress report */
  if( *moni_vflag == NCS_FLAG_OFF ){
    ncsf_repostart();
  }else{
    printf("\n");
  }
}


void CalcInit( char *moni_vflag, int *iflag, int *index, double report_end ){
  double  time;

  /* ----------------------------------------------------------------- */
  /* Start of Calculation for Time = 0                 */
  /* ----------------------------------------------------------------- */
  /* Integration taking one step ( initialize ) */
  *iflag = 1;    /* for initialize   */

  ncsg_cdpoint = 0;
  time = 0.0;

  /* Initialize of Module output (time = -1) */
  ncsg_delay_flag = OUTPUT_INIT;

#ifdef DSPTEC_EXTENSION
  /*
   * setting up of external input values
   *   if sweep function, ...
   */
  ncsf_xinset_init(ncsg_nxinc, ncsg_xindata, ncsg_xfpdata);
#endif
  /*
   * setting up of external input values 
   */
  ncsf_xinset( ncsg_nxinc, ncsg_xindata, ncsg_xfpdata, ncsg_xinfaddr );

  cf( time, ncs_y, ncs_yp );

  /* Initialize of queue */
  ncsg_delay_flag = DELAY_INIT_SET;

  /*
   * setting up of external input values 
   * ncssfunc.cpp
   */
  ncsf_xinset( ncsg_nxinc, ncsg_xindata, ncsg_xfpdata, ncsg_xinfaddr );

  cf( time, ncs_y, ncs_yp );
  /*
   * set_delay(); 
   */

  ncsg_delay_flag = CALCULATE;

  (* Integrl)( cf, time, ncsg_calstep, ncsg_cell, ncsg_neqn, ncs_y,
               &ncsg_relerr, iflag );
  /* ----------------------------------------------------------------- */
  /* End of Calculation for Time = 0                                   */
  /* ----------------------------------------------------------------- */
}

void Calculate( char *moni_vflag, int *iflag, int *index,
                double report_end, int calc_end, int str_period ){
  int            calc_cnt, strd_cnt=0;
  double          time;

#ifdef DSPTEC_EXTENSION
  realtime_calculate_init(moni_vflag, index, report_end, calc_end);
  realtime_calculate_mode2_init();
  if(realtime_calcucate_mode_check() == 1){
    realtime_calculate_mode2(moni_vflag, iflag, index, report_end, calc_end, str_period);
    return;
  }
  //realtime_calculate_mode2_init();
#endif

  /* ----------------------------------------------------------------- */
  /* Start of Calculation routine                                      */
  /* ----------------------------------------------------------------- */
  for( calc_cnt = 0; calc_cnt <= calc_end; calc_cnt++){
#ifdef DSPTEC_EXTENSION
    if(realtime_calculate_mode2_wait()) break;
#endif

    if(((ncsg_intg_method) == ('E'))&&(calc_cnt == 0)){
      calc_cnt = 1;
      calc_end++;
    }

    /* for interporation (Buffer Input) */
    ncsg_cdpoint = (int)calc_cnt;

    time = ((double) calc_cnt) * ncsg_calstep;

#ifdef DSPTEC_EXTENSION
    realtime_calculate_mode2_display(calc_cnt,time);
#endif

    /*
     * setting up of external input values
     * ncssfunc.cpp
     */
    ncsf_xinset(ncsg_nxinc, ncsg_xindata, ncsg_xfpdata, ncsg_xinfaddr);

    /* storing of results into memory area */
    if( strd_cnt % str_period == 0 ){

      ncsf_dataout( moni_vflag, (*index)++, strd_cnt );/*ncssfunc.cpp*/
      if( *moni_vflag == NCS_FLAG_OFF ){
        ncsf_repoprogress( calc_cnt, report_end ); /*ncssfunc.cpp*/
      }

    }
    strd_cnt = calc_cnt + 1;
    /*
     * Integration taking one step NETWORK() is produced by
     * preprocessor in cf(). 
     */

    (*Integrl)(cf, time, ncsg_calstep, ncsg_cell, ncsg_neqn, ncs_y,
               &ncsg_relerr, iflag);

#ifdef DSPTEC_EXTENSION
    realtime_calculate_mode2_next();
#endif    
  }
  /* ----------------------------------------------------------------- */
  /* End of Calculation routine                        */
  /* ----------------------------------------------------------------- */
}

void PostProcess( char *moni_vflag, int dpoint, int nstrd ){
  /* indicating of calculation progress report */
  if( *moni_vflag == NCS_FLAG_OFF ){
    ncsf_repoend();
  }

  /* indicating of end sign */
  if( *moni_vflag == NCS_FLAG_ON ){
    printf("\n");
  }
  ncsf_endsign();

  /* ----------------------------------------------------------------- */
  /* Saving  data into SATELITE Data Buffer                            */
  /* ----------------------------------------------------------------- */
  ncsf_StoreData( dpoint, nstrd );

  /* Free Memory */
  remove_all_exbuf_info( ncsg_buff_info );
  ncsf_FreeOutBuffer();

  /* free allocated memory for global variable of simulation program */
  ncsf_workcls();

#ifdef DSPTEC_EXTENSION
  /* close H/W */
  ncsf_scfclose();
#endif


}


/*----------------------------- Start of main -------------------------------*/
int main()
{
  char    moni_vflag;
  int     iflag, dpoint, index = 0;
  int     calc_end, str_period;
  double  report_end;

  load_syscom_file();

#ifdef NCS_MODEL_DYNAMIC_LOADING
  if(ncsd_model_open() != 0){
    return 1;
  }
#endif

  /* Setting of simulation condition */
  PreProcess( &moni_vflag, &dpoint, &calc_end, &str_period, &report_end );

#ifdef DSPTEC_EXTENSION
  realtime_calculation_open_mapfile();
#endif

  /* Integration taking one step ( initialize ) */
  CalcInit( &moni_vflag, &iflag, &index, report_end );

  /* Start Calculation */
  Calculate( &moni_vflag, &iflag, &index, 
       report_end, calc_end, str_period );

  /* Store Data */
  PostProcess( &moni_vflag, dpoint, ncsg_nstrd );

#ifdef DSPTEC_EXTENSION
  realtime_calculation_close_mapfile();
#endif

#ifdef NCS_MODEL_DYNAMIC_LOADING
  ncsd_model_close();
#endif
  return(0);
}

#ifdef WIN32
#ifdef BCC
int _matherr( struct _exception *except ){
#ifdef DSPTEC_EXTENSION
  realtime_calculation_close_mapfile();
#endif
  /* Create error file */
  FILE *fp;
  fp = fopen("ncscal.err","w");
  fprintf( fp, "%s\n", except->name );
  if(except->type == DOMAIN)
    fprintf( fp, "DOMAIN error\n" );
  else if(except->type == OVERFLOW)
    fprintf( fp, "OVERFLOW error\n" );
  else if(except->type == UNDERFLOW)
    fprintf( fp, "UNDERFLOW error\n" );
  else
    fprintf( fp, "Other error\n" );
  /* close*/
  fclose(fp);
  exit(999);
}
#endif
#endif

#ifdef __cplusplus
}
#endif

/*------------------------------- End of main -------------------------------*/
