/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/*****************************************************************************
  
                     NCS LIBRARY FOR SIMULATION PROGRAM
  
    $Id: ncsslib.cpp,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $

*****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define MOD_NCS_LIBNCSS_EXPORTS
#include "libncsc.h"
#include "libncss.h"
#include "ncssdata.h"

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************
FUNCTION  NCSL_INTGRL
******************************************************************************
NUMERICAL INTEGRATION
  
return( init ) : initial value ( when TIME = 0. );
return( f ) : integrated value ( when TIME > 0. );
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_intgrl( double init, double y, double f_old ){
  double f;

  if( NCS_TIME == 0. ){
    return (init);
  }
  f = f_old + y * ncsg_calstep;

  return( f );
}


/*****************************************************************************
FUNCTION  NCSL_DIFFER
******************************************************************************
NUMERICAL DIFFERENCIATION
  
return( init ) : initial value ( when TIME = 0. );
return( f ) : function value ( when TIME >= 0. );
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_differ( double init, double y, double y_old ){
  double f;

  if ( NCS_TIME == 0. ){
    return (init);
  }
  f = (y - y_old) / ncsg_calstep;

  return( f );
}


/*****************************************************************************
FUNCTION  NCSL_RAMP
******************************************************************************
RAMP FUNCTION
  
return( init_out ) : initial value ( when TIME < rampup_time );
return( f ) : function value ( when TIME >= rampup_tm );
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_ramp( double rampup_tm, double init_out, 
                            double gradient, double dummy1, double dummy2 ){
  double f;

  if( NCS_TIME < rampup_tm ){
    return( init_out );
  }
  f = gradient * (NCS_TIME - rampup_tm) + init_out;

  return( f );
}


/*****************************************************************************
FUNCTION  NCSL_PULSE
******************************************************************************
PULSE FUNCTION
  
return( init_out ) : initial value ( when TIME < start_tm );
return( init_out + height ) : function value ( when TIME >= start_tm );
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_pulse( double start_tm, double init_out, 
                             double height, double width, double period ){
  double  d_val, tm_mod;

  if( NCS_TIME < (start_tm - MachineEps / 2.0) ) {
    return (init_out);
  }
  if(period != 0.0)  {
    d_val = (NCS_TIME - start_tm) / period;
    d_val = (double)(long)d_val;

    tm_mod = NCS_TIME - start_tm - d_val * period;
  } else {
    tm_mod = NCS_TIME - start_tm;
  }


  if( tm_mod < (width - MachineEps / 2.0) ){
    return (init_out + height);
  } else {
    return (init_out);
  }
}


/*****************************************************************************
FUNCTION  NCSL_SIGMOID
******************************************************************************
SIGMOID FUNCTION
  
return( f ) : function value;
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_sigmoid( double a ){
  double  f;

  f = 1./ (1.+ exp(-a));

  return( f );
}


/*****************************************************************************
FUNCTION  NCSL_THOLD
******************************************************************************
THRESHOLD FUNCTION
  
return( f ) : function value;
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_thold( double input, double theta ){
    if( input < theta ){
    return( 0. );
  } else {
    return( 1. );
  }
}


/*****************************************************************************
FUNCTION  NCSL_RCAB
******************************************************************************
RATE CONSTANT OF IONIC CURRENT
  
return( g / f ) : function value;
  
MSC Ver.5.1                                     Coded by A.Anzai   06/12/1989
*****************************************************************************/

DLLEXPORT double ncsl_rcab(double c1, double c2, double c3, double c4,
                           double c5, double c6, double c7, double V){
  double  f, g;

  if( c6 == 0.&& c7 == -1. ){
    printf("Error Detected in 'rcab()' NCS Library !!\n");
    printf("Always Divided by Zero Because 'c6' = 0. and 'c7' = -1.0 !!\n");
    exit(NCS_NORMAL);
  } else {
    f = exp(c6 * (V + c3)) + c7;
    if( f == 0. ){
      f = c6 * exp(c6 * (V + c3));
      g = c1 * c2 * exp(c2 * (V + c3)) + c4;
      return( g / f );
    }
    g = c1 * exp(c2 * (V + c3)) + c4 * (V + c5);
    return( g / f );
  }
}

#ifdef __cplusplus
}
#endif
