/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/**************************************************************************
                                                                        
   Parser Program [npp.y] for yacc of NCS PreProcessor Program on Unix
                                                                        
    $Id: npp-yacc.y,v 1.1.1.1 2004/03/31 08:15:05 orrisroot Exp $

**************************************************************************/
%{

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#ifdef WIN32
# include <io.h>
# include <process.h>
# include <direct.h>
#else
# include <unistd.h>
#endif

#include "libncsc.h"
#include "chklist.h"
#include "list.h"
#include "shfilter.h"

#define NCS_ENDMARK_GPP  "@"
#define NCS_SPF_EXTN     "cpp"
#define NPP_VAL_LEN      16  /* LENGTH OF VALUE + EOS */

extern FILE *yyin;
int          yylex();
int          yyparse();
int          yyerror( char *s );

static void  routine_work( char c );
static char *getparameter( const char *ident );
static int   check_module_description();
static void  list_up_modules( List *mod );
static void  set_exinput( const char *var );
static void  set_input( const char *var );
static void  set_delay( char *var, char *time, char *init );
static void  set_observable( char *obs );
static void  mark_observable( char *ident );
static void  insert_observable( List *table );
static void  set_parameter( char *parm, char *val, int flag );
static void  init();
static void  make_header( char *input_file_name, char *usr_header_file );
static void  make_footer();
static void  make_headerfile( char *input_file_name );
static void  make_ncsf_constset();
static void  make_csources();
static void  make_initvalue();
static void  make_ncsf_prmset();
static void  make_network();
static void  make_getprmptr();
static void  make_prm_file( List *info );
static void  make_xin_file();
static void  make_out_file();
static void  make_obs_file( List *info );
static void  make_dly_file( List *info );
static void  make_mat_file();
static void  make_line( List *line_ptr );
static void  endprocess();

static void make_ncsf_workset();
static void make_ncsf_workcls();

/***** Variables to store information of model description  *****/
int npp_lineno = 1;           /* a number of lines                        */
int npp_charno = 0;           /* a number of characters                   */
int npp_tabno = 0;            /* a number of tab codes                    */
int npp_mline = 0;            /* difference number of line                */
int npp_integ_flag = 0;       /* flag of integration                      */
int npp_num_of_integ = 0;     /* a number of integration                  */
int npp_module_id = 0;        /* module ID                                */
int npp_num_of_xin = 0;       /* a number of exinputs                     */
int npp_num_of_obs = 0;       /* a number of observables                  */
int npp_num_of_input = 0;     /* a number of inputs                       */
int npp_num_of_delay = 0;     /* a number of delays                       */
int npp_max_input = 0;        /* maximal number of inputs                 */
int npp_module_count = 0;     /* a number of modules                      */

static char *mfile_path;      /* file name of model (work)                */
static char *model_file_name; /* file name of model (source)              */
static int error_num = 0;     /* a number of errors                       */

/***** Lists to store information of NETWORK description *****/
static int network_flag = 0;    /* flag for NETWORK description             */
static char net_name[32];       /* a name of NETWORK                        */
static List network;            /* NETWORK description written by C         */
static List net_info;           /* information of NETWORK description       */
static List net_def;            /* list of define for paramter in NETWORK   */
static List net_undef;          /* list of undef for paratmer in NETWORK    */
/* modules described left side equation */
static List net_module; /* in NETWORK        */
static List net_inputs; /* in NETWORK(input) */
static List net_loads;  /* in NETWORK(output)*/
static List cell_modules;       /* Cell modules defined in NETWORK          */
static List synapse_modules;    /* Synapse modules defined in NETWORK       */
static List gap_modules;        /* Gap modules defined in NETWORK           */

static List name;               /* list of name of modules                  */

/***** Lists to describe functions of module *****/
static List module;             /* final description of function            */
static List define;             /* define of paramter before function       */
static List integ;              /* for intagration                          */
static List f_header;           /* header of function                       */
static List declare;            /* variable declaration(double) in function */
static List constant;           /* description produced "constant:"         */
static List constant2;          /* description produced "constant:"         */
static List delay;              /* description produced delay statements    */
static List function;           /* description produced "function:"         */
static List exinput;            /* information of "exinput:"                */
static List input;              /* informaiton of "input:"                  */
static List observable;         /* description produced "observable:"       */
static List observ_table;       /* table of "observable:" in all module     */
static List initial;            /* for "initial:"                           */
static List f_end;           /* end of lines of function includes "output:" */
static List undef;              /* for undef                                */

static List output;             /* list of "output:"                        */
static List fact;               /* list of factor                           */
static List parameter;          /* for ncsf_prmset(), make_prm_file()       */
static List param_table;        /* table of "parameter:" in all module      */
static List initvalue;          /* for initvalue()                          */
static List getprmptr;          /* for getprmptr() (to link NPE)            */

/***** Lists to describe functions of module *****/
static List header;             /* final description for header file        */
static List ncs_xin;            /* list to write exinput (module ID -> ...) */
static List ncs_obs;            /* to write header file                     */
static List ncs_double;         /* for declaration of parameter in header   */

static List net_table;       /* list of component of module written NETWORK */
static List module_table;       /* module name and module ID                */
static List delay_file;         /* infomation of delay                      */
                                /*     mdl_name -> inp_name -> time -> init */

/***** Lists to directory describe in C language *****/
List npp_clang;
static int  csflag;

/***** work memory *****/
static FILE *fp;                /* file pointer of *.c                      */
char npp_buffer[100000];        /* working space                            */
static const char *work_dir;    /* path name of Temporary                   */

%}

%union {
    int num;
    char *ptr;
    double dval;
    int ival;
}

%token <num> SCOLON COLON IF ELSE CELL GAP SYNAPSE NETWORK
%token <num> EXINPUTM NCS_TYPE CONEX FOR CELLN SYNN GAPN NAME EXINPUT
%token <num> PARAMETER INIT FUNCTION Input OUTPUT OBSERVABLE END
%token <num> CONSTANT BRCL BRCR BRKL BRKR MODULE
%token <ptr> LP RP EQUAL COMMA PLEQ MIEQ MLEQ DVEQ
%token <ptr> IDENT NUMBER

%left <ptr> OR
%left <ptr> AND
%left <ptr> EQ GR LESS LEQ NE GEQ
%left <ptr> PLUS MINUS
%left <ptr> DIV MULT MOD
%token <ptr> INC DEC

%nonassoc UMINUS

%type <ptr> factor2 exp1 substitution net_name
%type <ptr> character characters
%type <ptr> if_state if_state2 function
%type <ptr> parameter parameter2
%type <ptr> exinput exinput2 exinput3 
%type <ptr> cell module_name output observable observable2
%type <ptr> constant constant2 
%type <ptr> input1 input2 input3
%type <ptr> initial initial2 initial3 
%type <ptr> inputs inputs2
%type <ptr> mname2 cellname mname mnames
%type <ptr> num_cal fnumber
%type <ptr> id_exp id_assign_exp assign_operator
%type <ptr> id_condition_exp id_logic_or_exp id_logic_and_exp
%type <ptr> id_eqn_exp id_relation_exp id_add_exp id_multi_exp
%type <ptr> id_unary_exp id_post_exp id_prim_exp
%type <dval> va_exp va_add_exp va_multi_exp
%type <dval> va_unary_exp va_post_exp va_prim_exp
%type <ival> iexp iexps
%type <ptr> id_add_exp1 id_multi_exp1 id_unary_exp1 id_prim_exp1
%type <ptr> cell_body syn_body


%%
/*** Model descrition ***/
model   : network modules { ; }
        ;

modules : /* empty */
        | modules module
        ;

module  : cell            { npp_module_count++; }
        | syn             { npp_module_count++; }
        | gap             { npp_module_count++; }
        ;

/*** NETWORK description ***/
network : NCS_TYPE COLON NETWORK SCOLON net_name net_body_list 
          connect END SCOLON
          { 
            /* for paramters */
            network_flag = 0;
            clearlist( &define );
            clearlist( &declare );
            clearlist( &parameter );
            clearlist( &undef );
            clearlist( &fact );
            
            insert( NCS_ENDMARK_GPP, endlist(&getprmptr), &getprmptr );
            
            /* check matching modules */
            if( check_module_description() == NCS_FALSE ){
              /* yyerror( "module mismatch" ); */
            }
          }
        | error END SCOLON
          {
            /* fatal error in NETWORK description */
            exit(100);
          }
        ;

net_body_list : net_body                { ; }
              | net_body_list net_body  { ; }
              ;

net_body : cellname                { ; }
         | synname                 { ; }
         | gapname                 { ; }
         | parameter               { ; }
         | error SCOLON            { yyerrok; }
         ;

net_name : MODULE COLON IDENT SCOLON
           {
             network_flag = 1;
             strcpy( net_name, $3 );  
             insert( "0", endlist(&name), &name ); 
             insert( $3, endlist(&name), &name ); 
           }
         ;

/*** CELL MODULE description ***/
cell : NCS_TYPE COLON CELL SCOLON module_name cell_body_list
       function END SCOLON
       { 
         int i;
         routine_work('b');
         insert( "1", endlist(&module_table), &module_table );

         /* insert observable information to lists */
         insert_observable( &observ_table );
         
         unique( &declare );
         
         /* connect lists to write CELL modules */
         sprintf( npp_buffer, "/*** module ***/\n" );
         insert( npp_buffer, endlist(&module), &module );
         listcat( &module, &define );
         listcat( &module, &integ );
         listcat( &module, &f_header );
         listcat( &module, &declare );
         listcat( &module, &constant );
         listcat( &module, &delay );
         listcat( &module, &initial );
         listcat( &module, &function );
         listcat( &module, &observable );
         listcat( &module, &f_end );
         listcat( &module, &undef );

         if( csflag == 0 ){
           make_csources();
           csflag++;
         }
         
         for( i = 1; i < endlist(&module); i++ ){
           fprintf( fp, "%s", see(i,&module) );
         }
         npp_num_of_input = 0;
         routine_work('e');
         
         /* to link NPE */
         insert( NCS_ENDMARK_GPP, endlist(&getprmptr), &getprmptr );
       }
     ;

cell_body_list : cell_body                  { ; }
               | cell_body_list cell_body   { ; }
               ;

cell_body : input1                { ; }
          | output                { ; }
          | observable            { ; }
          | constant              { ; }
          | parameter             { ; }
          | initial               { ; }
          | exinput               { ; }
          | error SCOLON          { yyerrok; }
          ;

/*** SYNAPSE MODULE description ***/
/*syn     : *//* empty */                      /* { ; }*/
syn : NCS_TYPE COLON SYNAPSE SCOLON module_name syn_body_list 
      function END SCOLON
      { 
        int i;
        routine_work('b');
        insert( "2", endlist(&module_table), &module_table );
        
        /* insert observable information to lists */
        insert_observable( &observ_table );
        
        /* connect lists to write SYNAPSE modules */
        listcat( &module, &define );
        listcat( &module, &integ );
        listcat( &module, &f_header );
        listcat( &module, &declare );
        listcat( &module, &constant );
        listcat( &module, &delay );
        listcat( &module, &initial );
        listcat( &module, &function );
        listcat( &module, &observable );
        listcat( &module, &f_end );
        listcat( &module,&undef );
        for( i = 1; i < endlist( &module ); i++){
          fprintf( fp, "%s", see(i,&module) );
        }
        routine_work('e');
        
        /* to link NPE */
        insert( NCS_ENDMARK_GPP, endlist(&getprmptr), &getprmptr );
      }
    ;

syn_body_list : syn_body                    { ; }
              | syn_body_list syn_body      { ; }
              ;

syn_body : inputs                  { ; }
         | output                  { ; }
         | observable              { ; }
         | constant                { ; }
         | parameter               { ; }
         | initial                 { ; }
         ;

/*** GAP MODULE description ***/
/*gap : *//* empty */                      /* { ; }*/
gap : NCS_TYPE COLON GAP SCOLON module_name syn_body_list function END SCOLON
      {
        int i;
        routine_work('b');
        insert( "3", endlist(&module_table), &module_table );
        
        /* insert observable information to lists */
        insert_observable( &observ_table );
        
        /* connect lists to write GAP modules */
        listcat( &module, &define );
        listcat( &module, &integ );
        listcat( &module, &f_header );
        listcat( &module, &declare );
        listcat( &module, &constant );
        listcat( &module, &delay );
        listcat( &module, &initial );
        listcat( &module, &function );
        listcat( &module, &observable );
        listcat( &module, &f_end );
        listcat( &module, &undef );
        for( i = 1; i < endlist(&module); i++ ){
          fprintf( fp, "%s", see(i,&module) );
        }
        routine_work('e');  
        
        /* to link NPE */
        insert( NCS_ENDMARK_GPP, endlist(&getprmptr), &getprmptr );
      }
    ;

/*** "module" statement ***/
module_name : MODULE COLON IDENT SCOLON
      { 
        sprintf( npp_buffer,
          "double %s(double NcsTime, double **Ncs_y, double **Ncs_dy)\n{\n",
          $3 );
        insert( npp_buffer, 1, &f_header );

        /* module name and module ID are stored in module_table */
        insert( $3, endlist(&module_table), &module_table );
        if( locate( $3, &net_table ) == 0 ){
          exit(101);
        }
        sprintf( npp_buffer, "%s", see(locate($3,&net_table)+2,&net_table) );
        insert( npp_buffer, endlist(&module_table), &module_table );
        npp_module_id = atoi( npp_buffer );
        
        /* to link NPE */
        sprintf( npp_buffer, "%d", npp_module_id );
        insert( npp_buffer, endlist(&name), &name ); 
        insert( $3, endlist(&name), &name ); 
      }
    ;

/*** for "cell", "synapse" and "gap" statement ***/
mnames : mnames COMMA mname 
         {
           sprintf( npp_buffer, "%s:%s", $1, $3 );
           $$ = strdup( npp_buffer );  
         }
       | mname  { $$ = $1; }
       ;

mname : IDENT BRKL NUMBER BRKR
        { /* module name, number of component, module ID are stored  */
          /* in net_table                                            */
          insert( $1, endlist(&net_table), &net_table );
          insert( $3, endlist(&net_table), &net_table );
          sprintf( npp_buffer, "%d", npp_module_id );
          insert( npp_buffer, endlist(&net_table), &net_table );
          npp_module_id++; 
          sprintf( npp_buffer, "%s,%s", $1, $3 );
          $$ = strdup( npp_buffer );
        }
      ;

/*** "cell" statement ***/
cellname : CELLN COLON mnames SCOLON
           { 
             insert( $3, endlist(&cell_modules), &cell_modules );
             list_up_modules( &cell_modules );
           }
         ;

/*** "synapse" statement ***/
synname : SYNN COLON mnames SCOLON                      
          { 
            insert( $3, endlist(&synapse_modules), &synapse_modules );
            list_up_modules( &synapse_modules );
          }
        ;

/*** "gap" statement ***/
gapname : GAPN COLON mnames SCOLON { ; }
          { 
            insert( $3, endlist(&gap_modules), &gap_modules );
            list_up_modules( &gap_modules );
          }
        ;

/*** "connection" statement **/
connect : CONEX COLON jexps { ; }
        ;

jexps   : jexps jexp
          { insert( ";", endlist(&net_info), &net_info ); }
        | jexp
          { insert( ";", endlist(&net_info), &net_info ); }
        ;

jexp    : mname2 LESS iexps SCOLON
          { 
            insert( "*", endlist(&net_info), &net_info );
            insert( $1, endlist(&net_info), &net_info );

            sprintf( npp_buffer, "%d", $3 );
            insert( $1, endlist(&net_module), &net_module );
            insert( npp_buffer, endlist(&net_module), &net_module );
          }
        | for     { ; }
        | error   { yyerrok; }
        ;

mname2  : IDENT BRKL id_exp BRKR
          { 
            /* define Check*/
            int dflag = 0, pt, dim;

            if( (pt = locate( $1, &cell_modules )) == 0 ){
              dflag++;
            }else{
              dim = atoi(see( pt+1, &cell_modules ));
              if( atoi($3) >= dim ){
                sprintf( npp_buffer,
                         "module \"%s\" dimension mismatch",$1 );
                yyerror( npp_buffer );
              }
            }
            if( (pt = locate( $1, &synapse_modules )) == 0 ){
              dflag++;
            }else{
              dim = atoi(see( pt+1, &synapse_modules ));
              if( atoi($3) >= dim ){
                sprintf( npp_buffer,
                         "module \"%s\" dimension mismatch",$1 );
                yyerror( npp_buffer );
              }
            }
            if( (pt = locate( $1, &gap_modules )) == 0 ){
              dflag++;
            }else{
              dim = atoi(see( pt+1, &gap_modules ));
              if( atoi($3) >= dim ){
                sprintf( npp_buffer,
                         "module \"%s\" dimension mismatch",$1 );
                yyerror( npp_buffer );
              }
            }
            if( dflag == 3 ){
              sprintf( npp_buffer, "module \"%s\" is not defined", $1 );
              yyerror( npp_buffer );
            }
            
            sprintf(npp_buffer,"%s,%s",$1,$3);
            $$=strdup(npp_buffer);
          }
        ;

iexps : iexps iexp  { $$ = $1 + $2; }
      | iexp        { $$ = $1; }
      ;

iexp  : LP jterms RP  
        { insert( "!", endlist(&net_info), &net_info );  $$ = 1; }
      | LP RP         { $$ = 1; }
      ;

jterms : jterm                          { ; }
       | jterms PLUS jterms             { ; }
       ;

jterm : mname2 LESS mname2
        { 
          sprintf( npp_buffer,"%s<%s", $1, $3 );
          insert( npp_buffer, endlist(&net_info), &net_info );

          insert( $1, endlist(&net_loads), &net_loads );
          insert( $3, endlist(&net_inputs), &net_inputs );
        }
      | LP jterms RP                    { ; }
      ;

/** "for" statement **/
for   : FOR scope BRCL jexps BRCR
        { 
          insert( "$", endlist(&net_info)-1, &net_info );
          /* take 1 from endlist(), because there is "$" after ";" */  
        }
      ;

scope : LP id_exp SCOLON id_exp SCOLON id_exp RP
        { 
          insert( "for", endlist(&net_info), &net_info );
          sprintf( npp_buffer, "int %s;", $2 );
          insert( npp_buffer, endlist(&net_info), &net_info );
          sprintf( npp_buffer, "%s ; %s ; %s", $2, $4, $6 );
          insert( npp_buffer, endlist(&net_info), &net_info );
        }
      ;

id_exp :  id_assign_exp
          { sprintf( npp_buffer, "%s",$1 ); $$=strdup( npp_buffer ); }
       ;

id_assign_exp : id_condition_exp
                {
                  sprintf( npp_buffer, "%s",$1 );
                  $$=strdup( npp_buffer );
                }
              | id_unary_exp assign_operator id_assign_exp
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$=strdup( npp_buffer );
                }
              ;

assign_operator  : EQUAL               { ; }
                 | PLEQ                { ; }
                 | MIEQ                { ; }
                 | MLEQ                { ; }
                 | DVEQ                { ; }
                 ;

id_condition_exp : id_logic_or_exp
                   { 
                     sprintf( npp_buffer, "%s",$1 );
                     $$ = strdup( npp_buffer );
                   }
                 ;

id_logic_or_exp  : id_logic_and_exp
                   { 
                     sprintf( npp_buffer, "%s",$1 );
                     $$ = strdup( npp_buffer );  
                   }
                 | id_logic_or_exp OR id_logic_or_exp
                   { 
                     sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                     $$ = strdup( npp_buffer );  
                   }
                 ;

id_logic_and_exp : id_eqn_exp
                   { 
                     sprintf( npp_buffer, "%s",$1 );
                     $$ = strdup( npp_buffer );
                   }
                 | id_logic_or_exp AND id_logic_or_exp
                   {
                     sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                     $$ = strdup( npp_buffer );
                   }
                 ;

id_eqn_exp : id_relation_exp
             {
               sprintf( npp_buffer, "%s",$1 );
               $$ = strdup( npp_buffer );
             }
           | id_eqn_exp EQ id_relation_exp
             {
               sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
               $$ = strdup( npp_buffer );
             }
           | id_eqn_exp NE id_relation_exp
             {
               sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
               $$ = strdup( npp_buffer );
             }
           ;

id_relation_exp : id_add_exp
                  {
                    sprintf( npp_buffer, "%s",$1 );
                    $$ = strdup( npp_buffer );
                  }
                | id_relation_exp LESS id_add_exp
                  {
                    sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                    $$ = strdup( npp_buffer );
                  }
                | id_relation_exp GR id_add_exp
                  {
                    sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                    $$ = strdup( npp_buffer );
                  }
                | id_relation_exp LEQ id_add_exp
                  {
                    sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                    $$ = strdup( npp_buffer );
                  }
                | id_relation_exp GEQ id_add_exp
                  {
                    sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                    $$ = strdup( npp_buffer );
                  }
                ;

id_add_exp : id_multi_exp
             {
               sprintf( npp_buffer, "%s",$1 );
               $$ = strdup( npp_buffer );
             }
           | id_add_exp PLUS id_multi_exp
             {
               sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
               $$ = strdup( npp_buffer );
             }
           | id_add_exp MINUS id_multi_exp
             {
               sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
               $$ = strdup( npp_buffer );
             }
           ;

id_multi_exp : id_unary_exp
               {
                    sprintf( npp_buffer, "%s",$1 );
                    $$ = strdup( npp_buffer );
                }
              | id_multi_exp MULT id_unary_exp
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$ = strdup( npp_buffer );
                }
              | id_multi_exp DIV id_unary_exp
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$ = strdup( npp_buffer );
                }
              | id_multi_exp MOD id_unary_exp
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$ = strdup( npp_buffer );
                }
              ;

id_unary_exp : id_post_exp
               {
                 sprintf( npp_buffer, "%s",$1 );
                 $$ = strdup( npp_buffer );
               }
             | INC id_unary_exp  /* %prec UINC */
               {
                 sprintf( npp_buffer, "%s%s", $1, $2 );
                 $$ = strdup( npp_buffer );
               }
             | DEC id_unary_exp  /* %prec UINC */
               {
                 sprintf( npp_buffer, "%s%s", $1, $2 );
                 $$ = strdup( npp_buffer );
               }
             | PLUS id_multi_exp %prec UMINUS
               {
                 sprintf( npp_buffer, "%s%s", $1, $2 );
                 $$ = strdup( npp_buffer );
               }
             | MINUS id_multi_exp %prec UMINUS 
               {
                 sprintf( npp_buffer, "%s%s", $1, $2 );
                 $$ = strdup( npp_buffer );
               }
             ;

id_post_exp : id_prim_exp
              {
                sprintf( npp_buffer, "%s",$1 );
                $$ = strdup( npp_buffer );
              }
            | id_post_exp BRKL id_exp BRKR
              {
                sprintf( npp_buffer, "%s%c%s%c", $1, $2, $3, $4 );
                $$ = strdup( npp_buffer );
              }
            | id_post_exp INC
              {
                sprintf( npp_buffer, "%s%s", $1, $2 );
                $$ = strdup( npp_buffer );
              }
            | id_post_exp DEC
              {
                sprintf( npp_buffer, "%s%s", $1, $2 );
                $$ = strdup( npp_buffer );
              }
            ;

id_prim_exp : NUMBER 
              {
                sprintf( npp_buffer, "%s",$1 );
                $$ = strdup( npp_buffer );
              }
            | IDENT
              {
                sprintf( npp_buffer, "%s",$1 );
                $$ = strdup( npp_buffer );
              }
            | LP id_exp RP
              {
                sprintf( npp_buffer, "%s%s%s",$1,$2,$3 );
                $$ = strdup( npp_buffer );
              }
            | IDENT LP exp1 factor2 RP
              { 
                /* working space for internal function */
                char buf4func[255];
                sprintf( npp_buffer, "%s%s", $3, $4 );
                if( check_lib_func_list( $1, npp_buffer ) == NCS_FALSE ){
                  sprintf( npp_buffer, "%s%s%s%s%s", $1, $2, $3, $4, $5 );
                }else{
                  strcpy( buf4func, npp_buffer );
                  sprintf( npp_buffer, "ncsl_%s%s%s%s", $1, $2, buf4func, $5 );
                }
                $$ = strdup( npp_buffer );
              }
           ;

/*** "exinput" statement ***/
exinput : EXINPUT COLON exinput2 SCOLON     { ; }
        ;
exinput2 : exinput2 COMMA exinput3          { ; }
         | exinput3                         { ; }
         ;

exinput3 : IDENT
           {
             set_exinput( $1 );
             npp_num_of_xin++; 
           }
         | IDENT LP NUMBER COMMA NUMBER RP  { ; }
         ;

/*** "input" statement for cell module ***/
input1   : Input COLON input2 SCOLON        { ; }
         ;

input2   : input2 COMMA input3              { ; }
         | input3                           { ; }
         ;

input3   : IDENT
           {
             sprintf( npp_buffer, "#define\t%s\tNET[%d]\n", $1, 
                      npp_num_of_input );
             insert( npp_buffer, endlist(&define), &define );
             npp_num_of_input++;
             /* maximum number of input */
             if( npp_num_of_input > npp_max_input ){
               npp_max_input = npp_num_of_input;
             }
             
             sprintf( npp_buffer, "#undef\t%s\n", $1 );
             insert( npp_buffer, endlist(&undef), &undef );
             
             set_input( $1 );
             /* can't use delay in CELL module */ 
           }
         ;

/*** "input" statement for synapse and gap module ***/
inputs   : Input COLON inputs2 SCOLON       { ; }
         ;

inputs2  : IDENT
           { 
             sprintf( npp_buffer, "#define\t%s\tINPUT\n", $1 );
             insert( npp_buffer, endlist(&define), &define );
             sprintf( npp_buffer, "#undef\t%s\n",$1 );
             insert( npp_buffer, endlist(&undef), &undef);
             set_input( $1 );
           }
         | IDENT LP num_cal COMMA num_cal RP
           {
             set_input( $1 );
             /* description for delay */
             set_delay( $1, $3, $5 );
           }
         ;

/*** "output" statement ***/
output   : OUTPUT COLON IDENT SCOLON
          {
            sprintf( npp_buffer, "#define\t%s\tncsm_outs%02d[CN]\n", 
                     $3, npp_module_id );
            insert( npp_buffer, endlist(&define), &define );
            sprintf( npp_buffer,
                     "\tncsm_outs%02d[CN] = %s;\n\treturn( %s );\n}\n",
                     npp_module_id, $3, $3 );
            insert( npp_buffer, 1, &f_end );
            insert( $3, 1, &output ); /* prevention multiple define */
            sprintf( npp_buffer, "#undef\t%s\n", $3 );
            insert( npp_buffer, endlist(&undef), &undef );
          }
         ;

/*** "observable" statement ***/
/*observable : */ /* empty */
/*             { $$ = strdup(""); } */
/*           | OBSERVABLE COLON observable2 SCOLON */

observable : OBSERVABLE COLON observable2 SCOLON
             { insert( "\n" , 1 , &observable ); }
           ;

observable2: IDENT
             { set_observable( $1 ); }
           | observable2 COMMA IDENT
             { set_observable( $3 ); }
           ;

/*** "constant" statement ***/
constant : CONSTANT COLON constant2 SCOLON  { ; }
         ;

constant2: IDENT EQUAL num_cal
           { 
             sprintf( npp_buffer, "%s", $1);
             insert( npp_buffer, endlist(&constant2), &constant2 );
             sprintf( npp_buffer, "\tstatic double %s = %s;\n", $1, $3 );
             insert( npp_buffer, endlist(&constant), &constant );
           }
         | constant2 COMMA IDENT EQUAL num_cal
           { 
             sprintf( npp_buffer, "%s", $3);
             insert( npp_buffer, endlist(&constant2), &constant2 );
             sprintf(npp_buffer,"\tstatic double %s = %s;\n", $3, $5 );
             $$ = strdup( npp_buffer );
             insert( $$, endlist(&constant), &constant );
           }
         ;

/*** "parameter" statement ***/
parameter : PARAMETER COLON parameter3 SCOLON { ; }
          ;

parameter3 : parameter3 COMMA parameter2 {;}
           | parameter2  {;}
           ;

parameter2: IDENT EQUAL num_cal
            { set_parameter( $1, $3, network_flag ); }
          ;

/** calculate value of parameter **/
num_cal    : va_exp
             { 
               sprintf( npp_buffer, "%e", $1 );
               $$ = strdup( npp_buffer );
             }
           ;

va_exp          : va_add_exp
                 { $$ = $1;  }
                ;

va_add_exp      : va_multi_exp
                 { $$ = $1;  }
                | va_add_exp PLUS va_multi_exp
                 { $$ = $1 + $3;  }
                | va_add_exp MINUS va_multi_exp
                 { $$ = $1 - $3;  }
                ;

va_multi_exp    : va_unary_exp
                 { $$ = $1;  }
                | va_multi_exp MULT va_unary_exp
                 { $$= $1 * $3;  }
                | va_multi_exp DIV va_unary_exp
                 { $$= $1 / $3;  }
                ;

va_unary_exp    : va_post_exp
                 { $$ = $1;  }
                | PLUS va_multi_exp  %prec UMINUS 
                 { $$ = $2;  }
                | MINUS va_multi_exp %prec UMINUS 
                 { $$ = -1.0 * $2;  }
                ;

va_post_exp     : va_prim_exp
                 { $$ = $1;  }
                ;

va_prim_exp     : NUMBER 
                 { $$ = atof($1);  }
                | LP va_exp RP
                 { $$ = $2;  }
                | IDENT  /* add by T.N 95/6/15 */
                  { 
                    sprintf( npp_buffer, "%s", getparameter($1) );
                    $$ = atof(npp_buffer);
                  }
                ;

/*** "initial" statement. this statement can not use, now. ***/
initial   : INIT COLON initial2 SCOLON
            { 
              sprintf( npp_buffer, "\tif ( TIME == 0. ) {\n" );
              insert( npp_buffer, 1, &initial );
              insert( "\t}\n\n", endlist(&initial), &initial );
            }
          ;

initial2  : initial2 COMMA initial3
          | initial3
          ;

initial3  : IDENT EQUAL num_cal
            {
              sprintf( npp_buffer, "\t\t%s = %s;\n", $1, $3 );
              insert( npp_buffer, endlist(&initial), &initial );
            }
          ;

/*** "function" statement ***/
function  : FUNCTION COLON characters
           { 
             char npp_buffer2[256];
             int i, j, flag = 0;
        
             insert( $3, 1, &function );

             /* check multiple define of variable */
             for( i = 1; i < endlist(&declare); i++ ){
               sprintf( npp_buffer2, "%s", strchr(see(i,&declare),' ')+1 );
               sprintf( npp_buffer2, "%s", strtok(npp_buffer2,";") );
    
               for( j = 1; j < endlist(&define); j++ ){
                 sprintf( npp_buffer, "%s", strchr(see(j,&define),'\t')+1 );
                 sprintf( npp_buffer, "%s", strtok(npp_buffer,"\t") );
                 if( strcmp(npp_buffer,npp_buffer2) == 0 ){
                   SL_delete( i, &declare );
                   flag = 1;
                 }
               }
               
               /* check define of variable equaled to integral */
               for( j = 1; j < endlist( &integ ); j++ ){
                 if( strchr( see( j, &integ ), '_' ) != NULL ){
                   sprintf( npp_buffer, "%s", strchr(see(j,&integ),'_')+1 );
                   sprintf( npp_buffer, "%s", strtok(npp_buffer,"\t") );
                   if( strcmp(npp_buffer,npp_buffer2) == 0 ){
                     SL_delete( i, &declare );
                     flag = 1;
                   }
                 }
               }
               if( flag == 1 ){
                 i--;
                 flag = 0;
               }
             }
           }
         ;

characters : character                       { ; }
           | characters character
             { 
               sprintf( npp_buffer, "%s%s", $1, $2 );
               $$ = strdup( npp_buffer );
             }
           ;

character : substitution SCOLON
            { 
              sprintf( npp_buffer, "\t%s;\n", $1 );
              $$ = strdup( npp_buffer );
            }
          | if_state                         { ; }
          | error SCOLON                     { yyerrok; }
          ;

substitution : IDENT EQUAL exp1
   { 
     int i, flag, ttt, flag2 = 0;
     char *st, *ed, tmpst[1024];
     flag = 0;   /* flag to check multiple define of variable */
     
     if( npp_integ_flag == 1 ){    /* exp1 == "integral(..." */
       
       /* definition for integration before function */
       sprintf( npp_buffer,"#undef\t%s\t/* for integration */\n", $1 );
       insert( npp_buffer, endlist(&integ), &integ );
       sprintf( npp_buffer, "#define\t%s\tNcs_y[CN][%d]\n", 
                $1, npp_num_of_integ );
       insert( npp_buffer, endlist(&integ), &integ );
       sprintf( npp_buffer, "#define\tncsd_%s\tNcs_dy[CN][%d]\n", 
                $1, npp_num_of_integ );
       insert( npp_buffer, endlist(&integ), &integ );
       
       /* definition for integration after function */
       sprintf( npp_buffer, "#undef\t%s\n", $1 );
       insert( npp_buffer, endlist(&undef), &undef );
       sprintf( npp_buffer, "#undef\tncsd_%s\n", $1 );
       insert( npp_buffer, endlist(&undef), &undef );
       
       /* definition of variable for integration in function */
       sprintf( npp_buffer,"\tstatic double ncs_%s[NCS_OUT%02d_NC];\n",
                $1, npp_module_id );
       insert( npp_buffer, endlist(&declare), &declare );
       /* function for integration */
       npp_integ_flag = 0;
       npp_num_of_integ++;
       st = strpbrk( $3, "," )+1;
       ed = strrchr( $3, ')' );
       ttt = 0;
       while( st != ed ){
         tmpst[ttt] = *st;
         ttt++;
         st++;
       }
       tmpst[ttt] = '\0';
       sprintf( npp_buffer, "ncsd_%s = %s", $1, tmpst );
       $$ = strdup( npp_buffer );

     }else{        /* exp1 != "integral(..." */

       if( (locate($1,&fact) != 0) && (locate($1,&output)==0) ){

         /* check defined of parameter statement */
         for( i = 1; i < endlist(&define); i++ ){
           sprintf( npp_buffer, "%s", strchr(see(i,&define),'\t')+1 );
           sprintf( npp_buffer, "%s", strtok(npp_buffer,"\t") );
           if( strcmp(npp_buffer,$1) == 0 ){
             flag2 = 1;
             break;
           }
         }
         
         /* check variable defined output statement */
         if( flag2 == 0 ){
           sprintf( npp_buffer, "#define\t%s\tncs_%s[CN]\n", $1, $1 );
           insert( npp_buffer, endlist(&define), &define );
           sprintf( npp_buffer, "#undef\t%s\n", $1 );
           insert( npp_buffer, endlist(&undef), &undef );
           sprintf( npp_buffer,
                    "\tstatic double ncs_%s[NCS_OUT%02d_NC];\n",
                    $1, npp_module_id );
           insert( npp_buffer, endlist(&declare), &declare );
         }
       }
       
       sprintf( npp_buffer, "\tdouble %s;\n", $1 );
       $$ = strdup( npp_buffer );

       /* check if statement */
       for( i = 1; i < endlist(&declare); i++ ){
         if( strcmp(see(i,&declare),$$) == 0 ){
           flag = 1;
           break;
         }
       }
       /* check defined of parameter statement */
       for( i = 1; i < endlist(&define); i++ ){
         sprintf( npp_buffer, "%s", strchr(see(i,&define),'\t')+1 );
         sprintf( npp_buffer, "%s", strtok(npp_buffer,"\t") );
         if( strcmp(npp_buffer,$1) == 0 ){
           flag = 1;
           break;
         }
       }
       /* check defined of constant statement */
       for(i = 1; i < endlist(&constant2); i++){
         if( strcmp(see(i,&constant2),$1) == 0){
           flag = 1;
           break;
         }
       }
       
       if( flag == 0 ){
         insert( $$, endlist(&declare), &declare );
       }
       flag = 0;
       sprintf( npp_buffer, "%s = %s", $1, $3 );
       $$ = strdup( npp_buffer );
     }
     mark_observable( $1 );
   }
             | IDENT PLUS PLUS
               {
                 sprintf( npp_buffer, "%s++", $1 );
                 $$ = strdup( npp_buffer );
               }
             ;

/** for equation **/
exp1 : id_add_exp1
       { 
         sprintf( npp_buffer, "%s",$1 );
         $$=strdup( npp_buffer );
       }
     ;

id_add_exp1 : id_multi_exp1
              { 
                sprintf( npp_buffer, "%s",$1 );
                $$=strdup( npp_buffer );
              }
            | id_add_exp1 PLUS id_multi_exp1
              {
                sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                $$=strdup( npp_buffer );
              }
            | id_add_exp1 MINUS id_multi_exp1
              {
                sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                $$=strdup( npp_buffer );
              }
            ;

id_multi_exp1 : id_unary_exp1
                {
                  sprintf( npp_buffer, "%s",$1 );
                  $$=strdup( npp_buffer );
                }
              | id_multi_exp1 MULT id_unary_exp1
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$=strdup( npp_buffer );
                }
              | id_multi_exp1 DIV id_unary_exp1
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$=strdup( npp_buffer );
                }
              | id_multi_exp1 MOD id_unary_exp1
                {
                  sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                  $$=strdup( npp_buffer );
                }
              ;

id_unary_exp1 : id_prim_exp1
                { 
                  sprintf( npp_buffer, "%s",$1 );
                  $$=strdup( npp_buffer ); 
                }
              | PLUS id_multi_exp1 /* %prec UMINUS */
                { 
                  sprintf( npp_buffer, "%s%s", $1, $2 );
                  $$=strdup( npp_buffer );
                }
              | MINUS id_multi_exp1 /* %prec UMINUS */
                {
                  sprintf( npp_buffer, "%s%s", $1, $2 );
                  $$=strdup( npp_buffer );
                 }
              ;

id_prim_exp1 : fnumber        { ; }
             | IDENT
               { 
                 /* list of factor */
                 if( locate($1,&fact) == 0 ){
                   insert( $1, 1, &fact );
                 }
               }
             | IDENT LP exp1 factor2 RP
               { 
                 /* working space for internal function */
                 char buf4func[255];
                 /* for initvalue() */
                 if( strncmp($1,"integral",8) == 0 ){
                   npp_integ_flag =1;
                   if( isalpha((int)$3[0]) != 0 ){
                     sprintf( npp_buffer, "\t\tNcs_y[i][%d] = m%02d%s;\n",
                              npp_num_of_integ, npp_module_id, $3 );
                   }else{
                     sprintf( npp_buffer, "\t\tNcs_y[i][%d] = %s;\n",
                              npp_num_of_integ, $3 );
                   }
                   insert(npp_buffer,endlist(&initvalue),&initvalue);
                 }
                 /*  for check Library Function */
                 sprintf( npp_buffer, "%s%s", $3, $4 );
                 if( check_lib_func_list( $1, npp_buffer ) == NCS_FALSE ){
                   sprintf( npp_buffer, "%s%s%s%s%s", $1, $2, $3, $4, $5 );
                 }else{
                   strcpy( buf4func, npp_buffer );
                   sprintf( npp_buffer, "ncsl_%s%s%s%s", $1, $2, 
                            buf4func, $5 );
                 }
                 $$ = strdup( npp_buffer );
               }
             | LP exp1 RP
               {
                 sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
                 $$ = strdup( npp_buffer );
               }
             ;

fnumber   : NUMBER 
            {
              sprintf( npp_buffer, "%e", atof($1) );
              $$ = strdup( npp_buffer );
            }
          ;

factor2   : /* empty */
            { $$ = strdup(""); }
          | factor2 COMMA exp1              
            {
              sprintf( npp_buffer, "%s%s%s", $1, $2, $3 );
              $$ = strdup( npp_buffer );
            }
          ;

/** "if" statement **/
if_state  : IF LP id_exp RP BRCL characters BRCR if_state2
            {
              sprintf( npp_buffer, "\tif (%s) {\n\t%s\t}%s\n", $3, $6, $8 );
              $$ = strdup( npp_buffer );
            }
          ;

if_state2 : /* empty */    
            { $$ = strdup(""); }
          | ELSE BRCL characters BRCR
            {
              sprintf( npp_buffer, "else{\n\t%s\t}", $3 );
              $$ = strdup( npp_buffer );
            }
          ;
%%

/*************************************************************************
  Functions to set information accepted by yacc 
*************************************************************************/

/* for start and end of line in function */
static void routine_work(char c){
  switch(c){
  case 'b':
    /* beginning of module */
    insert( "\n", endlist(&integ), &integ );
    insert( "\n", endlist(&declare), &declare );
    break;
  case 'e':
    /* end of module */
    clearlist( &module );
    clearlist( &fact );
    clearlist( &output );
    insert( "\n", 1, &module );
    insert( "\n", 1, &delay );
    break;
  }
}

/*
 * Function to check definition cell module in network description.
 *   Return  NCS_TRUE  : network description is correct
 *           NCS_FALSE : a used module in right side is not defined 
 *                      in left side.
 */
static int
check_module_description()
{
  char *brkpt;
  int   i;
  List  mtmp;
  int   eflag = 0;
  char  enpp_buffer[100];
  init_list( &mtmp );

  /* get module name in right side of equation */
  for( i = 1; i < endlist(&net_module); i+=2 ){
    strcpy( npp_buffer, see( i, &net_module ) );
    brkpt = strchr( npp_buffer, (int)(',') );
    *brkpt = '\0';
    insert( npp_buffer, endlist(&mtmp), &mtmp );
  }
  while( unique( &mtmp ) == NCS_TRUE );
  
  /* check module name in left side of equation */
  for( i = 1; i < endlist(&net_inputs); i++ ){
    strcpy( npp_buffer, see( i, &net_inputs ) );
    brkpt = strchr( npp_buffer, (int)(',') );
    *brkpt = '\0';
    if( locate( npp_buffer, &mtmp ) == 0 ){
      sprintf( enpp_buffer, "cell module \"%s\" is not define", npp_buffer );
      yyerror( enpp_buffer );
      eflag++;
    }
  }
  
  clearlist( &mtmp );
  
  if( eflag != 0 ){
    return( NCS_FALSE );
  }else{
    return( NCS_TRUE );
  }
}

/*
 * Function to rebuild list of module name.
 *  Cut each single module name in stirng
 */
static void
list_up_modules( List *mod )
{
  char   *brkpt1, *brkpt2, *brkpt3;
  char   module_info[100];
  char   module_info2[100];
  
  if( endlist(mod) > 1 ){
    strcpy( npp_buffer, see( 1, mod ) );
    SL_delete( 1, mod );
    brkpt1 = npp_buffer;
    while( (brkpt2 = strchr( brkpt1, (int)(':')))!=NULL ){
      strncpy( module_info, brkpt1, (int)(brkpt2-brkpt1) );
      brkpt3 = strchr( module_info, (int)(','));
      strncpy( module_info2, module_info, (int)(brkpt3-module_info) );
      module_info2[(int)(brkpt3-module_info)] = '\0';
      insert( module_info2, endlist(mod), mod );
      brkpt3++;
      insert( brkpt3, endlist(mod), mod );
      brkpt1 = brkpt2+1;
    }
    brkpt3 = strchr( brkpt1, (int)(','));
    strncpy( module_info2, brkpt1, (int)(brkpt3-brkpt1) );
    module_info2[(int)(brkpt3-brkpt1)] = '\0';
    insert( module_info2, endlist(mod), mod );
    brkpt3++;
    insert( brkpt3, endlist(mod), mod );
  }
}

/* 
 * When "IDENT" check, if a variable defined already in parameter, 
 * the variable accept.
 *          add by T.N 95/6/15
 */
static char *
getparameter( const char *ident )
{
  int  id;    /* module ID */
  int  position;  /* position of "IDENT" in list of parameter */
  char tmp[256];

  /* get module ID include the parameter */
  id = atoi( see(endlist(&module_table)-1,&module_table) );
  sprintf( tmp, "m%02d%s", id, ident );

  /* get position of the parameter in list */
  position = locate( tmp, &parameter )+1;

  /* if the parameter is undefined parameter, then ERROR display */
  if( position == 1 ){
    exit(102);
  }
  return( see(position,&parameter) );
}


/* 
 *  Procedure to set information of exinput
 *  const char *var : name of variable.
 */
static void 
set_exinput( const char *var )
{
  sprintf( npp_buffer, "#define\t%s\tncsm_xin%02d[CN]\n", var, 
           npp_num_of_xin );
  insert( npp_buffer, endlist(&define), &define );
  sprintf( npp_buffer, "#undef\t%s\n", var );
  insert( npp_buffer, endlist(&undef), &undef );
  
  /* for header file */
  sprintf( npp_buffer, "%d", npp_module_id );
  insert( npp_buffer, endlist(&ncs_xin), &ncs_xin );
  
  sprintf( npp_buffer, "%s", var );
  insert( npp_buffer, endlist(&exinput), &exinput );
  sprintf( npp_buffer, "%d", npp_module_id );
  insert( npp_buffer, endlist(&exinput), &exinput );
  insert( see(endlist(&module_table)-2, &module_table), 
          endlist(&exinput), &exinput );
}

/* 
 *  Procedure to set information of input
 *  char *var : name of variable.
 */
static void 
set_input( const char *var )
{
  sprintf( npp_buffer, "%s", var );
  insert( npp_buffer, endlist(&input), &input );
  sprintf( npp_buffer, "%d", npp_module_id );
  insert( npp_buffer, endlist(&input), &input );
  insert( see(endlist(&module_table)-2,&module_table), 
          endlist(&input), &input );
}

/* 
 *  Procedure to set information of delay
 *  char *var : name of variable.
 *  char *time: delay time.
 *  char *init: initial value.
 */
static void 
set_delay( char *var, char *time, char *init )
{
  sprintf( npp_buffer, "\tstatic long *ncspt%02d;\n", npp_num_of_delay );
  insert( npp_buffer, endlist(&declare), &declare );
  sprintf( npp_buffer, "\tswitch(ncsg_delay_flag){\n" );
  insert( npp_buffer, endlist(&delay), &delay );
  sprintf( npp_buffer, 
           "\tcase CALCULATE:\n\t\t%s = delay(ncs_dly[%d].q[CN],&ncspt%02d[CN],ncsg_qsize[%d],%s);\n\t\tbreak;\n\n",
           var, npp_num_of_delay, npp_num_of_delay, npp_num_of_delay, var );
  insert( npp_buffer, endlist(&delay), &delay );
  sprintf( npp_buffer, 
           "\tcase DELAY_INIT_SET:\n\t\tncspt%02d = (long *)calloc(ncsg_cell,sizeof(long));\n\t\tif(ncsg_dlyinit_flg[%d] == DELAY_INIT_AUTO_SET)\n\t\t\tinit_queue(ncs_dly[%d].q[CN] , ncsg_qsize[%d],%s);\n\t\tbreak;\n\n",
           npp_num_of_delay, npp_num_of_delay, npp_num_of_delay,
           npp_num_of_delay, var );
  insert( npp_buffer, endlist(&delay), &delay );
  sprintf( npp_buffer, "\tcase OUTPUT_INIT:\n\t\tbreak;\t\n\t}\n\n" );
  insert( npp_buffer, endlist(&delay), &delay );
                            
  /* delay information is stored for *.DLY */
  insert( see(endlist(&module_table)-2,&module_table),
          endlist(&delay_file), &delay_file );
  insert( var, endlist(&delay_file), &delay_file );
  if( atoi( time ) < 0 ){
    printf( "Wanning: Improper delay time, delay time is set 0\n" );
    insert( "0", endlist(&delay_file), &delay_file );
  }else{
    insert( time, endlist(&delay_file), &delay_file );
  }
  insert( init, endlist(&delay_file), &delay_file );

  /* insert define and undef in function */
  sprintf( npp_buffer, "#define\t%s\tINPUT\n", var );
  insert( npp_buffer, endlist(&define), &define );
  sprintf( npp_buffer, "#undef\t%s\n", var );
  insert( npp_buffer, endlist(&undef), &undef );
  npp_num_of_delay++;
}

/* 
 *  Function to set information of observable statement
 *  char *obs: name of variable.
 */
static void
set_observable( char *obs )
{
  /* for file of observable infomation    */
  /* 1 set = [variable]->[module ID]->[module]->[flag]->[line] */
  insert( obs, endlist( &observ_table ), &observ_table );
  sprintf( npp_buffer, "%d", npp_module_id );
  insert( npp_buffer, endlist( &observ_table ), &observ_table );
  insert( see(endlist(&module_table)-2,&module_table), 
    endlist( &observ_table ), &observ_table );
  sprintf( npp_buffer, "%c%c", NCS_FLAG_OFF, '\0' );
  insert( npp_buffer, endlist( &observ_table ), &observ_table );
  sprintf( npp_buffer, "%d", npp_lineno );
  insert( npp_buffer, endlist( &observ_table ), &observ_table );
}

/* 
 *  Procedure to mark the table of observable variable
 *  char *ident: variable used in funtion.
 */
static void
mark_observable( char *ident )
{
    int   i, id;
  
    for( i = 1; i < endlist(&observ_table); i += 5 ){
    id = atoi( see(i+1,&observ_table) );
    if( id == npp_module_id ){
      if( strcmp( ident, see(i,&observ_table) ) == 0 ){
        sprintf( npp_buffer, "%c%c", NCS_FLAG_ON, '\0' );
        writelist( npp_buffer, i+3, &observ_table );
      }
    }
  }
}


/* 
 *  Procedure to insert observable information to lists
 *  List *table: table of observable variables.
 */
static void
insert_observable( List *table )
{
    int  i, j;
  char  flag;
  int     id;


  for( i = 1; i < endlist( table ); i += 5 ){

    id = atoi( see(i+1,table) );
    if( id == npp_module_id ){

      /* mark variables described exinput */
      if( npp_num_of_xin != 0 ){
        for( j = 1; j < endlist(&exinput); j += 3 ){
    id = atoi( see(j+1,&exinput) );
    if( id == npp_module_id ){
      if( strcmp( see(j,&exinput), see(i,&observ_table) ) == 0 ){
        sprintf( npp_buffer, "%c%c", NCS_FLAG_ON, '\0' );
        writelist( npp_buffer, i+3, &observ_table );
      }
    }
        }
      }

      /* mark variables described input */
      if( npp_num_of_input != 0 ){
        for( j = 1; j < endlist(&input); j += 3 ){
    id = atoi( see(j+1,&input) );
    if( id == npp_module_id ){
      if( strcmp( see(j,&input), see(i,&observ_table) ) == 0 ){
        sprintf( npp_buffer, "%c%c", NCS_FLAG_ON, '\0' );
        writelist( npp_buffer, i+3, &observ_table );
      }
    }
        }
      }

      /* mark variables described parameter */
      if( endlist(&param_table) > 1 ){
        for( j = 1; j < endlist(&param_table); j += 4 ){
    id = atoi( see(j+2,&param_table) );
    if( id == npp_module_id ){
      if( strcmp(see(j,&param_table),see(i,&observ_table)) == 0 ){
        sprintf( npp_buffer, "%c%c", NCS_FLAG_ON, '\0' );
        writelist( npp_buffer, i+3, &observ_table );
      }
    }
        }
      }

      /* insert script for observable information */
      flag = (char)*see( i+3, table );
      if( flag == NCS_FLAG_ON ){
        sprintf( npp_buffer, "\tncsm_obs%02d[CN] = %s;\n", 
           npp_num_of_obs, see( i, table ) );
        insert( npp_buffer , endlist(&observable), &observable );

        /* for header file */
        sprintf( npp_buffer, "%d", npp_num_of_obs );
        insert( npp_buffer, endlist(&ncs_obs), &ncs_obs );
        sprintf( npp_buffer, "%s", see( i+2, table ) );
        insert( npp_buffer, endlist(&ncs_obs), &ncs_obs );
        sprintf( npp_buffer, "o%02d%s", npp_module_id, see( i, table ) );
        insert( npp_buffer, endlist(&ncs_obs), &ncs_obs );
        npp_num_of_obs++;
      }else{
        fprintf(stderr,
         "Warnning: observable \"%s\" not found in function near line %s\n",
          see(i,table), see(i+4,table));
      }
    }
  }
}

/* 
 *  Function to set information of parameter statement
 *  char *param: name of parameter.
 *  char *val  : value of parameter.
 *  int  flag  : if there is parameter statement in NETWORK description,
 *                   then flag = 1.
 *                   if there is parameter statement in module description,
 *                   then flag = 0.
 */
static void
set_parameter( char *param, char *val, int flag )
{
    if( flag == 1 ){  /* NETWORK description */

    /* begin of line and end of line in function */
    sprintf( npp_buffer, "#define\t%s\tn00%s\n", param, param );
    insert( npp_buffer, endlist(&net_def), &net_def );
    sprintf( npp_buffer, "#undef\t%s\n", param );
    insert( npp_buffer, endlist(&net_undef), &net_undef );
    /* for header file */
    sprintf( npp_buffer, "int n00%s;\n", param );
    insert( npp_buffer, endlist(&ncs_double), &ncs_double );
    /* for ncsf_prmset() */
    sprintf( npp_buffer, "n00%s", param );
    insert( npp_buffer, endlist(&parameter), &parameter );
    sprintf( npp_buffer, "%s", val );
    insert( npp_buffer, endlist(&parameter), &parameter );
    insert( net_name, endlist(&parameter), &parameter );

  } else {    /* MODULE description */

    /* begin of line and end of line in function */
    sprintf( npp_buffer, "#define\t%s\tm%02d%s\n", param, 
             npp_module_id, param );
    insert( npp_buffer, endlist(&define), &define );
    sprintf( npp_buffer, "#undef\t%s\n", param );
    insert( npp_buffer , endlist(&undef), &undef );
    /* for header file */
    sprintf( npp_buffer, "double m%02d%s;\n", npp_module_id, param );
    insert( npp_buffer, endlist(&ncs_double), &ncs_double );
    /* for ncsf_prmset() */
    sprintf( npp_buffer, "m%02d%s", npp_module_id, param );
    insert( npp_buffer, endlist(&parameter), &parameter );
    sprintf( npp_buffer, "%s", val );
    insert( npp_buffer, endlist(&parameter), &parameter );
    insert( see(endlist(&module_table)-2, &module_table ),
      endlist(&parameter), &parameter);
    /* parameter information */
    sprintf( npp_buffer, "%s", param );
    insert( npp_buffer, endlist(&param_table), &param_table );
    sprintf( npp_buffer, "%s", val );
    insert( npp_buffer, endlist(&param_table), &param_table );
    sprintf( npp_buffer, "%d", npp_module_id );
    insert( npp_buffer, endlist(&param_table), &param_table );
    insert( see(endlist(&module_table)-2,&module_table), 
            endlist(&param_table), &param_table );
  }
  insert( param, endlist(&getprmptr), &getprmptr );
}


/*
 * Function to display error message
 */
int
yyerror( char *s )
{
  int  eline;

  if( error_num++ == 0 ){
    printf( "In model file \"%s\"\n", model_file_name );
  }
  eline = npp_lineno;

  printf( "Error: %s near line %d\n", s, eline+npp_mline );
  return( 0 );
}

char *get_error_code()
{
  static char *s = NULL;
  return( s );
}

/**********************************************************************
      Main Routine
 **********************************************************************/
int main(int argc, char *argv[])
{
  char   *argument;              /* argument 1 of npp command */
  char   *header;                /* argument 2 of npp command */
  char    scf_model[NCS_FNAME_WD];
  char    scf_header[NCS_FNAME_WD];
  char    scf_lib[NCS_FNAME_WD];
  char    tmp_fname[NCS_FNAME_WD];

  /* get name of model source file  */
  if(argc >= 2)
    argument = argv[1]; /* GetString(0); */
  else
    argument = NULL;

  /* get name of user's header file */
  if(argc >= 3)
    header = argv[2];   /* GetString(1); */
  else
    header = NULL;

  work_dir = getenv(NCS_TEMPDIR_ENV);

  /* initialize */
  strcpy(scf_model,"");
  strcpy(scf_header,"");
  strcpy(scf_lib,"");

  /* say wellcome */
  printf("NCS Preprocessor, ");

  if(work_dir == NULL || *work_dir == '\0'){
    printf("Error : Failed to set temporary directory.\n");
    printf("        Please set the `%s' environment variable.\n", 
           NCS_TEMPDIR_ENV);
    return 15;
  }

  printf("compiling now.\n");

  /* set path name of working directory */
  if( SetWorkDirSCF( work_dir ) == NCS_FALSE ){
    return (15);
  }
  /* Did a model file name assigned or not? */
  if( GetSCFN( scf_model, scf_header, scf_lib ) == NULL ){
    if( ( argument == NULL ) &&
        ( strlen(argument) == 0 ) &&
        ( scf_model[0] == NCS_EOS ) ) {
      return (1);
    }
  }

  /* Did a header file name assigned or not? */
  if( (header != NULL) && (strlen(header)!=0) ){
    strcpy( scf_header, header );
  } else {
    if( (scf_header == NULL) || (strlen(scf_header)==0 )){
      scf_header[0] = '\0';
    } 
  }

  /* When first argument was assigned */
  if( ( argument != NULL ) && (strlen(argument) != 0) ){
    if( SetSCFN( argument, scf_header, scf_lib ) == NCS_FALSE ){
      return (1);
    } else {
      sprintf( tmp_fname, "%s.%s", argument, SCF_MODEL_EXTN );
      model_file_name = strdup( tmp_fname );
    }
  } else {
    sprintf( tmp_fname, "%s.%s", scf_model, SCF_MODEL_EXTN );
    model_file_name = strdup( tmp_fname );
  }
    
  if( ModelNameSCFN( scf_model ) == NULL ){
    return (4);
  } else {
    sprintf( tmp_fname, "%s/%s", work_dir, scf_model );
    mfile_path = strdup( tmp_fname );
  }

  /* model file -> filter -> final model file */
  if(npp_sh_filter(model_file_name, mfile_path) != 0)
    return(3);

  /* open model file */
  if( (yyin = fopen( mfile_path,"r" )) == NULL ){
    fprintf( stderr, "Model file name = %s\n", mfile_path );
    return (3);
  }

        /* initialize list structure */
  init();
  init_list(&net_table);
  init_list(&module_table);
  init_list(&initvalue);
  init_list(&getprmptr);

  /* write header to model source file */
  make_header( scf_model, scf_header );

  yyparse();    /* LR(1) parsing routine */

  fclose( yyin ); /* close model file */

  /* good-by  */
  if( error_num == 0 ){
    /* write to the model source file */
    make_headerfile( scf_model );  /* header file      */
    make_network();                /* NETWORK()        */
    make_initvalue();              /* initvalue()      */
    make_ncsf_constset();          /* ncsf_constset()  */
    make_ncsf_prmset();            /* ncsf_prmset()   */
    make_getprmptr();              /* GetPrmPtr()     */

    make_ncsf_workset();
    make_ncsf_workcls();

    /* write footer to model source file */
    make_footer();
    fclose( fp );             

    /* write to simulation condition file */
    make_prm_file( &parameter );    /* for parameter      */
    make_xin_file();                /* for external input */
    make_dly_file( &delay_file );   /* for delay          */
    make_out_file();                /* for output         */
    make_obs_file( &observ_table ); /* for output         */
    make_mat_file();                /* for time           */

    printf("done.\n" );
  }else{
    if( error_num == 1 ){
      printf("There is a error detected.\n" );
    }else{
      printf("There are %d errors detected.\n", error_num );
    }
    free( model_file_name );
    free( mfile_path );
    return 7;
  }

  /* free list structure */
  endprocess();

  free( model_file_name );
  free( mfile_path );
  return( 0 );
}


/*
 * Function to initialize all lists.
 */
static void
init()
{
  init_list( &net_def );
  init_list( &net_undef );
  init_list( &network );
  init_list( &net_info );
  init_list( &module );
  insert( "\n", 1, &module );
  init_list( &f_header );
  init_list( &name );
  init_list( &exinput );
  init_list( &input );
  init_list( &output );
  init_list( &observable );
  init_list( &observ_table );
  init_list( &integ );
  insert( "\n", 1, &integ );
  init_list( &constant );
  init_list( &constant2 );
  init_list( &parameter );
  init_list( &param_table );
  init_list( &initial );
  init_list( &define );
  init_list( &declare );
  init_list( &delay );
  insert( "\n", 1, &delay );
  init_list( &function );
  init_list( &undef );
  init_list( &f_end );
  init_list( &ncs_double );
  init_list( &ncs_obs );
  init_list( &ncs_xin );
  init_list( &delay_file );
  init_list( &fact );

  init_list( &net_module );
  init_list( &net_inputs );
  init_list( &net_loads );

  init_list( &cell_modules );
  init_list( &synapse_modules );
  init_list( &gap_modules );

  init_list( &npp_clang );
  csflag = 0;
}

/*
 * Function to write description of include file to model source file
 */
static void 
make_header( char input_file_name[], char usr_header_file[] )
{
  char out_file_name[256];

  /* open the model source file */
  sprintf( out_file_name, "%s/%s.%s", 
     work_dir, input_file_name, NCS_SPF_EXTN );
  if( (fp = fopen(out_file_name,"w")) == NULL){
    printf( "Can't open '%s'\n", out_file_name );
    exit(103);
  }
  
  fprintf(fp,"#include <stdio.h>\n");
  fprintf(fp,"#include <string.h>\n");
  fprintf(fp,"#include <math.h>\n");
  fprintf(fp,"#include <stdlib.h>\n");

  fprintf(fp,"#define NCS_MODEL_SOURCE\n");
  fprintf(fp,"#include \"libncsc.h\"\n");
  fprintf(fp,"#include \"libncss.h\"\n");

  fprintf(fp,"#include \"%s.h\"\n" , input_file_name );

  if( usr_header_file[0] != '\0' ){
    fprintf(fp,"#include \"%s\"\n",usr_header_file);
  }else{
    fprintf(fp,"\n");
  }
  fprintf(fp,"#ifdef __cplusplus\n");
  fprintf(fp,"extern \"C\" {\n");
  fprintf(fp,"#endif\n");
  fprintf(fp,"\n");
}

/*
 * Function to write directory described C-language sources to model source file
 */
static void
make_csources()
{
  int i;

  fprintf( fp, "/*** this lines described in model file directory ***/" );
  for( i = 1; i < endlist(&npp_clang); i++){
    fprintf( fp, "%s", see(i,&npp_clang) );
  }
}

/*
 * Function to write footer of model source file
 */
static void
make_footer()
{
  fprintf(fp,"#ifdef __cplusplus\n");
  fprintf(fp,"}\n");
  fprintf(fp,"#endif\n");
}

/*
 * Function to write initvalue() to model source file
 */
static void
make_initvalue()
{
  int   i;

  insert( "\n", 1, &initvalue );
  /* set integration constant */
  insert( "DLLEXPORT void initvalue( double **Ncs_y )\n{\n\tint i,k;\n\n\tfor(i=0 ; i < ncsg_cell ; i++){\n", 1, &initvalue );
  insert( "\t}\n\n", endlist(&initvalue), &initvalue );

  /* for delay information */
  if( npp_num_of_delay != 0 ){
    sprintf( npp_buffer, "\tfor(k=0 ; k < NCS_DLY_NT ; k++){\n" );
    insert( npp_buffer, endlist(&initvalue), &initvalue );
    sprintf( npp_buffer,"\t\tncs_dly[k].q = (double **)alloc2d(ncsg_cell,ncsg_qsize[k],sizeof(double));\n" );
    insert( npp_buffer, endlist(&initvalue), &initvalue );
    sprintf( npp_buffer, "\t\tfor(i=0 ; i < ncsg_cell; i++){\n\t\t\tinit_queue(ncs_dly[k].q[i],ncsg_qsize[k],ncsg_dlyinit[k]);\n\t\t}\n\t}\n\n");
    insert( npp_buffer, endlist(&initvalue), &initvalue );
  }
  /* end of lines of initvalue() */
  insert("}\n\n",endlist(&initvalue),&initvalue);

  /* write initvalue() to model source file */
  for( i = 1; i < endlist(&initvalue); i++){
    fprintf( fp, "%s", see(i,&initvalue) );
  }
}

/*
 * Function to write header file for model source file
 */
static void
make_headerfile( char *input_file_name )
{
  int   i, j;
  int   cell_module_max;   /* maximum number of cell module */
  int   xin_max;           /* NCS_XIN_MAX */
  FILE *ptr;
  char tmp[NPP_VAL_LEN+3];
  char out_file_name[256];    /* file name of header file */

  cell_module_max = 0;
  xin_max = 0;

  /* rewrite list                                             */
  /* (module name->module ID->module type->component number ) */
  for( i = 1; i <= endlist(&net_table)-1; i = i + 3 ){
    if( locate(see(i,&net_table),&module_table) != 0 ){
      insert( see(i+1,&net_table),
        locate(see(i,&net_table),&module_table)+3,&module_table );
    }
  }

  /* calculate maximum number of cell module */
  for( i = 1; i < endlist(&module_table) ; i = i + 4 ){
    if( (cell_module_max < atoi(see(i+3,&module_table))) 
        && (atoi(see(i+2,&module_table)) == 1) ){
      cell_module_max = atoi( see(i+3,&module_table) );
    }
  }

  /* write begin of lines of header file */
  init_list( &header );

  insert( "#define\tNCELL\t", endlist(&header), &header );
  sprintf( npp_buffer, "%d\n", cell_module_max );
  insert( npp_buffer, endlist(&header), &header );

  insert( "#define\tNEQN\t", endlist(&header), &header );
  /* if there is no integration in model file, then integration is 1 */
  if( npp_num_of_integ == 0 ){
    sprintf( npp_buffer, "%d\n", 1 );
  } else {
    sprintf( npp_buffer, "%d\n", npp_num_of_integ );
  }
  insert( npp_buffer, endlist(&header), &header );

  insert( "#define\tNCS_OUT_NT\t", endlist(&header), &header );
  sprintf( npp_buffer, "%d\n", npp_module_count );
  insert( npp_buffer, endlist(&header), &header );

  /* if there is no exinput in model file, then exinput is 1 */
  insert( "#define\tNCS_XIN_NT\t", endlist(&header), &header );
  if( npp_num_of_xin == 0 ){
    sprintf( npp_buffer, "%d\n", 1 );
  } else {
    sprintf(npp_buffer,"%d\n", npp_num_of_xin);
  }
  insert( npp_buffer, endlist(&header), &header );

  /* if there is no observable in model file, then observable is 1 */
  insert( "#define\tNCS_OBS_NT\t", endlist(&header), &header );
  if( npp_num_of_obs == 0 ){
    sprintf( npp_buffer, "%d\n", 1 );
  } else {
    sprintf( npp_buffer, "%d\n", npp_num_of_obs );
  }
  insert( npp_buffer, endlist(&header), &header );

  /* if there is no input in model file, then input is 1 */
  insert( "#define\tNCS_NNET\t", endlist(&header), &header );
  if( npp_max_input == 0 ){
    sprintf( npp_buffer, "%d\n", 1 );
  } else {
    sprintf( npp_buffer, "%d\n", npp_max_input );
  }
  insert( npp_buffer, endlist(&header), &header );

  /* this value is 2 that is true? */
  insert("#define\tNCS_XFM_FADDR\t",endlist(&header),&header);
  insert("2\n",endlist(&header),&header);

  /* write NCS_XIN_MAX */
  for( i = 1; i < endlist(&ncs_xin); i++ ){
    sprintf( tmp, "%s", see(i,&ncs_xin) );
    for( j = 2; j < endlist(&module_table) ; j += 4 ){
      if( strcmp(tmp,see(j,&module_table)) == 0 ){
        xin_max += atoi( see(j+2,&module_table) );
        break;
      }
    }
  }
  /* calculation of NCS_XIN_MAX */
  if( xin_max == 0 ){
    xin_max = 1;
  }
  insert( "#define\tNCS_XIN_MAX\t", endlist(&header), &header );
  sprintf( npp_buffer, "%d\n", xin_max );
  insert( npp_buffer, endlist(&header), &header );

  /* for output include routine to match module */
  for( i = 0; i< npp_module_count; i++ ){
    for( j = 2; j < endlist(&module_table) ; j += 4 ){
      sprintf( tmp, "%d", i );
      if( strcmp(tmp,see(j,&module_table)) == 0 ){
        /* name of module */
        sprintf( tmp, "%s", see(j-1,&module_table) ); 
        break;
      }
    }
    
          /*                      module ID v       v comp number */
    sprintf(npp_buffer, "#define\tNCS_OUT%02d_NC\t%s\n", 
      i, see(locate(tmp,&module_table)+3,&module_table) );
    insert( npp_buffer, endlist(&header), &header );
    sprintf( npp_buffer, "double ncsm_outm%02d[%s];\n",
       i, see(locate(tmp,&module_table)+3,&module_table ));
    insert( npp_buffer, endlist(&header), &header );
    sprintf( npp_buffer, "double ncsm_outs%02d[%s];\n",
       i, see(locate(tmp,&module_table)+3,&module_table) );
    insert( npp_buffer, endlist(&header), &header );
  }

  /* for exinput */
  for( i = 0; i < npp_num_of_xin; i++ ){
    /*             v module ID */
    sprintf( tmp,"%s",see(i+1,&ncs_xin) );
    for( j = 2; j < endlist(&module_table); j += 4 ){
      if( strcmp(tmp,see(j,&module_table)) == 0 ){
        strcpy( tmp,see(j+2,&module_table));
        break;
      }
    }
    sprintf( npp_buffer, "#define\tNCS_XIN%02d_NC\t%s\n", i, tmp );
    insert( npp_buffer, endlist(&header), &header );
    sprintf( npp_buffer, "double ncsm_xin%02d[%s];\n", i, tmp );
    insert( npp_buffer, endlist(&header), &header );
  }

  /* for observable */
  for( i = 0; i < npp_num_of_obs; i++ ){
    strcpy( tmp, see(3*i+2,&ncs_obs) );
    sprintf( npp_buffer, "#define\tNCS_OBS%02d_NC\t%s\n",
       i, see(locate(tmp,&module_table)+3, &module_table) );
    insert( npp_buffer, endlist(&header), &header );
    sprintf( npp_buffer, "double ncsm_obs%02d[%s];\n",
       i, see(locate(tmp,&module_table)+3, &module_table) );
    insert( npp_buffer, endlist(&header), &header );
  }

  /* declaration of parameter */
  listcat( &header, &ncs_double );

  /* for delay */
  insert( "#define\tNCS_DLY_NT\t", endlist(&header), &header );
  if( npp_num_of_delay == 0 ){
    insert( "1\n", endlist(&header), &header );
  } else {
    sprintf( npp_buffer, "%d\n", npp_num_of_delay );
    insert( npp_buffer, endlist(&header), &header );
    insert( "int queue_size[NCS_DLY_NT];\n", endlist(&header), &header );
    sprintf( npp_buffer,
       "struct\tncs_queue{\n\tdouble **q;\n}\tncs_dly[NCS_DLY_NT];\n");
    insert( npp_buffer, endlist(&header), &header );
  }

  /* write to file */
  sprintf( npp_buffer, "%s/%s.h", work_dir, input_file_name );
  strcpy( out_file_name, npp_buffer );
  if( (ptr=fopen(out_file_name,"w")) == NULL ){
    exit(104);
  }
  for( i = 1; i < endlist(&header) ; i++ ){
    fprintf( ptr, "%s", see(i,&header) );
  }
  fclose( ptr );
  clearlist( &header );
}
  
/*
 * Function to write ncsf_constset() to the model source file
 */
static void
make_ncsf_constset()
{
  int   i;
  char   tmp[NPP_VAL_LEN+3];
  List   f_const;     /* to store final script of ncsf_constset()  */
  List   local_table; /* table translated module name -> module ID */

  /* making of local tabel */
  init_list( &f_const );
  init_list( &local_table );
  for( i = 1; i < endlist(&module_table) ; i = i + 4 ){
    insert( see(i,&module_table), endlist(&local_table), &local_table );
    insert(see(i+1,&module_table), endlist(&local_table), &local_table );
  }

  /* write output information to the function */
  for( i = 0; i < npp_module_count; i++ ){
    sprintf( npp_buffer, "\tncsm_outptr[%d] = ncsm_outm%02d;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsm_subptr[%d] = ncsm_outs%02d;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( tmp, "%d", i );
    sprintf( npp_buffer, "\tstrcpy( ncsg_outtbl[%d] , \"%s\" );\n",
       i, see(locate(tmp,&local_table)-1,&local_table) );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_ncout[%d] = NCS_OUT%02d_NC;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_outtblptr[%d] = ncsg_outtbl[%d];\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    insert( "\n", endlist(&f_const), &f_const );
  }

  /* write exinput informaiton to the function */
  for( i = 0; i < npp_num_of_xin; i++ ){
    sprintf( npp_buffer, "\tncsm_xinptr[%d] = ncsm_xin%02d;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    strcpy( tmp, see(i+1,&ncs_xin) );
    sprintf( npp_buffer, "\tstrcpy(ncsg_xintbl[%d] , \"%s\");\n",
       i, see(locate(tmp,&local_table)-1,&local_table) );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_ncxin[%d] = NCS_XIN%02d_NC;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_xintblptr[%d] = ncsg_xintbl[%d];\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    insert( "\n", endlist(&f_const), &f_const );
  }

  /* write observable information to the function */
  for( i = 0; i < npp_num_of_obs; i++ ){
    sprintf( npp_buffer, "\tncsm_obsptr[%d] = ncsm_obs%02d;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    strcpy( tmp, see(3*i+3,&ncs_obs) );
    sprintf( npp_buffer, "\tstrcpy(ncsg_obstbl[%d] , \"%s\");\n", i, tmp );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_ncobs[%d] = NCS_OBS%02d_NC;\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    sprintf( npp_buffer, "\tncsg_obstblptr[%d] = ncsg_obstbl[%d];\n", i, i );
    insert( npp_buffer, endlist(&f_const), &f_const );
    insert( "\n", endlist(&f_const), &f_const );
  }

  /* write end of lines of the function */
  sprintf( npp_buffer, "\t*(ncsg_outtbl[%d]) = \'\\0\';\n",
                 npp_module_count );
  insert( npp_buffer,endlist(&f_const),&f_const);
  sprintf( npp_buffer, "\t*(ncsg_xintbl[%d]) = \'\\0\';\n", 
                 npp_num_of_xin );
  insert( npp_buffer, endlist(&f_const), &f_const );
  sprintf( npp_buffer, "\t*(ncsg_obstbl[%d]) = \'\\0\';\n",
                 npp_num_of_obs );
  insert( npp_buffer, endlist(&f_const), &f_const );
  insert( "\n", endlist(&f_const), &f_const );

  sprintf( npp_buffer, "\tncsg_ntout = NCS_OUT_NT;\n" );
  insert( npp_buffer, endlist(&f_const), &f_const );
  /* if the model has exinput information, */
  if( npp_num_of_xin != 0 ){
    sprintf( npp_buffer, "\tncsg_ntxin = NCS_XIN_NT;\n" );
    insert( npp_buffer, endlist(&f_const), &f_const );
  }
  /* if the model has observable information */
  if( npp_num_of_obs != 0 ){
    sprintf( npp_buffer, "\tncsg_ntobs = NCS_OBS_NT;\n" );
    insert( npp_buffer, endlist(&f_const), &f_const );
  }
  sprintf( npp_buffer, "\tncsg_cell = NCELL;\n" );
  insert( npp_buffer, endlist(&f_const), &f_const );
  sprintf( npp_buffer, "\tncsg_neqn = NEQN;\n" );
  insert( npp_buffer, endlist(&f_const), &f_const );
  sprintf( npp_buffer, "\tncsg_relerr = 1.0e-04;\n" );
  insert( npp_buffer, endlist(&f_const), &f_const );

  /* write begin and end of line to the function */
  insert( "DLLEXPORT void ncsf_constset()\n{\n", 1, &f_const );
  insert( "}\n\n", endlist(&f_const), &f_const );

  /* write the function from list to the model source file */
  for( i = 1; i < endlist(&f_const); i++ ){
    fprintf( fp, "%s", see(i,&f_const) );
  }
  clearlist( &f_const );
  clearlist( &local_table );
}

/*
 * Function to write ncsf_prmset() to the model source file
 */
void make_ncsf_prmset()
{
  int   i;
  List   prmset;          /* to store final script of ncsf_prmset() */

  init_list( &prmset );

  /* write the first line of the function to list */
  sprintf( npp_buffer, 
     "int ncsf_prmset( struct scf_prm *ncs_prm )\n{\n");
  insert( npp_buffer, 1, &prmset );
  sprintf( npp_buffer, 
     "\tint ncs_dest;\n\tstatic char *ncs_prmtbl[] = {\n" );
  insert( npp_buffer, 2, &prmset );

  /* write parameters */
  for( i = 1; i < endlist(&parameter); i += 3 ){
    sprintf( npp_buffer, "\t\t\"%s\",\n", see(i,&parameter) );
    insert( npp_buffer, endlist(&prmset), &prmset );
  }
  sprintf( npp_buffer, "\t\t NCS_EOS  };\n\n" );
  insert( npp_buffer, endlist(&prmset), &prmset );

  sprintf( npp_buffer, "\tncs_dest = string_ana(ncs_prm->prm_name,ncs_prmtbl,NCS_MODE0);\n\n\tswitch(ncs_dest){\n");
  insert( npp_buffer, endlist(&prmset), &prmset );
  
  /* write case sentence */
  for( i = 1; i < endlist(&parameter); i += 3 ){
    sprintf( npp_buffer,
       "\tcase %d: %s = (double)ncs_prm->prm_val.prm_real;\n\t\tbreak;\n",
       (i+2)/3, see(i,&parameter) );
    insert( npp_buffer, endlist(&prmset), &prmset );
  }
  sprintf( npp_buffer, 
     "\tdefault: return (-1);\tbreak;\n\t}\n\treturn (0);\n}\n" );
  insert( npp_buffer, endlist(&prmset), &prmset );

  /* write the function from list to the model source file */
  for( i = 1; i < endlist(&prmset); i++ ){
    fprintf( fp, "%s", see(i,&prmset) );
  }
}

/*
 * Function to write NETWORK() to the model source file
 */
static void
make_network()
{
  int   i, j;
  List   line;
  char   local_npp_buffer[4096];
  char   poscn[127];
  int   iflag;

  init_list(&line);

  /* pre-process(modified from ";"->";" to ";" in list */
  for( i = 1; i < endlist(&net_info)-1; i++ ){
    sprintf( npp_buffer, "%s", see(i,&net_info) );
    sprintf( poscn, "%s", see(i+1,&net_info) );
    if( (strcmp(npp_buffer,";") == 0) && (strcmp(poscn,";") == 0) ){
      SL_delete( i, &net_info );
    }
  }

  /* write the first lines to the function */
  sprintf( npp_buffer, "DLLEXPORT void NETWORK( double NcsTime, double **Ncs_y , double **Ncs_dy)\n" );
  insert( npp_buffer, 1, &network );
  sprintf( npp_buffer, "{\n" );
  insert( npp_buffer, 2, &network );
  /** work befor ";" **/
  /* for "for" sentence */
  while( endlist(&net_info) != 1 ){
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    if( strcmp(npp_buffer,"for") == 0 ){
      /* write "int" to the function */
      sprintf( npp_buffer, "\t%s\n",see(2,&net_info) );
      /* don't write the same name variable */
      iflag = 0;
      for( j = 3; j < endlist(&net_info); j++ ){
        if( strncmp( see(j,&net_info),see(2,&net_info), 5 ) == 0 ){
    iflag++;
    break;
        }
      }
      if( iflag == 0 ){
        insert( npp_buffer, 3, &network );
      }
      sprintf( local_npp_buffer, "\tfor (%s){\n\n", see(3,&net_info) );
      insert( local_npp_buffer, endlist(&network), &network );
      SL_delete( 1, &net_info );
      SL_delete( 1, &net_info );
      SL_delete( 1, &net_info );
    }
    make_line( &line );

    listcat( &network, &line );
  }

  /* write end of lines to the function */
  insert("} /* network */\n\n",endlist(&network),&network);

  /* difine for the function */
  for( i = 1 ; i < endlist(&net_def); i++ ){
    insert( see(i,&net_def), 1, &network );
  }

  /* undef for the function  */
  listcat( &network, &net_undef );

  /* write the function from list to the model source file */
  for( i = 1; i < endlist(&network); i++ ){
    fprintf( fp, "%s", see(i,&network) );
  }
}

/*
 * Function to write lines before ";" to another list 
 * and delete the lines from the source list.
 */
void make_line( List *line_ptr )
{
  int    i, j;
  size_t k;
  int    net_number;       /* NET[net_number] */
  int    local_id;
  char   npp_buffer2[1024];
  char   npp_buffer2_1[1024];
  char   npp_buffer3[1024];
  char   end_line[1024];   /* store left side of equation */
  char   poscn[8];
  char   precn[8];
  char   cn[8];
  int    check_num;
  int    module_num;
  char   *npp_buffer2_1p;

  net_number = 0;

  /* for left side of equation */
  i = locate( "*", &net_info );
  sprintf( npp_buffer2, "%s", see(i+1,&net_info) );
  sprintf( npp_buffer2_1, "%s", see(i+1,&net_info) );
  sprintf( npp_buffer3, "%s", see(i+1,&net_info) );
  npp_buffer2_1p = npp_buffer2_1;
  /* because a next strtok() can be terminated by null */
  strtok( npp_buffer2_1, "," );

  /* end of lines */
  sprintf( end_line, "\tCN = %s;\n\t%s(NcsTime , Ncs_y , Ncs_dy);\n\n",
     strchr(npp_buffer2,',')+1, npp_buffer2_1p );
  /* free npp_buffer2 */

  /* to ready when the model is described multiple cell module */
  /* get POSCN  */
  sprintf( poscn, "%s", strchr(npp_buffer3,',')+1 );

  /* get module ID */
  sprintf( npp_buffer2, "%s", strtok(npp_buffer3,",") );
  /* free npp_buffer3 */

  local_id = atoi( see(locate(npp_buffer2,&module_table)+1,&module_table) );

  module_num = atoi( see(locate(npp_buffer2,&net_table)+1, &net_table) );

  /* keep mark of "*" because this mark indicate left side of equation */
  SL_delete(i+1,&net_info); /* delete the line next to "*" */


  sprintf(npp_buffer3,"\tPOSOUT = ncsm_outm%02d[%s];\n",local_id,poscn);

  /* when the model is described multiple cell modules */
  if( locate("*",&net_info) != 1 ){
    sprintf( npp_buffer, "%s", poscn );
    check_num = atoi( poscn );
    if( strchr("0123456789", (int)npp_buffer[0]) == NULL ){
      sprintf( npp_buffer, "\tif( %s >= %d ){\n", poscn, module_num );
      insert( npp_buffer, endlist( line_ptr ), line_ptr );
      sprintf( npp_buffer, "\t\texit(23); }\n" );
      insert( npp_buffer, endlist( line_ptr ), line_ptr );
    }else{
      /*
      if( atoi(poscn) >= module_num ){
        exit( 157 );
      }
      */
    }
    sprintf( npp_buffer2, "\tPOSCN = %s;\n", poscn );
    insert( npp_buffer2, endlist(line_ptr), line_ptr ); /* poscn  */
    insert( npp_buffer3, endlist(line_ptr), line_ptr ); /* posout */
    sprintf( npp_buffer3, "\tNET[%d] = 0.0;\n\n", net_number );
    insert( npp_buffer3, endlist(line_ptr), line_ptr );
  }

  while( strcmp(see(1,&net_info),"*") != 0 ){
    /* get PRECN */
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    sprintf( npp_buffer2, "%s", strchr(npp_buffer,'<')+1 );
    sprintf( precn, "%s", strchr(npp_buffer2,',')+1 );
    
    /* get module ID which is right side INPUT */
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    sprintf( npp_buffer2, "%s", strchr(npp_buffer,'<')+1 );
    sprintf( npp_buffer3, "%s", strtok(npp_buffer2,",") );
    local_id = atoi(see(locate(npp_buffer3,&module_table)+1,&module_table));
    module_num = atoi( see(locate(npp_buffer3,&net_table)+1,&net_table) );

    /* get CN */
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    sprintf( npp_buffer2, "%s", strchr(npp_buffer,',')+1 );
    sprintf( cn, "%s", strtok(npp_buffer2,"<") );

    /* get the module name */
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    sprintf( npp_buffer2, "%s", strtok(npp_buffer,",") );
    
    /* write all lines to the list */
    sprintf( npp_buffer, "\tPRECN = %s;\n", precn );
    insert( npp_buffer, endlist(line_ptr), line_ptr );

    sprintf( npp_buffer, "%s", precn );
    check_num = atoi( precn );
    if( strchr("0123456789",(int)npp_buffer[0]) == NULL ){
      sprintf( npp_buffer, "\tif( %s >= %d ){\n", precn, module_num );
      insert( npp_buffer, endlist(line_ptr), line_ptr );
      sprintf( npp_buffer, "\t\texit(23); }\n" );
      insert( npp_buffer, endlist(line_ptr), line_ptr );
    }else{
      /*
        if( check_num >= module_num ){
        exit( 157 );
        }
        */
    }

    sprintf( npp_buffer, "\tINPUT = ncsm_outm%02d[%s];\n", local_id, precn );
    insert( npp_buffer, endlist(line_ptr), line_ptr );
    sprintf( npp_buffer, "\tCN = %s;\n", cn );
    insert( npp_buffer, endlist(line_ptr), line_ptr );
    sprintf( npp_buffer,
       "\tNET[%d] += %s(NcsTime , Ncs_y , Ncs_dy);\n\n",
       net_number, npp_buffer2 );
    insert( npp_buffer, endlist(line_ptr), line_ptr );

    SL_delete( 1, &net_info );
    
    k = strlen( see(2,&net_info) );
    sprintf( npp_buffer, "%s", see(1,&net_info) );
    if( strcmp(npp_buffer,"!") == 0 ){
      /* delete "!" from the list */
      SL_delete( 1, &net_info );
      if( k > 1 ){
        net_number++;
        sprintf( npp_buffer, "\tNET[%d] = 0.0;\n\n", net_number );
        insert( npp_buffer, endlist(line_ptr), line_ptr );
      }
    }
  }

  /* delete "*" from the list */
  SL_delete( 1, &net_info );

  /* finally, write the left side of equation */
  insert( end_line, endlist(line_ptr), line_ptr );

  /* check end of "for" sentence */
  j = locate( ";", &net_info );
  for( i = 1; i < j; i++ ){
    sprintf( npp_buffer, "%s", see(i,&net_info) );
    if( strcmp(npp_buffer,"$") == 0 ){
      insert( "\t} /* for */\n\n", endlist(line_ptr), line_ptr );
      SL_delete( i, &net_info );
      break;
    }
  }
  /* delete ";" from the list */
  SL_delete( 1, &net_info );
}

/*
 * Function to write GetPrmPtr() for linking to NPE
 *                                        [ Oct.3,1994 ]
 */
static void
make_getprmptr()
{
    int  i, j;
    List  flist;

  init_list( &flist );

  /* module name */
  for( i = 1, j = 1; i < endlist(&name); i+=2 ){
  /* parameter name */
    while( (endlist(&getprmptr) > j) && 
        (strncmp(see( j, &getprmptr ), NCS_ENDMARK_GPP, 1) != 0) ){

      /* first half of script of "if" sentence */
      sprintf( npp_buffer, "\tif(!strcmp(\"%s\",mdl_name) ", 
        see( i+1, &name) );
      insert( npp_buffer, endlist(&flist), &flist );

      /* second half of script of "if" sentence */
      sprintf( npp_buffer, "&& !strcmp(\"%s\",prm_name)){\n", 
        see( j, &getprmptr) );
      insert( npp_buffer, endlist(&flist), &flist );
      if( i == 1 ){
        sprintf( npp_buffer, "\t\t\treturn( &n00%s );\n\t}\n", 
          see( j, &getprmptr ) );
      } else {
        sprintf( npp_buffer, "\t\t\treturn( &m%02d%s );\n\t}\n", 
          atoi(see(i,&name)),see( j, &getprmptr ) );
      }
      insert( npp_buffer, endlist(&flist), &flist );
      
      /* next parameter */
      j++;
    }
    /* move for NCS_ENDMARK_GPP */
    j++;
  }

  /* the first lines of the function */
  insert( "{\n", 1, &flist );
  insert( "DLLEXPORT double *GetPrmPtr( char *mdl_name, char *prm_name )\n", 1, &flist );
  insert( "/*\t       \t\t*/\n", 1, &flist );
  insert( "/*\tfor NPE\t\t*/\n", 1, &flist );
  insert( "\n/*\t       \t\t*/\n", 1, &flist );

  /* the end of lines of the function */
  insert( "\t/* Failded */\n\treturn( NULL );\n", 
    endlist(&flist), &flist);
  insert( "\n}\t/* GetPrmPtr() */\n\n", endlist(&flist), &flist );
    
  for( i = 1; i < endlist(&flist); i++ ){
    fprintf( fp, "%s", see(i,&flist) );
  }
    
  clearlist( &flist );
}


/* 
 * Write simulation condition file for parameter information
 */
static void 
make_prm_file( List *info )
{
  int    i;
  PrmFile prm;
  char   mdl_name[SCF_NAME_LEN];         
  char     prm_name[SCF_NAME_WID];
  double   value;
  char  type;


  if( PrmMakeSCF() == NCS_FALSE ){
    exit(105);
  }

  for( i = 1; i < endlist(info); i += 3 ){
    sprintf( mdl_name, "%s", see(i+2,info) );
    sprintf( prm_name, "%s", see(i,info) );
    value = atof( see(i+1,info) );
    type = SCF_PRM_TYPE_NORMAL;

    PrmSetInfoSCF(&prm, &type, mdl_name, prm_name, value );
    if( PrmAddInfoSCF( &prm ) == NCS_FALSE ){
      exit(106);
    }

  }
}
    
/* 
 * Write simulation condition file for external input information
 */
static void
make_xin_file()
{
  /* making of base of SCF file for exinput */
    if( XinMakeSCF() == NCS_FALSE ){
    exit(107);
  }

  /* making of base of SCF file for external input function */
    if( XfpMakeSCF() == NCS_FALSE ){
    exit(108);
  }
}

/* 
 * Write simulation condition file for delay information
 */
static void
make_dly_file( List *info )
{
  int   i;
  DlyFile dly;
  char  flag;

  /* making of base of SCF file for delay */
    if( DlyMakeSCF() == NCS_FALSE ){
    exit(109);
  }

  /* setting of SCF file for delay */
  flag = SCF_DLY_FLAG_FIX; 
  if( endlist(info) != 1 ){
    for( i = 1; i < endlist(info); i += 4 ){
      DlySetInfoSCF( &dly, see(i,info), see(i+1,info),
         atof( see( i+2, info ) ), atof( see( i+3, info ) ), 
         &flag );
      if( DlyAddInfoSCF( &dly ) == NCS_FALSE ){
        exit(110);
      } 
    }
  }
}

/* 
 * Write simulation condition file for output information
 */
static void
make_out_file()
{
  /* making of base of SCF file for observable */
    if( OutMakeSCF() == NCS_FALSE ){
    exit(111);
  }
}

/* 
 * Write simulation condition file for observable information
 */
void 
make_obs_file( List *info )
{
  int  i;
  int  lp;
  int  mod;
  int  component = 0;
  char flag;
  ObsFile obs;

  if( ObsMakeSCF() == NCS_FALSE ){
    exit(112);
  }

  /* store all observable informaiton */
  obs.type = SCF_OBS_NORMAL;
  for( i = 1; i < endlist(info); i += 5 ){
    mod = 0;
    flag = (char)*see( i+3, info );
    /* There is the variable in function description */
    if( flag == NCS_FLAG_ON ){  
      strcpy( obs.info.var, see(i,info) );
      strcpy( obs.info.module, see(i+2,info) );
      
      if( (lp=locate(obs.info.module,&cell_modules)) != 0 ){
        mod = 1;
        component = atoi( see(lp+1,&cell_modules) );
      }
      if( ( mod == 0 ) && 
          ( (lp=locate(obs.info.module,&synapse_modules)) != 0 ) ){
        mod = 2;
        component = atoi( see(lp+1,&synapse_modules) );
      }
      if( ( mod == 0 ) && 
          ( (lp=locate(obs.info.module,&gap_modules)) != 0 ) ){
        mod = 3;
        component = atoi( see(lp+1,&gap_modules) );
      }
      
      if( mod != 0 ){
        obs.info.max_comp = component;
      } else {
        printf(
          "Wanning: \"%s\" module is not defined in network description", 
          obs.info.module );
        obs.info.max_comp = 0;
      }
      
      if( ObsAddInfoSCF( &obs ) == NCS_FALSE ){
        exit(112);
      }
    }
  }
  
  clearlist( info );
}


/* 
 * Write simulation condition file for calculation information
 */
static void
make_mat_file()
{
  MatFile  mat;
  int   cell_module_max = 0;
  int   i;
  char  method;

  if( MatMakeSCF() == NCS_FALSE ){
    exit(105);
  }

  for( i = 1; i < endlist(&module_table); i += 4 ){
    if( (cell_module_max < atoi(see(i+3,&module_table))) 
        && (atoi(see(i+2,&module_table))==1) ){
      cell_module_max = atoi( see(i+3,&module_table) );
    }
  }
  method = SCF_MAT_INTEG_FERBERG;
  
  MatSetInfoSCF( &mat, 1.0, 0.1, 0.1, 1.0, &method,
          cell_module_max, 1.0e-5 );

  if( MatAddInfoSCF( &mat ) == NCS_FALSE ){
    exit(106);
  }
}

/* 
 * end of command
 */
static void
endprocess()
{
  clearlist( &net_def );
  clearlist( &net_undef );
  clearlist( &network );
  clearlist( &net_info );
  clearlist( &module );
  clearlist( &f_header );
  clearlist( &name );
  clearlist( &exinput );
  clearlist( &input );
  clearlist( &output );
  clearlist( &observable );
  clearlist( &observ_table );
  clearlist( &integ );
  clearlist( &constant );
  clearlist( &parameter );
  clearlist( &param_table );
  clearlist( &initial );
  clearlist( &define );
  clearlist( &declare );
  clearlist( &delay );
  clearlist( &function );
  clearlist( &undef );
  clearlist( &f_end );
  clearlist( &ncs_double );
  clearlist( &ncs_obs );
  clearlist( &ncs_xin );
  clearlist( &delay_file );
  clearlist( &fact );

  clearlist( &net_module );
  clearlist( &net_inputs );
  clearlist( &net_loads );

  clearlist( &cell_modules );
  clearlist( &synapse_modules );
  clearlist( &gap_modules );
}

/* 
 * Function to write ncsf_workset() to the model source file
 */
static void make_ncsf_workset()
{
  int   i;
  List  f_global;     /* to store final script of ncsf_global()  */
  
  /* making of local tabel */
  init_list( &f_global );

  /* write begin of line to the function */
  insert("DLLEXPORT void ncsf_workset()\n{\n", endlist(&f_global), &f_global );
  insert("\tint i;\n", endlist(&f_global), &f_global );

  /* module information table */
  insert("\tncsg_ncout = (int*)malloc(sizeof(int)*NCS_OUT_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_ncxin = (int*)malloc(sizeof(int)*NCS_XIN_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_ncobs = (int*)malloc(sizeof(int)*NCS_OBS_NT);\n",
         endlist(&f_global), &f_global);

  /* memory address table */
  insert("\tncsm_outptr = (double**)malloc(sizeof(double*)*NCS_OUT_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsm_xinptr = (double**)malloc(sizeof(double*)*NCS_XIN_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsm_obsptr = (double**)malloc(sizeof(double*)*NCS_OBS_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsm_subptr = (double**)malloc(sizeof(double*)*NCS_OUT_NT);\n",
         endlist(&f_global), &f_global);

  /* module name table & address table */
  
  insert("\tncsg_outtbl = (char**)malloc(sizeof(char*)*(NCS_OUT_NT+1));\n",
         endlist(&f_global), &f_global);
  insert("\tfor(i=0; i<NCS_OUT_NT+1; i++)\n", endlist(&f_global), &f_global);
  insert("\t\tncsg_outtbl[i] = (char*)malloc(sizeof(char)*NAME_SIZE);\n",
         endlist(&f_global), &f_global);

  insert("\tncsg_xintbl = (char**)malloc(sizeof(char*)*(NCS_XIN_NT+1));\n",
         endlist(&f_global), &f_global);
  insert("\tfor(i=0; i<NCS_XIN_NT+1; i++)\n", endlist(&f_global), &f_global);
  insert("\t\tncsg_xintbl[i] = (char*)malloc(sizeof(char)*NAME_SIZE);\n",
         endlist(&f_global), &f_global);

  insert("\tncsg_obstbl = (char**)malloc(sizeof(char*)*(NCS_OBS_NT+1));\n",
         endlist(&f_global), &f_global);
  insert("\tfor(i=0; i<NCS_OBS_NT+1; i++)\n",
         endlist(&f_global), &f_global);
  insert("\t\tncsg_obstbl[i] = (char*)malloc(sizeof(char)*NAME_SIZE);\n",
         endlist(&f_global), &f_global);

  insert("\tncsg_outtblptr = (char**)malloc(sizeof(char*)*(NCS_OUT_NT+1));\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_xintblptr = (char**)malloc(sizeof(char*)*(NCS_XIN_NT+1));\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_obstblptr = (char**)malloc(sizeof(char*)*(NCS_OBS_NT+1));\n",
         endlist(&f_global), &f_global);

  /* EXTERNAL INPUT INFORMATION */
  insert("\tncsg_xindata = (xininfo_t*)malloc(sizeof(xininfo_t)*NCS_XIN_MAX);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_xfpdata = (xfpinfo_t*)malloc(sizeof(xfpinfo_t)*NCS_XIN_MAX);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_xinfaddr = (XIN_FUNC*)malloc(sizeof(XIN_FUNC)*NCS_XF_MAX);\n",
         endlist(&f_global), &f_global);

  /* For NCS_NET */
  insert( "\tNET = (double *)malloc(NCS_NNET * sizeof(double));\n",
		endlist(&f_global), &f_global );
  insert( "\t\tfor(i=0;i<NCS_NNET;i++)\n",
		endlist(&f_global), &f_global );
  insert( "\t\t\tNET[i]=0.0;\n",
		endlist(&f_global), &f_global );

  /* For Delay */
  insert("\tncsg_dlytime = (double*)malloc(sizeof(double)*NCS_DLY_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_dlyinit = (double*)malloc(sizeof(double)*NCS_DLY_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_qsize = (int*)malloc(sizeof(int)*NCS_DLY_NT);\n",
         endlist(&f_global), &f_global);
  insert("\tncsg_dlyinit_flg = (char*)malloc(sizeof(char)*NCS_DLY_NT);\n",
         endlist(&f_global), &f_global);

  /* for external functions */
  insert("\tncsf_prmset_func = ncsf_prmset;\n",
         endlist(&f_global), &f_global);

  /* write end of line to the function */
  insert( "}\n\n", endlist(&f_global), &f_global );
  
  /* write the function from list to the model source file */
  for( i = 1; i < endlist(&f_global); i++ ){
    fprintf( fp, "%s", see(i,&f_global) );
  }
  clearlist( &f_global );
}

/*
 * Function to write ncsf_workcls() to the model source file
 */
void make_ncsf_workcls()
{
  int  i;
  List f_global;     /* to store final script of ncsf_constset()  */
  
  /* making of local tabel */
  init_list( &f_global );
  
  /* write end of line to the function */
  insert("DLLEXPORT void ncsf_workcls()\n{\n", endlist(&f_global), &f_global );
  insert("\tint i;\n", endlist(&f_global), &f_global );

  /* module information table */
  insert("\tfree(ncsg_ncout);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_ncxin);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_ncobs);\n", endlist(&f_global), &f_global);

  /* memory address table */
  insert("\tfree(ncsm_outptr);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsm_xinptr);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsm_obsptr);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsm_subptr);\n", endlist(&f_global), &f_global);

  /* module name table & address table */
  insert("\tfor(i=0; i<NCS_OUT_NT+1; i++)\n", endlist(&f_global), &f_global);
  insert("\t\tfree(ncsg_outtbl[i]);\n", endlist(&f_global), &f_global);
  insert("\tfor(i=0; i<NCS_XIN_NT+1; i++)\n", endlist(&f_global), &f_global);
  insert("\t\tfree(ncsg_xintbl[i]);\n", endlist(&f_global), &f_global);
  insert("\tfor(i=0; i<NCS_OBS_NT+1; i++)\n", endlist(&f_global), &f_global);
  insert("\t\tfree(ncsg_obstbl[i]);\n", endlist(&f_global), &f_global);

  insert("\tfree(ncsg_outtblptr);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_xintblptr);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_obstblptr);\n", endlist(&f_global), &f_global);

  /* EXTERNAL INPUT INFORMATION */
  insert("\tfree(ncsg_xindata);\n",  endlist(&f_global), &f_global);
  insert("\tfree(ncsg_xfpdata);\n",  endlist(&f_global), &f_global);
  insert("\tfree(ncsg_xinfaddr);\n", endlist(&f_global), &f_global);

  /* For Delay */
  insert("\tfree(ncsg_dlytime);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_dlyinit);\n", endlist(&f_global), &f_global);
  insert("\tfree(ncsg_qsize);\n",   endlist(&f_global), &f_global);
  insert("\tfree(ncsg_dlyinit_flg);\n", endlist(&f_global), &f_global);

  /* write end of line to the function */
  insert( "}\n\n", endlist(&f_global), &f_global );
  
  /* write the function from list to the model source file */
  for( i = 1; i < endlist(&f_global); i++ ){
    fprintf( fp, "%s", see(i,&f_global) );
  }
  clearlist( &f_global );
}
