/* 
 * Copyright (c) 2003-2005 RIKEN Japan, All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: npemake.cpp,v 1.6 2005/11/11 07:43:47 orrisroot Exp $ */
/**********************************
** create NPE executable program **
**********************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#ifdef WIN32
# include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h> /* chmod() */
#endif

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libsatellite.h"
#include "libnpec.h"
#include "npepriv.h"

#ifdef __cplusplus
extern "C" {
#endif

#define CONSOLE(x) syscom->console->x

#ifdef WIN32
# define MAKEWRAPPER  "_npe_make.bat"
# define MDLLTEMP     "_npe_model.dll"
# define PDLLTEMP     "_npe_penalty.dll"
#else
# define MAKEWRAPPER  "_npe_make.sh"
#endif
#define MAKEFILE     "_npe_makefile"
#define MAKEERROR    "_npe_make.err"
#define MOBJTEMP     "_npe_model.o"
#define POBJTEMP     "_npe_penalty.o"

#ifdef WIN32
static int  create_npe_makefile_mingw(npe_condition_t *ncond);
static char *filename_dos2unix(const char *path);
#else
static int  create_npe_makefile_unix(npe_condition_t *ncond);
#endif
static void print_npe_make_errmsg(const char *fname);
static int  create_npe_makewrapper(npe_condition_t *ncond);

int npe_make(npe_condition_t *ncond){
  int errcode, pid;
  char *makewrapper, *makeerror;
#ifdef WIN32
  if((errcode = create_npe_makefile_mingw(ncond)) != 0)
    return errcode;
#else
  if((errcode = create_npe_makefile_unix(ncond)) != 0)
    return errcode;
#endif
  if((errcode = create_npe_makewrapper(ncond)) != 0)
    return errcode;
  makewrapper = npe_tmpfile_path(get_tmpdir(), MAKEWRAPPER);
  if(makewrapper == NULL)
    return 2; /* out of memory */
  makeerror  = npe_tmpfile_path(get_tmpdir(), MAKEERROR);
  if(makeerror == NULL){
    free(makewrapper);
    return 2; /* out of memory */
  }
#ifdef WIN32
  {
    char *p, *tmp;
    for(p=makewrapper; *p!='\0'; p++) if(*p=='/') *p='\\';
    for(p=makeerror; *p!='\0'; p++) if(*p=='/') *p='\\';
    tmp = (char*)malloc(strlen(makewrapper) + 3);
    if(tmp == NULL){
      free(makewrapper); free(makeerror);
      return 2; /* out of memory */
    }
    sprintf(tmp, "\"%s\"", makewrapper);
    free(makewrapper);
    makewrapper = tmp;
  }
#endif
  pid = create_child_process(makewrapper, CONSOLE(ifd), CONSOLE(ofd), 
                             CONSOLE(efd), NULL);
  free(makewrapper);
  if(pid == -1){
    errcode = 230; /* can't create make wrapper process */
  }else{
    int status;
    status = wait_child_process(pid, &errcode);
    switch(status){
    case -1: errcode = 231; break; /* process not found */
    case 0:
      switch(errcode){
      case 0: break;
      case 101: errcode = 232; break; /* npe compile error */
      case 102: errcode = 233; break; /* npe link error */
      }
      break; /* normal exit */
    case 1:  errcode = 234; break; /* signal caught in make command wrapper */
    case 2:  errcode = 235; break; /* make cmd wrapper process is stoping */
    case 3:  errcode = 236; break; /* make cmd wrapper was core dumped */
    default: errcode = 237; break; /* unknown error in make cmd wrapper */
    }
  }
  if(errcode != 0)
    print_npe_make_errmsg(makeerror);
  free(makeerror);
  return errcode;
}

char *npe_tmpfile_path(const char *tmpdir, const char *fname){
  char *ret;
  size_t len;
  len = strlen(tmpdir) + strlen(fname) + 2;
  ret = (char*)malloc(len);
  if(ret == NULL)
    return NULL;
  sprintf(ret, "%s/%s", tmpdir, fname);
  return ret;
}

static void print_npe_make_errmsg(const char *fname){
  FILE *fp;
  char  buf[1024];
  fp = fopen(fname,"r");
  if(fp){
    while(feof(fp)==0){
      fgets(buf,1023,fp);
      buf[1023]='\0';
      printf("%s",buf);
    }
    fclose(fp);
  }
}

static int create_npe_makewrapper(npe_condition_t *ncond){
  FILE *fp;
  char *makewrapper;
#ifdef WIN32
  const char *mingw_path="tools\\bin";
  char  tmpdir[_MAX_PATH], *p;
  char  sl4drv[_MAX_DRIVE], sl4prefix[_MAX_DIR], sl4path[_MAX_PATH];
  /* get full path info */
  ::GetModuleFileName(NULL, sl4path, _MAX_PATH);
  /* split drive and directory */
  _splitpath(sl4path, sl4drv, sl4prefix, NULL, NULL);
  strncpy(tmpdir,get_tmpdir(),_MAX_PATH);
  for(p=tmpdir;*p!='\0';p++) if(*p == '/') *p='\\';
#endif
  makewrapper = npe_tmpfile_path(get_tmpdir(), MAKEWRAPPER);
  if(makewrapper == NULL)
    return 2; /* out of memory */
  fp = fopen(makewrapper, "w");
  if(fp == NULL){
    free(makewrapper);
    return 228; /* can't create make command wrapper */
  }
#ifdef WIN32
  fprintf(fp,"@echo off\n");
  /* set path */
  fprintf(fp,"SET PATH=%s%s%s;%%PATH%%\n", sl4drv, sl4prefix, mingw_path);
  fprintf(fp,"if exist \"%s\\%s\" del /f \"%s\\%s\"\n", 
          tmpdir, MAKEERROR, tmpdir, MAKEERROR);
  fprintf(fp,"make -f \"%s\\%s\" > \"%s\\%s\" 2>&1\n",
          tmpdir, MAKEFILE, tmpdir, MAKEERROR);
  fprintf(fp,"if errorlevel 1 goto linkerr\n");
  fprintf(fp,"exit 0\n");
  fprintf(fp,":linkerr\n");
  fprintf(fp,"exit 102\n");
#else
  fprintf(fp,"#!/bin/sh\n");
  fprintf(fp,"rm -f \"%s/%s\"\n", get_tmpdir(), MAKEERROR);
  fprintf(fp,"make -f \"%s/%s\" clean\n", get_tmpdir(), MAKEFILE);
  fprintf(fp,"make -f \"%s/%s\" npe-compile > \"%s/%s\" 2>&1\n",
          get_tmpdir(), MAKEFILE, get_tmpdir(), MAKEERROR);
  fprintf(fp,"RETCODE=$?\n");
  fprintf(fp,"if [ $RETCODE -eq 1 ]; then\n");
  fprintf(fp," exit 101\n");
  fprintf(fp,"fi\n");
  fprintf(fp,"make -f \"%s/%s\" npe-link >> \"%s/%s\" 2>&1\n", 
          get_tmpdir(), MAKEFILE, get_tmpdir(), MAKEERROR);
  fprintf(fp,"RETCODE=$?\n");
  fprintf(fp,"if [ $RETCODE -eq 1 ]; then\n");
  fprintf(fp," exit 102\n");
  fprintf(fp,"fi\n");
  fprintf(fp,"exit 0\n");
#endif
  fclose(fp);

#ifndef WIN32
  if(chmod(makewrapper, 0700) == -1){
    free(makewrapper);
    return 229; /* can't change file attribute of make command wrapper */
  }
#endif
  free(makewrapper);
  return 0;
}

#ifdef WIN32
static int create_npe_makefile_mingw(npe_condition_t *ncond){
  /* for WIN32 */
  FILE  *fp;
  char  *cc="mingw32-gcc";
  char  *cxx="mingw32-g++";
  char  *tmp;
  char *mobj, *pobj;
  char *mfile, *pfile;
  char *sl4drv, *sl4prefix;
  char *tmpdir, *makefile;
  char   _sl4drv[_MAX_DRIVE], _sl4prefix[_MAX_DIR], sl4path[_MAX_PATH];

  mobj  = pobj  = NULL;
  mfile = pfile = NULL;
  sl4drv = sl4prefix = NULL;
  tmpdir = makefile = NULL;

  mfile = filename_dos2unix(npe_common->model_file);
  pfile = filename_dos2unix(npe_common->penalty_file);

  /* tmpdir */
  tmpdir = filename_dos2unix(get_tmpdir());
  if(tmpdir == NULL) goto memoryerror;

  /* get full path info */
  ::GetModuleFileName(NULL, sl4path, _MAX_PATH);
  /* split drive and directory */
  _splitpath(sl4path, _sl4drv, _sl4prefix, NULL, NULL);
  sl4drv    = filename_dos2unix(_sl4drv);
  if(sl4drv == NULL) goto memoryerror;
  sl4prefix = filename_dos2unix(_sl4prefix);
  if(sl4prefix == NULL) goto memoryerror;

  /* model object */
  if(ncond->model_file_type != NPE_FILETYPE_OBJECT){
    tmp = npe_tmpfile_path(get_tmpdir(), MOBJTEMP);
    if(tmp == NULL) goto memoryerror;
    mobj = filename_dos2unix(tmp);
  }else{
    mobj = strdup(mfile);
  }
  if(mobj == NULL) goto memoryerror;

  /* penalty object */
  if(ncond->penalty_file_type != NPE_FILETYPE_OBJECT){
    tmp = npe_tmpfile_path(get_tmpdir(), POBJTEMP);
    if(tmp == NULL) goto memoryerror;
    pobj = filename_dos2unix(tmp);
  }else{
    pobj = strdup(pfile);
  }
  if(pobj == NULL) goto memoryerror;
  
  /* create a makefile */
  makefile = npe_tmpfile_path(get_tmpdir(), MAKEFILE);
  if(makefile == NULL) goto memoryerror;
  fp = fopen(makefile, "w");
  if(fp == NULL){
    free(mobj);   free(pobj);
    free(pfile);  free(mfile);
    free(sl4drv); free(sl4prefix);
    free(tmpdir); free(makefile);
    return 227; /* can't create makefile */
  }
  
  fprintf(fp,"CC = %s\n", cc);
  fprintf(fp,"CXX = %s\n", cxx);
  fprintf(fp,"DFLAGS = -Wall -g -ggdb\n");
  
  fprintf(fp,"NPE_CFLAGS   = \n");
  fprintf(fp,"NPE_CXXFLAGS = \n");
  fprintf(fp,"NPE_LDFLAGS  = -shared -Wl,--export-all-symbols -Wl,--enable-auto-import -Wl,--no-whole-archive\n");
  
  fprintf(fp,"OPTS = -O2\n");

  fprintf(fp,"SL4_INCLUDES = -I%s%sinclude\n", sl4drv, sl4prefix);
  fprintf(fp,"NCS_INCLUDES = -I%s%sinclude/ncs\n", sl4drv, sl4prefix);
  fprintf(fp,"NPE_INCLUDES = -I%s%sinclude/npe\n", sl4drv, sl4prefix);

  fprintf(fp,"CXXFLAGS = $(DFLAGS) $(OPTS) $(NPE_CXXFLAGS) $(INCLUDES)\n");
  fprintf(fp,"CFLAGS   = $(DFLAGS) $(OPTS) $(NPE_CFLAGS) $(INCLUDES)\n");
  
  fprintf(fp,"\n");
  fprintf(fp,"SL4LIB  = %s%slibsatellite.dll\n", sl4drv, sl4prefix);
  fprintf(fp,"NCSLIBS = %s%smodules/ncs/libncss.dll\n", sl4drv, sl4prefix);
  fprintf(fp,"NPELIBS = %s%smodules/npe/libnpee.dll\n", sl4drv, sl4prefix);

  switch(npe_common->model_type){
  case NPE_MODEL_USR:
    fprintf(fp,"INCLUDES = -I. $(SL4_INCLUDES) $(NPE_INCLUDES)\n");
    fprintf(fp,"LDFLAGS = $(NPELIBS)  $(SL4LIB) $(NPE_LDFLAGS)\n");
    break;
  case NPE_MODEL_NCS:
    fprintf(fp,"INCLUDES = -I. $(SL4_INCLUDES) $(NPE_INCLUDES) $(NCS_INCLUDES)\n");
    fprintf(fp,"LDFLAGS = $(NPELIBS) $(NCSLIBS) $(SL4LIB) $(NPE_LDFLAGS)\n");
    break;
  }
  fprintf(fp,"\n");
  if(npe_common->model_type == NPE_MODEL_NCS && 
     ncond->model_file_type == NPE_FILETYPE_DLL){
    fprintf(fp,"MDLL = %s\n", mfile);
  }else{
    fprintf(fp,"MDLL = %s/%s\n", tmpdir, MDLLTEMP);
  }
  fprintf(fp,"PDLL = %s/%s\n", tmpdir, PDLLTEMP);
  fprintf(fp,"MOBJ = %s\n", mobj);
  fprintf(fp,"POBJ = %s\n", pobj);
  fprintf(fp,"\n");

  fprintf(fp,"all: $(MDLL) $(PDLL)\n");
  fprintf(fp,"\n");
  fprintf(fp,"npe-compile: $(OBJS)\n");
  fprintf(fp,"\n");
  fprintf(fp,"npe-link: $(MDLL) $(PDLL)\n");
  fprintf(fp,"\n");

  /* model object creation rule */
  switch(ncond->model_file_type){
  case NPE_FILETYPE_CSOURCE:
    fprintf(fp,"$(MOBJ) : %s\n", mfile);
    fprintf(fp,"\t$(CC) $(CFLAGS) -c %s -o $(MOBJ)\n", mfile);
    break;
  case NPE_FILETYPE_CXXSOURCE:
    fprintf(fp,"$(MOBJ) : %s\n", mfile);
    fprintf(fp,"\t$(CXX) $(CXXFLAGS) -c %s -o $(MOBJ)\n", mfile);
    break;
  }
  fprintf(fp,"\n");

  /* penalty object creation rule */
  switch(ncond->penalty_file_type){
  case NPE_FILETYPE_CSOURCE:
    fprintf(fp,"$(POBJ) : %s\n", pfile);
    fprintf(fp,"\t$(CC) $(CFLAGS) -c %s -o $(POBJ)\n", pfile);
    break;
  case NPE_FILETYPE_CXXSOURCE:
    fprintf(fp,"$(POBJ) : %s\n", pfile);
    fprintf(fp,"\t$(CXX) $(CXXFLAGS) -c %s -o $(POBJ)\n", pfile);
    break;
  }
  fprintf(fp,"\n");
  if(npe_common->model_type != NPE_MODEL_NCS ||
     ncond->model_file_type != NPE_FILETYPE_DLL){
    fprintf(fp,"$(MDLL): $(MOBJ)\n");
    fprintf(fp,"\t$(CXX) $(CXXFLAGS) -o $(MDLL) $(MOBJ) $(LDFLAGS)\n");
    fprintf(fp,"\n");
  }
  fprintf(fp,"$(PDLL): $(POBJ)\n");
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) -o $(PDLL) $(POBJ) $(LDFLAGS)\n");
  fprintf(fp,"\n");
  fprintf(fp,".PHONY: all npe-compile npe-link clean\n");
  fclose(fp);

  free(mobj);   free(pobj);
  free(pfile);  free(mfile);
  free(sl4drv); free(sl4prefix);
  free(tmpdir); free(makefile);
  return(0);

  memoryerror:
  if(mobj)      free(mobj);
  if(pobj)      free(pobj);
  if(mfile)     free(mfile);
  if(pfile)     free(pfile);
  if(sl4drv)    free(sl4drv);
  if(sl4prefix) free(sl4prefix);
  if(makefile)  free(makefile);
  if(tmpdir)    free(tmpdir);
  return 2; /* out of memory */
}
#else
static int create_npe_makefile_unix(npe_condition_t *ncond){
  /* for UNIX */
  FILE  *fp;
  char  *cc_tmp, *makefile;
  char  *mobj, *pobj;

  /* model object */
  if(ncond->model_file_type != NPE_FILETYPE_OBJECT){
    mobj = npe_tmpfile_path(get_tmpdir(), MOBJTEMP);
  }else{
    mobj = strdup(npe_common->model_file);
  }
  if(mobj == NULL){
    return 2; /* out of memory */
  }

  /* penalty object */
  if(ncond->penalty_file_type != NPE_FILETYPE_OBJECT){
    pobj = npe_tmpfile_path(get_tmpdir(), POBJTEMP);
  }else{
    pobj = strdup(npe_common->penalty_file);
  }
  if(pobj == NULL){
    free(mobj);
    return 2; /* out of memory */
  }
  
  /* create a makefile */
  makefile = npe_tmpfile_path(get_tmpdir(), MAKEFILE);
  if(makefile == NULL){
    free(mobj); free(pobj);
    return 2; /* out of memory */
  }
  fp = fopen(makefile, "w");
  free(makefile);
  if(fp == NULL){
    free(mobj); free(pobj);
    return 227; /* can't create makefile */
  }
  
  cc_tmp = getenv("CC");
  fprintf(fp,"CC = %s\n",(cc_tmp!=0) ? cc_tmp : NPE_CC);
  cc_tmp = getenv("CXX");
  fprintf(fp,"CXX = %s\n",(cc_tmp!=0) ? cc_tmp : NPE_CXX);
  fprintf(fp,"DFLAGS = -Wall -g -ggdb\n");
  fprintf(fp,"RM = rm -f\n");
  
  fprintf(fp,"NPE_CFLAGS = %s\n",   NPE_CFLAGS);
  fprintf(fp,"NPE_CXXFLAGS = %s\n", NPE_CXXFLAGS);
  fprintf(fp,"NPE_LDFLAGS = %s\n",  NPE_LDFLAGS);
  
  fprintf(fp,"OPTS = -O2\n");

  fprintf(fp,"SL4_INCLUDES = -I%s\n", SL4_INCLUDE_DIR);
  fprintf(fp,"NCS_INCLUDES = -I%s/ncs\n", SL4_INCLUDE_DIR);
  fprintf(fp,"NPE_INCLUDES = -I%s/npe\n", SL4_INCLUDE_DIR);

  fprintf(fp,"CXXFLAGS = $(DFLAGS) $(OPTS) $(NPE_CXXFLAGS) $(INCLUDES)\n");
  fprintf(fp,"CFLAGS   = $(DFLAGS) $(OPTS) $(NPE_CFLAGS) $(INCLUDES)\n");
  
  fprintf(fp,"\n");
  fprintf(fp,"SL4LIB  = -L%s -lsatellite -lm\n", SL4_LIB_DIR);
  fprintf(fp,"NCSLIBC = -L%s/ncs -lncsc\n", SL4_MODULE_DIR);
  fprintf(fp,"NCSLIBS = -L%s/ncs -lncss\n", SL4_MODULE_DIR);
  fprintf(fp,"NPELIBC = -L%s/npe -lnpec\n", SL4_MODULE_DIR);
  fprintf(fp,"NPELIBS = -L%s/npe -lnpee\n", SL4_MODULE_DIR);
  fprintf(fp,"NPELIBU = -L%s/npe -lnpeu\n", SL4_MODULE_DIR);
  fprintf(fp,"NPELIBN = -L%s/npe -lnpen\n", SL4_MODULE_DIR);
  fprintf(fp,"TERMCAP_LIBS = %s\n", TERMCAP_LIBS);

  switch(npe_common->model_type){
  case NPE_MODEL_USR:
    fprintf(fp,"INCLUDES = -I. $(SL4_INCLUDES) $(NPE_INCLUDES)\n");
    fprintf(fp,"LDFLAGS = $(NPELIBU) $(NPELIBS) $(NPELIBC) $(SL4LIB) $(NPE_LDFLAGS)\n");
    break;
  case NPE_MODEL_NCS:
    fprintf(fp,"INCLUDES = -I. $(SL4_INCLUDES) $(NPE_INCLUDES) $(NCS_INCLUDES)\n");
    fprintf(fp,"LDFLAGS = $(NPELIBN) $(NPELIBS) $(NPELIBC) $(NCSLIBS) $(NCSLIBC) $(SL4LIB) $(TERMCAP_LIBS) $(NPE_LDFLAGS)\n");
    break;
  }
  fprintf(fp,"\n");
  fprintf(fp,"PROG = %s/%s\n", get_tmpdir(), NPE_COMMAND);
  fprintf(fp,"OBJS = %s %s\n", mobj, pobj);
  fprintf(fp,"\n");

  fprintf(fp,"all: $(PROG)\n");
  fprintf(fp,"\n");
  fprintf(fp,"npe-compile: $(OBJS)\n");
  fprintf(fp,"\n");
  fprintf(fp,"npe-link: $(PROG)\n");
  fprintf(fp,"\n");
  fprintf(fp,"clean:\n");
  fprintf(fp,"\t@$(RM) %s/%s %s/%s %s/%s\n", get_tmpdir(), NPE_COMMAND,
          get_tmpdir(), MOBJTEMP, get_tmpdir(), POBJTEMP);
  fprintf(fp,"\n");

  /* model object creation rule */
  switch(ncond->model_file_type){
  case NPE_FILETYPE_CSOURCE:
    fprintf(fp,"%s : %s\n", mobj, npe_common->model_file);
    fprintf(fp,"\t$(CC) $(CFLAGS) -c %s -o %s\n",
            npe_common->model_file, mobj);
    break;
  case NPE_FILETYPE_CXXSOURCE:
    fprintf(fp,"%s : %s\n", mobj, npe_common->model_file);
    fprintf(fp,"\t$(CXX) $(CXXFLAGS) -c %s -o %s\n",
            npe_common->model_file, mobj);
    break;
  }
  fprintf(fp,"\n");

  /* penalty object creation rule */
  switch(ncond->penalty_file_type){
  case NPE_FILETYPE_CSOURCE:
    fprintf(fp,"%s : %s\n", pobj, npe_common->penalty_file);
    fprintf(fp,"\t$(CC) $(CFLAGS) -c %s -o %s\n",
            npe_common->penalty_file, pobj);
    break;
  case NPE_FILETYPE_CXXSOURCE:
    fprintf(fp,"%s : %s\n", pobj, npe_common->penalty_file);
    fprintf(fp,"\t$(CXX) $(CXXFLAGS) -c %s -o %s\n",
            npe_common->penalty_file, pobj);
    break;
  }
  fprintf(fp,"\n");
  fprintf(fp,"$(PROG): $(OBJS)\n");
  fprintf(fp,"\t$(CXX) $(CXXFLAGS) -o $@ $(OBJS) $(LDFLAGS)\n");
  fprintf(fp,"\n");
  fprintf(fp,".PHONY: all npe-compile npe-link clean\n");
  fclose(fp);

  free(mobj);
  free(pobj);

  return(0);
}
#endif

#ifdef WIN32
static char *filename_dos2unix(const char *dos){
  const char *p;
  char *unix;
  size_t len;
  len = strlen(dos); 
  for(p=dos; *p!='\0'; p++) if(*p==' ') len++;
  unix = (char*)malloc(sizeof(char) * (len + 1));
  if(unix == NULL) return NULL;
  len = 0;
  for(p=dos; *p!='\0'; p++){
    unix[len++] = *p;
    if(*p == '\\'){
      unix[len-1] = '/';
    }else if(*p == ' '){
      unix[len-1] = '\\';
      unix[len++] = ' ';
    }
  }
  unix[len] = '\0';
  return unix;
}
#endif

#ifdef __cplusplus
}
#endif
