/* 
 * Copyright (c) 2003 RIKEN (The Institute of Physical and Chemical Research)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY RIKEN AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL RIKEN OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: nperun.cpp,v 1.2 2004/04/19 17:07:52 orrisroot Exp $ */
/*****************************
**    run NPE program       **
*****************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>

#include "SL_macro.h"
#include "SL_cmd.h"

#include "libsatellite.h"
#include "libnpec.h"
#include "npepriv.h"

#define CONSOLE(x) syscom->console->x
#ifndef _MAX_PATH
# ifdef  MAXPATHLEN
#  define _MAX_PATH  MAXPATHLEN
# else
#  define _MAX_PATH  1024
# endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

static int _npe_common_memory_store(npe_condition_t *ncond);
static int _npe_library_errorcode(int err);

int npe_run(npe_condition_t *ncond){
  int  pid, errcode;
  char bin[_MAX_PATH], libpath[_MAX_PATH], moddir[_MAX_PATH];

  /* write npe common memory to temporary file */
  if((errcode = _npe_common_memory_store(ncond)) != 0)
    return errcode;

  /* set current path */
#ifdef WIN32
  GetModuleDirectory(moddir, _MAX_PATH);
  if(npe_common->model_type == NPE_MODEL_USR){
    snprintf(bin, _MAX_PATH, "\"%s\\npe\\npeestu.exe\"", moddir);
  }else{
    snprintf(bin, _MAX_PATH, "\"%s\\npe\\npeestn.exe\"", moddir);
  }
  snprintf(libpath, _MAX_PATH, "%s\\..;%s\\ncs;%s\\npe", moddir, moddir);
#else
  snprintf(bin, _MAX_PATH, "%s/%s", get_tmpdir(), NPE_COMMAND);
  snprintf(libpath, _MAX_PATH, "%s:%s/ncs:%s/npe", 
           SL4_LIB_DIR, SL4_MODULE_DIR, SL4_MODULE_DIR);
#endif

  store_syscom_file();
  pid = create_child_process(bin, CONSOLE(ifd), CONSOLE(ofd), CONSOLE(efd),
                             libpath);
  errcode = 0;
  if(pid == -1){
    errcode = 258; /* could not create process */
  }else{
    int status;
    status = wait_child_process(pid, &errcode);
    switch(status){
    case -1: errcode = 238; break; /* NPE cmd process not found */
    case  0:  /* normal exit */
      errcode = _npe_library_errorcode(errcode); break;
    case  1: errcode = 239; break; /* signal caught in NPE cmd */
    case  2: errcode = 240; break; /* NPE cmd process is stoping */
    case  3: errcode = 241; break; /* NPE cmd was core dumped */
    default: errcode = 242; break; /* unknown error in NPE cmd */
    }
  }
  unlink_syscom_file();
  return errcode; /*succeed*/
}

static int _npe_common_memory_store(npe_condition_t *ncond){
  char *common_file;
  int   errcode;
  /* get common file path */
  common_file = npe_tmpfile_path(get_tmpdir(), NPECOMMONAREA);
  if(common_file == NULL)
    return 2; /* out of memory */
  errcode = npe_common_store(npe_common, common_file);
  free(common_file);
  return errcode;
}

static int _npe_library_errorcode(int err){
  int ret = 0;
  switch(err){
  case 0:   ret = 0;   break; /* success */
  case 101: ret = 2;   break; /* out of memory */
  case 102: ret = 243; break; /* fatal error */
  case 103: ret = 244; break; /* too many restart */
  case 104: ret = 245; break; /* too many data */
  case 105: ret = 246; break; /* can't open data file */
  case 106: ret = 247; break; /* illegal data point of data file */
  case 107: ret = 248; break; /* illegal block size of data file header */
  case 108: ret = 249; break; /* can't open weight file */
  case 109: ret = 250; break; /* illegal data point of weight file */
  case 110: ret = 251; break; /* illegal block size of weight file header */
  case 111: ret = 252; break; /* illegal logical expression of term */
  case 112: ret = 253; break; /* can't get parameter of NCS model */
  case 113: ret = 254; break; /* can't read NCS simulation condition file */
  case 114: ret = 255; break; /* can't load model library */
  case 115: ret = 256; break; /* can't load penalty library */
  default:  ret = 257; break; /* unknown error in NCS model */
  }
  return ret;
}

#ifdef __cplusplus
}
#endif

